/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.List;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.checkpoint.CompletedCheckpoint;
import org.apache.flink.runtime.checkpoint.CompletedCheckpointStore;
import org.apache.flink.runtime.state.StateHandle;
import org.apache.flink.runtime.zookeeper.StateStorageHelper;
import org.apache.flink.runtime.zookeeper.ZooKeeperStateHandleStore;
import org.apache.flink.shaded.com.google.common.base.Preconditions;
import org.apache.flink.shaded.org.apache.curator.framework.CuratorFramework;
import org.apache.flink.shaded.org.apache.curator.framework.api.BackgroundCallback;
import org.apache.flink.shaded.org.apache.curator.framework.api.CuratorEvent;
import org.apache.flink.shaded.org.apache.curator.framework.api.CuratorEventType;
import org.apache.flink.shaded.org.apache.curator.utils.ZKPaths;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZooKeeperCompletedCheckpointStore
implements CompletedCheckpointStore {
    private static final Logger LOG = LoggerFactory.getLogger(ZooKeeperCompletedCheckpointStore.class);
    private final CuratorFramework client;
    private final ZooKeeperStateHandleStore<CompletedCheckpoint> checkpointsInZooKeeper;
    private final int maxNumberOfCheckpointsToRetain;
    private final ClassLoader userClassLoader;
    private final ArrayDeque<Tuple2<StateHandle<CompletedCheckpoint>, String>> checkpointStateHandles;

    public ZooKeeperCompletedCheckpointStore(int maxNumberOfCheckpointsToRetain, ClassLoader userClassLoader, CuratorFramework client, String checkpointsPath, StateStorageHelper<CompletedCheckpoint> stateStorage) throws Exception {
        Preconditions.checkArgument(maxNumberOfCheckpointsToRetain >= 1, "Must retain at least one checkpoint.");
        Preconditions.checkNotNull(stateStorage, "State storage");
        this.maxNumberOfCheckpointsToRetain = maxNumberOfCheckpointsToRetain;
        this.userClassLoader = Preconditions.checkNotNull(userClassLoader, "User class loader");
        Preconditions.checkNotNull(client, "Curator client");
        Preconditions.checkNotNull(checkpointsPath, "Checkpoints path");
        client.newNamespaceAwareEnsurePath(checkpointsPath).ensure(client.getZookeeperClient());
        this.client = client.usingNamespace(client.getNamespace() + checkpointsPath);
        this.checkpointsInZooKeeper = new ZooKeeperStateHandleStore(this.client, stateStorage);
        this.checkpointStateHandles = new ArrayDeque(maxNumberOfCheckpointsToRetain + 1);
        LOG.info("Initialized in '{}'.", (Object)checkpointsPath);
    }

    @Override
    public void recover() throws Exception {
        List<Tuple2<StateHandle<CompletedCheckpoint>, String>> initialCheckpoints;
        LOG.info("Recovering checkpoints from ZooKeeper.");
        while (true) {
            try {
                initialCheckpoints = this.checkpointsInZooKeeper.getAllSortedByName();
            }
            catch (ConcurrentModificationException e) {
                LOG.warn("Concurrent modification while reading from ZooKeeper. Retrying.");
                continue;
            }
            break;
        }
        int numberOfInitialCheckpoints = initialCheckpoints.size();
        LOG.info("Found {} checkpoints in ZooKeeper.", (Object)numberOfInitialCheckpoints);
        if (numberOfInitialCheckpoints > 0) {
            Tuple2<StateHandle<CompletedCheckpoint>, String> latest = initialCheckpoints.get(numberOfInitialCheckpoints - 1);
            CompletedCheckpoint latestCheckpoint = (CompletedCheckpoint)((StateHandle)latest.f0).getState(this.userClassLoader);
            this.checkpointStateHandles.add(latest);
            LOG.info("Initialized with {}. Removing all older checkpoints.", (Object)latestCheckpoint);
            for (int i = 0; i < numberOfInitialCheckpoints - 1; ++i) {
                try {
                    this.removeFromZooKeeperAndDiscardCheckpoint(initialCheckpoints.get(i));
                    continue;
                }
                catch (Exception e) {
                    LOG.error("Failed to discard checkpoint", e);
                }
            }
        }
    }

    @Override
    public void addCheckpoint(CompletedCheckpoint checkpoint) throws Exception {
        Preconditions.checkNotNull(checkpoint, "Checkpoint");
        String path = String.format("/%s", checkpoint.getCheckpointID());
        StateHandle<CompletedCheckpoint> stateHandle = this.checkpointsInZooKeeper.add(path, checkpoint);
        this.checkpointStateHandles.addLast((Tuple2<StateHandle<CompletedCheckpoint>, String>)new Tuple2(stateHandle, (Object)path));
        if (this.checkpointStateHandles.size() > this.maxNumberOfCheckpointsToRetain) {
            this.removeFromZooKeeperAndDiscardCheckpoint(this.checkpointStateHandles.removeFirst());
        }
        LOG.debug("Added {} to {}.", (Object)checkpoint, (Object)path);
    }

    @Override
    public CompletedCheckpoint getLatestCheckpoint() throws Exception {
        if (this.checkpointStateHandles.isEmpty()) {
            return null;
        }
        return (CompletedCheckpoint)((StateHandle)this.checkpointStateHandles.getLast().f0).getState(this.userClassLoader);
    }

    @Override
    public List<CompletedCheckpoint> getAllCheckpoints() throws Exception {
        ArrayList<CompletedCheckpoint> checkpoints = new ArrayList<CompletedCheckpoint>(this.checkpointStateHandles.size());
        for (Tuple2<StateHandle<CompletedCheckpoint>, String> stateHandle : this.checkpointStateHandles) {
            checkpoints.add((CompletedCheckpoint)((StateHandle)stateHandle.f0).getState(this.userClassLoader));
        }
        return checkpoints;
    }

    @Override
    public int getNumberOfRetainedCheckpoints() {
        return this.checkpointStateHandles.size();
    }

    @Override
    public void discardAllCheckpoints() throws Exception {
        for (Tuple2<StateHandle<CompletedCheckpoint>, String> checkpoint : this.checkpointStateHandles) {
            try {
                this.removeFromZooKeeperAndDiscardCheckpoint(checkpoint);
            }
            catch (Exception e) {
                LOG.error("Failed to discard checkpoint.", e);
            }
        }
        this.checkpointStateHandles.clear();
        String path = "/" + this.client.getNamespace();
        LOG.info("Removing {} from ZooKeeper", (Object)path);
        ZKPaths.deleteChildren(this.client.getZookeeperClient().getZooKeeper(), path, true);
    }

    private void removeFromZooKeeperAndDiscardCheckpoint(final Tuple2<StateHandle<CompletedCheckpoint>, String> stateHandleAndPath) throws Exception {
        BackgroundCallback callback = new BackgroundCallback(){

            @Override
            public void processResult(CuratorFramework client, CuratorEvent event) throws Exception {
                try {
                    if (event.getType() == CuratorEventType.DELETE) {
                        if (event.getResultCode() != 0) {
                            throw new IllegalStateException("Unexpected result code " + event.getResultCode() + " in '" + event + "' callback.");
                        }
                    } else {
                        throw new IllegalStateException("Unexpected event type " + (Object)((Object)event.getType()) + " in '" + event + "' callback.");
                    }
                    CompletedCheckpoint checkpoint = (CompletedCheckpoint)((StateHandle)stateHandleAndPath.f0).getState(ZooKeeperCompletedCheckpointStore.this.userClassLoader);
                    checkpoint.discard(ZooKeeperCompletedCheckpointStore.this.userClassLoader);
                    ((StateHandle)stateHandleAndPath.f0).discardState();
                    LOG.debug("Discarded " + checkpoint);
                }
                catch (Exception e) {
                    LOG.error("Failed to discard checkpoint.", e);
                }
            }
        };
        this.checkpointsInZooKeeper.remove((String)stateHandleAndPath.f1, callback);
    }
}

