/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.client.program;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.ProgramDescription;
import org.apache.flink.client.ClientUtils;
import org.apache.flink.client.program.PackagedProgramUtils;
import org.apache.flink.client.program.ProgramInvocationException;
import org.apache.flink.client.program.ProgramParametrizationException;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.core.security.FlinkSecurityManager;
import org.apache.flink.runtime.jobgraph.SavepointRestoreSettings;
import org.apache.flink.util.InstantiationUtil;
import org.apache.flink.util.JarUtils;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PackagedProgram
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(PackagedProgram.class);
    public static final String MANIFEST_ATTRIBUTE_ASSEMBLER_CLASS = "program-class";
    public static final String MANIFEST_ATTRIBUTE_MAIN_CLASS = "Main-Class";
    private final URL jarFile;
    private final String[] args;
    private final Class<?> mainClass;
    private final List<File> extractedTempLibraries;
    private final List<URL> classpaths;
    private final URLClassLoader userCodeClassLoader;
    private final SavepointRestoreSettings savepointSettings;
    private final boolean isPython;

    private PackagedProgram(@Nullable File jarFile, List<URL> classpaths, @Nullable String entryPointClassName, Configuration configuration, SavepointRestoreSettings savepointRestoreSettings, String ... args) throws ProgramInvocationException {
        this.classpaths = (List)Preconditions.checkNotNull(classpaths);
        this.savepointSettings = (SavepointRestoreSettings)Preconditions.checkNotNull((Object)savepointRestoreSettings);
        this.args = (String[])Preconditions.checkNotNull((Object)args);
        Preconditions.checkArgument((jarFile != null || entryPointClassName != null ? 1 : 0) != 0, (Object)"Either the jarFile or the entryPointClassName needs to be non-null.");
        this.isPython = PackagedProgramUtils.isPython(entryPointClassName);
        this.jarFile = PackagedProgram.loadJarFile(jarFile);
        assert (this.jarFile != null || entryPointClassName != null);
        this.extractedTempLibraries = this.jarFile == null ? Collections.emptyList() : PackagedProgram.extractContainedLibraries(this.jarFile);
        this.userCodeClassLoader = ClientUtils.buildUserCodeClassLoader(this.getJobJarAndDependencies(), classpaths, this.getClass().getClassLoader(), configuration);
        this.mainClass = PackagedProgram.loadMainClass(entryPointClassName != null ? entryPointClassName : PackagedProgram.getEntryPointClassNameFromJar(this.jarFile), this.userCodeClassLoader);
        if (!PackagedProgram.hasMainMethod(this.mainClass)) {
            throw new ProgramInvocationException("The given program class does not have a main(String[]) method.");
        }
    }

    public SavepointRestoreSettings getSavepointSettings() {
        return this.savepointSettings;
    }

    public String[] getArguments() {
        return this.args;
    }

    public String getMainClassName() {
        return this.mainClass.getName();
    }

    @Nullable
    public String getDescription() throws ProgramInvocationException {
        if (ProgramDescription.class.isAssignableFrom(this.mainClass)) {
            ProgramDescription descr;
            try {
                descr = (ProgramDescription)InstantiationUtil.instantiate(this.mainClass.asSubclass(ProgramDescription.class), ProgramDescription.class);
            }
            catch (Throwable t) {
                return null;
            }
            try {
                return descr.getDescription();
            }
            catch (Throwable t) {
                throw new ProgramInvocationException("Error while getting the program description" + (String)(t.getMessage() == null ? "." : ": " + t.getMessage()), t);
            }
        }
        return null;
    }

    public void invokeInteractiveModeForExecution() throws ProgramInvocationException {
        FlinkSecurityManager.monitorUserSystemExitForCurrentThread();
        try {
            PackagedProgram.callMainMethod(this.mainClass, this.args);
        }
        finally {
            FlinkSecurityManager.unmonitorUserSystemExitForCurrentThread();
        }
    }

    public List<URL> getClasspaths() {
        return this.classpaths;
    }

    public ClassLoader getUserCodeClassLoader() {
        return this.userCodeClassLoader;
    }

    public List<URL> getJobJarAndDependencies() {
        ArrayList<URL> libs = new ArrayList<URL>(this.extractedTempLibraries.size() + 1);
        if (this.jarFile != null) {
            libs.add(this.jarFile);
        }
        for (File tmpLib : this.extractedTempLibraries) {
            try {
                libs.add(tmpLib.getAbsoluteFile().toURI().toURL());
            }
            catch (MalformedURLException e) {
                throw new RuntimeException("URL is invalid. This should not happen.", e);
            }
        }
        if (this.isPython) {
            libs.add(PackagedProgramUtils.getPythonJar());
        }
        return libs;
    }

    public static List<URL> getJobJarAndDependencies(File jarFile, @Nullable String entryPointClassName) throws ProgramInvocationException {
        URL jarFileUrl = PackagedProgram.loadJarFile(jarFile);
        List<Object> extractedTempLibraries = jarFileUrl == null ? Collections.emptyList() : PackagedProgram.extractContainedLibraries(jarFileUrl);
        ArrayList<URL> libs = new ArrayList<URL>(extractedTempLibraries.size() + 1);
        if (jarFileUrl != null) {
            libs.add(jarFileUrl);
        }
        for (File file : extractedTempLibraries) {
            try {
                libs.add(file.getAbsoluteFile().toURI().toURL());
            }
            catch (MalformedURLException e) {
                throw new RuntimeException("URL is invalid. This should not happen.", e);
            }
        }
        if (PackagedProgramUtils.isPython(entryPointClassName).booleanValue()) {
            libs.add(PackagedProgramUtils.getPythonJar());
        }
        return libs;
    }

    private void deleteExtractedLibraries() {
        PackagedProgram.deleteExtractedLibraries(this.extractedTempLibraries);
        this.extractedTempLibraries.clear();
    }

    private static boolean hasMainMethod(Class<?> entryClass) {
        Method mainMethod;
        try {
            mainMethod = entryClass.getMethod("main", String[].class);
        }
        catch (NoSuchMethodException e) {
            return false;
        }
        catch (Throwable t) {
            throw new RuntimeException("Could not look up the main(String[]) method from the class " + entryClass.getName() + ": " + t.getMessage(), t);
        }
        return Modifier.isStatic(mainMethod.getModifiers()) && Modifier.isPublic(mainMethod.getModifiers());
    }

    private static void callMainMethod(Class<?> entryClass, String[] args) throws ProgramInvocationException {
        Method mainMethod;
        if (!Modifier.isPublic(entryClass.getModifiers())) {
            throw new ProgramInvocationException("The class " + entryClass.getName() + " must be public.");
        }
        try {
            mainMethod = entryClass.getMethod("main", String[].class);
        }
        catch (NoSuchMethodException e) {
            throw new ProgramInvocationException("The class " + entryClass.getName() + " has no main(String[]) method.");
        }
        catch (Throwable t) {
            throw new ProgramInvocationException("Could not look up the main(String[]) method from the class " + entryClass.getName() + ": " + t.getMessage(), t);
        }
        if (!Modifier.isStatic(mainMethod.getModifiers())) {
            throw new ProgramInvocationException("The class " + entryClass.getName() + " declares a non-static main method.");
        }
        if (!Modifier.isPublic(mainMethod.getModifiers())) {
            throw new ProgramInvocationException("The class " + entryClass.getName() + " declares a non-public main method.");
        }
        try {
            mainMethod.invoke(null, new Object[]{args});
        }
        catch (IllegalArgumentException e) {
            throw new ProgramInvocationException("Could not invoke the main method, arguments are not matching.", e);
        }
        catch (IllegalAccessException e) {
            throw new ProgramInvocationException("Access to the main method was denied: " + e.getMessage(), e);
        }
        catch (InvocationTargetException e) {
            Throwable exceptionInMethod = e.getTargetException();
            if (exceptionInMethod instanceof Error) {
                throw (Error)exceptionInMethod;
            }
            if (exceptionInMethod instanceof ProgramParametrizationException) {
                throw (ProgramParametrizationException)exceptionInMethod;
            }
            if (exceptionInMethod instanceof ProgramInvocationException) {
                throw (ProgramInvocationException)exceptionInMethod;
            }
            throw new ProgramInvocationException("The main method caused an error: " + exceptionInMethod.getMessage(), exceptionInMethod);
        }
        catch (Throwable t) {
            throw new ProgramInvocationException("An error occurred while invoking the program's main method: " + t.getMessage(), t);
        }
    }

    private static String getEntryPointClassNameFromJar(URL jarFile) throws ProgramInvocationException {
        JarFile jar;
        try {
            jar = new JarFile(new File(jarFile.toURI()));
        }
        catch (URISyntaxException use) {
            throw new ProgramInvocationException("Invalid file path '" + jarFile.getPath() + "'", use);
        }
        catch (IOException ioex) {
            throw new ProgramInvocationException("Error while opening jar file '" + jarFile.getPath() + "'. " + ioex.getMessage(), ioex);
        }
        try {
            Manifest manifest;
            try {
                manifest = jar.getManifest();
            }
            catch (IOException ioex) {
                throw new ProgramInvocationException("The Manifest in the jar file could not be accessed '" + jarFile.getPath() + "'. " + ioex.getMessage(), ioex);
            }
            if (manifest == null) {
                throw new ProgramInvocationException("No manifest found in jar file '" + jarFile.getPath() + "'. The manifest is need to point to the program's main class.");
            }
            Attributes attributes = manifest.getMainAttributes();
            String className = attributes.getValue(MANIFEST_ATTRIBUTE_ASSEMBLER_CLASS);
            if (className != null) {
                String string = className;
                return string;
            }
            className = attributes.getValue(MANIFEST_ATTRIBUTE_MAIN_CLASS);
            if (className != null) {
                String string = className;
                return string;
            }
            throw new ProgramInvocationException("Neither a 'Main-Class', nor a 'program-class' entry was found in the jar file.");
        }
        finally {
            try {
                jar.close();
            }
            catch (Throwable t) {
                throw new ProgramInvocationException("Could not close the JAR file: " + t.getMessage(), t);
            }
        }
    }

    @Nullable
    private static URL loadJarFile(File jar) throws ProgramInvocationException {
        if (jar != null) {
            URL jarFileUrl;
            try {
                jarFileUrl = jar.getAbsoluteFile().toURI().toURL();
            }
            catch (MalformedURLException e1) {
                throw new IllegalArgumentException("The jar file path is invalid.");
            }
            PackagedProgram.checkJarFile(jarFileUrl);
            return jarFileUrl;
        }
        return null;
    }

    private static Class<?> loadMainClass(String className, ClassLoader cl) throws ProgramInvocationException {
        ClassLoader contextCl = null;
        try {
            contextCl = Thread.currentThread().getContextClassLoader();
            Thread.currentThread().setContextClassLoader(cl);
            Class<?> clazz = Class.forName(className, false, cl);
            return clazz;
        }
        catch (ClassNotFoundException e) {
            throw new ProgramInvocationException("The program's entry point class '" + className + "' was not found in the jar file.", e);
        }
        catch (ExceptionInInitializerError e) {
            throw new ProgramInvocationException("The program's entry point class '" + className + "' threw an error during initialization.", e);
        }
        catch (LinkageError e) {
            throw new ProgramInvocationException("The program's entry point class '" + className + "' could not be loaded due to a linkage failure.", e);
        }
        catch (Throwable t) {
            throw new ProgramInvocationException("The program's entry point class '" + className + "' caused an exception during initialization: " + t.getMessage(), t);
        }
        finally {
            if (contextCl != null) {
                Thread.currentThread().setContextClassLoader(contextCl);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static List<File> extractContainedLibraries(URL jarFile) throws ProgramInvocationException {
        try (JarFile jar = new JarFile(new File(jarFile.toURI()));){
            List<JarEntry> containedJarFileEntries = PackagedProgram.getContainedJarEntries(jar);
            if (containedJarFileEntries.isEmpty()) {
                List<File> list = Collections.emptyList();
                return list;
            }
            ArrayList<File> extractedTempLibraries = new ArrayList<File>(containedJarFileEntries.size());
            boolean incomplete = true;
            try {
                Random rnd = new Random();
                byte[] buffer = new byte[4096];
                for (JarEntry entry : containedJarFileEntries) {
                    String name = entry.getName().replace('/', '_');
                    File tempFile = PackagedProgram.copyLibToTempFile(name, rnd, jar, entry, buffer);
                    extractedTempLibraries.add(tempFile);
                }
                incomplete = false;
            }
            finally {
                if (incomplete) {
                    PackagedProgram.deleteExtractedLibraries(extractedTempLibraries);
                }
            }
            ArrayList<File> arrayList = extractedTempLibraries;
            return arrayList;
        }
        catch (Throwable t) {
            throw new ProgramInvocationException("Unknown I/O error while extracting contained jar files.", t);
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static File copyLibToTempFile(String name, Random rnd, JarFile jar, JarEntry input, byte[] buffer) throws ProgramInvocationException {
        File output = PackagedProgram.createTempFile(rnd, input, name);
        try (FileOutputStream out = new FileOutputStream(output);){
            File file;
            try (BufferedInputStream in = new BufferedInputStream(jar.getInputStream(input));){
                int numRead = 0;
                while ((numRead = ((InputStream)in).read(buffer)) != -1) {
                    ((OutputStream)out).write(buffer, 0, numRead);
                }
                file = output;
            }
            return file;
        }
        catch (IOException e) {
            throw new ProgramInvocationException("An I/O error occurred while extracting nested library '" + input.getName() + "' to temporary file '" + output.getAbsolutePath() + "'.");
        }
    }

    private static File createTempFile(Random rnd, JarEntry entry, String name) throws ProgramInvocationException {
        try {
            File tempFile = File.createTempFile(rnd.nextInt(Integer.MAX_VALUE) + "_", name);
            tempFile.deleteOnExit();
            return tempFile;
        }
        catch (IOException e) {
            throw new ProgramInvocationException("An I/O error occurred while creating temporary file to extract nested library '" + entry.getName() + "'.", e);
        }
    }

    private static List<JarEntry> getContainedJarEntries(JarFile jar) {
        return jar.stream().filter(jarEntry -> {
            String name = jarEntry.getName();
            return name.length() > 8 && name.startsWith("lib/") && name.endsWith(".jar");
        }).collect(Collectors.toList());
    }

    private static void deleteExtractedLibraries(List<File> tempLibraries) {
        for (File f : tempLibraries) {
            f.delete();
        }
    }

    private static void checkJarFile(URL jarfile) throws ProgramInvocationException {
        try {
            JarUtils.checkJarFile((URL)jarfile);
        }
        catch (IOException e) {
            throw new ProgramInvocationException(e.getMessage(), e);
        }
        catch (Throwable t) {
            throw new ProgramInvocationException("Cannot access jar file" + (String)(t.getMessage() == null ? "." : ": " + t.getMessage()), t);
        }
    }

    @Override
    public void close() {
        try {
            this.userCodeClassLoader.close();
        }
        catch (IOException e) {
            LOG.debug("Error while closing user-code classloader.", (Throwable)e);
        }
        try {
            this.deleteExtractedLibraries();
        }
        catch (Exception e) {
            LOG.debug("Error while deleting jars extracted from user-jar.", (Throwable)e);
        }
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        @Nullable
        private File jarFile;
        @Nullable
        private String entryPointClassName;
        private String[] args = new String[0];
        private List<URL> userClassPaths = Collections.emptyList();
        private Configuration configuration = new Configuration();
        private SavepointRestoreSettings savepointRestoreSettings = SavepointRestoreSettings.none();

        public Builder setJarFile(@Nullable File jarFile) {
            this.jarFile = jarFile;
            return this;
        }

        public Builder setUserClassPaths(List<URL> userClassPaths) {
            this.userClassPaths = userClassPaths;
            return this;
        }

        public Builder setEntryPointClassName(@Nullable String entryPointClassName) {
            this.entryPointClassName = entryPointClassName;
            return this;
        }

        public Builder setArguments(String ... args) {
            this.args = args;
            return this;
        }

        public Builder setConfiguration(Configuration configuration) {
            this.configuration = configuration;
            return this;
        }

        public Builder setSavepointRestoreSettings(SavepointRestoreSettings savepointRestoreSettings) {
            this.savepointRestoreSettings = savepointRestoreSettings;
            return this;
        }

        @VisibleForTesting
        public List<URL> getUserClassPaths() {
            return this.userClassPaths;
        }

        public PackagedProgram build() throws ProgramInvocationException {
            if (this.jarFile == null && this.entryPointClassName == null) {
                throw new IllegalArgumentException("The jarFile and entryPointClassName can not be null at the same time.");
            }
            return new PackagedProgram(this.jarFile, this.userClassPaths, this.entryPointClassName, this.configuration, this.savepointRestoreSettings, this.args);
        }

        private Builder() {
        }
    }
}

