/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.cdc.common.utils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;
import org.apache.flink.cdc.common.utils.ThreadLocalCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DateTimeUtils {
    private static final Logger LOG = LoggerFactory.getLogger(DateTimeUtils.class);
    public static final int EPOCH_JULIAN = 2440588;
    public static final long MILLIS_PER_DAY = 86400000L;
    private static final String DATE_FORMAT_STRING = "yyyy-MM-dd";
    private static final String TIME_FORMAT_STRING = "HH:mm:ss";
    private static final String TIMESTAMP_FORMAT_STRING = "yyyy-MM-dd HH:mm:ss";
    private static final ThreadLocalCache<String, SimpleDateFormat> FORMATTER_CACHE = ThreadLocalCache.of(SimpleDateFormat::new);

    public static int timestampMillisToDate(long ts) {
        int days = (int)(ts / 86400000L);
        if (days < 0) {
            --days;
        }
        return days;
    }

    public static int timestampMillisToTime(long ts) {
        return (int)(ts % 86400000L);
    }

    public static int parseDate(String dateStr, String fromFormat) {
        long ts = DateTimeUtils.internalParseTimestampMillis(dateStr, fromFormat, TimeZone.getTimeZone("UTC"));
        ZoneId zoneId = ZoneId.of("UTC");
        Instant instant = Instant.ofEpochMilli(ts);
        ZonedDateTime zdt = ZonedDateTime.ofInstant(instant, zoneId);
        return DateTimeUtils.ymdToUnixDate(zdt.getYear(), zdt.getMonthValue(), zdt.getDayOfMonth());
    }

    public static int parseDate(String dateStr, String fromFormat, String timezone) {
        long ts = DateTimeUtils.internalParseTimestampMillis(dateStr, fromFormat, TimeZone.getTimeZone(timezone));
        ZoneId zoneId = ZoneId.of(timezone);
        Instant instant = Instant.ofEpochMilli(ts);
        ZonedDateTime zdt = ZonedDateTime.ofInstant(instant, zoneId);
        return DateTimeUtils.ymdToUnixDate(zdt.getYear(), zdt.getMonthValue(), zdt.getDayOfMonth());
    }

    private static long internalParseTimestampMillis(String dateStr, String format, TimeZone tz) {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(tz);
        try {
            Date date = formatter.parse(dateStr);
            return date.getTime();
        }
        catch (ParseException e) {
            LOG.error(String.format("Exception when parsing datetime string '%s' in format '%s', the default value Long.MIN_VALUE(-9223372036854775808) will be returned.", dateStr, format), (Throwable)e);
            return Long.MIN_VALUE;
        }
    }

    private static int ymdToUnixDate(int year, int month, int day) {
        int julian = DateTimeUtils.ymdToJulian(year, month, day);
        return julian - 2440588;
    }

    private static int ymdToJulian(int year, int month, int day) {
        int a = (14 - month) / 12;
        int y = year + 4800 - a;
        int m = month + 12 * a - 3;
        return day + (153 * m + 2) / 5 + 365 * y + y / 4 - y / 100 + y / 400 - 32045;
    }

    public static String formatUnixTimestamp(long unixTime, TimeZone timeZone) {
        return DateTimeUtils.formatUnixTimestamp(unixTime, TIMESTAMP_FORMAT_STRING, timeZone);
    }

    public static String formatUnixTimestamp(long unixTime, String format, TimeZone timeZone) {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(timeZone);
        Date date = new Date(unixTime * 1000L);
        try {
            return formatter.format(date);
        }
        catch (Exception e) {
            LOG.error("Exception when formatting.", (Throwable)e);
            return null;
        }
    }

    public static long unixTimestamp(String dateStr, TimeZone timeZone) {
        return DateTimeUtils.unixTimestamp(dateStr, TIMESTAMP_FORMAT_STRING, timeZone);
    }

    public static long unixTimestamp(String dateStr, String format, TimeZone timeZone) {
        long ts = DateTimeUtils.internalParseTimestampMillis(dateStr, format, timeZone);
        if (ts == Long.MIN_VALUE) {
            return Long.MIN_VALUE;
        }
        return ts / 1000L;
    }

    public static String formatTimestampMillis(long ts, String format, TimeZone timeZone) {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(timeZone);
        Date dateTime = new Date(ts);
        return formatter.format(dateTime);
    }

    public static Integer timestampDiff(String timeIntervalUnit, long fromDate, String fromTimezone, long toDate, String toTimezone) {
        Calendar from = Calendar.getInstance(TimeZone.getTimeZone(fromTimezone));
        from.setTime(new Date(fromDate));
        Calendar to = Calendar.getInstance(TimeZone.getTimeZone(toTimezone));
        to.setTime(new Date(toDate));
        long second = (to.getTimeInMillis() - from.getTimeInMillis()) / 1000L;
        switch (timeIntervalUnit) {
            case "SECOND": {
                if (second > Integer.MAX_VALUE) {
                    return null;
                }
                return (int)second;
            }
            case "MINUTE": {
                if (second > Integer.MAX_VALUE) {
                    return null;
                }
                return (int)second / 60;
            }
            case "HOUR": {
                if (second > Integer.MAX_VALUE) {
                    return null;
                }
                return (int)second / 3600;
            }
            case "DAY": {
                if (second > Integer.MAX_VALUE) {
                    return null;
                }
                return (int)second / 86400;
            }
            case "MONTH": {
                return to.get(1) * 12 + to.get(2) - (from.get(1) * 12 + from.get(2));
            }
            case "YEAR": {
                return to.get(1) - from.get(1);
            }
        }
        throw new RuntimeException(String.format("Unsupported timestamp interval unit %s. Supported units are: SECOND, MINUTE, HOUR, DAY, MONTH, YEAR", timeIntervalUnit));
    }

    public static long timestampAdd(String timeIntervalUnit, int interval, long timePoint, String timezone) {
        int field;
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeZone(TimeZone.getTimeZone(timezone));
        calendar.setTime(new Date(timePoint));
        switch (timeIntervalUnit) {
            case "SECOND": {
                field = 13;
                break;
            }
            case "MINUTE": {
                field = 12;
                break;
            }
            case "HOUR": {
                field = 10;
                break;
            }
            case "DAY": {
                field = 5;
                break;
            }
            case "MONTH": {
                field = 2;
                break;
            }
            case "YEAR": {
                field = 1;
                break;
            }
            default: {
                throw new RuntimeException(String.format("Unsupported timestamp interval unit %s. Supported units are: SECOND, MINUTE, HOUR, DAY, MONTH, YEAR", timeIntervalUnit));
            }
        }
        calendar.add(field, interval);
        return calendar.getTimeInMillis();
    }
}

