/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.java.util.http.client.pool;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.io.Closer;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.java.util.http.client.pool.ResourceContainer;
import org.apache.druid.java.util.http.client.pool.ResourceFactory;
import org.apache.druid.java.util.http.client.pool.ResourcePoolConfig;

public class ResourcePool<K, V>
implements Closeable {
    private static final Logger log = new Logger(ResourcePool.class);
    private final LoadingCache<K, ResourceHolderPerKey<K, V>> pool;
    private final AtomicBoolean closed = new AtomicBoolean(false);

    public ResourcePool(final ResourceFactory<K, V> factory, final ResourcePoolConfig config, final boolean eagerInitialization) {
        this.pool = CacheBuilder.newBuilder().build(new CacheLoader<K, ResourceHolderPerKey<K, V>>(){

            public ResourceHolderPerKey<K, V> load(K input) {
                if (eagerInitialization) {
                    return new EagerCreationResourceHolder(config.getMaxPerKey(), config.getUnusedConnectionTimeoutMillis(), input, factory);
                }
                return new LazyCreationResourceHolder(config.getMaxPerKey(), config.getUnusedConnectionTimeoutMillis(), input, factory);
            }
        });
    }

    @Nullable
    public ResourceContainer<V> take(final K key) {
        ResourceHolderPerKey holder;
        if (this.closed.get()) {
            log.error(StringUtils.format("take(%s) called even though I'm closed.", key), new Object[0]);
            return null;
        }
        try {
            holder = (ResourceHolderPerKey)this.pool.get(key);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
        final Object value = holder.get();
        return new ResourceContainer<V>(){
            private final AtomicBoolean returned = new AtomicBoolean(false);

            @Override
            public V get() {
                Preconditions.checkState((!this.returned.get() ? 1 : 0) != 0, (String)"Resource for key[%s] has been returned, cannot get().", (Object)key);
                return value;
            }

            @Override
            public void returnResource() {
                if (this.returned.getAndSet(true)) {
                    log.warn("Resource at key[%s] was returned multiple times?", key);
                } else {
                    holder.giveBack(value);
                }
            }

            protected void finalize() throws Throwable {
                if (!this.returned.get()) {
                    log.warn(StringUtils.format("Resource[%s] at key[%s] was not returned before Container was finalized, potential resource leak.", value, key), new Object[0]);
                    this.returnResource();
                }
                super.finalize();
            }
        };
    }

    @Override
    public void close() {
        this.closed.set(true);
        ConcurrentMap mapView = this.pool.asMap();
        Closer closer = Closer.create();
        Iterator iterator = mapView.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry e = iterator.next();
            iterator.remove();
            closer.register((ResourceHolderPerKey)e.getValue());
        }
        try {
            closer.close();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static class ResourceHolder<V> {
        private final long lastAccessedTime;
        private final V resource;

        private ResourceHolder(long lastAccessedTime, V resource) {
            this.resource = resource;
            this.lastAccessedTime = lastAccessedTime;
        }

        private long getLastAccessedTime() {
            return this.lastAccessedTime;
        }

        public V getResource() {
            return this.resource;
        }
    }

    private static class ResourceHolderPerKey<K, V>
    implements Closeable {
        protected final int maxSize;
        private final K key;
        private final ResourceFactory<K, V> factory;
        private final long unusedResourceTimeoutMillis;
        protected final ArrayDeque<ResourceHolder<V>> resourceHolderList;
        private int numLentResources = 0;
        private boolean closed = false;

        protected ResourceHolderPerKey(int maxSize, long unusedResourceTimeoutMillis, K key, ResourceFactory<K, V> factory) {
            this.maxSize = maxSize;
            this.key = key;
            this.factory = factory;
            this.unusedResourceTimeoutMillis = unusedResourceTimeoutMillis;
            this.resourceHolderList = new ArrayDeque();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Nullable
        V get() {
            V retVal;
            V poolVal;
            boolean expired = false;
            ResourceHolderPerKey resourceHolderPerKey = this;
            synchronized (resourceHolderPerKey) {
                while (!this.closed && this.numLentResources == this.maxSize) {
                    try {
                        this.wait();
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        return null;
                    }
                }
                if (this.closed) {
                    log.info(StringUtils.format("get() called even though I'm closed. key[%s]", this.key), new Object[0]);
                    return null;
                }
                if (this.numLentResources < this.maxSize) {
                    if (this.resourceHolderList.isEmpty()) {
                        poolVal = this.factory.generate(this.key);
                    } else {
                        ResourceHolder<V> holder = this.resourceHolderList.removeFirst();
                        poolVal = holder.getResource();
                        if (System.currentTimeMillis() - holder.getLastAccessedTime() > this.unusedResourceTimeoutMillis) {
                            expired = true;
                        }
                    }
                    ++this.numLentResources;
                } else {
                    throw new IllegalStateException("Unexpected state: More objects lent than permissible");
                }
            }
            try {
                if (poolVal != null && !expired && this.factory.isGood(poolVal)) {
                    retVal = poolVal;
                } else {
                    if (poolVal != null) {
                        this.factory.close(poolVal);
                    }
                    retVal = this.factory.generate(this.key);
                }
            }
            catch (Throwable e) {
                ResourceHolderPerKey resourceHolderPerKey2 = this;
                synchronized (resourceHolderPerKey2) {
                    --this.numLentResources;
                    this.notifyAll();
                }
                Throwables.propagateIfPossible((Throwable)e);
                throw new RuntimeException(e);
            }
            return retVal;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void giveBack(V object) {
            Preconditions.checkNotNull(object, (Object)"object");
            ResourceHolderPerKey resourceHolderPerKey = this;
            synchronized (resourceHolderPerKey) {
                if (this.closed) {
                    log.info(StringUtils.format("giveBack called after being closed. key[%s]", this.key), new Object[0]);
                    this.factory.close(object);
                    return;
                }
                if (this.resourceHolderList.size() >= this.maxSize) {
                    if (this.holderListContains(object)) {
                        log.warn(new Exception("Exception for stacktrace"), StringUtils.format("Returning object[%s] at key[%s] that has already been returned!? Skipping", object, this.key), new Object[0]);
                    } else {
                        log.warn(new Exception("Exception for stacktrace"), StringUtils.format("Returning object[%s] at key[%s] even though we already have all that we can hold[%s]!? Skipping", object, this.key, this.resourceHolderList), new Object[0]);
                    }
                    return;
                }
                this.resourceHolderList.addLast(new ResourceHolder<V>(System.currentTimeMillis(), object));
                --this.numLentResources;
                this.notifyAll();
            }
        }

        private boolean holderListContains(V object) {
            return this.resourceHolderList.stream().anyMatch(a -> a.getResource().equals(object));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() {
            ResourceHolderPerKey resourceHolderPerKey = this;
            synchronized (resourceHolderPerKey) {
                this.closed = true;
                this.resourceHolderList.forEach((Consumer<ResourceHolder<V>>)((Consumer<ResourceHolder>)v -> this.factory.close(v.getResource())));
                this.resourceHolderList.clear();
                this.notifyAll();
            }
        }
    }

    private static class LazyCreationResourceHolder<K, V>
    extends ResourceHolderPerKey<K, V> {
        private LazyCreationResourceHolder(int maxSize, long unusedResourceTimeoutMillis, K key, ResourceFactory<K, V> factory) {
            super(maxSize, unusedResourceTimeoutMillis, key, factory);
        }
    }

    private static class EagerCreationResourceHolder<K, V>
    extends LazyCreationResourceHolder<K, V> {
        private EagerCreationResourceHolder(int maxSize, long unusedResourceTimeoutMillis, K key, ResourceFactory<K, V> factory) {
            super(maxSize, unusedResourceTimeoutMillis, key, factory);
            for (int i = 0; i < maxSize; ++i) {
                this.resourceHolderList.add(new ResourceHolder<Object>(System.currentTimeMillis(), Preconditions.checkNotNull(factory.generate(key), (Object)"factory.generate(key)")));
            }
        }
    }
}

