/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.compress;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ParameterizedClass;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.IVersionedSerializer;
import org.apache.cassandra.io.compress.DeflateCompressor;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.compress.LZ4Compressor;
import org.apache.cassandra.io.compress.SnappyCompressor;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompressionParameters {
    private static final Logger LOGGER = LoggerFactory.getLogger(CompressionParameters.class);
    private static volatile boolean hasLoggedSsTableCompressionWarning;
    private static volatile boolean hasLoggedChunkLengthWarning;
    public static final int DEFAULT_CHUNK_LENGTH = 65536;
    public static final double DEFAULT_CRC_CHECK_CHANCE = 1.0;
    public static final IVersionedSerializer<CompressionParameters> serializer;
    public static final String CLASS = "class";
    public static final String CHUNK_LENGTH_IN_KB = "chunk_length_in_kb";
    public static final String ENABLED = "enabled";
    @Deprecated
    public static final String SSTABLE_COMPRESSION = "sstable_compression";
    @Deprecated
    public static final String CHUNK_LENGTH_KB = "chunk_length_kb";
    public static final String CRC_CHECK_CHANCE = "crc_check_chance";
    public static final Set<String> GLOBAL_OPTIONS;
    private final ICompressor sstableCompressor;
    private final Integer chunkLength;
    private volatile double crcCheckChance;
    private final ImmutableMap<String, String> otherOptions;
    private CFMetaData liveMetadata;

    public static CompressionParameters fromMap(Map<? extends CharSequence, ? extends CharSequence> opts) {
        String sstableCompressionClass;
        Map<String, String> options = CompressionParameters.copyOptions(opts);
        if (!CompressionParameters.removeEnabled(options)) {
            sstableCompressionClass = null;
            if (!options.isEmpty()) {
                throw new ConfigurationException("If the 'enabled' option is set to false no other options must be specified");
            }
        } else {
            sstableCompressionClass = CompressionParameters.removeSstableCompressionClass(options);
        }
        Integer chunkLength = CompressionParameters.removeChunkLength(options);
        CompressionParameters cp = new CompressionParameters(sstableCompressionClass, chunkLength, options);
        cp.validate();
        return cp;
    }

    public static CompressionParameters noCompression() {
        return new CompressionParameters((ICompressor)null, (Integer)65536, Collections.emptyMap());
    }

    public static CompressionParameters snappy() {
        return CompressionParameters.snappy(null);
    }

    public static CompressionParameters snappy(Integer chunkLength) {
        return new CompressionParameters(SnappyCompressor.instance, chunkLength, Collections.emptyMap());
    }

    public static CompressionParameters deflate() {
        return CompressionParameters.deflate(null);
    }

    public static CompressionParameters deflate(Integer chunkLength) {
        return new CompressionParameters(DeflateCompressor.instance, chunkLength, Collections.emptyMap());
    }

    public static CompressionParameters lz4() {
        return CompressionParameters.lz4(null);
    }

    public static CompressionParameters lz4(Integer chunkLength) {
        return new CompressionParameters(LZ4Compressor.instance, chunkLength, Collections.emptyMap());
    }

    CompressionParameters(String sstableCompressorClass, Integer chunkLength, Map<String, String> otherOptions) throws ConfigurationException {
        this(CompressionParameters.createCompressor(CompressionParameters.parseCompressorClass(sstableCompressorClass), otherOptions), chunkLength, otherOptions);
    }

    private CompressionParameters(ICompressor sstableCompressor, Integer chunkLength, Map<String, String> otherOptions) throws ConfigurationException {
        this.sstableCompressor = sstableCompressor;
        this.chunkLength = chunkLength;
        this.otherOptions = ImmutableMap.copyOf(otherOptions);
        String chance = otherOptions.get(CRC_CHECK_CHANCE);
        this.crcCheckChance = chance == null ? 1.0 : CompressionParameters.parseCrcCheckChance(chance);
    }

    public CompressionParameters copy() {
        return new CompressionParameters(this.sstableCompressor, this.chunkLength, (Map<String, String>)this.otherOptions);
    }

    public void setLiveMetadata(CFMetaData liveMetadata) {
        if (liveMetadata == null) {
            return;
        }
        this.liveMetadata = liveMetadata;
    }

    public void setCrcCheckChance(double crcCheckChance) throws ConfigurationException {
        CompressionParameters.validateCrcCheckChance(crcCheckChance);
        this.crcCheckChance = crcCheckChance;
        if (this.liveMetadata != null && this != this.liveMetadata.compressionParameters) {
            this.liveMetadata.compressionParameters.setCrcCheckChance(crcCheckChance);
        }
    }

    public boolean isEnabled() {
        return this.sstableCompressor != null;
    }

    public ICompressor getSstableCompressor() {
        return this.sstableCompressor;
    }

    public ImmutableMap<String, String> getOtherOptions() {
        return this.otherOptions;
    }

    public double getCrcCheckChance() {
        return this.liveMetadata == null ? this.crcCheckChance : this.liveMetadata.compressionParameters.crcCheckChance;
    }

    private static double parseCrcCheckChance(String crcCheckChance) throws ConfigurationException {
        try {
            double chance = Double.parseDouble(crcCheckChance);
            CompressionParameters.validateCrcCheckChance(chance);
            return chance;
        }
        catch (NumberFormatException e) {
            throw new ConfigurationException("crc_check_chance should be a double");
        }
    }

    private static void validateCrcCheckChance(double crcCheckChance) throws ConfigurationException {
        if (crcCheckChance < 0.0 || crcCheckChance > 1.0) {
            throw new ConfigurationException("crc_check_chance should be between 0.0 and 1.0");
        }
    }

    public int chunkLength() {
        return this.chunkLength == null ? 65536 : this.chunkLength;
    }

    private static Class<?> parseCompressorClass(String className) throws ConfigurationException {
        if (className == null || className.isEmpty()) {
            return null;
        }
        className = className.contains(".") ? className : "org.apache.cassandra.io.compress." + className;
        try {
            return Class.forName(className);
        }
        catch (Exception e) {
            throw new ConfigurationException("Could not create Compression for type " + className, e);
        }
    }

    private static ICompressor createCompressor(Class<?> compressorClass, Map<String, String> compressionOptions) throws ConfigurationException {
        if (compressorClass == null) {
            if (!compressionOptions.isEmpty()) {
                throw new ConfigurationException("Unknown compression options (" + compressionOptions.keySet() + ") since no compression class found");
            }
            return null;
        }
        try {
            Method method = compressorClass.getMethod("create", Map.class);
            ICompressor compressor = (ICompressor)method.invoke(null, compressionOptions);
            Sets.SetView supportedOpts = Sets.union(compressor.supportedOptions(), GLOBAL_OPTIONS);
            for (String provided : compressionOptions.keySet()) {
                if (supportedOpts.contains(provided)) continue;
                throw new ConfigurationException("Unknown compression options " + provided);
            }
            return compressor;
        }
        catch (NoSuchMethodException e) {
            throw new ConfigurationException("create method not found", e);
        }
        catch (SecurityException e) {
            throw new ConfigurationException("Access forbiden", e);
        }
        catch (IllegalAccessException e) {
            throw new ConfigurationException("Cannot access method create in " + compressorClass.getName(), e);
        }
        catch (InvocationTargetException e) {
            Throwable cause = e.getCause();
            throw new ConfigurationException(String.format("%s.create() threw an error: %s", compressorClass.getSimpleName(), cause == null ? e.getClass().getName() + " " + e.getMessage() : cause.getClass().getName() + " " + cause.getMessage()), e);
        }
        catch (ExceptionInInitializerError e) {
            throw new ConfigurationException("Cannot initialize class " + compressorClass.getName());
        }
    }

    public static ICompressor createCompressor(ParameterizedClass compression) throws ConfigurationException {
        return CompressionParameters.createCompressor(CompressionParameters.parseCompressorClass(compression.class_name), CompressionParameters.copyOptions(compression.parameters));
    }

    private static Map<String, String> copyOptions(Map<? extends CharSequence, ? extends CharSequence> co) {
        if (co == null || co.isEmpty()) {
            return Collections.emptyMap();
        }
        HashMap<String, String> compressionOptions = new HashMap<String, String>();
        for (Map.Entry<? extends CharSequence, ? extends CharSequence> entry : co.entrySet()) {
            compressionOptions.put(entry.getKey().toString(), entry.getValue().toString());
        }
        return compressionOptions;
    }

    private static Integer parseChunkLength(String chLengthKB) throws ConfigurationException {
        if (chLengthKB == null) {
            return null;
        }
        try {
            int parsed = Integer.parseInt(chLengthKB);
            if (parsed > 0x1FFFFF) {
                throw new ConfigurationException("Value of chunk_length_in_kb is too large (" + parsed + ")");
            }
            return 1024 * parsed;
        }
        catch (NumberFormatException e) {
            throw new ConfigurationException("Invalid value for chunk_length_in_kb", e);
        }
    }

    private static Integer removeChunkLength(Map<String, String> options) {
        if (options.containsKey(CHUNK_LENGTH_IN_KB)) {
            if (options.containsKey(CHUNK_LENGTH_KB)) {
                throw new ConfigurationException(String.format("The '%s' option must not be used if the chunk length is already specified by the '%s' option", CHUNK_LENGTH_KB, CHUNK_LENGTH_IN_KB));
            }
            return CompressionParameters.parseChunkLength(options.remove(CHUNK_LENGTH_IN_KB));
        }
        if (options.containsKey(CHUNK_LENGTH_KB)) {
            if (options.containsKey(CHUNK_LENGTH_KB) && !hasLoggedChunkLengthWarning) {
                hasLoggedChunkLengthWarning = true;
                LOGGER.warn(String.format("The %s option has been deprecated. You should use %s instead", CHUNK_LENGTH_KB, CHUNK_LENGTH_IN_KB));
            }
            return CompressionParameters.parseChunkLength(options.remove(CHUNK_LENGTH_KB));
        }
        return null;
    }

    public static boolean containsSstableCompressionClass(Map<String, String> options) {
        return options.containsKey(CLASS) || options.containsKey(SSTABLE_COMPRESSION);
    }

    private static String removeSstableCompressionClass(Map<String, String> options) {
        if (options.containsKey(CLASS)) {
            if (options.containsKey(SSTABLE_COMPRESSION)) {
                throw new ConfigurationException(String.format("The '%s' option must not be used if the compression algorithm is already specified by the '%s' option", SSTABLE_COMPRESSION, CLASS));
            }
            String clazz = options.remove(CLASS);
            if (clazz.isEmpty()) {
                throw new ConfigurationException(String.format("The '%s' option must not be empty. To disable compression use 'enabled' : false", CLASS));
            }
            return clazz;
        }
        if (options.containsKey(SSTABLE_COMPRESSION) && !hasLoggedSsTableCompressionWarning) {
            hasLoggedSsTableCompressionWarning = true;
            LOGGER.warn(String.format("The %s option has been deprecated. You should use %s instead", SSTABLE_COMPRESSION, CLASS));
        }
        return options.remove(SSTABLE_COMPRESSION);
    }

    public static boolean isEnabled(Map<String, String> options) {
        String enabled = options.get(ENABLED);
        return enabled == null || Boolean.parseBoolean(enabled);
    }

    private static boolean removeEnabled(Map<String, String> options) {
        String enabled = options.remove(ENABLED);
        return enabled == null || Boolean.parseBoolean(enabled);
    }

    public void validate() throws ConfigurationException {
        if (this.chunkLength != null) {
            if (this.chunkLength <= 0) {
                throw new ConfigurationException("Invalid negative or null chunk_length_in_kb");
            }
            boolean found = false;
            for (int c = this.chunkLength.intValue(); c != 0; c >>= 1) {
                if ((c & 1) == 0) continue;
                if (found) {
                    throw new ConfigurationException("chunk_length_in_kb must be a power of 2");
                }
                found = true;
            }
        }
        CompressionParameters.validateCrcCheckChance(this.crcCheckChance);
    }

    public Map<String, String> asMap() {
        if (!this.isEnabled()) {
            return Collections.singletonMap(ENABLED, "false");
        }
        HashMap<String, String> options = new HashMap<String, String>((Map<String, String>)this.otherOptions);
        options.put(CLASS, this.sstableCompressor.getClass().getName());
        options.put(CHUNK_LENGTH_IN_KB, this.chunkLengthInKB());
        return options;
    }

    public String chunkLengthInKB() {
        return String.valueOf(this.chunkLength() / 1024);
    }

    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj == null || obj.getClass() != this.getClass()) {
            return false;
        }
        CompressionParameters cp = (CompressionParameters)obj;
        return new EqualsBuilder().append((Object)this.sstableCompressor, (Object)cp.sstableCompressor).append(this.chunkLength(), cp.chunkLength()).append(this.otherOptions, cp.otherOptions).isEquals();
    }

    public int hashCode() {
        return new HashCodeBuilder(29, 1597).append((Object)this.sstableCompressor).append(this.chunkLength()).append(this.otherOptions).toHashCode();
    }

    static {
        serializer = new Serializer();
        GLOBAL_OPTIONS = ImmutableSet.of((Object)CRC_CHECK_CHANCE);
    }

    static class Serializer
    implements IVersionedSerializer<CompressionParameters> {
        Serializer() {
        }

        @Override
        public void serialize(CompressionParameters parameters, DataOutputPlus out, int version) throws IOException {
            out.writeUTF(parameters.sstableCompressor.getClass().getSimpleName());
            out.writeInt(parameters.otherOptions.size());
            for (Map.Entry entry : parameters.otherOptions.entrySet()) {
                out.writeUTF((String)entry.getKey());
                out.writeUTF((String)entry.getValue());
            }
            out.writeInt(parameters.chunkLength());
        }

        @Override
        public CompressionParameters deserialize(DataInputPlus in, int version) throws IOException {
            CompressionParameters parameters;
            String compressorName = in.readUTF();
            int optionCount = in.readInt();
            HashMap<String, String> options = new HashMap<String, String>();
            for (int i = 0; i < optionCount; ++i) {
                String key = in.readUTF();
                String value = in.readUTF();
                options.put(key, value);
            }
            int chunkLength = in.readInt();
            try {
                parameters = new CompressionParameters(compressorName, (Integer)chunkLength, options);
            }
            catch (ConfigurationException e) {
                throw new RuntimeException("Cannot create CompressionParameters for parameters", e);
            }
            return parameters;
        }

        @Override
        public long serializedSize(CompressionParameters parameters, int version) {
            long size = TypeSizes.sizeof(parameters.sstableCompressor.getClass().getSimpleName());
            size += (long)TypeSizes.sizeof(parameters.otherOptions.size());
            for (Map.Entry entry : parameters.otherOptions.entrySet()) {
                size += (long)TypeSizes.sizeof((String)entry.getKey());
                size += (long)TypeSizes.sizeof((String)entry.getValue());
            }
            return size += (long)TypeSizes.sizeof(parameters.chunkLength());
        }
    }
}

