/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.collect.ImmutableSortedSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.common.primitives.Ints;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.dht.Bounds;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.util.FileUtils;
import org.codehaus.jackson.JsonEncoding;
import org.codehaus.jackson.JsonFactory;
import org.codehaus.jackson.JsonGenerator;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeveledManifest {
    private static final Logger logger = LoggerFactory.getLogger(LeveledManifest.class);
    public static final String EXTENSION = ".json";
    static final int MAX_COMPACTING_L0 = 32;
    private final ColumnFamilyStore cfs;
    private final List<SSTableReader>[] generations;
    private final Map<SSTableReader, Integer> sstableGenerations;
    private final RowPosition[] lastCompactedKeys;
    private final int maxSSTableSizeInBytes;

    private LeveledManifest(ColumnFamilyStore cfs, int maxSSTableSizeInMB) {
        this.cfs = cfs;
        this.maxSSTableSizeInBytes = maxSSTableSizeInMB * 1024 * 1024;
        int n = (int)Math.log10(1000000000 / maxSSTableSizeInMB);
        this.generations = new List[n];
        this.lastCompactedKeys = new RowPosition[n];
        for (int i = 0; i < this.generations.length; ++i) {
            this.generations[i] = new ArrayList<SSTableReader>();
            this.lastCompactedKeys[i] = ((Token)cfs.partitioner.getMinimumToken()).minKeyBound();
        }
        this.sstableGenerations = new HashMap<SSTableReader, Integer>();
    }

    static LeveledManifest create(ColumnFamilyStore cfs, int maxSSTableSize) {
        return LeveledManifest.create(cfs, maxSSTableSize, cfs.getSSTables());
    }

    public static LeveledManifest create(ColumnFamilyStore cfs, int maxSSTableSize, Iterable<SSTableReader> sstables) {
        LeveledManifest manifest = new LeveledManifest(cfs, maxSSTableSize);
        LeveledManifest.load(cfs, manifest, sstables);
        for (SSTableReader ssTableReader : sstables) {
            if (manifest.levelOf(ssTableReader) >= 0) continue;
            manifest.add(ssTableReader);
        }
        return manifest;
    }

    private static void load(ColumnFamilyStore cfs, LeveledManifest manifest, Iterable<SSTableReader> sstables) {
        File manifestFile = LeveledManifest.tryGetManifest(cfs);
        if (manifestFile == null) {
            return;
        }
        ObjectMapper m = new ObjectMapper();
        try {
            JsonNode rootNode = (JsonNode)m.readValue(manifestFile, JsonNode.class);
            JsonNode generations = rootNode.get("generations");
            assert (generations.isArray());
            for (JsonNode generation : generations) {
                int level = generation.get("generation").getIntValue();
                JsonNode generationValues = generation.get("members");
                for (JsonNode generationValue : generationValues) {
                    for (SSTableReader ssTableReader : sstables) {
                        if (ssTableReader.descriptor.generation != generationValue.getIntValue()) continue;
                        logger.debug("Loading {} at L{}", (Object)ssTableReader, (Object)level);
                        manifest.add(ssTableReader, level);
                    }
                }
            }
        }
        catch (Exception e) {
            logger.error("Manifest present but corrupt. Cassandra will compact levels from scratch", (Throwable)e);
        }
    }

    public synchronized void add(SSTableReader reader) {
        this.logDistribution();
        logger.debug("Adding {} to L0", (Object)reader);
        this.add(reader, 0);
        this.serialize();
    }

    private int skipLevels(int newLevel, Iterable<SSTableReader> added) {
        while (this.maxBytesForLevel(newLevel) < SSTableReader.getTotalBytes(added) && this.generations[newLevel + 1].isEmpty()) {
            ++newLevel;
        }
        return newLevel;
    }

    public synchronized void promote(Iterable<SSTableReader> removed, Iterable<SSTableReader> added) {
        int newLevel;
        assert (!Iterables.isEmpty(removed));
        this.logDistribution();
        if (logger.isDebugEnabled()) {
            logger.debug("Replacing [" + this.toString(removed) + "]");
        }
        int minimumLevel = Integer.MAX_VALUE;
        int maximumLevel = 0;
        for (SSTableReader sstable : removed) {
            int thisLevel = this.remove(sstable);
            assert (thisLevel >= 0);
            maximumLevel = Math.max(maximumLevel, thisLevel);
            minimumLevel = Math.min(minimumLevel, thisLevel);
        }
        if (!added.iterator().hasNext()) {
            return;
        }
        if (minimumLevel == 0 && maximumLevel == 0 && SSTable.getTotalBytes(removed) <= (long)this.maxSSTableSizeInBytes) {
            newLevel = 0;
        } else {
            newLevel = minimumLevel == maximumLevel ? maximumLevel + 1 : maximumLevel;
            newLevel = this.skipLevels(newLevel, added);
            assert (newLevel > 0);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Adding [{}] at L{}", (Object)this.toString(added), (Object)newLevel);
        }
        this.lastCompactedKeys[minimumLevel] = ((SSTableReader)SSTable.sstableOrdering.max(added)).last;
        for (SSTableReader ssTableReader : added) {
            this.add(ssTableReader, newLevel);
        }
        if (newLevel != 0) {
            this.repairOverlappingSSTables(newLevel);
        }
        this.serialize();
    }

    public synchronized void repairOverlappingSSTables(int level) {
        SSTableReader previous = null;
        Collections.sort(this.generations[level], SSTable.sstableComparator);
        ArrayList<SSTableReader> outOfOrderSSTables = new ArrayList<SSTableReader>();
        for (SSTableReader current : this.generations[level]) {
            if (previous != null && current.first.compareTo(previous.last) <= 0) {
                logger.error(String.format("At level %d, %s [%s, %s] overlaps %s [%s, %s].  This is caused by a bug in Cassandra 1.1.0 .. 1.1.3.  Sending back to L0.  If you have not yet run scrub, you should do so since you may also have rows out-of-order within an sstable", level, previous, previous.first, previous.last, current, current.first, current.last));
                outOfOrderSSTables.add(current);
                continue;
            }
            previous = current;
        }
        if (!outOfOrderSSTables.isEmpty()) {
            for (SSTableReader sstable : outOfOrderSSTables) {
                this.sendBackToL0(sstable);
            }
            this.serialize();
        }
    }

    public synchronized void replace(Iterable<SSTableReader> removed, Iterable<SSTableReader> added) {
        assert (Iterables.size(removed) == 1) : Iterables.size(removed);
        assert (Iterables.size(added) <= 1) : Iterables.size(added);
        this.logDistribution();
        logger.debug("Replacing {} with {}", removed, added);
        int level = this.remove(removed.iterator().next());
        if (!Iterables.isEmpty(added)) {
            this.add(added.iterator().next(), level);
        }
        this.serialize();
    }

    private synchronized void sendBackToL0(SSTableReader sstable) {
        this.remove(sstable);
        this.add(sstable, 0);
    }

    private String toString(Iterable<SSTableReader> sstables) {
        StringBuilder builder = new StringBuilder();
        for (SSTableReader sstable : sstables) {
            builder.append(sstable.descriptor.cfname).append('-').append(sstable.descriptor.generation).append("(L").append(this.levelOf(sstable)).append("), ");
        }
        return builder.toString();
    }

    private long maxBytesForLevel(int level) {
        if (level == 0) {
            return 4L * (long)this.maxSSTableSizeInBytes;
        }
        double bytes = Math.pow(10.0, level) * (double)this.maxSSTableSizeInBytes;
        if (bytes > 9.223372036854776E18) {
            throw new RuntimeException("At most 9223372036854775807 bytes may be in a compaction level; your maxSSTableSize must be absurdly high to compute " + bytes);
        }
        return (long)bytes;
    }

    public synchronized Collection<SSTableReader> getCompactionCandidates() {
        for (int i = this.generations.length - 1; i >= 0; --i) {
            List<SSTableReader> sstables = this.generations[i];
            if (sstables.isEmpty()) continue;
            double score = (double)SSTableReader.getTotalBytes(sstables) / (double)this.maxBytesForLevel(i);
            logger.debug("Compaction score for level {} is {}", (Object)i, (Object)score);
            if (!(score > 1.001) && (i != 0 || sstables.size() <= 1)) continue;
            Collection<SSTableReader> candidates = this.getCandidatesFor(i);
            if (logger.isDebugEnabled()) {
                logger.debug("Compaction candidates for L{} are {}", (Object)i, (Object)this.toString(candidates));
            }
            if (candidates.isEmpty()) continue;
            return candidates;
        }
        return Collections.emptyList();
    }

    public int getLevelSize(int i) {
        return this.generations.length > i ? this.generations[i].size() : 0;
    }

    public synchronized int[] getAllLevelSize() {
        int[] counts = new int[this.generations.length];
        for (int i = 0; i < counts.length; ++i) {
            counts[i] = this.generations[i].size();
        }
        return counts;
    }

    private void logDistribution() {
        if (logger.isDebugEnabled()) {
            for (int i = 0; i < this.generations.length; ++i) {
                if (this.generations[i].isEmpty()) continue;
                logger.debug("L{} contains {} SSTables ({} bytes) in {}", new Object[]{i, this.generations[i].size(), SSTableReader.getTotalBytes(this.generations[i]), this});
            }
        }
    }

    int levelOf(SSTableReader sstable) {
        Integer level = this.sstableGenerations.get(sstable);
        if (level == null) {
            return -1;
        }
        return level;
    }

    private int remove(SSTableReader reader) {
        int level = this.levelOf(reader);
        assert (level >= 0) : reader + " not present in manifest";
        this.generations[level].remove(reader);
        this.sstableGenerations.remove(reader);
        return level;
    }

    private void add(SSTableReader sstable, int level) {
        assert (level < this.generations.length) : "Invalid level " + level + " out of " + (this.generations.length - 1);
        this.generations[level].add(sstable);
        this.sstableGenerations.put(sstable, level);
    }

    private static Set<SSTableReader> overlapping(Collection<SSTableReader> candidates, Iterable<SSTableReader> others) {
        assert (!candidates.isEmpty());
        Iterator<SSTableReader> iter = candidates.iterator();
        SSTableReader sstable = iter.next();
        Token first = sstable.first.token;
        Token last = sstable.last.token;
        while (iter.hasNext()) {
            sstable = iter.next();
            first = first.compareTo(sstable.first.token) <= 0 ? first : sstable.first.token;
            last = last.compareTo(sstable.last.token) >= 0 ? last : sstable.last.token;
        }
        return LeveledManifest.overlapping(first, last, others);
    }

    private static Set<SSTableReader> overlapping(SSTableReader sstable, Iterable<SSTableReader> others) {
        return LeveledManifest.overlapping(sstable.first.token, sstable.last.token, others);
    }

    private static Set<SSTableReader> overlapping(Token start, Token end, Iterable<SSTableReader> sstables) {
        assert (start.compareTo(end) <= 0);
        HashSet<SSTableReader> overlapped = new HashSet<SSTableReader>();
        Bounds<Token> promotedBounds = new Bounds<Token>(start, end);
        for (SSTableReader candidate : sstables) {
            Bounds<Token> candidateBounds = new Bounds<Token>(candidate.first.token, candidate.last.token);
            if (!candidateBounds.intersects(promotedBounds)) continue;
            overlapped.add(candidate);
        }
        return overlapped;
    }

    /*
     * Enabled aggressive block sorting
     */
    private Collection<SSTableReader> getCandidatesFor(int level) {
        SSTableReader sstable;
        int i;
        int start;
        assert (!this.generations[level].isEmpty());
        logger.debug("Choosing candidates for L{}", (Object)level);
        if (level != 0) {
            Collections.sort(this.generations[level], SSTable.sstableComparator);
            start = 0;
        } else {
            HashSet<SSTableReader> candidates;
            block10: {
                candidates = new HashSet<SSTableReader>();
                HashSet<SSTableReader> remaining = new HashSet<SSTableReader>(this.generations[0]);
                ArrayList<SSTableReader> ageSortedSSTables = new ArrayList<SSTableReader>(this.generations[0]);
                Collections.sort(ageSortedSSTables, SSTable.maxTimestampComparator);
                Iterator i$ = ageSortedSSTables.iterator();
                while (i$.hasNext()) {
                    SSTableReader sstable2 = (SSTableReader)i$.next();
                    if (candidates.contains(sstable2)) continue;
                    for (SSTableReader newCandidate : Sets.union(Collections.singleton(sstable2), LeveledManifest.overlapping(sstable2, remaining))) {
                        if (newCandidate.isMarkedSuspect()) continue;
                        candidates.add(newCandidate);
                        remaining.remove(newCandidate);
                    }
                    if (candidates.size() > 32) {
                        ArrayList<SSTableReader> ageSortedCandidates = new ArrayList<SSTableReader>(candidates);
                        Collections.sort(ageSortedCandidates, SSTable.maxTimestampComparator);
                        candidates = new HashSet(ageSortedCandidates.subList(0, 32));
                        if (SSTable.getTotalBytes(candidates) <= (long)this.maxSSTableSizeInBytes) return candidates;
                        candidates.addAll(LeveledManifest.overlapping(candidates, this.generations[1]));
                        return candidates;
                    }
                    if (SSTable.getTotalBytes(candidates) <= (long)this.maxSSTableSizeInBytes) {
                        continue;
                    }
                    break block10;
                }
                return candidates;
            }
            candidates.addAll(LeveledManifest.overlapping(candidates, this.generations[1]));
            return candidates;
        }
        for (i = 0; i < this.generations[level].size(); ++i) {
            sstable = this.generations[level].get(i);
            if (sstable.first.compareTo(this.lastCompactedKeys[level]) <= 0) continue;
            start = i;
            break;
        }
        i = start;
        do {
            SSTableReader candidate;
            sstable = this.generations[level].get(i);
            Sets.SetView candidates = Sets.union(Collections.singleton(sstable), LeveledManifest.overlapping(sstable, this.generations[level + 1]));
            Iterator i$ = candidates.iterator();
            do {
                if (!i$.hasNext()) return candidates;
            } while (!(candidate = (SSTableReader)i$.next()).isMarkedSuspect());
        } while ((i = (i + 1) % this.generations[level].size()) != start);
        return Collections.emptyList();
    }

    public static File tryGetManifest(ColumnFamilyStore cfs) {
        return cfs.directories.tryGetLeveledManifest();
    }

    public synchronized void serialize() {
        File manifestFile = this.cfs.directories.getOrCreateLeveledManifest();
        File oldFile = new File(manifestFile.getPath().replace(EXTENSION, "-old.json"));
        File tmpFile = new File(manifestFile.getPath().replace(EXTENSION, "-tmp.json"));
        JsonFactory f = new JsonFactory();
        try {
            JsonGenerator g = f.createJsonGenerator(tmpFile, JsonEncoding.UTF8);
            g.useDefaultPrettyPrinter();
            g.writeStartObject();
            g.writeArrayFieldStart("generations");
            for (int level = 0; level < this.generations.length; ++level) {
                g.writeStartObject();
                g.writeNumberField("generation", level);
                g.writeArrayFieldStart("members");
                for (SSTableReader ssTableReader : this.generations[level]) {
                    g.writeNumber(ssTableReader.descriptor.generation);
                }
                g.writeEndArray();
                g.writeEndObject();
            }
            g.writeEndArray();
            g.writeEndObject();
            g.close();
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, tmpFile);
        }
        if (oldFile.exists() && manifestFile.exists()) {
            FileUtils.deleteWithConfirm(oldFile);
        }
        if (manifestFile.exists()) {
            FileUtils.renameWithConfirm(manifestFile, oldFile);
        }
        assert (tmpFile.exists());
        FileUtils.renameWithConfirm(tmpFile, manifestFile);
        logger.debug("Saved manifest {}", (Object)manifestFile);
    }

    public String toString() {
        return "Manifest@" + this.hashCode();
    }

    public int getLevelCount() {
        for (int i = this.generations.length - 1; i >= 0; --i) {
            if (this.generations[i].size() <= 0) continue;
            return i;
        }
        return 0;
    }

    public synchronized SortedSet<SSTableReader> getLevelSorted(int level, Comparator<SSTableReader> comparator) {
        return ImmutableSortedSet.copyOf(comparator, this.generations[level]);
    }

    public List<SSTableReader> getLevel(int i) {
        return this.generations[i];
    }

    public synchronized int getEstimatedTasks() {
        long tasks = 0L;
        long[] estimated = new long[this.generations.length];
        for (int i = this.generations.length - 1; i >= 0; --i) {
            List<SSTableReader> sstables = this.generations[i];
            estimated[i] = Math.max(0L, SSTableReader.getTotalBytes(sstables) - this.maxBytesForLevel(i)) / (long)this.maxSSTableSizeInBytes;
            tasks += estimated[i];
        }
        logger.debug("Estimating {} compactions to do for {}.{}", new Object[]{Arrays.asList(new long[][]{estimated}), this.cfs.table.name, this.cfs.columnFamily});
        return Ints.checkedCast((long)tasks);
    }
}

