/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.discover;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.common.concurrent.FutureUtils;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.discover.BookieServiceInfo;
import org.apache.bookkeeper.discover.RegistrationManager;
import org.apache.bookkeeper.discover.ZKRegistrationClient;
import org.apache.bookkeeper.meta.AbstractZkLedgerManagerFactory;
import org.apache.bookkeeper.meta.LayoutManager;
import org.apache.bookkeeper.meta.LedgerManagerFactory;
import org.apache.bookkeeper.meta.ZkLayoutManager;
import org.apache.bookkeeper.meta.ZkLedgerUnderreplicationManager;
import org.apache.bookkeeper.meta.zk.ZKMetadataDriverBase;
import org.apache.bookkeeper.net.BookieId;
import org.apache.bookkeeper.proto.DataFormats;
import org.apache.bookkeeper.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.bookkeeper.shaded.com.google.common.collect.Lists;
import org.apache.bookkeeper.util.BookKeeperConstants;
import org.apache.bookkeeper.util.ZkUtils;
import org.apache.bookkeeper.versioning.LongVersion;
import org.apache.bookkeeper.versioning.Version;
import org.apache.bookkeeper.versioning.Versioned;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Op;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZKUtil;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKRegistrationManager
implements RegistrationManager {
    private static final Logger log = LoggerFactory.getLogger(ZKRegistrationManager.class);
    private static final Function<Throwable, BKException> EXCEPTION_FUNC = cause -> {
        if (cause instanceof BKException) {
            log.error("Failed to get bookie list : ", cause);
            return (BKException)cause;
        }
        if (cause instanceof InterruptedException) {
            log.error("Interrupted reading bookie list : ", cause);
            return new BKException.BKInterruptedException();
        }
        return new BKException.MetaStoreException();
    };
    private final ServerConfiguration conf;
    private final ZooKeeper zk;
    private final List<ACL> zkAcls;
    private final LayoutManager layoutManager;
    private volatile boolean zkRegManagerInitialized = false;
    private final String ledgersRootPath;
    private final String cookiePath;
    protected final String bookieRegistrationPath;
    protected final String bookieReadonlyRegistrationPath;
    private final int zkTimeoutMs;
    private final List<RegistrationManager.RegistrationListener> listeners = new ArrayList<RegistrationManager.RegistrationListener>();

    public ZKRegistrationManager(ServerConfiguration conf, ZooKeeper zk) {
        this(conf, zk, ZKMetadataDriverBase.resolveZkLedgersRootPath(conf));
    }

    public ZKRegistrationManager(ServerConfiguration conf, ZooKeeper zk, String ledgersRootPath) {
        this.conf = conf;
        this.zk = zk;
        this.zkAcls = ZkUtils.getACLs(conf);
        this.ledgersRootPath = ledgersRootPath;
        this.cookiePath = ledgersRootPath + "/" + "cookies";
        this.bookieRegistrationPath = ledgersRootPath + "/" + "available";
        this.bookieReadonlyRegistrationPath = this.bookieRegistrationPath + "/" + "readonly";
        this.zkTimeoutMs = conf.getZkTimeout();
        this.layoutManager = new ZkLayoutManager(zk, ledgersRootPath, this.zkAcls);
        this.zk.register(event -> {
            if (!this.zkRegManagerInitialized) {
                return;
            }
            if (event.getType().equals((Object)Watcher.Event.EventType.None) && event.getState().equals((Object)Watcher.Event.KeeperState.Expired)) {
                this.listeners.forEach(RegistrationManager.RegistrationListener::onRegistrationExpired);
            }
        });
    }

    @Override
    public void close() {
    }

    public String getCookiePath(BookieId bookieId) {
        return this.cookiePath + "/" + bookieId;
    }

    protected boolean checkRegNodeAndWaitExpired(String regPath) throws IOException {
        final CountDownLatch prevNodeLatch = new CountDownLatch(1);
        Watcher zkPrevRegNodewatcher = new Watcher(){

            public void process(WatchedEvent event) {
                if (Watcher.Event.EventType.NodeDeleted == event.getType()) {
                    prevNodeLatch.countDown();
                }
            }
        };
        try {
            Stat stat = this.zk.exists(regPath, zkPrevRegNodewatcher);
            if (null != stat) {
                if (stat.getEphemeralOwner() != this.zk.getSessionId()) {
                    log.info("Previous bookie registration znode: {} exists, so waiting zk sessiontimeout: {} ms for znode deletion", (Object)regPath, (Object)this.zkTimeoutMs);
                    if (!prevNodeLatch.await(this.zkTimeoutMs, TimeUnit.MILLISECONDS)) {
                        throw new KeeperException.NodeExistsException(regPath);
                    }
                    return false;
                }
                return true;
            }
            return false;
        }
        catch (KeeperException ke) {
            log.error("ZK exception checking and wait ephemeral znode {} expired : ", (Object)regPath, (Object)ke);
            throw new IOException("ZK exception checking and wait ephemeral znode " + regPath + " expired", ke);
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            log.error("Interrupted checking and wait ephemeral znode {} expired : ", (Object)regPath, (Object)ie);
            throw new IOException("Interrupted checking and wait ephemeral znode " + regPath + " expired", ie);
        }
    }

    @Override
    public void registerBookie(BookieId bookieId, boolean readOnly, BookieServiceInfo bookieServiceInfo) throws BookieException {
        if (!readOnly) {
            String regPath = this.bookieRegistrationPath + "/" + bookieId;
            this.doRegisterBookie(regPath, bookieServiceInfo);
        } else {
            this.doRegisterReadOnlyBookie(bookieId, bookieServiceInfo);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @VisibleForTesting
    static byte[] serializeBookieServiceInfo(BookieServiceInfo bookieServiceInfo) {
        if (log.isDebugEnabled()) {
            log.debug("serialize BookieServiceInfo {}", (Object)bookieServiceInfo);
        }
        try (ByteArrayOutputStream os = new ByteArrayOutputStream();){
            DataFormats.BookieServiceInfoFormat.Builder builder = DataFormats.BookieServiceInfoFormat.newBuilder();
            List bsiEndpoints = bookieServiceInfo.getEndpoints().stream().map(e -> DataFormats.BookieServiceInfoFormat.Endpoint.newBuilder().setId(e.getId()).setPort(e.getPort()).setHost(e.getHost()).setProtocol(e.getProtocol()).addAllAuth(e.getAuth()).addAllExtensions(e.getExtensions()).build()).collect(Collectors.toList());
            builder.addAllEndpoints(bsiEndpoints);
            builder.putAllProperties(bookieServiceInfo.getProperties());
            builder.build().writeTo(os);
            byte[] byArray = os.toByteArray();
            return byArray;
        }
        catch (IOException err) {
            log.error("Cannot serialize bookieServiceInfo from " + bookieServiceInfo);
            throw new RuntimeException(err);
        }
    }

    private void doRegisterBookie(String regPath, BookieServiceInfo bookieServiceInfo) throws BookieException {
        try {
            if (!this.checkRegNodeAndWaitExpired(regPath)) {
                this.zk.create(regPath, ZKRegistrationManager.serializeBookieServiceInfo(bookieServiceInfo), this.zkAcls, CreateMode.EPHEMERAL);
                this.zkRegManagerInitialized = true;
            }
        }
        catch (KeeperException ke) {
            log.error("ZK exception registering ephemeral Znode for Bookie!", (Throwable)ke);
            throw new BookieException.MetadataStoreException(ke);
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            log.error("Interrupted exception registering ephemeral Znode for Bookie!", (Throwable)ie);
            throw new BookieException.MetadataStoreException(ie);
        }
        catch (IOException e) {
            throw new BookieException.MetadataStoreException(e);
        }
    }

    private void doRegisterReadOnlyBookie(BookieId bookieId, BookieServiceInfo bookieServiceInfo) throws BookieException {
        try {
            if (null == this.zk.exists(this.bookieReadonlyRegistrationPath, false)) {
                try {
                    this.zk.create(this.bookieReadonlyRegistrationPath, ZKRegistrationManager.serializeBookieServiceInfo(bookieServiceInfo), this.zkAcls, CreateMode.PERSISTENT);
                }
                catch (KeeperException.NodeExistsException nodeExistsException) {
                    // empty catch block
                }
            }
            String regPath = this.bookieReadonlyRegistrationPath + "/" + bookieId;
            this.doRegisterBookie(regPath, bookieServiceInfo);
            regPath = this.bookieRegistrationPath + "/" + bookieId;
            try {
                this.zk.delete(regPath, -1);
            }
            catch (KeeperException.NoNodeException nne) {
                log.warn("No writable bookie registered node {} when transitioning to readonly", (Object)regPath, (Object)nne);
            }
        }
        catch (InterruptedException | KeeperException e) {
            throw new BookieException.MetadataStoreException(e);
        }
    }

    @Override
    public void unregisterBookie(BookieId bookieId, boolean readOnly) throws BookieException {
        String regPath = !readOnly ? this.bookieRegistrationPath + "/" + bookieId : this.bookieReadonlyRegistrationPath + "/" + bookieId;
        this.doUnregisterBookie(regPath);
    }

    private void doUnregisterBookie(String regPath) throws BookieException {
        try {
            this.zk.delete(regPath, -1);
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            throw new BookieException.MetadataStoreException(ie);
        }
        catch (KeeperException e) {
            throw new BookieException.MetadataStoreException(e);
        }
    }

    @Override
    public void writeCookie(BookieId bookieId, Versioned<byte[]> cookieData) throws BookieException {
        block10: {
            String zkPath = this.getCookiePath(bookieId);
            try {
                if (Version.NEW == cookieData.getVersion()) {
                    if (this.zk.exists(this.cookiePath, false) == null) {
                        try {
                            this.zk.create(this.cookiePath, new byte[0], this.zkAcls, CreateMode.PERSISTENT);
                        }
                        catch (KeeperException.NodeExistsException nne) {
                            log.info("More than one bookie tried to create {} at once. Safe to ignore.", (Object)this.cookiePath);
                        }
                    }
                    this.zk.create(zkPath, cookieData.getValue(), this.zkAcls, CreateMode.PERSISTENT);
                    break block10;
                }
                if (!(cookieData.getVersion() instanceof LongVersion)) {
                    throw new BookieException.BookieIllegalOpException("Invalid version type, expected it to be LongVersion");
                }
                this.zk.setData(zkPath, cookieData.getValue(), (int)((LongVersion)cookieData.getVersion()).getLongVersion());
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                throw new BookieException.MetadataStoreException("Interrupted writing cookie for bookie " + bookieId, (Throwable)ie);
            }
            catch (KeeperException.NoNodeException nne) {
                throw new BookieException.CookieNotFoundException(bookieId.toString());
            }
            catch (KeeperException.NodeExistsException nee) {
                throw new BookieException.CookieExistException(bookieId.toString());
            }
            catch (KeeperException e) {
                throw new BookieException.MetadataStoreException("Failed to write cookie for bookie " + bookieId);
            }
        }
    }

    @Override
    public Versioned<byte[]> readCookie(BookieId bookieId) throws BookieException {
        String zkPath = this.getCookiePath(bookieId);
        try {
            Stat stat = this.zk.exists(zkPath, false);
            byte[] data = this.zk.getData(zkPath, false, stat);
            LongVersion version = new LongVersion(stat.getVersion());
            return new Versioned<byte[]>(data, version);
        }
        catch (KeeperException.NoNodeException nne) {
            throw new BookieException.CookieNotFoundException(bookieId.toString());
        }
        catch (InterruptedException | KeeperException e) {
            throw new BookieException.MetadataStoreException("Failed to read cookie for bookie " + bookieId);
        }
    }

    @Override
    public void removeCookie(BookieId bookieId, Version version) throws BookieException {
        String zkPath = this.getCookiePath(bookieId);
        try {
            this.zk.delete(zkPath, (int)((LongVersion)version).getLongVersion());
        }
        catch (KeeperException.NoNodeException e) {
            throw new BookieException.CookieNotFoundException(bookieId.toString());
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new BookieException.MetadataStoreException("Interrupted deleting cookie for bookie " + bookieId, (Throwable)e);
        }
        catch (KeeperException e) {
            throw new BookieException.MetadataStoreException("Failed to delete cookie for bookie " + bookieId);
        }
        log.info("Removed cookie from {} for bookie {}.", (Object)this.cookiePath, (Object)bookieId);
    }

    @Override
    public String getClusterInstanceId() throws BookieException {
        String instanceId = null;
        try {
            if (this.zk.exists(this.ledgersRootPath, null) == null) {
                log.error("BookKeeper metadata doesn't exist in zookeeper. Has the cluster been initialized? Try running bin/bookkeeper shell metaformat");
                throw new KeeperException.NoNodeException("BookKeeper metadata");
            }
            try {
                byte[] data = this.zk.getData(this.ledgersRootPath + "/" + "INSTANCEID", false, null);
                instanceId = new String(data, StandardCharsets.UTF_8);
            }
            catch (KeeperException.NoNodeException e) {
                log.info("INSTANCEID not exists in zookeeper. Not considering it for data verification");
            }
        }
        catch (InterruptedException | KeeperException e) {
            throw new BookieException.MetadataStoreException("Failed to get cluster instance id", e);
        }
        return instanceId;
    }

    @Override
    public boolean prepareFormat() throws Exception {
        boolean availableNodeExists;
        boolean ledgerRootExists = null != this.zk.exists(this.ledgersRootPath, false);
        boolean bl = availableNodeExists = null != this.zk.exists(this.bookieRegistrationPath, false);
        if (!ledgerRootExists) {
            ZkUtils.createFullPathOptimistic(this.zk, this.ledgersRootPath, "".getBytes(StandardCharsets.UTF_8), this.zkAcls, CreateMode.PERSISTENT);
        }
        if (!availableNodeExists) {
            this.zk.create(this.bookieRegistrationPath, "".getBytes(StandardCharsets.UTF_8), this.zkAcls, CreateMode.PERSISTENT);
        }
        if (null == this.zk.exists(this.bookieReadonlyRegistrationPath, false)) {
            this.zk.create(this.bookieReadonlyRegistrationPath, new byte[0], this.zkAcls, CreateMode.PERSISTENT);
        }
        return ledgerRootExists;
    }

    @Override
    public boolean initNewCluster() throws Exception {
        boolean ledgerRootExists;
        String zkServers = ZKMetadataDriverBase.resolveZkServers(this.conf);
        String instanceIdPath = this.ledgersRootPath + "/" + "INSTANCEID";
        log.info("Initializing ZooKeeper metadata for new cluster, ZKServers: {} ledger root path: {}", (Object)zkServers, (Object)this.ledgersRootPath);
        boolean bl = ledgerRootExists = null != this.zk.exists(this.ledgersRootPath, false);
        if (ledgerRootExists) {
            log.error("Ledger root path: {} already exists", (Object)this.ledgersRootPath);
            return false;
        }
        ArrayList<Op> multiOps = Lists.newArrayListWithExpectedSize(4);
        multiOps.add(Op.create((String)this.ledgersRootPath, (byte[])BookKeeperConstants.EMPTY_BYTE_ARRAY, this.zkAcls, (CreateMode)CreateMode.PERSISTENT));
        multiOps.add(Op.create((String)this.bookieRegistrationPath, (byte[])BookKeeperConstants.EMPTY_BYTE_ARRAY, this.zkAcls, (CreateMode)CreateMode.PERSISTENT));
        multiOps.add(Op.create((String)this.bookieReadonlyRegistrationPath, (byte[])BookKeeperConstants.EMPTY_BYTE_ARRAY, this.zkAcls, (CreateMode)CreateMode.PERSISTENT));
        String instanceId = UUID.randomUUID().toString();
        multiOps.add(Op.create((String)instanceIdPath, (byte[])instanceId.getBytes(StandardCharsets.UTF_8), this.zkAcls, (CreateMode)CreateMode.PERSISTENT));
        this.zk.multi(multiOps);
        AbstractZkLedgerManagerFactory.newLedgerManagerFactory(this.conf, this.layoutManager);
        log.info("Successfully initiated cluster. ZKServers: {} ledger root path: {} instanceId: {}", new Object[]{zkServers, this.ledgersRootPath, instanceId});
        return true;
    }

    @Override
    public boolean nukeExistingCluster() throws Exception {
        boolean ledgerRootExists;
        String zkServers = ZKMetadataDriverBase.resolveZkServers(this.conf);
        log.info("Nuking ZooKeeper metadata of existing cluster, ZKServers: {} ledger root path: {}", (Object)zkServers, (Object)this.ledgersRootPath);
        boolean bl = ledgerRootExists = null != this.zk.exists(this.ledgersRootPath, false);
        if (!ledgerRootExists) {
            log.info("There is no existing cluster with ledgersRootPath: {} in ZKServers: {}, so exiting nuke operation", (Object)this.ledgersRootPath, (Object)zkServers);
            return true;
        }
        boolean availableNodeExists = null != this.zk.exists(this.bookieRegistrationPath, false);
        try (ZKRegistrationClient regClient = new ZKRegistrationClient(this.zk, this.ledgersRootPath, null, false);){
            if (availableNodeExists) {
                Collection roBookies;
                boolean readonlyNodeExists;
                Collection rwBookies = FutureUtils.result(regClient.getWritableBookies(), EXCEPTION_FUNC).getValue();
                if (rwBookies != null && !rwBookies.isEmpty()) {
                    log.error("Bookies are still up and connected to this cluster, stop all bookies before nuking the cluster");
                    boolean bl2 = false;
                    return bl2;
                }
                boolean bl3 = readonlyNodeExists = null != this.zk.exists(this.bookieReadonlyRegistrationPath, false);
                if (readonlyNodeExists && (roBookies = (Collection)FutureUtils.result(regClient.getReadOnlyBookies(), EXCEPTION_FUNC).getValue()) != null && !roBookies.isEmpty()) {
                    log.error("Readonly Bookies are still up and connected to this cluster, stop all bookies before nuking the cluster");
                    boolean bl4 = false;
                    return bl4;
                }
            }
        }
        LedgerManagerFactory ledgerManagerFactory = AbstractZkLedgerManagerFactory.newLedgerManagerFactory(this.conf, this.layoutManager);
        return ledgerManagerFactory.validateAndNukeExistingCluster(this.conf, this.layoutManager);
    }

    @Override
    public boolean format() throws Exception {
        block11: {
            block10: {
                block9: {
                    block8: {
                        try {
                            ZKUtil.deleteRecursive((ZooKeeper)this.zk, (String)(ZkLedgerUnderreplicationManager.getBasePath(this.ledgersRootPath) + "/ledgers"));
                        }
                        catch (KeeperException.NoNodeException e) {
                            if (!log.isDebugEnabled()) break block8;
                            log.debug("underreplicated ledgers root path node not exists in zookeeper to delete");
                        }
                    }
                    try {
                        ZKUtil.deleteRecursive((ZooKeeper)this.zk, (String)(ZkLedgerUnderreplicationManager.getBasePath(this.ledgersRootPath) + '/' + "locks"));
                    }
                    catch (KeeperException.NoNodeException e) {
                        if (!log.isDebugEnabled()) break block9;
                        log.debug("underreplicatedledger locks node not exists in zookeeper to delete");
                    }
                }
                try {
                    ZKUtil.deleteRecursive((ZooKeeper)this.zk, (String)this.cookiePath);
                }
                catch (KeeperException.NoNodeException e) {
                    if (!log.isDebugEnabled()) break block10;
                    log.debug("cookies node not exists in zookeeper to delete");
                }
            }
            try {
                this.zk.delete(this.ledgersRootPath + "/" + "INSTANCEID", -1);
            }
            catch (KeeperException.NoNodeException e) {
                if (!log.isDebugEnabled()) break block11;
                log.debug("INSTANCEID not exists in zookeeper to delete");
            }
        }
        String instanceId = UUID.randomUUID().toString();
        this.zk.create(this.ledgersRootPath + "/" + "INSTANCEID", instanceId.getBytes(StandardCharsets.UTF_8), this.zkAcls, CreateMode.PERSISTENT);
        log.info("Successfully formatted BookKeeper metadata");
        return true;
    }

    @Override
    public boolean isBookieRegistered(BookieId bookieId) throws BookieException {
        String regPath = this.bookieRegistrationPath + "/" + bookieId;
        String readonlyRegPath = this.bookieReadonlyRegistrationPath + "/" + bookieId;
        try {
            return null != this.zk.exists(regPath, false) || null != this.zk.exists(readonlyRegPath, false);
        }
        catch (KeeperException e) {
            log.error("ZK exception while checking registration ephemeral znodes for BookieId: {}", (Object)bookieId, (Object)e);
            throw new BookieException.MetadataStoreException(e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            log.error("InterruptedException while checking registration ephemeral znodes for BookieId: {}", (Object)bookieId, (Object)e);
            throw new BookieException.MetadataStoreException(e);
        }
    }

    @Override
    public void addRegistrationListener(RegistrationManager.RegistrationListener listener) {
        this.listeners.add(listener);
    }
}

