package org.antora.maven;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.apache.maven.plugin.logging.Log;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Paths;
import java.util.Calendar;
import java.util.Date;
import java.util.Optional;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

public class NodeVersionResolver {
    private static String DIST_BASE_URL = "https://nodejs.org/dist";

    private static String FALLBACK_VERSION = "v16.20.2";

    private Log log;

    private FileDownloader fileDownloader;

    private File releasesDataCache;

    private URL releasesDataSource;

    public NodeVersionResolver(Log log) {
        this.log = log;
        this.fileDownloader = new FileDownloader();
        this.releasesDataCache = resolveReleasesCache();
        this.releasesDataSource = toURL(DIST_BASE_URL + "/index.json");
    }

    public String resolveVersion(String version) {
        return resolveVersion(version, FALLBACK_VERSION);
    }

    public String resolveVersion(String version, String fallbackVersion) {
        boolean offline = false;
        String requestedVersion = version;
        if (version == null || version.isEmpty()) {
            if (System.getProperty("org.apache.maven.test.invoker") != null) return FALLBACK_VERSION;
            version = requestedVersion = "lts";
        } else {
            if (version.startsWith("v") && version.length() > 1 && (version.charAt(1) + "").matches("\\d")) {
                version = version.substring(1);
            }
            if (version.split("[.]").length > 2) return "v" + version;
        }
        String resolvedVersion;
        if (fallbackVersion.startsWith("v")) fallbackVersion = fallbackVersion.substring(1);
        try {
            boolean releasesDataCached = this.releasesDataCache.exists();
            if (offline) {
                if (!releasesDataCached) {
                    this.log.warn("No cached version of Node.js release data available to resolve Node.js version " +
                        "while in offline mode. Reverting to fallback version.");
                }
            } else if (!releasesDataCached || isOlderThanOneDay(this.releasesDataCache.lastModified())) {
                if (!releasesDataCached) this.releasesDataCache.getParentFile().mkdirs();
                fileDownloader.download(this.releasesDataSource, this.releasesDataCache);
            }
            JsonArray releases;
            try (FileReader reader = new FileReader(this.releasesDataCache)) {
                releases = JsonParser.parseReader(reader).getAsJsonArray();
            }
            Stream<JsonObject> releasesStream =
                StreamSupport.stream(releases.spliterator(), false).map(JsonElement::getAsJsonObject).map(it -> {
                    String rawVersion = it.get("version").getAsString();
                    if (rawVersion.startsWith("v")) {
                        it.remove("version");
                        it.addProperty("version", rawVersion.substring(1));
                    }
                    return it;
                });
            Optional<JsonObject> release;
            if (version.equals("latest")) {
                release = releasesStream.findFirst();
            } else if (version.equals("lts")) {
                release = releasesStream.filter(it -> it.getAsJsonPrimitive("lts").isString()).findFirst();
            } else {
                String prefix = version + ".";
                release = releasesStream.filter(it -> it.get("version").getAsString().startsWith(prefix)).findFirst();
            }
            if (release.isPresent()) {
                resolvedVersion = release.get().get("version").getAsString();
            } else {
                this.log.warn("Could not resolve version " + requestedVersion + " in Node.js releases data. " +
                    "Reverting to fallback version.");
                resolvedVersion = fallbackVersion;
            }
        } catch (IOException ex) {
            resolvedVersion = fallbackVersion;
        }
        return "v" + resolvedVersion;
    }

    private File resolveReleasesCache() {
        String appDataDir = System.getenv("APPDATA");
        if (appDataDir == null) {
            String homeDir = System.getProperty("user.home");
            if (System.getProperty("os.name").contains("darwin")) {
                appDataDir = Paths.get(homeDir, "Library", "Preferences").toString();
            } else {
                appDataDir = Paths.get(homeDir, ".local", "share").toString();
            }
        }
        return Paths.get(appDataDir, "node", "releases.json").toFile();
    }

    private boolean isOlderThanOneDay(long lastModified) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        cal.add(Calendar.HOUR, -24);
        Date dayOld = cal.getTime();
        return new Date(lastModified).before(dayOld);
    }

    private URL toURL(String url) {
        try {
            return new URL(url);
        } catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
    }
}
