package org.antora.maven;

import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.BuildPluginManager;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import static org.twdata.maven.mojoexecutor.MojoExecutor.Element;
import static org.twdata.maven.mojoexecutor.MojoExecutor.element;

/**
 * Sets up and runs Antora on your Maven project using the specified playbook file or playbook provider.
 */
@Mojo(name = "antora", defaultPhase = LifecyclePhase.NONE, requiresOnline = true)
public class AntoraMavenPlugin extends AbstractMojo {
    private static final String ANTORA_PACKAGE_NAME = "antora";

    private static final String ANTORA_CLI_PACKAGE_NAME = "@antora/cli";

    private static final String ANTORA_COMMAND_NAME = "antora";

    private static final String PACKAGE_FILE = "package.json";

    private static final String PACKAGE_LOCK_FILE = "package-lock.json";

    /**
     * The project currently being build.
     */
    @Parameter(defaultValue = "${project}", readonly = true)
    private MavenProject mavenProject;

    /**
     * The current Maven session.
     */
    @Parameter(defaultValue = "${session}", readonly = true)
    private MavenSession mavenSession;

    /**
     * The Maven BuildPluginManager component.
     */
    @Component
    private BuildPluginManager pluginManager;

    /**
     * Additional environment variables to set when invoking Antora.
     */
    @Parameter(alias = "env")
    private Map<String, String> environmentVariables;

    /**
     * Sets the name or path of the Node.js executable to use. Implicitly selects which Node.js installation to use to
     * invoke npm and npx. When specified, Node.js is not installed locally and the nodeVersion parameter is ignored.
     */
    @Parameter(property = "node.executable")
    private String nodeExecutable;

    /**
     * The location where Node.js is installed or linked. A node directory will be created inside this directory that
     * hosts the installation files or symlinks.
     */
    @Parameter(defaultValue = "${project.build.directory}", required = true)
    private File nodeInstallDirectory;

    /**
     * The version of Node.js to install. Can be an exact or fuzzy version number with an optional leading "v". Also
     * accepts the keyword "lts", which resolves to the latest LTS release, and "latest", which resolves to the latest
     * available release.
     */
    @Parameter
    private String nodeVersion;

    /**
     * Sets the list of CLI options to pass to Antora. If the option is a flag, the form should be either
     * name?=true|false or name (which implies name?=true). Otherwise, the form should be name=value. In either case,
     * the equals sign can be replaced with a space. The value should never be enclosed in quotes.
     */
    @Parameter
    private List<String> options;

    /**
     * Sets the list of additional npm packages to install when installing Antora. The form should be the same as used
     * with npm install (e.g., @asciidoctor/pdf or @asciidoctor/pdf@latest). The antora package should not be included
     * in this list.
     */
    @Parameter
    private List<String> packages;

    /**
     * Sets the Antora playbook file to pass to Antora.
     */
    @Parameter(defaultValue = "antora-playbook.yml", property = "antora.playbook", required = true)
    private File playbook;

    /**
     * Sets the location of a playbook template to sideload or download to provided-antora-playbook.yml to use as the
     * Antora playbook. When set, the playbook parameter is ignored.
     */
    @Parameter
    private PlaybookProvider playbookProvider;

    /**
     * Sets the version of Antora to use.
     */
    @Parameter(defaultValue = "latest")
    private String version;

    public void execute() throws MojoExecutionException, MojoFailureException {
        validate();
        File basedir = this.mavenProject.getBasedir();
        File nodeHomeDir = new File(this.nodeInstallDirectory, "node");
        File npmCacheDir = null;
        FrontendMojoExecutor frontendMojoExecutor = new FrontendMojoExecutor(this.pluginManager, this.mavenSession,
            element("installDirectory", this.nodeInstallDirectory.getPath()));
        SystemNodeLinker systemNodeLinker = new SystemNodeLinker(getLog(), nodeHomeDir);
        if (this.nodeExecutable == null) {
            systemNodeLinker.unlinkNode();
            frontendMojoExecutor.executeMojo("install-node-and-npm",
                element("nodeVersion", new NodeVersionResolver(getLog()).resolveVersion(this.nodeVersion)));
            npmCacheDir = new File(nodeHomeDir, "_npm");
        } else {
            systemNodeLinker.linkNode(this.nodeExecutable);
        }
        List<String> allPackages = new ArrayList<>();
        String playbookArgument;
        if (this.playbookProvider == null) {
            playbookArgument = basedir.toPath().relativize(this.playbook.toPath()).toString();
            if (packageFileExistsWithDependencies(new File(basedir, PACKAGE_FILE))) {
                boolean hasPackageLockFile = new File(basedir, PACKAGE_LOCK_FILE).exists();
                frontendMojoExecutor.executeMojo("npm", element("arguments", hasPackageLockFile ? "ci" : "i"),
                    element("environmentVariables", npmInstallEnvironmentVariables(hasPackageLockFile, npmCacheDir)));
                String antoraCommandName = isWindows() ? ANTORA_COMMAND_NAME + ".cmd" : ANTORA_COMMAND_NAME;
                if (!Paths.get(basedir.getPath(), "node_modules", ".bin", antoraCommandName).toFile().exists()) {
                    allPackages.add(antoraCliPackageSpec(this.version));
                }
            } else {
                allPackages.add(antoraPackageSpec(this.version));
                if (this.packages != null) allPackages.addAll(this.packages);
            }
        } else {
            File providedPlaybook = new File(basedir, "provided-antora-playbook.yml");
            playbookArgument = providedPlaybook.getName();
            Map<String, String> providedPackages = new ProvidedPlaybookRetriever(getLog(), basedir)
                .retrievePlaybook(this.playbookProvider, providedPlaybook);
            allPackages.add(antoraPackageSpec(providedPackages.containsKey(ANTORA_PACKAGE_NAME)
                ? providedPackages.remove(ANTORA_PACKAGE_NAME)
                : this.version));
            providedPackages.forEach((name, versionSpec) -> allPackages.add(packageSpec(name, versionSpec)));
        }
        try {
            frontendMojoExecutor.executeMojo("npx",
                element("arguments", npxAntoraArguments(allPackages, this.options, playbookArgument)),
                element("environmentVariables", antoraEnvironmentVariables(this.environmentVariables, npmCacheDir)));
        } catch (MojoExecutionException e) {
            throw new MojoFailureException("Antora failed to generate site successfully", e);
        }
    }

    private void validate() throws MojoExecutionException {
        File homeNodeModulesDir = new File(System.getProperty("user.home"), "node_modules");
        if (homeNodeModulesDir.isDirectory()) {
            boolean directoryIsEmpty = true;
            try (Stream<Path> entries = Files.list(homeNodeModulesDir.toPath())) {
                directoryIsEmpty = entries.findFirst().isEmpty();
            } catch (IOException ioe) {
                directoryIsEmpty = false;
            }
            if (!directoryIsEmpty) getLog().warn(
                "Detected the existence of $HOME/node_modules, which is not compatible with this plugin. Please remove it.");
        }
        this.playbookProvider = PlaybookProvider.validate(this.playbookProvider);
        if (this.playbookProvider == null && !this.playbook.exists()) {
            String message = "Antora playbook file not found: " + this.playbook;
            getLog().error("Cannot run Antora because " + message);
            throw new MojoExecutionException(message);
        }
    }

    private Element[] npmInstallEnvironmentVariables(boolean hasPackageLockFile, File npmCacheDir) {
        Map<String, String> env = new HashMap<>();
        if (npmCacheDir != null) env.put("npm_config_cache", npmCacheDir.getPath());
        env.put("npm_config_fund", "false");
        env.put("npm_config_omit", "optional");
        if (!hasPackageLockFile) env.put("npm_config_package_lock", "false");
        env.put("npm_config_update_notifier", "false");
        return mapToElementArray(env);
    }

    private String npxAntoraArguments(List<String> allPackages, List<String> rawOptions, String playbookArgument) {
        List<String> arguments = new ArrayList<>(List.of("--yes"));
        if (allPackages.isEmpty()) {
            arguments.set(0, "--offline");
        } else {
            allPackages.forEach(packageSpec -> arguments.addAll(List.of("--package", packageSpec)));
        }
        arguments.add(ANTORA_COMMAND_NAME);
        arguments.addAll(antoraOptions(rawOptions));
        arguments.add(playbookArgument);
        return String.join(" ", arguments);
    }

    private List<String> antoraOptions(List<String> rawOptions) {
        List<String> opts = new ArrayList<>();
        if (rawOptions == null) {
            Path relativeBuildDirectory = this.mavenProject.getBasedir()
                .toPath()
                .relativize(Paths.get(this.mavenProject.getBuild().getDirectory(), "site"));
            appendOptions(List.of("--to-dir=" + relativeBuildDirectory), opts);
        } else if (!rawOptions.isEmpty()) {
            appendOptions(rawOptions, opts);
        }
        if (this.mavenSession.getRequest().isShowErrors() && !opts.contains("--stacktrace")) opts.add("--stacktrace");
        return opts;
    }

    private Element[] antoraEnvironmentVariables(Map<String, String> env, File npmCacheDir) {
        env = env == null ? new HashMap<>() : new HashMap<>(env);
        String isTTY = System.getenv("IS_TTY");
        if (isTTY == null) env.put("IS_TTY", (isTTY = "true"));
        if ("true".equals(isTTY) && System.getenv("FORCE_COLOR") == null &&
            this.mavenSession.getRequest().isInteractiveMode()) {
            env.put("FORCE_COLOR", "true");
        }
        env.put("NODE_OPTIONS", "--no-global-search-paths"); // attempt to isolate execution from system
        if (npmCacheDir != null) env.put("npm_config_cache", npmCacheDir.getPath());
        env.put("npm_config_fund", "false");
        env.put("npm_config_lockfile_version", "3");
        env.put("npm_config_omit", "optional");
        env.put("npm_config_update_notifier", "false");
        return mapToElementArray(env);
    }

    private String antoraPackageSpec(String versionSpec) {
        return packageSpec(ANTORA_PACKAGE_NAME, versionSpec);
    }

    private String antoraCliPackageSpec(String versionSpec) {
        return packageSpec(ANTORA_CLI_PACKAGE_NAME, versionSpec);
    }

    private String packageSpec(String name, String versionSpec) {
        return name + (versionSpec == null || "latest".equals(versionSpec) ? "" : "@" + versionSpec);
    }

    private boolean packageFileExistsWithDependencies(File packageFile) {
        if (!packageFile.exists()) return false;
        try (FileReader reader = new FileReader(packageFile)) {
            JsonObject pkg = JsonParser.parseReader(reader).getAsJsonObject();
            return (pkg.has("dependencies") && !pkg.getAsJsonObject("dependencies").isEmpty()) ||
                (pkg.has("devDependencies") && !pkg.getAsJsonObject("devDependencies").isEmpty());
        } catch (IOException ex) {
            return true;
        }
    }

    private Element[] mapToElementArray(Map<String, String> map) {
        return map.entrySet()
            .stream()
            .map(envEntry -> element(envEntry.getKey(), envEntry.getValue()))
            .toArray(Element[]::new);
    }

    private void appendOptions(List<String> from, List<String> to) {
        from.forEach((text) -> {
            if ((text = text.strip()).isEmpty()) return;
            String[] segments = text.split("[ =]", 2);
            String name = segments[0];
            if (name.startsWith("--")) name = name.substring(2);
            String value = segments.length == 2 ? segments[1] : null;
            if (name.endsWith("?")) {
                name = name.substring(0, name.length() - 1);
                if (value != null && value.contains(":")) {
                    String[] valueAndFallback = value.split(":", 2);
                    if ((value = valueAndFallback[0]).isEmpty() && valueAndFallback.length == 2) {
                        value = valueAndFallback[1];
                    }
                }
                if ("false".equals(value)) return;
                to.add("--" + name);
            } else if (value == null) {
                to.add("--" + name);
            } else if (Pattern.compile("[ \"]").matcher(value).find()) {
                String q = "\"";
                if (value.contains(q)) {
                    if (value.contains("'")) {
                        // NOTE frontend plugin doesn't support escaped quotes; revert to character reference
                        value = value.replace(q, "&quot;");
                    } else {
                        q = "'";
                    }
                }
                to.add("--" + name + "=" + q + value + q);
            } else {
                to.add("--" + name + "=" + value);
            }
        });
    }

    private boolean isWindows() {
        return System.getProperty("os.name").toLowerCase().contains("windows");
    }
}
