/*
 * Decompiled with CFR 0.152.
 */
package alluxio.cli.fsadmin.report;

import alluxio.cli.fsadmin.FileSystemAdminShellUtils;
import alluxio.client.block.BlockMasterClient;
import alluxio.client.block.options.GetWorkerReportOptions;
import alluxio.exception.status.InvalidArgumentException;
import alluxio.util.FormatUtils;
import alluxio.wire.WorkerInfo;
import com.google.common.base.Strings;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.apache.commons.cli.CommandLine;

public class CapacityCommand {
    private static final int INDENT_SIZE = 4;
    private BlockMasterClient mBlockMasterClient;
    private PrintStream mPrintStream;
    private int mIndentationLevel = 0;
    private long mSumCapacityBytes;
    private long mSumUsedBytes;
    private Map<String, Long> mSumCapacityBytesOnTierMap;
    private Map<String, Long> mSumUsedBytesOnTierMap;
    private TreeMap<String, Map<String, String>> mCapacityTierInfoMap;
    private Map<String, Map<String, String>> mUsedTierInfoMap;

    public CapacityCommand(BlockMasterClient blockMasterClient, PrintStream printStream) {
        this.mBlockMasterClient = blockMasterClient;
        this.mPrintStream = printStream;
    }

    public int run(CommandLine cl) throws IOException {
        if (cl.hasOption("h")) {
            System.out.println(CapacityCommand.getUsage());
            return 0;
        }
        GetWorkerReportOptions options = this.getOptions(cl);
        this.generateCapacityReport(options);
        return 0;
    }

    public void generateCapacityReport(GetWorkerReportOptions options) throws IOException {
        List workerInfoList = this.mBlockMasterClient.getWorkerReport(options);
        if (workerInfoList.size() == 0) {
            this.print("No workers found.");
            return;
        }
        Collections.sort(workerInfoList, new WorkerInfo.LastContactSecComparator());
        this.collectWorkerInfo(workerInfoList);
        this.printAggregatedInfo(options);
        this.printWorkerInfo(workerInfoList);
    }

    private void collectWorkerInfo(List<WorkerInfo> workerInfoList) {
        this.initVariables();
        for (WorkerInfo workerInfo : workerInfoList) {
            long usedBytes = workerInfo.getUsedBytes();
            long capacityBytes = workerInfo.getCapacityBytes();
            this.mSumCapacityBytes += capacityBytes;
            this.mSumUsedBytes += usedBytes;
            String workerName = workerInfo.getAddress().getHost();
            Map totalBytesOnTiers = workerInfo.getCapacityBytesOnTiers();
            for (Map.Entry totalBytesTier : totalBytesOnTiers.entrySet()) {
                String tier = (String)totalBytesTier.getKey();
                long value = (Long)totalBytesTier.getValue();
                this.mSumCapacityBytesOnTierMap.put(tier, value + this.mSumCapacityBytesOnTierMap.getOrDefault(tier, 0L));
                Map map = this.mCapacityTierInfoMap.getOrDefault(tier, new HashMap());
                map.put(workerName, FormatUtils.getSizeFromBytes((long)value));
                this.mCapacityTierInfoMap.put(tier, map);
            }
            Map usedBytesOnTiers = workerInfo.getUsedBytesOnTiers();
            for (Map.Entry usedBytesTier : usedBytesOnTiers.entrySet()) {
                String tier = (String)usedBytesTier.getKey();
                long value = (Long)usedBytesTier.getValue();
                this.mSumUsedBytesOnTierMap.put(tier, value + this.mSumUsedBytesOnTierMap.getOrDefault(tier, 0L));
                Map map = this.mUsedTierInfoMap.getOrDefault(tier, new HashMap());
                map.put(workerName, FormatUtils.getSizeFromBytes((long)value));
                this.mUsedTierInfoMap.put(tier, map);
            }
        }
    }

    private void printAggregatedInfo(GetWorkerReportOptions options) {
        long value;
        this.mIndentationLevel = 0;
        this.print(String.format("Capacity information for %s workers: ", options.getWorkerRange().toString().toLowerCase()));
        ++this.mIndentationLevel;
        this.print("Total Capacity: " + FormatUtils.getSizeFromBytes((long)this.mSumCapacityBytes));
        ++this.mIndentationLevel;
        for (Map.Entry<String, Long> totalBytesTier : this.mSumCapacityBytesOnTierMap.entrySet()) {
            value = totalBytesTier.getValue();
            this.print("Tier: " + totalBytesTier.getKey() + "  Size: " + FormatUtils.getSizeFromBytes((long)value));
        }
        --this.mIndentationLevel;
        this.print("Used Capacity: " + FormatUtils.getSizeFromBytes((long)this.mSumUsedBytes));
        ++this.mIndentationLevel;
        for (Map.Entry<String, Long> usedBytesTier : this.mSumUsedBytesOnTierMap.entrySet()) {
            value = usedBytesTier.getValue();
            this.print("Tier: " + usedBytesTier.getKey() + "  Size: " + FormatUtils.getSizeFromBytes((long)value));
        }
        --this.mIndentationLevel;
        if (this.mSumCapacityBytes != 0L) {
            int usedPercentage = (int)(100L * this.mSumUsedBytes / this.mSumCapacityBytes);
            this.print(String.format("Used Percentage: %s%%", usedPercentage));
            this.print(String.format("Free Percentage: %s%%", 100 - usedPercentage));
        }
    }

    private void printWorkerInfo(List<WorkerInfo> workerInfoList) {
        this.mIndentationLevel = 0;
        if (this.mCapacityTierInfoMap.size() == 0) {
            return;
        }
        if (this.mCapacityTierInfoMap.size() == 1) {
            this.printShortWorkerInfo(workerInfoList);
            return;
        }
        Set<String> tiers = this.mCapacityTierInfoMap.keySet();
        String tiersInfo = String.format(Strings.repeat((String)"%-14s", (int)tiers.size()), tiers.toArray());
        String longInfoFormat = this.getInfoFormat(workerInfoList, false);
        this.print(String.format("%n" + longInfoFormat, "Worker Name", "Last Heartbeat", "Storage", "Total", tiersInfo));
        for (WorkerInfo info : workerInfoList) {
            String workerName = info.getAddress().getHost();
            long usedBytes = info.getUsedBytes();
            long capacityBytes = info.getCapacityBytes();
            String usedPercentageInfo = "";
            if (capacityBytes != 0L) {
                int usedPercentage = (int)(100L * usedBytes / capacityBytes);
                usedPercentageInfo = String.format(" (%s%%)", usedPercentage);
            }
            String capacityTierInfo = CapacityCommand.getWorkerFormattedTierValues(this.mCapacityTierInfoMap, workerName);
            String usedTierInfo = CapacityCommand.getWorkerFormattedTierValues(this.mUsedTierInfoMap, workerName);
            this.print(String.format(longInfoFormat, workerName, info.getLastContactSec(), "capacity", FormatUtils.getSizeFromBytes((long)capacityBytes), capacityTierInfo));
            this.print(String.format(longInfoFormat, "", "", "used", FormatUtils.getSizeFromBytes((long)usedBytes) + usedPercentageInfo, usedTierInfo));
        }
    }

    private void printShortWorkerInfo(List<WorkerInfo> workerInfoList) {
        String tier = this.mCapacityTierInfoMap.firstKey();
        String shortInfoFormat = this.getInfoFormat(workerInfoList, true);
        this.print(String.format("%n" + shortInfoFormat, "Worker Name", "Last Heartbeat", "Storage", tier));
        for (WorkerInfo info : workerInfoList) {
            long capacityBytes = info.getCapacityBytes();
            long usedBytes = info.getUsedBytes();
            String usedPercentageInfo = "";
            if (capacityBytes != 0L) {
                int usedPercentage = (int)(100L * usedBytes / capacityBytes);
                usedPercentageInfo = String.format(" (%s%%)", usedPercentage);
            }
            this.print(String.format(shortInfoFormat, info.getAddress().getHost(), info.getLastContactSec(), "capacity", FormatUtils.getSizeFromBytes((long)capacityBytes)));
            this.print(String.format(shortInfoFormat, "", "", "used", FormatUtils.getSizeFromBytes((long)usedBytes) + usedPercentageInfo));
        }
    }

    private String getInfoFormat(List<WorkerInfo> workerInfoList, boolean isShort) {
        int firstIndent = 16;
        int maxWorkerNameLength = workerInfoList.stream().map(w -> w.getAddress().getHost().length()).max(Comparator.comparing(Integer::intValue)).get();
        if (firstIndent <= maxWorkerNameLength) {
            firstIndent = maxWorkerNameLength + 5;
        }
        if (isShort) {
            return "%-" + firstIndent + "s %-16s %-13s %s";
        }
        return "%-" + firstIndent + "s %-16s %-13s %-16s %s";
    }

    private GetWorkerReportOptions getOptions(CommandLine cl) throws IOException {
        if (cl.getOptions().length > 1) {
            System.out.println(CapacityCommand.getUsage());
            throw new InvalidArgumentException("Too many arguments passed in.");
        }
        GetWorkerReportOptions workerOptions = GetWorkerReportOptions.defaults();
        HashSet<GetWorkerReportOptions.WorkerInfoField> fieldRange = new HashSet<GetWorkerReportOptions.WorkerInfoField>(Arrays.asList(GetWorkerReportOptions.WorkerInfoField.ADDRESS, GetWorkerReportOptions.WorkerInfoField.WORKER_CAPACITY_BYTES, GetWorkerReportOptions.WorkerInfoField.WORKER_CAPACITY_BYTES_ON_TIERS, GetWorkerReportOptions.WorkerInfoField.LAST_CONTACT_SEC, GetWorkerReportOptions.WorkerInfoField.WORKER_USED_BYTES, GetWorkerReportOptions.WorkerInfoField.WORKER_USED_BYTES_ON_TIERS));
        workerOptions.setFieldRange(fieldRange);
        if (cl.hasOption("live")) {
            workerOptions.setWorkerRange(GetWorkerReportOptions.WorkerRange.LIVE);
        } else if (cl.hasOption("lost")) {
            workerOptions.setWorkerRange(GetWorkerReportOptions.WorkerRange.LOST);
        } else if (cl.hasOption("workers")) {
            workerOptions.setWorkerRange(GetWorkerReportOptions.WorkerRange.SPECIFIED);
            String addressString = cl.getOptionValue("workers");
            String[] addressArray = addressString.split(",");
            workerOptions.setAddresses(new HashSet<String>(Arrays.asList(addressArray)));
        }
        return workerOptions;
    }

    private static String getWorkerFormattedTierValues(Map<String, Map<String, String>> map, String workerName) {
        return map.values().stream().map(tierMap -> String.format("%-14s", tierMap.getOrDefault(workerName, "-"))).collect(Collectors.joining(""));
    }

    private void initVariables() {
        this.mSumCapacityBytes = 0L;
        this.mSumUsedBytes = 0L;
        this.mSumCapacityBytesOnTierMap = new TreeMap<String, Long>(FileSystemAdminShellUtils::compareTierNames);
        this.mSumUsedBytesOnTierMap = new TreeMap<String, Long>(FileSystemAdminShellUtils::compareTierNames);
        this.mCapacityTierInfoMap = new TreeMap(FileSystemAdminShellUtils::compareTierNames);
        this.mUsedTierInfoMap = new TreeMap<String, Map<String, String>>(FileSystemAdminShellUtils::compareTierNames);
    }

    private void print(String text) {
        String indent = Strings.repeat((String)" ", (int)(this.mIndentationLevel * 4));
        this.mPrintStream.println(indent + text);
    }

    public static String getUsage() {
        return "alluxio fsadmin report capacity [filter arg]\nReport Alluxio capacity information.\nWhere [filter arg] is an optional argument. If no arguments passed in, capacity information of all workers will be printed out.\n[filter arg] can be one of the following:\n    -live                   Live workers\n    -lost                   Lost workers\n    -workers <worker_names>  Specified workers, host names or ip addresses separated by \",\"\n";
    }
}

