/*
 * Decompiled with CFR 0.152.
 */
package alluxio.cli.fs.command;

import alluxio.AlluxioURI;
import alluxio.cli.Command;
import alluxio.cli.CommandUtils;
import alluxio.cli.fs.FileSystemShellUtils;
import alluxio.cli.fs.command.AbstractFileSystemCommand;
import alluxio.client.file.FileInStream;
import alluxio.client.file.FileOutStream;
import alluxio.client.file.FileSystem;
import alluxio.client.file.FileSystemContext;
import alluxio.client.file.URIStatus;
import alluxio.exception.AlluxioException;
import alluxio.exception.ExceptionMessage;
import alluxio.exception.FileAlreadyExistsException;
import alluxio.exception.FileDoesNotExistException;
import alluxio.exception.InvalidPathException;
import alluxio.exception.status.InvalidArgumentException;
import alluxio.grpc.DeletePOptions;
import alluxio.grpc.SetAclAction;
import alluxio.grpc.SetAttributePOptions;
import alluxio.security.authorization.Mode;
import alluxio.util.io.PathUtils;
import com.google.common.base.Joiner;
import com.google.common.io.Closer;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class CpCommand
extends AbstractFileSystemCommand {
    private static final Logger LOG = LoggerFactory.getLogger(CpCommand.class);
    private static final String COPY_SUCCEED_MESSAGE = "Copied %s to %s";
    private static final String COPY_FAIL_MESSAGE = "Failed to copy %s to %s";
    private static final int COPY_FROM_LOCAL_BUFFER_SIZE_DEFAULT = 0x800000;
    private static final int COPY_TO_LOCAL_BUFFER_SIZE_DEFAULT = 0x4000000;
    private static final Option RECURSIVE_OPTION = Option.builder((String)"R").longOpt("recursive").required(false).hasArg(false).desc("copy files in subdirectories recursively").build();
    private static final Option RECURSIVE_ALIAS_OPTION = Option.builder((String)"r").required(false).hasArg(false).desc("copy files in subdirectories recursively").build();
    public static final Option THREAD_OPTION = Option.builder().longOpt("thread").required(false).hasArg(true).numberOfArgs(1).argName("threads").type(Number.class).desc("Number of threads used to copy files in parallel, default value is CPU cores * 2").build();
    public static final Option BUFFER_SIZE_OPTION = Option.builder().longOpt("buffersize").required(false).hasArg(true).numberOfArgs(1).argName("buffer size").type(Number.class).desc("Read buffer size in bytes, default is 8MB when copying from local, and 64MB when copying to local").build();
    private static final Option PRESERVE_OPTION = Option.builder((String)"p").longOpt("preserve").required(false).desc("Preserve file permission attributes when copying files. All ownership, permissions and ACLs will be preserved").build();
    private int mCopyFromLocalBufferSize = 0x800000;
    private int mCopyToLocalBufferSize = 0x4000000;
    private int mThread = Runtime.getRuntime().availableProcessors() * 2;
    private boolean mPreservePermissions = false;

    public CpCommand(FileSystemContext fsContext) {
        super(fsContext);
    }

    public String getCommandName() {
        return "cp";
    }

    public void validateArgs(CommandLine cl) throws InvalidArgumentException {
        CommandUtils.checkNumOfArgsEquals((Command)this, (CommandLine)cl, (int)2);
        if (cl.hasOption(BUFFER_SIZE_OPTION.getLongOpt())) {
            try {
                int bufSize = ((Number)cl.getParsedOptionValue(BUFFER_SIZE_OPTION.getLongOpt())).intValue();
                if (bufSize < 0) {
                    throw new InvalidArgumentException(BUFFER_SIZE_OPTION.getLongOpt() + " must be > 0");
                }
                this.mCopyFromLocalBufferSize = bufSize;
                this.mCopyToLocalBufferSize = bufSize;
            }
            catch (ParseException e) {
                throw new InvalidArgumentException("Failed to parse option " + BUFFER_SIZE_OPTION.getLongOpt() + " into an integer", (Throwable)e);
            }
        }
        if (cl.hasOption(THREAD_OPTION.getLongOpt())) {
            try {
                this.mThread = ((Number)cl.getParsedOptionValue(THREAD_OPTION.getLongOpt())).intValue();
                if (this.mThread <= 0) {
                    throw new InvalidArgumentException(THREAD_OPTION.getLongOpt() + " must be > 0");
                }
            }
            catch (ParseException e) {
                throw new InvalidArgumentException("Failed to parse option " + THREAD_OPTION.getLongOpt() + " into an integer", (Throwable)e);
            }
        }
        this.mPreservePermissions = cl.hasOption(PRESERVE_OPTION.getLongOpt());
    }

    public Options getOptions() {
        return new Options().addOption(RECURSIVE_OPTION).addOption(RECURSIVE_ALIAS_OPTION).addOption(THREAD_OPTION).addOption(PRESERVE_OPTION);
    }

    public int run(CommandLine cl) throws AlluxioException, IOException {
        String[] args = cl.getArgs();
        AlluxioURI srcPath = new AlluxioURI(args[0]);
        AlluxioURI dstPath = new AlluxioURI(args[1]);
        if ((dstPath.getScheme() == null || CpCommand.isAlluxio(dstPath.getScheme())) && CpCommand.isFile(srcPath.getScheme())) {
            ArrayList<AlluxioURI> srcPaths = new ArrayList<AlluxioURI>();
            if (srcPath.containsWildcard()) {
                List<File> srcFiles = FileSystemShellUtils.getFiles(srcPath.getPath());
                if (srcFiles.size() == 0) {
                    throw new IOException(ExceptionMessage.PATH_DOES_NOT_EXIST.getMessage(new Object[]{srcPath}));
                }
                for (File file : srcFiles) {
                    srcPaths.add(new AlluxioURI(srcPath.getScheme(), srcPath.getAuthority(), file.getPath()));
                }
            } else {
                File src = new File(srcPath.getPath());
                if (src.isDirectory()) {
                    File[] files = src.listFiles();
                    if (files == null) {
                        throw new IOException(String.format("Failed to list files for directory %s", src));
                    }
                    for (File file : files) {
                        srcPaths.add(new AlluxioURI(srcPath.getScheme(), srcPath.getAuthority(), file.getPath()));
                    }
                } else {
                    srcPaths.add(srcPath);
                }
            }
            if (srcPaths.size() == 1 && !new File(((AlluxioURI)srcPaths.get(0)).getPath()).isDirectory()) {
                this.copyFromLocalFile((AlluxioURI)srcPaths.get(0), dstPath);
            } else {
                CopyThreadPoolExecutor pool = new CopyThreadPoolExecutor(this.mThread, System.out, System.err, this.mFileSystem, this.mFileSystem.exists(dstPath) ? null : dstPath);
                try {
                    this.createDstDir(dstPath);
                    for (AlluxioURI alluxioURI : srcPaths) {
                        AlluxioURI dst = new AlluxioURI(dstPath, new AlluxioURI(alluxioURI.getName()));
                        this.asyncCopyLocalPath(pool, alluxioURI, dst);
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException(e);
                }
                finally {
                    pool.shutdown();
                }
            }
            System.out.println(String.format(COPY_SUCCEED_MESSAGE, srcPath, dstPath));
        } else if ((srcPath.getScheme() == null || CpCommand.isAlluxio(srcPath.getScheme())) && CpCommand.isFile(dstPath.getScheme())) {
            List<AlluxioURI> srcPaths = FileSystemShellUtils.getAlluxioURIs(this.mFileSystem, srcPath);
            if (srcPaths.size() == 0) {
                throw new IOException(ExceptionMessage.PATH_DOES_NOT_EXIST.getMessage(new Object[]{srcPath}));
            }
            if (srcPath.containsWildcard()) {
                this.copyWildcardToLocal(srcPaths, dstPath);
            } else {
                this.copyToLocal(srcPath, dstPath);
            }
        } else if ((srcPath.getScheme() == null || CpCommand.isAlluxio(srcPath.getScheme())) && (dstPath.getScheme() == null || CpCommand.isAlluxio(dstPath.getScheme()))) {
            boolean recursive;
            List<AlluxioURI> srcPaths = FileSystemShellUtils.getAlluxioURIs(this.mFileSystem, srcPath);
            if (srcPaths.size() == 0) {
                throw new FileDoesNotExistException(ExceptionMessage.PATH_DOES_NOT_EXIST.getMessage(new Object[]{srcPath.getPath()}));
            }
            boolean bl = recursive = cl.hasOption(RECURSIVE_OPTION.getOpt()) || cl.hasOption(RECURSIVE_ALIAS_OPTION.getOpt());
            if (srcPath.containsWildcard()) {
                this.copyWildcard(srcPaths, dstPath, recursive);
            } else {
                this.copy(srcPath, dstPath, recursive);
            }
        } else {
            throw new InvalidPathException("Schemes must be either file or alluxio, and at most one file scheme is allowed.");
        }
        return 0;
    }

    private void copyWildcard(List<AlluxioURI> srcPaths, AlluxioURI dstPath, boolean recursive) throws AlluxioException, IOException {
        URIStatus dstStatus = null;
        try {
            dstStatus = this.mFileSystem.getStatus(dstPath);
        }
        catch (FileDoesNotExistException fileDoesNotExistException) {
            // empty catch block
        }
        if (dstStatus != null && !dstStatus.isFolder()) {
            throw new InvalidPathException(ExceptionMessage.DESTINATION_CANNOT_BE_FILE.getMessage(new Object[0]));
        }
        if (dstStatus == null) {
            this.mFileSystem.createDirectory(dstPath);
            System.out.println("Created directory: " + dstPath);
        }
        ArrayList<String> errorMessages = new ArrayList<String>();
        for (AlluxioURI srcPath : srcPaths) {
            try {
                this.copy(srcPath, new AlluxioURI(dstPath.getScheme(), dstPath.getAuthority(), PathUtils.concatPath((Object)dstPath.getPath(), (Object)srcPath.getName())), recursive);
            }
            catch (AlluxioException | IOException e) {
                errorMessages.add(e.getMessage());
            }
        }
        if (errorMessages.size() != 0) {
            throw new IOException(Joiner.on((char)'\n').join(errorMessages));
        }
    }

    private void copy(AlluxioURI srcPath, AlluxioURI dstPath, boolean recursive) throws AlluxioException, IOException {
        URIStatus srcStatus = this.mFileSystem.getStatus(srcPath);
        URIStatus dstStatus = null;
        try {
            dstStatus = this.mFileSystem.getStatus(dstPath);
        }
        catch (FileDoesNotExistException fileDoesNotExistException) {
            // empty catch block
        }
        if (!srcStatus.isFolder()) {
            if (dstStatus != null && dstStatus.isFolder()) {
                dstPath = new AlluxioURI(PathUtils.concatPath((Object)dstPath.getPath(), (Object)srcPath.getName()));
            }
            this.copyFile(srcPath, dstPath);
        } else {
            if (!recursive) {
                throw new IOException(srcPath.getPath() + " is a directory, to copy it please use \"cp -R/-r/--recursive <src> <dst>\"");
            }
            List statuses = this.mFileSystem.listStatus(srcPath);
            if (dstStatus != null) {
                if (!dstStatus.isFolder()) {
                    throw new InvalidPathException(ExceptionMessage.DESTINATION_CANNOT_BE_FILE.getMessage(new Object[0]));
                }
                if (srcStatus.isFolder()) {
                    dstPath = new AlluxioURI(PathUtils.concatPath((Object)dstPath.getPath(), (Object)srcPath.getName()));
                    this.mFileSystem.createDirectory(dstPath);
                    System.out.println("Created directory: " + dstPath);
                }
            }
            if (dstStatus == null) {
                this.mFileSystem.createDirectory(dstPath);
                System.out.println("Created directory: " + dstPath);
            }
            this.preserveAttributes(srcPath, dstPath);
            ArrayList<String> errorMessages = new ArrayList<String>();
            for (URIStatus status : statuses) {
                try {
                    this.copy(new AlluxioURI(srcPath.getScheme(), srcPath.getAuthority(), status.getPath()), new AlluxioURI(dstPath.getScheme(), dstPath.getAuthority(), PathUtils.concatPath((Object)dstPath.getPath(), (Object)status.getName())), recursive);
                }
                catch (IOException e) {
                    errorMessages.add(e.getMessage());
                }
            }
            if (errorMessages.size() != 0) {
                throw new IOException(Joiner.on((char)'\n').join(errorMessages));
            }
        }
    }

    private void copyFile(AlluxioURI srcPath, AlluxioURI dstPath) throws AlluxioException, IOException {
        try (FileInStream is = this.mFileSystem.openFile(srcPath);
             FileOutStream os = this.mFileSystem.createFile(dstPath);){
            try {
                IOUtils.copyLarge((InputStream)is, (OutputStream)os, (byte[])new byte[0x800000]);
            }
            catch (Exception e) {
                os.cancel();
                this.mFileSystem.delete(dstPath, DeletePOptions.newBuilder().setUnchecked(true).build());
                throw e;
            }
            System.out.println(String.format(COPY_SUCCEED_MESSAGE, srcPath, dstPath));
        }
        this.preserveAttributes(srcPath, dstPath);
    }

    private void preserveAttributes(AlluxioURI srcPath, AlluxioURI dstPath) throws IOException, AlluxioException {
        if (this.mPreservePermissions) {
            URIStatus srcStatus = this.mFileSystem.getStatus(srcPath);
            this.mFileSystem.setAttribute(dstPath, SetAttributePOptions.newBuilder().setOwner(srcStatus.getOwner()).setGroup(srcStatus.getGroup()).setMode(new Mode((short)srcStatus.getMode()).toProto()).build());
            this.mFileSystem.setAcl(dstPath, SetAclAction.REPLACE, srcStatus.getAcl().getEntries());
        }
    }

    private void createDstDir(AlluxioURI dstPath) throws AlluxioException, IOException {
        try {
            this.mFileSystem.createDirectory(dstPath);
        }
        catch (FileAlreadyExistsException fileAlreadyExistsException) {
            // empty catch block
        }
        URIStatus dstStatus = this.mFileSystem.getStatus(dstPath);
        if (!dstStatus.isFolder()) {
            throw new InvalidPathException(ExceptionMessage.DESTINATION_CANNOT_BE_FILE.getMessage(new Object[0]));
        }
    }

    private void copyFromLocalFile(AlluxioURI srcPath, AlluxioURI dstPath) throws AlluxioException, IOException {
        File src = new File(srcPath.getPath());
        if (src.isDirectory()) {
            throw new IOException("Source " + src.getAbsolutePath() + " is not a file.");
        }
        if (this.mFileSystem.exists(dstPath) && this.mFileSystem.getStatus(dstPath).isFolder()) {
            dstPath = dstPath.join(src.getName());
        }
        FileOutStream os = null;
        try (Closer closer = Closer.create();){
            os = (FileOutStream)closer.register((Closeable)this.mFileSystem.createFile(dstPath));
            FileInputStream in = (FileInputStream)closer.register((Closeable)new FileInputStream(src));
            FileChannel channel = (FileChannel)closer.register((Closeable)in.getChannel());
            ByteBuffer buf = ByteBuffer.allocate(this.mCopyFromLocalBufferSize);
            while (channel.read(buf) != -1) {
                buf.flip();
                os.write(buf.array(), 0, buf.limit());
            }
        }
        catch (Exception e) {
            if (os != null) {
                os.cancel();
                if (this.mFileSystem.exists(dstPath)) {
                    this.mFileSystem.delete(dstPath);
                }
            }
            throw e;
        }
    }

    private void asyncCopyLocalPath(CopyThreadPoolExecutor pool, AlluxioURI srcPath, AlluxioURI dstPath) throws InterruptedException {
        File src = new File(srcPath.getPath());
        if (!src.isDirectory()) {
            pool.submit(() -> {
                try {
                    this.copyFromLocalFile(srcPath, dstPath);
                    pool.succeed(srcPath, dstPath);
                }
                catch (Exception e) {
                    pool.fail(srcPath, dstPath, e);
                }
                return null;
            });
        } else {
            try {
                this.mFileSystem.createDirectory(dstPath);
            }
            catch (Exception e) {
                pool.fail(srcPath, dstPath, e);
                return;
            }
            File[] fileList = src.listFiles();
            if (fileList == null) {
                pool.fail(srcPath, dstPath, new IOException(String.format("Failed to list directory %s.", src)));
                return;
            }
            for (File srcFile : fileList) {
                AlluxioURI newURI = new AlluxioURI(dstPath, new AlluxioURI(srcFile.getName()));
                this.asyncCopyLocalPath(pool, new AlluxioURI(srcPath.getScheme(), srcPath.getAuthority(), srcFile.getPath()), newURI);
            }
        }
    }

    private void copyWildcardToLocal(List<AlluxioURI> srcPaths, AlluxioURI dstPath) throws AlluxioException, IOException {
        File dstFile = new File(dstPath.getPath());
        if (dstFile.exists() && !dstFile.isDirectory()) {
            throw new InvalidPathException(ExceptionMessage.DESTINATION_CANNOT_BE_FILE.getMessage(new Object[0]));
        }
        if (!dstFile.exists()) {
            if (!dstFile.mkdirs()) {
                throw new IOException("Fail to create directory: " + dstPath);
            }
            System.out.println("Create directory: " + dstPath);
        }
        ArrayList<String> errorMessages = new ArrayList<String>();
        for (AlluxioURI srcPath : srcPaths) {
            try {
                File dstSubFile = new File(dstFile.getAbsoluteFile(), srcPath.getName());
                this.copyToLocal(srcPath, new AlluxioURI(dstPath.getScheme(), dstPath.getAuthority(), dstSubFile.getPath()));
            }
            catch (IOException e) {
                errorMessages.add(e.getMessage());
            }
        }
        if (errorMessages.size() != 0) {
            throw new IOException(Joiner.on((char)'\n').join(errorMessages));
        }
    }

    private void copyToLocal(AlluxioURI srcPath, AlluxioURI dstPath) throws AlluxioException, IOException {
        URIStatus srcStatus = this.mFileSystem.getStatus(srcPath);
        File dstFile = new File(dstPath.getPath());
        if (srcStatus.isFolder()) {
            List statuses;
            if (!dstFile.exists()) {
                if (!dstFile.mkdirs()) {
                    throw new IOException("mkdir failure for directory: " + dstPath);
                }
                System.out.println("Create directory: " + dstPath);
            }
            try {
                statuses = this.mFileSystem.listStatus(srcPath);
            }
            catch (AlluxioException e) {
                throw new IOException(e.getMessage());
            }
            ArrayList<String> errorMessages = new ArrayList<String>();
            for (URIStatus status : statuses) {
                try {
                    File subDstFile = new File(dstFile.getAbsolutePath(), status.getName());
                    this.copyToLocal(new AlluxioURI(srcPath.getScheme(), srcPath.getAuthority(), status.getPath()), new AlluxioURI(dstPath.getScheme(), dstPath.getAuthority(), subDstFile.getPath()));
                }
                catch (IOException e) {
                    errorMessages.add(e.getMessage());
                }
            }
            if (errorMessages.size() != 0) {
                throw new IOException(Joiner.on((char)'\n').join(errorMessages));
            }
        } else {
            this.copyFileToLocal(srcPath, dstPath);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void copyFileToLocal(AlluxioURI srcPath, AlluxioURI dstPath) throws AlluxioException, IOException {
        File dstFile = new File(dstPath.getPath());
        String randomSuffix = String.format(".%s_copyToLocal_", RandomStringUtils.randomAlphanumeric((int)8));
        File outputFile = dstFile.isDirectory() ? new File(PathUtils.concatPath((Object)dstFile.getAbsolutePath(), (Object)srcPath.getName())) : dstFile;
        File tmpDst = new File(outputFile.getPath() + randomSuffix);
        try (Closer closer = Closer.create();){
            FileInStream is = (FileInStream)closer.register((Closeable)this.mFileSystem.openFile(srcPath));
            FileOutputStream out = (FileOutputStream)closer.register((Closeable)new FileOutputStream(tmpDst));
            byte[] buf = new byte[this.mCopyToLocalBufferSize];
            int t = is.read(buf);
            while (t != -1) {
                out.write(buf, 0, t);
                t = is.read(buf);
            }
            if (!tmpDst.renameTo(outputFile)) {
                throw new IOException("Failed to rename " + tmpDst.getPath() + " to destination " + outputFile.getPath());
            }
            System.out.println("Copied " + srcPath + " to file://" + outputFile.getPath());
        }
        finally {
            tmpDst.delete();
        }
    }

    public String getUsage() {
        return "cp [-R/-r/--recursive] [--buffersize <bytes>] <src> <dst>";
    }

    public String getDescription() {
        return "Copies a file or a directory in the Alluxio filesystem or between local filesystem and Alluxio filesystem. The -R/-r/--recursive flags are needed to copydirectories in the Alluxio filesystem. Local Path with schema \"file\".";
    }

    private static boolean isAlluxio(String scheme) {
        return "alluxio".equals(scheme);
    }

    private static boolean isFile(String scheme) {
        return "file".equals(scheme);
    }

    @ThreadSafe
    private static final class CopyThreadPoolExecutor {
        private static final Object MESSAGE_DONE = new Object();
        private final ThreadPoolExecutor mPool;
        private final BlockingQueue<Object> mMessages;
        private final ConcurrentLinkedQueue<Exception> mExceptions;
        private final PrintStream mStdout;
        private final PrintStream mStderr;
        private final Thread mPrinter;
        private final FileSystem mFileSystem;
        private final AlluxioURI mPath;

        public CopyThreadPoolExecutor(int threads, PrintStream stdout, PrintStream stderr, FileSystem fileSystem, AlluxioURI path) {
            this.mPool = new ThreadPoolExecutor(threads, threads, 1L, TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(threads * 2), new ThreadPoolExecutor.CallerRunsPolicy());
            this.mMessages = new LinkedBlockingQueue<Object>();
            this.mExceptions = new ConcurrentLinkedQueue();
            this.mStdout = stdout;
            this.mStderr = stderr;
            this.mPrinter = new Thread(() -> {
                while (!Thread.currentThread().isInterrupted()) {
                    try {
                        Object message = this.mMessages.take();
                        if (message == MESSAGE_DONE) break;
                        if (message instanceof String) {
                            this.mStdout.println(message);
                            continue;
                        }
                        if (message instanceof CopyException) {
                            CopyException e = (CopyException)message;
                            this.mStderr.println(this.messageAndCause(e));
                            continue;
                        }
                        LOG.error("Unsupported message type " + message.getClass() + " in message queue of copy thread pool");
                    }
                    catch (InterruptedException e) {
                        break;
                    }
                }
            });
            this.mPrinter.start();
            this.mFileSystem = fileSystem;
            this.mPath = path;
        }

        public <T> void submit(Callable<T> task) {
            this.mPool.submit(task);
        }

        public void succeed(AlluxioURI src, AlluxioURI dst) throws InterruptedException {
            this.mMessages.put(String.format(CpCommand.COPY_SUCCEED_MESSAGE, src, dst));
        }

        public void fail(AlluxioURI src, AlluxioURI dst, Exception cause) throws InterruptedException {
            CopyException exception = new CopyException(src, dst, cause);
            this.mExceptions.add(exception);
            this.mMessages.put(exception);
        }

        public void shutdown() throws IOException {
            this.mPool.shutdown();
            try {
                this.mPool.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                LOG.warn("Copy thread pool is interrupted in shutdown.", (Throwable)e);
                Thread.currentThread().interrupt();
                this.mPool.shutdownNow();
            }
            try {
                this.mMessages.put(MESSAGE_DONE);
                this.mPrinter.join();
            }
            catch (InterruptedException e) {
                LOG.warn("Message queue or printer in copy thread pool is interrupted in shutdown.", (Throwable)e);
                Thread.currentThread().interrupt();
                this.mPrinter.interrupt();
            }
            try {
                if (this.mPath != null && this.mFileSystem.exists(this.mPath) && this.mFileSystem.getStatus(this.mPath).isFolder() && this.mFileSystem.listStatus(this.mPath).isEmpty()) {
                    this.mFileSystem.delete(this.mPath);
                }
            }
            catch (Exception e) {
                this.mExceptions.add(new IOException("Failed to delete path " + this.mPath.toString(), e));
            }
            if (!this.mExceptions.isEmpty()) {
                ArrayList<String> errors = new ArrayList<String>();
                for (Exception e : this.mExceptions) {
                    LOG.error(this.stacktrace(e));
                    errors.add(this.messageAndCause(e));
                }
                throw new IOException("ERRORS:\n" + Joiner.on((String)"\n").join(errors));
            }
        }

        private String messageAndCause(Exception e) {
            return e.getMessage() + ": " + e.getCause().getMessage();
        }

        private String stacktrace(Exception e) {
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            PrintStream ps = new PrintStream(os, true);
            e.printStackTrace(ps);
            ps.close();
            return os.toString();
        }

        private static final class CopyException
        extends Exception {
            public CopyException(AlluxioURI src, AlluxioURI dst, Exception cause) {
                super(String.format(CpCommand.COPY_FAIL_MESSAGE, src, dst), cause);
            }
        }
    }
}

