//
//  ========================================================================
//  Copyright (c) 1995-2022 Mort Bay Consulting Pty Ltd and others.
//  ------------------------------------------------------------------------
//  All rights reserved. This program and the accompanying materials
//  are made available under the terms of the Eclipse Public License v1.0
//  and Apache License v2.0 which accompanies this distribution.
//
//      The Eclipse Public License is available at
//      http://www.eclipse.alluxio.shaded.client.org.legal/epl-v10.html
//
//      The Apache License v2.0 is available at
//      http://www.opensource.alluxio.shaded.client.org.licenses/apache2.0.php
//
//  You may elect to redistribute this code under either of these licenses.
//  ========================================================================
//

package alluxio.shaded.client.org.eclipse.jetty.server.handler;

import java.alluxio.shaded.client.io.IOException;
import java.alluxio.shaded.client.io.OutputStream;
import java.alluxio.shaded.client.io.PrintStream;
import java.util.Locale;
import alluxio.shaded.client.javax.servlet.ServletException;
import alluxio.shaded.client.javax.servlet.http.HttpServletRequest;
import alluxio.shaded.client.javax.servlet.http.HttpServletResponse;

import alluxio.shaded.client.org.eclipse.jetty.alluxio.shaded.client.io.Connection;
import alluxio.shaded.client.org.eclipse.jetty.server.AbstractConnector;
import alluxio.shaded.client.org.eclipse.jetty.server.Connector;
import alluxio.shaded.client.org.eclipse.jetty.server.DebugListener;
import alluxio.shaded.client.org.eclipse.jetty.server.Request;
import alluxio.shaded.client.org.eclipse.jetty.server.Response;
import alluxio.shaded.client.org.eclipse.jetty.util.DateCache;
import alluxio.shaded.client.org.eclipse.jetty.util.RolloverFileOutputStream;

/**
 * Debug Handler.
 * A lightweight debug handler that can be used in production code.
 * Details of the request and response are written to an output stream
 * and the current thread name is updated with information that will link
 * to the details in that output.
 *
 * @deprecated Use {@link DebugListener}
 */
public class DebugHandler extends HandlerWrapper implements Connection.Listener
{
    private DateCache _date = new DateCache("HH:mm:ss", Locale.US);
    private OutputStream _out;
    private PrintStream _print;

    /*
     * @see alluxio.shaded.client.org.eclipse.jetty.server.Handler#handle(java.lang.String, alluxio.shaded.client.javax.servlet.http.HttpServletRequest, alluxio.shaded.client.javax.servlet.http.HttpServletResponse, int)
     */
    @Override
    public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response)
        throws IOException, ServletException
    {
        final Response base_response = baseRequest.getResponse();
        final Thread thread = Thread.currentThread();
        final String old_name = thread.getName();

        boolean retry = false;
        String name = (String)request.getAttribute("alluxio.shaded.client.org.eclipse.jetty.thread.name");
        if (name == null)
            name = old_name + ":" + baseRequest.getHttpURI();
        else
            retry = true;

        String ex = null;
        try
        {
            if (retry)
                print(name, "RESUME");
            else
                print(name, "REQUEST " + baseRequest.getRemoteAddr() + " " + request.getMethod() + " " + baseRequest.getHeader("Cookie") + "; " + baseRequest.getHeader("User-Agent"));
            thread.setName(name);

            getHandler().handle(target, baseRequest, request, response);
        }
        catch (IOException alluxio.shaded.client.io.)
        {
            ex = alluxio.shaded.client.io..toString();
            throw alluxio.shaded.client.io.;
        }
        catch (ServletException cause)
        {
            ex = cause.toString() + ":" + cause.getCause();
            throw cause;
        }
        catch (RuntimeException rte)
        {
            ex = rte.toString();
            throw rte;
        }
        catch (Error e)
        {
            ex = e.toString();
            throw e;
        }
        finally
        {
            thread.setName(old_name);
            if (baseRequest.getHttpChannelState().isAsyncStarted())
            {
                request.setAttribute("alluxio.shaded.client.org.eclipse.jetty.thread.name", name);
                print(name, "ASYNC");
            }
            else
                print(name, "RESPONSE " + base_response.getStatus() + (ex == null ? "" : ("/" + ex)) + " " + base_response.getContentType());
        }
    }

    private void print(String name, String message)
    {
        long now = System.currentTimeMillis();
        final String d = _date.formatNow(now);
        final int ms = (int)(now % 1000);

        _print.println(d + (ms > 99 ? "." : (ms > 9 ? ".0" : ".00")) + ms + ":" + name + " " + message);
    }

    /* (non-Javadoc)
     * @see alluxio.shaded.client.org.eclipse.jetty.server.handler.HandlerWrapper#doStart()
     */
    @Override
    protected void doStart() throws Exception
    {
        if (_out == null)
            _out = new RolloverFileOutputStream("./logs/yyyy_mm_dd.debug.log", true);
        _print = new PrintStream(_out);

        for (Connector connector : getServer().getConnectors())
        {
            if (connector instanceof AbstractConnector)
                connector.addBean(this, false);
        }

        super.doStart();
    }

    /* (non-Javadoc)
     * @see alluxio.shaded.client.org.eclipse.jetty.server.handler.HandlerWrapper#doStop()
     */
    @Override
    protected void doStop() throws Exception
    {
        super.doStop();
        _print.close();
        for (Connector connector : getServer().getConnectors())
        {
            if (connector instanceof AbstractConnector)
                connector.removeBean(this);
        }
    }

    /**
     * @return the out
     */
    public OutputStream getOutputStream()
    {
        return _out;
    }

    /**
     * @param out the out to set
     */
    public void setOutputStream(OutputStream out)
    {
        _out = out;
    }

    @Override
    public void onOpened(Connection connection)
    {
        print(Thread.currentThread().getName(), "OPENED " + connection.toString());
    }

    @Override
    public void onClosed(Connection connection)
    {
        print(Thread.currentThread().getName(), "CLOSED " + connection.toString());
    }
}
