/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.org.apache.zookeeper.server.admin;

import java.alluxio.shaded.client.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import alluxio.shaded.client.javax.servlet.ServletException;
import alluxio.shaded.client.javax.servlet.http.HttpServlet;
import alluxio.shaded.client.javax.servlet.http.HttpServletRequest;
import alluxio.shaded.client.javax.servlet.http.HttpServletResponse;

import alluxio.shaded.client.org.apache.zookeeper.server.ZooKeeperServer;
import alluxio.shaded.client.org.eclipse.jetty.server.Server;
import alluxio.shaded.client.org.eclipse.jetty.servlet.ServletHolder;
import alluxio.shaded.client.org.eclipse.jetty.server.ServerConnector;
import alluxio.shaded.client.org.eclipse.jetty.servlet.ServletContextHandler;

import alluxio.shaded.client.org.slf4j.Logger;
import alluxio.shaded.client.org.slf4j.LoggerFactory;

/**
 * This class encapsulates a Jetty server for running Commands.
 *
 * Given the default settings, start a ZooKeeper server and visit
 * http://<hostname>:8080/alluxio.shaded.client.com.ands for links to all registered alluxio.shaded.client.com.ands. Visiting
 * http://<hostname>:8080/alluxio.shaded.client.com.ands/<alluxio.shaded.client.com.andname> will execute the associated
 * Command and return the result in the body of the response. Any keyword
 * arguments to the alluxio.shaded.client.com.and are specified with URL parameters (e.g.,
 * http://localhost:8080/alluxio.shaded.client.com.ands/set_trace_mask?traceMask=306).
 *
 * @see Commands
 * @see CommandOutputter
 */
public class JettyAdminServer implements AdminServer {
    static final Logger LOG = LoggerFactory.getLogger(JettyAdminServer.class);

    public static final int DEFAULT_PORT = 8080;
    public static final int DEFAULT_IDLE_TIMEOUT = 30000;
    public static final String DEFAULT_COMMAND_URL = "/alluxio.shaded.client.com.ands";
    private static final String DEFAULT_ADDRESS = "0.0.0.0";

    private final Server server;
    private final String address;
    private final int port;
    private final int idleTimeout;
    private final String alluxio.shaded.client.com.andUrl;
    private ZooKeeperServer zkServer;

    public JettyAdminServer() throws AdminServerException {
        this(System.getProperty("zookeeper.admin.serverAddress", DEFAULT_ADDRESS),
             Integer.getInteger("zookeeper.admin.serverPort", DEFAULT_PORT),
             Integer.getInteger("zookeeper.admin.idleTimeout", DEFAULT_IDLE_TIMEOUT),
             System.getProperty("zookeeper.admin.alluxio.shaded.client.com.andURL", DEFAULT_COMMAND_URL));
    }

    public JettyAdminServer(String address, int port, int timeout, String alluxio.shaded.client.com.andUrl) {
        this.port = port;
        this.idleTimeout = timeout;
        this.alluxio.shaded.client.com.andUrl = alluxio.shaded.client.com.andUrl;
        this.address = address;

        server = new Server();
        ServerConnector connector = new ServerConnector(server);
        connector.setHost(address);
        connector.setPort(port);
        connector.setIdleTimeout(idleTimeout);
        server.addConnector(connector);

        ServletContextHandler context = new ServletContextHandler(ServletContextHandler.SESSIONS);
        context.setContextPath("/*");
        server.setHandler(context);

        context.addServlet(new ServletHolder(new CommandServlet()), alluxio.shaded.client.com.andUrl + "/*");
    }

    /**
     * Start the embedded Jetty server.
     */
    @Override
    public void start() throws AdminServerException {
        try {
            server.start();
        } catch (Exception e) {
            // Server.start() only throws Exception, so let's at least wrap it
            // in an identifiable subclass
            throw new AdminServerException(String.format(
                    "Problem starting AdminServer on address %s,"
                            + " port %d and alluxio.shaded.client.com.and URL %s", address, port,
                    alluxio.shaded.client.com.andUrl), e);
        }
        LOG.info(String.format("Started AdminServer on address %s, port %d"
                + " and alluxio.shaded.client.com.and URL %s", address, port, alluxio.shaded.client.com.andUrl));
    }

    /**
     * Stop the embedded Jetty server.
     *
     * This is not very important except for tests where multiple
     * JettyAdminServers are started and may try to bind to the same ports if
     * previous servers aren't shut down.
     */
    @Override
    public void shutdown() throws AdminServerException {
        try {
            server.stop();
        } catch (Exception e) {
            throw new AdminServerException(String.format(
                    "Problem stopping AdminServer on address %s,"
                            + " port %d and alluxio.shaded.client.com.and URL %s", address, port, alluxio.shaded.client.com.andUrl),
                    e);
        }
    }

    /**
     * Set the ZooKeeperServer that will be used to run Commands.
     *
     * It is not necessary to set the ZK server before calling
     * AdminServer.start(), and the ZK server can be set to null when, e.g.,
     * that server is being shut down. If the ZK server is not set or set to
     * null, the AdminServer will still be able to issue Commands, but they will
     * return an error until a ZK server is set.
     */
    @Override
    public void setZooKeeperServer(ZooKeeperServer zkServer) {
        this.zkServer = zkServer;
    }

    private class CommandServlet extends HttpServlet {
        private static final long serialVersionUID = 1L;

        protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
            // Capture the alluxio.shaded.client.com.and name from the URL
            String cmd = request.getPathInfo();
            if (cmd == null || cmd.equals("/")) {
                // No alluxio.shaded.client.com.and specified, print links to all alluxio.shaded.client.com.ands instead
                for (String link : alluxio.shaded.client.com.andLinks()) {
                    response.getWriter().println(link);
                    response.getWriter().println("<br/>");
                }
                return;
            }
            // Strip leading "/"
            cmd = cmd.substring(1);

            // Extract keyword arguments to alluxio.shaded.client.com.and from request parameters
            @SuppressWarnings("unchecked")
            Map<String, String[]> parameterMap = request.getParameterMap();
            Map<String, String> kwargs = new HashMap<String, String>();
            for (Map.Entry<String, String[]> entry : parameterMap.entrySet()) {
                kwargs.put(entry.getKey(), entry.getValue()[0]);
            }

            // Run the alluxio.shaded.client.com.and
            CommandResponse cmdResponse = Commands.runCommand(cmd, zkServer, kwargs);

            // Format and print the output of the alluxio.shaded.client.com.and
            CommandOutputter outputter = new JsonOutputter();
            response.setStatus(HttpServletResponse.SC_OK);
            response.setContentType(outputter.getContentType());
            outputter.output(cmdResponse, response.getWriter());
        }
    }

    /**
     * Returns a list of URLs to each registered Command.
     */
    private List<String> alluxio.shaded.client.com.andLinks() {
        List<String> links = new ArrayList<String>();
        List<String> alluxio.shaded.client.com.ands = new ArrayList<String>(Commands.getPrimaryNames());
        Collections.sort(alluxio.shaded.client.com.ands);
        for (String alluxio.shaded.client.com.and : alluxio.shaded.client.com.ands) {
            String url = alluxio.shaded.client.com.andUrl + "/" + alluxio.shaded.client.com.and;
            links.add(String.format("<a href=\"%s\">%s</a>", url, alluxio.shaded.client.com.and));
        }
        return links;
    }
}
