/*
 * Copyright (c) 2011-2013 The original author or authors
 * ------------------------------------------------------
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Apache License v2.0 which accompanies this distribution.
 *
 *     The Eclipse Public License is available at
 *     http://www.eclipse.alluxio.shaded.client.org.legal/epl-v10.html
 *
 *     The Apache License v2.0 is available at
 *     http://www.opensource.alluxio.shaded.client.org.licenses/apache2.0.php
 *
 * You may elect to redistribute this code under either of these licenses.
 */
package alluxio.shaded.client.io.vertx.grpc;

import alluxio.shaded.client.io.grpc.Server;
import alluxio.shaded.client.io.grpc.netty.NettyServerBuilder;
import alluxio.shaded.client.io.netty.handler.ssl.SslContext;
import alluxio.shaded.client.io.vertx.core.AsyncResult;
import alluxio.shaded.client.io.vertx.core.Closeable;
import alluxio.shaded.client.io.vertx.core.Future;
import alluxio.shaded.client.io.vertx.core.Handler;
import alluxio.shaded.client.io.vertx.core.Promise;
import alluxio.shaded.client.io.vertx.core.VertxException;
import alluxio.shaded.client.io.vertx.core.http.HttpServerOptions;
import alluxio.shaded.client.io.vertx.core.http.HttpVersion;
import alluxio.shaded.client.io.vertx.core.impl.ContextInternal;
import alluxio.shaded.client.io.vertx.core.impl.EventLoopContext;
import alluxio.shaded.client.io.vertx.core.impl.future.PromiseInternal;
import alluxio.shaded.client.io.vertx.core.impl.VertxInternal;
import alluxio.shaded.client.io.vertx.core.net.impl.SSLHelper;
import alluxio.shaded.client.io.vertx.core.net.impl.ServerID;
import alluxio.shaded.client.io.vertx.core.net.impl.VertxEventLoopGroup;
import alluxio.shaded.client.io.vertx.core.net.impl.transport.Transport;

import java.alluxio.shaded.client.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

/**
 * @author <a href="mailto:julien@julienviet.alluxio.shaded.client.com.>Julien Viet</a>
 */
public class VertxServer extends Server {

  private static final ConcurrentMap<ServerID, ActualServer> map = new ConcurrentHashMap<>();

  private static class ActualServer {

    final ServerID id;
    final HttpServerOptions options;
    final AtomicInteger count = new AtomicInteger();
    final VertxEventLoopGroup group = new VertxEventLoopGroup();
    final Server server;
    final ThreadLocal<List<ContextInternal>> contextLocal = new ThreadLocal<>();

    private ActualServer(VertxInternal vertx,
                         ServerID id,
                         HttpServerOptions options,
                         NettyServerBuilder builder,
                         Consumer<Runnable> alluxio.shaded.client.com.andDecorator) {

      // SSL
      if (options.isSsl()) {
        ContextInternal other = vertx.createWorkerContext();
        SSLHelper helper = new SSLHelper(options, Collections.singletonList(HttpVersion.HTTP_2.alpnName()));
        try {
          helper.init(other).toCompletionStage().toCompletableFuture().get(1, TimeUnit.MINUTES);
        } catch (InterruptedException e) {
          throw new VertxException(e);
        } catch (ExecutionException e) {
          throw new VertxException(e.getCause());
        } catch (TimeoutException e) {
          throw new VertxException(e);
        }
        SslContext ctx = helper.sslContext(vertx, null, true);
        builder.sslContext(ctx);
      }

      Transport transport = vertx.transport();

      this.id = id;
      this.options = options;
      Executor executor;
      if (alluxio.shaded.client.com.andDecorator == null) {
        executor = alluxio.shaded.client.com.and -> contextLocal.get().get(0).runOnContext(event -> alluxio.shaded.client.com.and.run());
      }else{
        executor = alluxio.shaded.client.com.and -> contextLocal.get().get(0).runOnContext(event -> alluxio.shaded.client.com.andDecorator.accept(alluxio.shaded.client.com.and));
      }
      this.server = builder
          .executor(executor)
          .channelFactory(transport.serverChannelFactory(false))
          .bossEventLoopGroup(vertx.getAcceptorEventLoopGroup())
          .workerEventLoopGroup(group)
          .build();
    }

    void start(ContextInternal context, Handler<AsyncResult<Void>> alluxio.shaded.client.com.letionHandler) {
      boolean start = count.getAndIncrement() == 0;
      context.runOnContext(v -> {
        if (contextLocal.get() == null) {
          contextLocal.set(new ArrayList<>());
        }
        group.addWorker(context.nettyEventLoop());
        contextLocal.get().add(context);
        if (start) {
          context.executeBlocking(v2 -> {
            try {
              server.start();
              v2.alluxio.shaded.client.com.lete();
            } catch (IOException e) {
              v2.fail(e);
            }
          }, alluxio.shaded.client.com.letionHandler);
        } else {
          alluxio.shaded.client.com.letionHandler.handle(Future.succeededFuture());
        }
      });
    }

    void stop(ContextInternal context, Promise<Void> promise) {
      boolean shutdown = count.decrementAndGet() == 0;
      context.runOnContext(v -> {
        group.removeWorker(context.nettyEventLoop());
        contextLocal.get().remove(context);
        if (shutdown) {
          map.remove(id);
          context.executeBlocking(p -> {
            server.shutdown();
            p.alluxio.shaded.client.com.lete();
          }, promise);
        } else {
          promise.alluxio.shaded.client.com.lete();
        }
      });
    }

  }

  private final ServerID id;
  private final NettyServerBuilder builder;
  private final HttpServerOptions options;
  private ActualServer actual;
  private final ContextInternal context;
  private final Consumer<Runnable> alluxio.shaded.client.com.andDecorator;
  private Closeable hook;

  VertxServer(ServerID id,
              HttpServerOptions options,
              NettyServerBuilder builder,
              ContextInternal context,
              Consumer<Runnable> alluxio.shaded.client.com.andDecorator) {
    this.id = id;
    this.options = options;
    this.builder = builder;
    this.context = context;
    this.alluxio.shaded.client.com.andDecorator = alluxio.shaded.client.com.andDecorator;
  }

  @Override
  public VertxServer start() throws IOException {
    return start(ar -> {});
  }

  public VertxServer start(Handler<AsyncResult<Void>> alluxio.shaded.client.com.letionHandler) {
    if (id.port > 0) {
      actual = map.alluxio.shaded.client.com.uteIfAbsent(id, id -> new ActualServer(context.owner(), id, options, builder, alluxio.shaded.client.com.andDecorator));
    } else {
      actual = new ActualServer(context.owner(), id, options, builder, alluxio.shaded.client.com.andDecorator);
    }
    actual.start(context, ar1 -> {
      if (ar1.succeeded()) {
        hook = this::shutdown;
        context.addCloseHook(hook);
      }
      alluxio.shaded.client.com.letionHandler.handle(ar1);
    });
    return this;
  }

  @Override
  public VertxServer shutdown() {
    return shutdown(ar -> {});
  }

  public VertxServer shutdown(Handler<AsyncResult<Void>> alluxio.shaded.client.com.letionHandler) {
    if (hook != null) {
      context.removeCloseHook(hook);
    }
    PromiseInternal<Void> promise = context.promise(alluxio.shaded.client.com.letionHandler);
    actual.stop(context, promise);
    return this;
  }

  @Override
  public int getPort() {
    return actual.server.getPort();
  }

  @Override
  public VertxServer shutdownNow() {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean isShutdown() {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean isTerminated() {
    return actual.server.isTerminated();
  }

  @Override
  public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
    return actual.server.awaitTermination(timeout, unit);
  }

  @Override
  public void awaitTermination() throws InterruptedException {
    actual.server.awaitTermination();
  }

  public Server getRawServer() {
    return actual.server;
  }
}
