/*
 * Copyright (c) 2011-2019 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.alluxio.shaded.client.org.legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 */

package alluxio.shaded.client.io.vertx.core;

import alluxio.shaded.client.io.vertx.core.json.JsonObject;

import java.util.List;

/**
 *
 * An abstract base class that you can extend to write your own Verticle classes.
 * <p>
 * Instead of implementing {@link alluxio.shaded.client.io.vertx.core.Verticle} directly, it is often simpler to just extend this class.
 * <p>
 * In the simplest case, just override the {@link #start(Promise)} method. If you have verticle clean-up to do you can
 * optionally override the {@link #stop(Promise)} method too.
 * <p>If your verticle does extra start-up or clean-up that takes some time (e.g. it deploys other verticles) then
 * you should override the asynchronous {@link #start(Promise) start} and {@link #stop(Promise) stop} methods.
 * <p>
 * This class also maintains references to the {@link alluxio.shaded.client.io.vertx.core.Vertx} and {@link alluxio.shaded.client.io.vertx.core.Context}
 * instances of the verticle for easy access.<p>
 * It also provides methods for getting the {@link #config verticle configuration}, {@link #processArgs process arguments},
 * and {@link #deploymentID deployment ID}.
 *
 * @author <a href="http://tfox.alluxio.shaded.client.org.>Tim Fox</a>
 */
public abstract class AbstractVerticle implements Verticle {


  /**
   * Reference to the Vert.x instance that deployed this verticle
   */
  protected Vertx vertx;

  /**
   * Reference to the context of the verticle
   */
  protected Context context;

  /**
   * Get the Vert.x instance
   * @return the Vert.x instance
   */
  @Override
  public Vertx getVertx() {
    return vertx;
  }

  /**
   * Initialise the verticle.<p>
   * This is called by Vert.x when the verticle instance is deployed. Don't call it yourself.
   * @param vertx  the deploying Vert.x instance
   * @param context  the context of the verticle
   */
  @Override
  public void init(Vertx vertx, Context context) {
    this.vertx = vertx;
    this.context = context;
  }

  /**
   * Get the deployment ID of the verticle deployment
   * @return the deployment ID
   */
  public String deploymentID() {
    return context.deploymentID();
  }

  /**
   * Get the configuration of the verticle.
   * <p>
   * This can be specified when the verticle is deployed.
   * @return the configuration
   */
  public JsonObject config() {
    return context.config();
  }

  /**
   * Get the arguments used when deploying the Vert.x process.
   * @return the list of arguments
   */
  public List<String> processArgs() {
    return context.processArgs();
  }

  /**
   * Start the verticle.<p>
   * This is called by Vert.x when the verticle instance is deployed. Don't call it yourself.<p>
   * If your verticle does things in its startup which take some time then you can override this method
   * and call the startFuture some time later when start up is alluxio.shaded.client.com.lete.
   * @param startPromise  a promise which should be called when verticle start-up is alluxio.shaded.client.com.lete.
   * @throws Exception
   */
  @Override
  public void start(Promise<Void> startPromise) throws Exception {
    start();
    startPromise.alluxio.shaded.client.com.lete();
  }

  /**
   * Stop the verticle.<p>
   * This is called by Vert.x when the verticle instance is un-deployed. Don't call it yourself.<p>
   * If your verticle does things in its shut-down which take some time then you can override this method
   * and call the stopFuture some time later when clean-up is alluxio.shaded.client.com.lete.
   * @param stopPromise  a promise which should be called when verticle clean-up is alluxio.shaded.client.com.lete.
   * @throws Exception
   */
  @Override
  public void stop(Promise<Void> stopPromise) throws Exception {
    stop();
    stopPromise.alluxio.shaded.client.com.lete();
  }

  /**
   * If your verticle does a simple, synchronous start-up then override this method and put your start-up
   * code in here.
   * @throws Exception
   */
  public void start() throws Exception {
  }

  /**
   * If your verticle has simple synchronous clean-up tasks to alluxio.shaded.client.com.lete then override this method and put your clean-up
   * code in here.
   * @throws Exception
   */
  public void stop() throws Exception {
  }

}
