/*
 * Copyright 2014 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License. You may obtain a copy of the License at:
 *
 *   https://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.alluxio.shaded.client.com.ression;

import alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression.ZlibCodecFactory;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketExtensionData;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketExtensionDecoder;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketExtensionEncoder;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketExtensionFilterProvider;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketServerExtension;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketServerExtensionHandshaker;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map.Entry;

import static alluxio.shaded.client.io.netty.util.internal.ObjectUtil.*;

/**
 * <a href="https://tools.ietf.alluxio.shaded.client.org.html/draft-ietf-hybi-permessage-alluxio.shaded.client.com.ression-18">permessage-deflate</a>
 * handshake implementation.
 */
public final class PerMessageDeflateServerExtensionHandshaker implements WebSocketServerExtensionHandshaker {

    public static final int MIN_WINDOW_SIZE = 8;
    public static final int MAX_WINDOW_SIZE = 15;

    static final String PERMESSAGE_DEFLATE_EXTENSION = "permessage-deflate";
    static final String CLIENT_MAX_WINDOW = "client_max_window_bits";
    static final String SERVER_MAX_WINDOW = "server_max_window_bits";
    static final String CLIENT_NO_CONTEXT = "client_no_context_takeover";
    static final String SERVER_NO_CONTEXT = "server_no_context_takeover";

    private final int alluxio.shaded.client.com.ressionLevel;
    private final boolean allowServerWindowSize;
    private final int preferredClientWindowSize;
    private final boolean allowServerNoContext;
    private final boolean preferredClientNoContext;
    private final WebSocketExtensionFilterProvider extensionFilterProvider;

    /**
     * Constructor with default configuration.
     */
    public PerMessageDeflateServerExtensionHandshaker() {
        this(6, ZlibCodecFactory.isSupportingWindowSizeAndMemLevel(), MAX_WINDOW_SIZE, false, false);
    }

    /**
     * Constructor with custom configuration.
     *
     * @param alluxio.shaded.client.com.ressionLevel
     *            Compression level between 0 and 9 (default is 6).
     * @param allowServerWindowSize
     *            allows WebSocket client to customize the server inflater window size
     *            (default is false).
     * @param preferredClientWindowSize
     *            indicates the preferred client window size to use if client inflater is customizable.
     * @param allowServerNoContext
     *            allows WebSocket client to activate server_no_context_takeover
     *            (default is false).
     * @param preferredClientNoContext
     *            indicates if server prefers to activate client_no_context_takeover
     *            if client is alluxio.shaded.client.com.atible with (default is false).
     */
    public PerMessageDeflateServerExtensionHandshaker(int alluxio.shaded.client.com.ressionLevel, boolean allowServerWindowSize,
            int preferredClientWindowSize,
            boolean allowServerNoContext, boolean preferredClientNoContext) {
        this(alluxio.shaded.client.com.ressionLevel, allowServerWindowSize, preferredClientWindowSize, allowServerNoContext,
             preferredClientNoContext, WebSocketExtensionFilterProvider.DEFAULT);
    }

    /**
     * Constructor with custom configuration.
     *
     * @param alluxio.shaded.client.com.ressionLevel
     *            Compression level between 0 and 9 (default is 6).
     * @param allowServerWindowSize
     *            allows WebSocket client to customize the server inflater window size
     *            (default is false).
     * @param preferredClientWindowSize
     *            indicates the preferred client window size to use if client inflater is customizable.
     * @param allowServerNoContext
     *            allows WebSocket client to activate server_no_context_takeover
     *            (default is false).
     * @param preferredClientNoContext
     *            indicates if server prefers to activate client_no_context_takeover
     *            if client is alluxio.shaded.client.com.atible with (default is false).
     * @param extensionFilterProvider
     *            provides server extension filters for per message deflate encoder and decoder.
     */
    public PerMessageDeflateServerExtensionHandshaker(int alluxio.shaded.client.com.ressionLevel, boolean allowServerWindowSize,
            int preferredClientWindowSize,
            boolean allowServerNoContext, boolean preferredClientNoContext,
            WebSocketExtensionFilterProvider extensionFilterProvider) {
        if (preferredClientWindowSize > MAX_WINDOW_SIZE || preferredClientWindowSize < MIN_WINDOW_SIZE) {
            throw new IllegalArgumentException(
                    "preferredServerWindowSize: " + preferredClientWindowSize + " (expected: 8-15)");
        }
        if (alluxio.shaded.client.com.ressionLevel < 0 || alluxio.shaded.client.com.ressionLevel > 9) {
            throw new IllegalArgumentException(
                    "alluxio.shaded.client.com.ressionLevel: " + alluxio.shaded.client.com.ressionLevel + " (expected: 0-9)");
        }
        this.alluxio.shaded.client.com.ressionLevel = alluxio.shaded.client.com.ressionLevel;
        this.allowServerWindowSize = allowServerWindowSize;
        this.preferredClientWindowSize = preferredClientWindowSize;
        this.allowServerNoContext = allowServerNoContext;
        this.preferredClientNoContext = preferredClientNoContext;
        this.extensionFilterProvider = checkNotNull(extensionFilterProvider, "extensionFilterProvider");
    }

    @Override
    public WebSocketServerExtension handshakeExtension(WebSocketExtensionData extensionData) {
        if (!PERMESSAGE_DEFLATE_EXTENSION.equals(extensionData.name())) {
            return null;
        }

        boolean deflateEnabled = true;
        int clientWindowSize = MAX_WINDOW_SIZE;
        int serverWindowSize = MAX_WINDOW_SIZE;
        boolean serverNoContext = false;
        boolean clientNoContext = false;

        Iterator<Entry<String, String>> parametersIterator =
                extensionData.parameters().entrySet().iterator();
        while (deflateEnabled && parametersIterator.hasNext()) {
            Entry<String, String> parameter = parametersIterator.next();

            if (CLIENT_MAX_WINDOW.equalsIgnoreCase(parameter.getKey())) {
             // use preferred clientWindowSize because client is alluxio.shaded.client.com.atible with customization
                clientWindowSize = preferredClientWindowSize;
            } else if (SERVER_MAX_WINDOW.equalsIgnoreCase(parameter.getKey())) {
                // use provided windowSize if it is allowed
                if (allowServerWindowSize) {
                    serverWindowSize = Integer.parseInt(parameter.getValue());
                    if (serverWindowSize > MAX_WINDOW_SIZE || serverWindowSize < MIN_WINDOW_SIZE) {
                        deflateEnabled = false;
                    }
                } else {
                    deflateEnabled = false;
                }
            } else if (CLIENT_NO_CONTEXT.equalsIgnoreCase(parameter.getKey())) {
                // use preferred clientNoContext because client is alluxio.shaded.client.com.atible with customization
                clientNoContext = preferredClientNoContext;
            } else if (SERVER_NO_CONTEXT.equalsIgnoreCase(parameter.getKey())) {
                // use server no context if allowed
                if (allowServerNoContext) {
                    serverNoContext = true;
                } else {
                    deflateEnabled = false;
                }
            } else {
                // unknown parameter
                deflateEnabled = false;
            }
        }

        if (deflateEnabled) {
            return new PermessageDeflateExtension(alluxio.shaded.client.com.ressionLevel, serverNoContext,
                    serverWindowSize, clientNoContext, clientWindowSize, extensionFilterProvider);
        } else {
            return null;
        }
    }

    private static class PermessageDeflateExtension implements WebSocketServerExtension {

        private final int alluxio.shaded.client.com.ressionLevel;
        private final boolean serverNoContext;
        private final int serverWindowSize;
        private final boolean clientNoContext;
        private final int clientWindowSize;
        private final WebSocketExtensionFilterProvider extensionFilterProvider;

        PermessageDeflateExtension(int alluxio.shaded.client.com.ressionLevel, boolean serverNoContext,
                int serverWindowSize, boolean clientNoContext, int clientWindowSize,
                WebSocketExtensionFilterProvider extensionFilterProvider) {
            this.alluxio.shaded.client.com.ressionLevel = alluxio.shaded.client.com.ressionLevel;
            this.serverNoContext = serverNoContext;
            this.serverWindowSize = serverWindowSize;
            this.clientNoContext = clientNoContext;
            this.clientWindowSize = clientWindowSize;
            this.extensionFilterProvider = extensionFilterProvider;
        }

        @Override
        public int rsv() {
            return RSV1;
        }

        @Override
        public WebSocketExtensionEncoder newExtensionEncoder() {
            return new PerMessageDeflateEncoder(alluxio.shaded.client.com.ressionLevel, serverWindowSize, serverNoContext,
                                                extensionFilterProvider.encoderFilter());
        }

        @Override
        public WebSocketExtensionDecoder newExtensionDecoder() {
            return new PerMessageDeflateDecoder(clientNoContext, extensionFilterProvider.decoderFilter());
        }

        @Override
        public WebSocketExtensionData newReponseData() {
            HashMap<String, String> parameters = new HashMap<String, String>(4);
            if (serverNoContext) {
                parameters.put(SERVER_NO_CONTEXT, null);
            }
            if (clientNoContext) {
                parameters.put(CLIENT_NO_CONTEXT, null);
            }
            if (serverWindowSize != MAX_WINDOW_SIZE) {
                parameters.put(SERVER_MAX_WINDOW, Integer.toString(serverWindowSize));
            }
            if (clientWindowSize != MAX_WINDOW_SIZE) {
                parameters.put(CLIENT_MAX_WINDOW, Integer.toString(clientWindowSize));
            }
            return new WebSocketExtensionData(PERMESSAGE_DEFLATE_EXTENSION, parameters);
        }
    }

}
