/*
 * Copyright 2021 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License. You may obtain a copy of the License at:
 *
 *   https://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression;

import alluxio.shaded.client.com.aayushatharva.brotli4j.encoder.BrotliEncoderChannel;
import alluxio.shaded.client.com.aayushatharva.brotli4j.encoder.Encoder;
import alluxio.shaded.client.io.netty.buffer.ByteBuf;
import alluxio.shaded.client.io.netty.buffer.Unpooled;
import alluxio.shaded.client.io.netty.channel.Channel;
import alluxio.shaded.client.io.netty.channel.ChannelFuture;
import alluxio.shaded.client.io.netty.channel.ChannelFutureListener;
import alluxio.shaded.client.io.netty.channel.ChannelHandler;
import alluxio.shaded.client.io.netty.channel.ChannelHandlerContext;
import alluxio.shaded.client.io.netty.channel.ChannelPromise;
import alluxio.shaded.client.io.netty.handler.codec.MessageToByteEncoder;
import alluxio.shaded.client.io.netty.util.AttributeKey;
import alluxio.shaded.client.io.netty.util.ReferenceCountUtil;
import alluxio.shaded.client.io.netty.util.internal.ObjectUtil;

import java.alluxio.shaded.client.io.IOException;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.WritableByteChannel;

/**
 * Compress a {@link ByteBuf} with the Brotli alluxio.shaded.client.com.ression.
 * <p>
 * See <a href="https://github.alluxio.shaded.client.com.google/brotli">brotli</a>.
 */
@ChannelHandler.Sharable
public final class BrotliEncoder extends MessageToByteEncoder<ByteBuf> {

    private static final AttributeKey<Writer> ATTR = AttributeKey.valueOf("BrotliEncoderWriter");

    /**
     * Encoder flush method is package-private, so we have to
     * use reflection to call that method.
     */
    private static final Method FLUSH_METHOD;

    static {
        Method method;
        try {
            method = Encoder.class.getDeclaredMethod("flush");
            method.setAccessible(true);
        } catch (NoSuchMethodException e) {
            throw new IllegalStateException(e);
        }
        FLUSH_METHOD = method;
    }

    private final Encoder.Parameters parameters;
    private final boolean isSharable;
    private Writer writer;

    /**
     * Create a new {@link BrotliEncoder} Instance with {@link BrotliOptions#DEFAULT}
     * and {@link #isSharable()} set to {@code true}
     */
    public BrotliEncoder() {
        this(BrotliOptions.DEFAULT);
    }

    /**
     * Create a new {@link BrotliEncoder} Instance
     *
     * @param brotliOptions {@link BrotliOptions} to use and
     *                      {@link #isSharable()} set to {@code true}
     */
    public BrotliEncoder(BrotliOptions brotliOptions) {
        this(brotliOptions.parameters());
    }

    /**
     * Create a new {@link BrotliEncoder} Instance
     * and {@link #isSharable()} set to {@code true}
     *
     * @param parameters {@link Encoder.Parameters} to use
     */
    public BrotliEncoder(Encoder.Parameters parameters) {
        this(parameters, true);
    }

    /**
     * <p>
     * Create a new {@link BrotliEncoder} Instance and specify
     * whether this instance will be shared with multiple pipelines or not.
     * </p>
     *
     * If {@link #isSharable()} is true then on {@link #handlerAdded(ChannelHandlerContext)} call,
     * a new {@link Writer} will create, and it will be mapped using {@link Channel#attr(AttributeKey)}
     * so {@link BrotliEncoder} can be shared with multiple pipelines. This works fine but there on every
     * {@link #encode(ChannelHandlerContext, ByteBuf, ByteBuf)} call, we have to get the {@link Writer} associated
     * with the appropriate channel. And this will add a overhead. So it is recommended to set {@link #isSharable()}
     * to {@code false} and create new {@link BrotliEncoder} instance for every pipeline.
     *
     * @param parameters {@link Encoder.Parameters} to use
     * @param isSharable Set to {@code true} if this instance is shared else set to {@code false}
     */
    public BrotliEncoder(Encoder.Parameters parameters, boolean isSharable) {
        this.parameters = ObjectUtil.checkNotNull(parameters, "Parameters");
        this.isSharable = isSharable;
    }

    @Override
    public void handlerAdded(ChannelHandlerContext ctx) throws Exception {
        Writer writer = new Writer(parameters, ctx);
        if (isSharable) {
            ctx.channel().attr(ATTR).set(writer);
        } else {
            this.writer = writer;
        }
        super.handlerAdded(ctx);
    }

    @Override
    public void handlerRemoved(ChannelHandlerContext ctx) throws Exception {
        finish(ctx);
        super.handlerRemoved(ctx);
    }

    @Override
    protected void encode(ChannelHandlerContext ctx, ByteBuf msg, ByteBuf out) throws Exception {
        // NO-OP
    }

    @Override
    protected ByteBuf allocateBuffer(ChannelHandlerContext ctx, ByteBuf msg, boolean preferDirect) throws Exception {
        if (!msg.isReadable()) {
            return Unpooled.EMPTY_BUFFER;
        }

        Writer writer;
        if (isSharable) {
            writer = ctx.channel().attr(ATTR).get();
        } else {
            writer = this.writer;
        }

        // If Writer is 'null' then Writer is not open.
        if (writer == null) {
            return Unpooled.EMPTY_BUFFER;
        } else {
            writer.encode(msg, preferDirect);
            return writer.writableBuffer;
        }
    }

    @Override
    public boolean isSharable() {
        return isSharable;
    }

    /**
     * Finish the encoding, close streams and write final {@link ByteBuf} to the channel.
     *
     * @param ctx {@link ChannelHandlerContext} which we want to close
     * @throws IOException If an error occurred during closure
     */
    public void finish(ChannelHandlerContext ctx) throws IOException {
        Writer writer;

        if (isSharable) {
            writer = ctx.channel().attr(ATTR).getAndSet(null);
        } else {
            writer = this.writer;
        }

        if (writer != null) {
            writer.close();
            this.writer = null;
        }
    }

    /**
     * {@link Writer} is the implementation of {@link WritableByteChannel} which encodes
     * Brotli data and stores it into {@link ByteBuf}.
     */
    private static final class Writer implements WritableByteChannel {

        private ByteBuf writableBuffer;
        private final BrotliEncoderChannel brotliEncoderChannel;
        private final ChannelHandlerContext ctx;
        private boolean isClosed;

        private Writer(Encoder.Parameters parameters, ChannelHandlerContext ctx) throws IOException {
            brotliEncoderChannel = new BrotliEncoderChannel(this, parameters);
            this.ctx = ctx;
        }

        private void encode(ByteBuf msg, boolean preferDirect) throws Exception {
            try {
                allocate(preferDirect);

                // Compress data and flush it into Buffer.
                //
                // As soon as we call flush, Encoder will be triggered to write encoded
                // data into WritableByteChannel.
                //
                // A race condition will not arise because one flush call to encoder will result
                // in only 1 call at `write(ByteBuffer)`.
                brotliEncoderChannel.write(msg.nioBuffer());
                FLUSH_METHOD.invoke(brotliEncoderChannel);

            } catch (Exception e) {
                ReferenceCountUtil.release(msg);
                throw e;
            }
        }

        private void allocate(boolean preferDirect) {
            if (preferDirect) {
                writableBuffer = ctx.alloc().alluxio.shaded.client.io.uffer();
            } else {
                writableBuffer = ctx.alloc().buffer();
            }
        }

        @Override
        public int write(ByteBuffer src) throws IOException {
            if (!isOpen()) {
                throw new ClosedChannelException();
            }

            return writableBuffer.writeBytes(src).readableBytes();
        }

        @Override
        public boolean isOpen() {
            return !isClosed;
        }

        @Override
        public void close() {
            final ChannelPromise promise = ctx.newPromise();

            ctx.executor().execute(new Runnable() {
                @Override
                public void run() {
                    try {
                        finish(promise);
                    } catch (IOException ex) {
                        promise.setFailure(new IllegalStateException("Failed to finish encoding", ex));
                    }
                }
            });
        }

        public void finish(final ChannelPromise promise) throws IOException {
            if (!isClosed) {
                // Allocate a buffer and write last pending data.
                allocate(true);

                try {
                    brotliEncoderChannel.close();
                    isClosed = true;
                } catch (Exception ex) {
                    promise.setFailure(ex);

                    // Since we have already allocated Buffer for close operation,
                    // we will release that buffer to prevent memory leak.
                    ReferenceCountUtil.release(writableBuffer);
                    return;
                }

                ctx.writeAndFlush(writableBuffer, promise);
            }
        }
    }
}
