/*
 * Copyright 2016-2021 The jetcd authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.etcd.jetcd.impl;

import java.util.concurrent.CompletableFuture;

import alluxio.shaded.client.io.etcd.jetcd.ByteSequence;
import alluxio.shaded.client.io.etcd.jetcd.KV;
import alluxio.shaded.client.io.etcd.jetcd.Txn;
import alluxio.shaded.client.io.etcd.jetcd.api.CompactionRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.VertxKVGrpc;
import alluxio.shaded.client.io.etcd.jetcd.kv.CompactResponse;
import alluxio.shaded.client.io.etcd.jetcd.kv.DeleteResponse;
import alluxio.shaded.client.io.etcd.jetcd.kv.GetResponse;
import alluxio.shaded.client.io.etcd.jetcd.kv.PutResponse;
import alluxio.shaded.client.io.etcd.jetcd.kv.TxnResponse;
import alluxio.shaded.client.io.etcd.jetcd.op.TxnImpl;
import alluxio.shaded.client.io.etcd.jetcd.options.CompactOption;
import alluxio.shaded.client.io.etcd.jetcd.options.DeleteOption;
import alluxio.shaded.client.io.etcd.jetcd.options.GetOption;
import alluxio.shaded.client.io.etcd.jetcd.options.PutOption;
import alluxio.shaded.client.io.etcd.jetcd.support.Errors;
import alluxio.shaded.client.io.etcd.jetcd.support.Requests;

import static alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.base.Preconditions.checkNotNull;

/**
 * Implementation of etcd kv client.
 */
final class KVImpl extends Impl implements KV {
    private final VertxKVGrpc.KVVertxStub stub;
    private final ByteSequence namespace;

    KVImpl(ClientConnectionManager connectionManager) {
        super(connectionManager);

        this.stub = connectionManager.newStub(VertxKVGrpc::newVertxStub);
        this.namespace = connectionManager.getNamespace();
    }

    @Override
    public CompletableFuture<PutResponse> put(ByteSequence key, ByteSequence value) {
        return this.put(key, value, PutOption.DEFAULT);
    }

    @Override
    public CompletableFuture<PutResponse> put(ByteSequence key, ByteSequence value, PutOption option) {
        checkNotNull(key, "key should not be null");
        checkNotNull(value, "value should not be null");
        checkNotNull(option, "option should not be null");
        return execute(
            () -> stub.put(Requests.mapPutRequest(key, value, option, namespace)),
            response -> new PutResponse(response, namespace),
            Errors::isRetryable);
    }

    @Override
    public CompletableFuture<GetResponse> get(ByteSequence key) {
        return this.get(key, GetOption.DEFAULT);
    }

    @Override
    public CompletableFuture<GetResponse> get(ByteSequence key, GetOption option) {
        checkNotNull(key, "key should not be null");
        checkNotNull(option, "option should not be null");

        return execute(
            () -> stub.range(Requests.mapRangeRequest(key, option, namespace)),
            response -> new GetResponse(response, namespace),
            Errors::isRetryable);
    }

    @Override
    public CompletableFuture<DeleteResponse> delete(ByteSequence key) {
        return this.delete(key, DeleteOption.DEFAULT);
    }

    @Override
    public CompletableFuture<DeleteResponse> delete(ByteSequence key, DeleteOption option) {
        checkNotNull(key, "key should not be null");
        checkNotNull(option, "option should not be null");

        return execute(
            () -> stub.deleteRange(Requests.mapDeleteRequest(key, option, namespace)),
            response -> new DeleteResponse(response, namespace),
            Errors::isRetryable);
    }

    @Override
    public CompletableFuture<CompactResponse> alluxio.shaded.client.com.act(long rev) {
        return this.alluxio.shaded.client.com.act(rev, CompactOption.DEFAULT);
    }

    @Override
    public CompletableFuture<CompactResponse> alluxio.shaded.client.com.act(long rev, CompactOption option) {
        checkNotNull(option, "option should not be null");

        CompactionRequest request = CompactionRequest.newBuilder()
            .setRevision(rev).setPhysical(option.isPhysical())
            .build();

        return execute(
            () -> stub.alluxio.shaded.client.com.act(request),
            CompactResponse::new,
            Errors::isRetryable);
    }

    @Override
    public Txn txn() {
        return TxnImpl.newTxn(
            request -> execute(
                () -> stub.txn(request),
                response -> new TxnResponse(response, namespace), Errors::isRetryable),
            namespace);
    }
}
