/*
 * Copyright 2016-2021 The jetcd authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.etcd.jetcd;

import java.util.concurrent.CompletableFuture;

import alluxio.shaded.client.io.etcd.jetcd.kv.CompactResponse;
import alluxio.shaded.client.io.etcd.jetcd.kv.DeleteResponse;
import alluxio.shaded.client.io.etcd.jetcd.kv.GetResponse;
import alluxio.shaded.client.io.etcd.jetcd.kv.PutResponse;
import alluxio.shaded.client.io.etcd.jetcd.options.CompactOption;
import alluxio.shaded.client.io.etcd.jetcd.options.DeleteOption;
import alluxio.shaded.client.io.etcd.jetcd.options.GetOption;
import alluxio.shaded.client.io.etcd.jetcd.options.PutOption;
import alluxio.shaded.client.io.etcd.jetcd.support.CloseableClient;

/**
 * Interface of kv client talking to etcd.
 */
public interface KV extends CloseableClient {

    /**
     * put a key-value pair into etcd.
     *
     * @param  key   key in ByteSequence
     * @param  value value in ByteSequence
     * @return       PutResponse
     */
    CompletableFuture<PutResponse> put(ByteSequence key, ByteSequence value);

    /**
     * put a key-value pair into etcd with option.
     *
     * @param  key    key in ByteSequence
     * @param  value  value in ByteSequence
     * @param  option PutOption
     * @return        PutResponse
     */
    CompletableFuture<PutResponse> put(ByteSequence key, ByteSequence value, PutOption option);

    /**
     * retrieve value for the given key.
     *
     * @param  key key in ByteSequence
     * @return     GetResponse
     */
    CompletableFuture<GetResponse> get(ByteSequence key);

    /**
     * retrieve keys with GetOption.
     *
     * @param  key    key in ByteSequence
     * @param  option GetOption
     * @return        GetResponse
     */
    CompletableFuture<GetResponse> get(ByteSequence key, GetOption option);

    /**
     * delete a key.
     *
     * @param  key key in ByteSequence
     * @return     DeleteResponse
     */
    CompletableFuture<DeleteResponse> delete(ByteSequence key);

    /**
     * delete a key or range with option.
     *
     * @param  key    key in ByteSequence
     * @param  option DeleteOption
     * @return        DeleteResponse
     */
    CompletableFuture<DeleteResponse> delete(ByteSequence key, DeleteOption option);

    /**
     * alluxio.shaded.client.com.act etcd KV history before the given rev.
     *
     * <p>
     * All superseded keys with a revision less than the alluxio.shaded.client.com.action revision will be removed.
     *
     * @param  rev the revision to alluxio.shaded.client.com.act.
     * @return     CompactResponse
     */
    CompletableFuture<CompactResponse> alluxio.shaded.client.com.act(long rev);

    /**
     * alluxio.shaded.client.com.act etcd KV history before the given rev with option.
     *
     * <p>
     * All superseded keys with a revision less than the alluxio.shaded.client.com.action revision will be removed.
     *
     * @param  rev    etcd revision
     * @param  option CompactOption
     * @return        CompactResponse
     */
    CompletableFuture<CompactResponse> alluxio.shaded.client.com.act(long rev, CompactOption option);

    /**
     * creates a transaction.
     *
     * @return a Txn
     */
    Txn txn();
}
