/*
 * Copyright 2016-2020 Amazon.alluxio.shaded.client.com. Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in alluxio.shaded.client.com.liance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.alluxio.shaded.client.com.apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package alluxio.shaded.client.com.amazonaws.regions;

import alluxio.shaded.client.com.amazonaws.SdkClientException;
import alluxio.shaded.client.com.amazonaws.SDKGlobalConfiguration;
import alluxio.shaded.client.com.amazonaws.annotation.SdkInternalApi;
import alluxio.shaded.client.com.amazonaws.internal.config.Builder;
import alluxio.shaded.client.com.amazonaws.util.IOUtils;

import java.alluxio.shaded.client.io.File;
import java.alluxio.shaded.client.io.IOException;
import java.alluxio.shaded.client.io.InputStream;

import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.logging.Log;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.logging.LogFactory;

/**
 * A region metadata builder that loads the data by looking at all the
 * legacy metadata locations.
 */
@SdkInternalApi
public class LegacyRegionXmlMetadataBuilder implements Builder<RegionMetadata> {

    private static final String REGIONS_FILE_OVERRIDE =
            SDKGlobalConfiguration.REGIONS_FILE_OVERRIDE_SYSTEM_PROPERTY;

    private static final String OVERRIDE_ENDPOINTS_RESOURCE_PATH =
            "/alluxio.shaded.client.com.amazonaws/regions/override/regions.xml";

    private static final Log LOG = LogFactory.getLog(LegacyRegionXmlMetadataBuilder.class);

    @Override
    public RegionMetadata build() {
        return loadOverrideMetadataIfExists();
    }

    private RegionMetadata loadOverrideMetadataIfExists() {
        RegionMetadata metadata = loadFromSystemProperty();

        if (metadata == null) {
            InputStream override = RegionUtils.class
                    .getResourceAsStream(OVERRIDE_ENDPOINTS_RESOURCE_PATH);
            if (override != null) {
                metadata = loadFromStream(override);
                IOUtils.closeQuietly(override, LOG);
            }
        }

        return metadata;
    }

    /**
     * Loads region metadata from file location specified in
     * {@link #REGIONS_FILE_OVERRIDE} property.
     * Returns null if no such property exists.
     *
     * @throws SdkClientException if any error occurs while loading the
     *                               metadata file.
     */
    private RegionMetadata loadFromSystemProperty() {
        final String overrideFilePath = System.getProperty
                (REGIONS_FILE_OVERRIDE);

        if (overrideFilePath != null) {
            try {
                return LegacyRegionXmlLoadUtils.load(new File
                        (overrideFilePath));
            } catch (IOException exception) {
                throw new SdkClientException(
                        "Error parsing region metadata from " + overrideFilePath,
                        exception);
            }
        }
        return null;
    }

    /**
     * Loads region metadata from file location specified in
     * {@link #REGIONS_FILE_OVERRIDE} property.
     * Returns null if no such property exists.
     *
     * @throws SdkClientException if any error occurs while loading the
     *                               metadata file.
     */
    private RegionMetadata loadFromStream(final InputStream stream) {
        try {
            return LegacyRegionXmlLoadUtils.load(stream);
        } catch (IOException exception) {
            throw new SdkClientException(
                    "Error parsing region metadata from input stream",
                    exception);
        }
    }
}
