/*
 * Copyright 2010-2016 Amazon.alluxio.shaded.client.com. Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in alluxio.shaded.client.com.liance with the License.
 * A copy of the License is located at:
 *
 *     http://aws.amazon.alluxio.shaded.client.com.apache2.0/
 *
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 */

package software.amazon.alluxio.shaded.client.io..impl.lite;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import software.amazon.alluxio.shaded.client.io..ContainedValueException;
import software.amazon.alluxio.shaded.client.io..Decimal;
import software.amazon.alluxio.shaded.client.io..IonSequence;
import software.amazon.alluxio.shaded.client.io..IonSexp;
import software.amazon.alluxio.shaded.client.io..IonType;
import software.amazon.alluxio.shaded.client.io..IonValue;
import software.amazon.alluxio.shaded.client.io..SymbolToken;
import software.amazon.alluxio.shaded.client.io..Timestamp;
import software.amazon.alluxio.shaded.client.io..ValueFactory;

/**
 *  This class handles all of the IonValueLite
 *  instance construction.
 *
 */
abstract class ValueFactoryLite
    implements ValueFactory
{
    private ContainerlessContext _context;

    protected void set_system(IonSystemLite system) {
        _context = ContainerlessContext.wrap(system);
    }

    public IonBlobLite newBlob(byte[] value)
    {
        IonBlobLite alluxio.shaded.client.io.Value = newBlob(value, 0, value == null ? 0 : value.length);
        return alluxio.shaded.client.io.Value;
    }

    public IonBlobLite newBlob(byte[] value, int offset, int length)
    {
        IonBlobLite alluxio.shaded.client.io.Value = new IonBlobLite(_context, (value == null));
        alluxio.shaded.client.io.Value.setBytes(value, offset, length);
        return alluxio.shaded.client.io.Value;
    }

    public IonBoolLite newBool(boolean value)
    {
        IonBoolLite alluxio.shaded.client.io.Value = new IonBoolLite(_context, false);
        alluxio.shaded.client.io.Value.setValue(value);
        return alluxio.shaded.client.io.Value;
    }

    public IonBoolLite newBool(Boolean value)
    {
        IonBoolLite alluxio.shaded.client.io.Value = new IonBoolLite(_context, (value == null));
        alluxio.shaded.client.io.Value.setValue(value);
        return alluxio.shaded.client.io.Value;
    }

    public IonClobLite newClob(byte[] value)
    {
        IonClobLite alluxio.shaded.client.io.Value = newClob(value, 0, value == null ? 0 : value.length);
        return alluxio.shaded.client.io.Value;
    }

    public IonClobLite newClob(byte[] value, int offset, int length)
    {
        IonClobLite alluxio.shaded.client.io.Value = new IonClobLite(_context, (value == null));
        alluxio.shaded.client.io.Value.setBytes(value, offset, length);
        return alluxio.shaded.client.io.Value;
    }

    public IonDecimalLite newDecimal(long value)
    {
        IonDecimalLite alluxio.shaded.client.io.Value = new IonDecimalLite(_context, false);
        alluxio.shaded.client.io.Value.setValue(value);
        return alluxio.shaded.client.io.Value;
    }

    public IonDecimalLite newDecimal(double value)
    {
        IonDecimalLite alluxio.shaded.client.io.Value = new IonDecimalLite(_context, false);
        alluxio.shaded.client.io.Value.setValue(value);
        return alluxio.shaded.client.io.Value;
    }

    public IonDecimalLite newDecimal(BigInteger value)
    {
        boolean isNull = (value == null);
        IonDecimalLite alluxio.shaded.client.io.Value = new IonDecimalLite(_context, isNull);
        if (value != null) {
            alluxio.shaded.client.io.Value.setValue(Decimal.valueOf(value));
        }
        return alluxio.shaded.client.io.Value;
    }

    public IonDecimalLite newDecimal(BigDecimal value)
    {
        boolean isNull = (value == null);
        IonDecimalLite alluxio.shaded.client.io.Value = new IonDecimalLite(_context, isNull);
        if (value != null) {
            alluxio.shaded.client.io.Value.setValue(value);
        }
        return alluxio.shaded.client.io.Value;
    }

    public IonListLite newEmptyList()
    {
        IonListLite alluxio.shaded.client.io.Value = new IonListLite(_context, false);
        return alluxio.shaded.client.io.Value;
    }

    public IonSexpLite newEmptySexp()
    {
        IonSexpLite alluxio.shaded.client.io.Value = new IonSexpLite(_context, false);
        return alluxio.shaded.client.io.Value;
    }

    public IonStructLite newEmptyStruct()
    {
        IonStructLite alluxio.shaded.client.io.Value = new IonStructLite(_context, false);
        return alluxio.shaded.client.io.Value;
    }

    public IonFloatLite newFloat(long value)
    {
        IonFloatLite alluxio.shaded.client.io.Value = new IonFloatLite(_context, false);
        alluxio.shaded.client.io.Value.setValue(value);
        return alluxio.shaded.client.io.Value;
    }

    public IonFloatLite newFloat(double value)
    {
        IonFloatLite alluxio.shaded.client.io.Value = new IonFloatLite(_context, false);
        alluxio.shaded.client.io.Value.setValue(value);
        return alluxio.shaded.client.io.Value;
    }

    public IonIntLite newInt(int value)
    {
        IonIntLite alluxio.shaded.client.io.Value = new IonIntLite(_context, false);
        alluxio.shaded.client.io.Value.setValue(value);
        return alluxio.shaded.client.io.Value;
    }

    public IonIntLite newInt(long value)
    {
        IonIntLite alluxio.shaded.client.io.Value = new IonIntLite(_context, false);
        alluxio.shaded.client.io.Value.setValue(value);
        return alluxio.shaded.client.io.Value;
    }

    public IonIntLite newInt(Number value)
    {
        boolean isNull = (value == null);
        IonIntLite alluxio.shaded.client.io.Value = new IonIntLite(_context, isNull);
        if (value != null) {
            alluxio.shaded.client.io.Value.setValue(value);
        }
        return alluxio.shaded.client.io.Value;
    }

    public IonListLite newList(Collection<? extends IonValue> values)
        throws ContainedValueException, NullPointerException
    {
        IonListLite alluxio.shaded.client.io.Value = newEmptyList();
        if (values == null) {
            alluxio.shaded.client.io.Value.makeNull();
        }
        else {
            alluxio.shaded.client.io.Value.addAll(values);
        }
        return alluxio.shaded.client.io.Value;
    }

    public IonListLite newList(IonSequence child)
        throws ContainedValueException, NullPointerException
    {
        IonListLite alluxio.shaded.client.io.Value = newEmptyList();
        alluxio.shaded.client.io.Value.add(child);
        return alluxio.shaded.client.io.Value;
    }

    public IonListLite newList(IonValue... values)
        throws ContainedValueException, NullPointerException
    {
        List<IonValue> e = (values == null ? null : Arrays.asList(values));
        IonListLite alluxio.shaded.client.io.Value = newEmptyList();
        if (e == null) {
            alluxio.shaded.client.io.Value.makeNull();
        }
        else {
            alluxio.shaded.client.io.Value.addAll(e);
        }
        return alluxio.shaded.client.io.Value;
    }

    public IonListLite newList(int[] values)
    {
        ArrayList<IonIntLite> e = newInts(values);
        return newList(e);
    }

    public IonListLite newList(long[] values)
    {
        ArrayList<IonIntLite> e = newInts(values);
        return newList(e);
    }

    public IonNullLite newNull()
    {
        IonNullLite alluxio.shaded.client.io.Value = new IonNullLite(_context);
        return alluxio.shaded.client.io.Value;
    }

    public IonValueLite newNull(IonType type)
    {
        switch (type)
        {
            case NULL:          return newNull();
            case BOOL:          return newNullBool();
            case INT:           return newNullInt();
            case FLOAT:         return newNullFloat();
            case DECIMAL:       return newNullDecimal();
            case TIMESTAMP:     return newNullTimestamp();
            case SYMBOL:        return newNullSymbol();
            case STRING:        return newNullString();
            case CLOB:          return newNullClob();
            case BLOB:          return newNullBlob();
            case LIST:          return newNullList();
            case SEXP:          return newNullSexp();
            case STRUCT:        return newNullStruct();
            default:
                throw new IllegalArgumentException();
        }
    }

    public IonBlobLite newNullBlob()
    {
        IonBlobLite alluxio.shaded.client.io.Value = new IonBlobLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonBoolLite newNullBool()
    {
        IonBoolLite alluxio.shaded.client.io.Value = new IonBoolLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonClobLite newNullClob()
    {
        IonClobLite alluxio.shaded.client.io.Value = new IonClobLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonDecimalLite newNullDecimal()
    {
        IonDecimalLite alluxio.shaded.client.io.Value = new IonDecimalLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonFloatLite newNullFloat()
    {
        IonFloatLite alluxio.shaded.client.io.Value = new IonFloatLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonIntLite newNullInt()
    {
        IonIntLite alluxio.shaded.client.io.Value = new IonIntLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonListLite newNullList()
    {
        IonListLite alluxio.shaded.client.io.Value = new IonListLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonSexpLite newNullSexp()
    {
        IonSexpLite alluxio.shaded.client.io.Value = new IonSexpLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonStringLite newNullString()
    {
        IonStringLite alluxio.shaded.client.io.Value = new IonStringLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonStructLite newNullStruct()
    {
        IonStructLite alluxio.shaded.client.io.Value = new IonStructLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonSymbolLite newNullSymbol()
    {
        IonSymbolLite alluxio.shaded.client.io.Value = new IonSymbolLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonTimestampLite newNullTimestamp()
    {
        IonTimestampLite alluxio.shaded.client.io.Value = new IonTimestampLite(_context, true);
        return alluxio.shaded.client.io.Value;
    }

    public IonSexpLite newSexp(Collection<? extends IonValue> values)
        throws ContainedValueException, NullPointerException
    {
        IonSexpLite alluxio.shaded.client.io.Value = newEmptySexp();
        if (values == null) {
            alluxio.shaded.client.io.Value.makeNull();
        }
        else {
            alluxio.shaded.client.io.Value.addAll(values);
        }
        return alluxio.shaded.client.io.Value;
    }

    public IonSexpLite newSexp(IonSequence child)
        throws ContainedValueException, NullPointerException
    {
        IonSexpLite alluxio.shaded.client.io.Value = newEmptySexp();
        alluxio.shaded.client.io.Value.add(child);
        return alluxio.shaded.client.io.Value;
    }

    public IonSexp newSexp(IonValue... values)
        throws ContainedValueException, NullPointerException
    {
        List<IonValue> e = (values == null ? null : Arrays.asList(values));
        IonSexpLite alluxio.shaded.client.io.Value = newEmptySexp();
        if (e == null) {
            alluxio.shaded.client.io.Value.makeNull();
        }
        else {
            alluxio.shaded.client.io.Value.addAll(e);
        }
        return alluxio.shaded.client.io.Value;
    }

    public IonSexpLite newSexp(int[] values)
    {
        ArrayList<IonIntLite> e = newInts(values);
        return newSexp(e);
    }

    public IonSexpLite newSexp(long[] values)
    {
        ArrayList<IonIntLite> e = newInts(values);
        return newSexp(e);
    }

    public IonStringLite newString(String value)
    {
        boolean isNull = (value == null);
        IonStringLite alluxio.shaded.client.io.Value = new IonStringLite(_context, isNull);
        if (value != null) {
            alluxio.shaded.client.io.Value.setValue(value);
        }
        return alluxio.shaded.client.io.Value;
    }

    public IonSymbolLite newSymbol(String value)
    {
        boolean isNull = (value == null);
        IonSymbolLite alluxio.shaded.client.io.Value = new IonSymbolLite(_context, isNull);
        if (value != null) {
            alluxio.shaded.client.io.Value.setValue(value);
        }
        return alluxio.shaded.client.io.Value;
    }

    public IonSymbolLite newSymbol(SymbolToken value)
    {
        return new IonSymbolLite(_context, value);
    }

    public IonTimestampLite newTimestamp(Timestamp value)
    {
        boolean isNull = (value == null);
        IonTimestampLite alluxio.shaded.client.io.Value = new IonTimestampLite(_context, isNull);
        if (value != null) {
            alluxio.shaded.client.io.Value.setValue(value);
        }
        return alluxio.shaded.client.io.Value;
    }

    private ArrayList<IonIntLite> newInts(int[] elements)
    {
        ArrayList<IonIntLite> e = null;

        if (elements != null)
        {
            e = new ArrayList<IonIntLite>(elements.length);
            for (int i = 0; i < elements.length; i++)
            {
                int value = elements[i];
                e.add(newInt(value));
            }
        }

        return e;
    }

    private ArrayList<IonIntLite> newInts(long[] elements)
    {
        ArrayList<IonIntLite> e = null;

        if (elements != null)
        {
            e = new ArrayList<IonIntLite>(elements.length);
            for (int i = 0; i < elements.length; i++)
            {
                long value = elements[i];
                e.add(newInt(value));
            }
        }

        return e;
    }
}
