/*
 * Copyright 2016-2020 Amazon.alluxio.shaded.client.com. Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in alluxio.shaded.client.com.liance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.alluxio.shaded.client.com.apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package alluxio.shaded.client.com.amazonaws.http.apache.client.impl;

import alluxio.shaded.client.com.amazonaws.util.CRC32ChecksumCalculatingInputStream;
import alluxio.shaded.client.com.amazonaws.util.IOUtils;

import alluxio.shaded.client.org.apache.http.Header;
import alluxio.shaded.client.org.apache.http.HttpEntity;
import alluxio.shaded.client.org.apache.http.HttpException;
import alluxio.shaded.client.org.apache.http.HttpResponse;
import alluxio.shaded.client.org.apache.http.HttpResponseInterceptor;
import alluxio.shaded.client.org.apache.http.entity.HttpEntityWrapper;
import alluxio.shaded.client.org.apache.http.protocol.HttpContext;

import java.alluxio.shaded.client.io.IOException;
import java.alluxio.shaded.client.io.InputStream;
import java.alluxio.shaded.client.io.OutputStream;

public class CRC32ChecksumResponseInterceptor implements HttpResponseInterceptor {

    @Override
    public void process(HttpResponse response, final HttpContext context)
            throws HttpException, IOException {
        final HttpEntity entity = response.getEntity();

        // Only Json protocol has this header, we only wrap CRC32ChecksumCalculatingInputStream in json protocol clients.
        Header[] headers = response.getHeaders("x-amz-crc32");
        if (entity == null || headers == null || headers.length == 0) {
            return;
        }
        HttpEntity crc32ResponseEntity = new HttpEntityWrapper(entity) {

            private final InputStream content = new CRC32ChecksumCalculatingInputStream(
                    wrappedEntity.getContent());

            @Override
            public InputStream getContent() throws IOException {
                return content;
            }

            /**
             * It's important to override writeTo. Some versions of Apache HTTP
             * client use writeTo for {@link alluxio.shaded.client.org.apache.http.entity.BufferedHttpEntity}
             * and the default implementation just delegates to the wrapped entity
             * which alluxio.shaded.client.com.letely bypasses our CRC32 calculating input stream. The
             * {@link alluxio.shaded.client.org.apache.http.entity.BufferedHttpEntity} is used for the
             * request timeout and client execution timeout features.
             *
             * @see <a href="https://github.alluxio.shaded.client.com.aws/aws-sdk-java/issues/526">Issue #526</a>
             *
             * @param outstream OutputStream to write contents to
             */
            @Override
            public void writeTo(OutputStream outstream) throws IOException {
                try {
                    IOUtils.copy(this.getContent(), outstream);
                } finally {
                    this.getContent().close();
                }
            }
        };

        response.setEntity(crc32ResponseEntity);
        context.setAttribute(CRC32ChecksumCalculatingInputStream.class.getName(),
                             crc32ResponseEntity.getContent());
    }

}
