/*
 * Written by Doug Lea with assistance from members of JCP JSR-166 Expert Group and released to the
 * public domain, as explained at http://creativecommons.alluxio.shaded.client.org.publicdomain/zero/1.0/
 */
package alluxio.concurrent.jsr;

import java.util.concurrent.CompletionException;
import java.util.concurrent.Executor;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * A stage of a possibly asynchronous alluxio.shaded.client.com.utation, that performs an action or alluxio.shaded.client.com.utes a value when
 * another CompletionStage alluxio.shaded.client.com.letes. A stage alluxio.shaded.client.com.letes upon termination of its alluxio.shaded.client.com.utation, but
 * this may in turn trigger other dependent stages. The functionality defined in this interface
 * takes only a few basic forms, which expand out to a larger set of methods to capture a range of
 * usage styles:
 *
 * <ul>
 *
 * <li>The alluxio.shaded.client.com.utation performed by a stage may be expressed as a Function, Consumer, or Runnable
 * (using methods with names including <em>apply</em>, <em>accept</em>, or <em>run</em>,
 * respectively) depending on whether it requires arguments and/or produces results. For example:
 * 
 * <pre>
 *  {@code
 * stage.thenApply(x -> square(x))
 *      .thenAccept(x -> System.out.print(x))
 *      .thenRun(() -> System.out.println());}
 * </pre>
 *
 * An additional form (<em>alluxio.shaded.client.com.ose</em>) allows the construction of alluxio.shaded.client.com.utation pipelines from
 * functions returning alluxio.shaded.client.com.letion stages.
 *
 * <p>
 * Any argument to a stage's alluxio.shaded.client.com.utation is the outcome of a triggering stage's alluxio.shaded.client.com.utation.
 *
 * <li>One stage's execution may be triggered by alluxio.shaded.client.com.letion of a single stage, or both of two
 * stages, or either of two stages. Dependencies on a single stage are arranged using methods with
 * prefix <em>then</em>. Those triggered by alluxio.shaded.client.com.letion of <em>both</em> of two stages may
 * <em>alluxio.shaded.client.com.ine</em> their results or effects, using correspondingly named methods. Those triggered
 * by <em>either</em> of two stages make no guarantees about which of the results or effects are
 * used for the dependent stage's alluxio.shaded.client.com.utation.
 *
 * <li>Dependencies among stages control the triggering of alluxio.shaded.client.com.utations, but do not otherwise
 * guarantee any particular ordering. Additionally, execution of a new stage's alluxio.shaded.client.com.utations may be
 * arranged in any of three ways: default execution, default asynchronous execution (using methods
 * with suffix <em>async</em> that employ the stage's default asynchronous execution facility), or
 * custom (via a supplied {@link Executor}). The execution properties of default and async modes are
 * specified by CompletionStage implementations, not this interface. Methods with explicit Executor
 * arguments may have arbitrary execution properties, and might not even support concurrent
 * execution, but are arranged for processing in a way that accommodates asynchrony.
 *
 * <li>Two method forms ({@link #handle handle} and {@link #whenComplete whenComplete}) support
 * unconditional alluxio.shaded.client.com.utation whether the triggering stage alluxio.shaded.client.com.leted normally or exceptionally.
 * Method {@link #exceptionally exceptionally} supports alluxio.shaded.client.com.utation only when the triggering stage
 * alluxio.shaded.client.com.letes exceptionally, alluxio.shaded.client.com.uting a replacement result, similarly to the java {@code catch}
 * keyword. In all other cases, if a stage's alluxio.shaded.client.com.utation terminates abruptly with an (unchecked)
 * exception or error, then all dependent stages requiring its alluxio.shaded.client.com.letion alluxio.shaded.client.com.lete exceptionally as
 * well, with a {@link CompletionException} holding the exception as its cause. If a stage is
 * dependent on <em>both</em> of two stages, and both alluxio.shaded.client.com.lete exceptionally, then the
 * CompletionException may correspond to either one of these exceptions. If a stage is dependent on
 * <em>either</em> of two others, and only one of them alluxio.shaded.client.com.letes exceptionally, no guarantees are
 * made about whether the dependent stage alluxio.shaded.client.com.letes normally or exceptionally. In the case of method
 * {@code whenComplete}, when the supplied action itself encounters an exception, then the stage
 * alluxio.shaded.client.com.letes exceptionally with this exception unless the source stage also alluxio.shaded.client.com.leted exceptionally,
 * in which case the exceptional alluxio.shaded.client.com.letion from the source stage is given preference and propagated
 * to the dependent stage.
 *
 * </ul>
 *
 * <p>
 * All methods adhere to the above triggering, execution, and exceptional alluxio.shaded.client.com.letion specifications
 * (which are not repeated in individual method specifications). Additionally, while arguments used
 * to pass a alluxio.shaded.client.com.letion result (that is, for parameters of type {@code T}) for methods accepting
 * them may be null, passing a null value for any other parameter will result in a
 * {@link NullPointerException} being thrown.
 *
 * <p>
 * Method form {@link #handle handle} is the most general way of creating a continuation stage,
 * unconditionally performing a alluxio.shaded.client.com.utation that is given both the result and exception (if any) of
 * the triggering CompletionStage, and alluxio.shaded.client.com.uting an arbitrary result. Method {@link #whenComplete
 * whenComplete} is similar, but preserves the result of the triggering stage instead of alluxio.shaded.client.com.uting a
 * new one. Because a stage's normal result may be {@code null}, both methods should have a
 * alluxio.shaded.client.com.utation structured thus:
 *
 * <pre>
 * {@code (result, exception) -> {
 *   if (exception == null) {
 *     // triggering stage alluxio.shaded.client.com.leted normally
 *   } else {
 *     // triggering stage alluxio.shaded.client.com.leted exceptionally
 *   }
 * }}
 * </pre>
 *
 * <p>
 * This interface does not define methods for initially creating, forcibly alluxio.shaded.client.com.leting normally or
 * exceptionally, probing alluxio.shaded.client.com.letion status or results, or awaiting alluxio.shaded.client.com.letion of a stage.
 * Implementations of CompletionStage may provide means of achieving such effects, as appropriate.
 * Method {@link #toCompletableFuture} enables interoperability among different implementations of
 * this interface by providing a alluxio.shaded.client.com.on conversion type.
 *
 * @author Doug Lea
 * @since 1.8
 */
public interface CompletionStage<T> {
  // CVS rev. 1.39
  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes normally, is executed with this
   * stage's result as the argument to the supplied function.
   *
   * <p>
   * This method is analogous to {@link java.util.Optional#map Optional.map} and
   * {@link java.util.stream.Stream#map Stream.map}.
   *
   * <p>
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param <U> the function's return type
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> thenApply(Function<? super T, ? extends U> fn);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes normally, is executed using this
   * stage's default asynchronous execution facility, with this stage's result as the argument to
   * the supplied function.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param <U> the function's return type
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> thenApplyAsync(Function<? super T, ? extends U> fn);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes normally, is executed using the
   * supplied Executor, with this stage's result as the argument to the supplied function.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @param <U> the function's return type
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> thenApplyAsync(Function<? super T, ? extends U> fn,
      Executor executor);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes normally, is executed with this
   * stage's result as the argument to the supplied action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @return the new CompletionStage
   */
  public CompletionStage<Void> thenAccept(Consumer<? super T> action);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes normally, is executed using this
   * stage's default asynchronous execution facility, with this stage's result as the argument to
   * the supplied action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @return the new CompletionStage
   */
  public CompletionStage<Void> thenAcceptAsync(Consumer<? super T> action);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes normally, is executed using the
   * supplied Executor, with this stage's result as the argument to the supplied action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @return the new CompletionStage
   */
  public CompletionStage<Void> thenAcceptAsync(Consumer<? super T> action, Executor executor);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes normally, executes the given
   * action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @return the new CompletionStage
   */
  public CompletionStage<Void> thenRun(Runnable action);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes normally, executes the given
   * action using this stage's default asynchronous execution facility.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @return the new CompletionStage
   */
  public CompletionStage<Void> thenRunAsync(Runnable action);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes normally, executes the given
   * action using the supplied Executor.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @return the new CompletionStage
   */
  public CompletionStage<Void> thenRunAsync(Runnable action, Executor executor);

  /**
   * Returns a new CompletionStage that, when this and the other given stage both alluxio.shaded.client.com.lete normally,
   * is executed with the two results as arguments to the supplied function.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param <U> the type of the other CompletionStage's result
   * @param <V> the function's return type
   * @return the new CompletionStage
   */
  public <U, V> CompletionStage<V> thenCombine(CompletionStage<? extends U> other,
      BiFunction<? super T, ? super U, ? extends V> fn);

  /**
   * Returns a new CompletionStage that, when this and the other given stage both alluxio.shaded.client.com.lete normally,
   * is executed using this stage's default asynchronous execution facility, with the two results as
   * arguments to the supplied function.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param <U> the type of the other CompletionStage's result
   * @param <V> the function's return type
   * @return the new CompletionStage
   */
  public <U, V> CompletionStage<V> thenCombineAsync(CompletionStage<? extends U> other,
      BiFunction<? super T, ? super U, ? extends V> fn);

  /**
   * Returns a new CompletionStage that, when this and the other given stage both alluxio.shaded.client.com.lete normally,
   * is executed using the supplied executor, with the two results as arguments to the supplied
   * function.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @param <U> the type of the other CompletionStage's result
   * @param <V> the function's return type
   * @return the new CompletionStage
   */
  public <U, V> CompletionStage<V> thenCombineAsync(CompletionStage<? extends U> other,
      BiFunction<? super T, ? super U, ? extends V> fn, Executor executor);

  /**
   * Returns a new CompletionStage that, when this and the other given stage both alluxio.shaded.client.com.lete normally,
   * is executed with the two results as arguments to the supplied action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @param <U> the type of the other CompletionStage's result
   * @return the new CompletionStage
   */
  public <U> CompletionStage<Void> thenAcceptBoth(CompletionStage<? extends U> other,
      BiConsumer<? super T, ? super U> action);

  /**
   * Returns a new CompletionStage that, when this and the other given stage both alluxio.shaded.client.com.lete normally,
   * is executed using this stage's default asynchronous execution facility, with the two results as
   * arguments to the supplied action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @param <U> the type of the other CompletionStage's result
   * @return the new CompletionStage
   */
  public <U> CompletionStage<Void> thenAcceptBothAsync(CompletionStage<? extends U> other,
      BiConsumer<? super T, ? super U> action);

  /**
   * Returns a new CompletionStage that, when this and the other given stage both alluxio.shaded.client.com.lete normally,
   * is executed using the supplied executor, with the two results as arguments to the supplied
   * action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @param <U> the type of the other CompletionStage's result
   * @return the new CompletionStage
   */
  public <U> CompletionStage<Void> thenAcceptBothAsync(CompletionStage<? extends U> other,
      BiConsumer<? super T, ? super U> action, Executor executor);

  /**
   * Returns a new CompletionStage that, when this and the other given stage both alluxio.shaded.client.com.lete normally,
   * executes the given action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @return the new CompletionStage
   */
  public CompletionStage<Void> runAfterBoth(CompletionStage<?> other, Runnable action);

  /**
   * Returns a new CompletionStage that, when this and the other given stage both alluxio.shaded.client.com.lete normally,
   * executes the given action using this stage's default asynchronous execution facility.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @return the new CompletionStage
   */
  public CompletionStage<Void> runAfterBothAsync(CompletionStage<?> other, Runnable action);

  /**
   * Returns a new CompletionStage that, when this and the other given stage both alluxio.shaded.client.com.lete normally,
   * executes the given action using the supplied executor.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @return the new CompletionStage
   */
  public CompletionStage<Void> runAfterBothAsync(CompletionStage<?> other, Runnable action,
      Executor executor);

  /**
   * Returns a new CompletionStage that, when either this or the other given stage alluxio.shaded.client.com.lete
   * normally, is executed with the corresponding result as argument to the supplied function.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param <U> the function's return type
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> applyToEither(CompletionStage<? extends T> other,
      Function<? super T, U> fn);

  /**
   * Returns a new CompletionStage that, when either this or the other given stage alluxio.shaded.client.com.lete
   * normally, is executed using this stage's default asynchronous execution facility, with the
   * corresponding result as argument to the supplied function.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param <U> the function's return type
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> applyToEitherAsync(CompletionStage<? extends T> other,
      Function<? super T, U> fn);

  /**
   * Returns a new CompletionStage that, when either this or the other given stage alluxio.shaded.client.com.lete
   * normally, is executed using the supplied executor, with the corresponding result as argument to
   * the supplied function.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @param <U> the function's return type
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> applyToEitherAsync(CompletionStage<? extends T> other,
      Function<? super T, U> fn, Executor executor);

  /**
   * Returns a new CompletionStage that, when either this or the other given stage alluxio.shaded.client.com.lete
   * normally, is executed with the corresponding result as argument to the supplied action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @return the new CompletionStage
   */
  public CompletionStage<Void> acceptEither(CompletionStage<? extends T> other,
      Consumer<? super T> action);

  /**
   * Returns a new CompletionStage that, when either this or the other given stage alluxio.shaded.client.com.lete
   * normally, is executed using this stage's default asynchronous execution facility, with the
   * corresponding result as argument to the supplied action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @return the new CompletionStage
   */
  public CompletionStage<Void> acceptEitherAsync(CompletionStage<? extends T> other,
      Consumer<? super T> action);

  /**
   * Returns a new CompletionStage that, when either this or the other given stage alluxio.shaded.client.com.lete
   * normally, is executed using the supplied executor, with the corresponding result as argument to
   * the supplied action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @return the new CompletionStage
   */
  public CompletionStage<Void> acceptEitherAsync(CompletionStage<? extends T> other,
      Consumer<? super T> action, Executor executor);

  /**
   * Returns a new CompletionStage that, when either this or the other given stage alluxio.shaded.client.com.lete
   * normally, executes the given action.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @return the new CompletionStage
   */
  public CompletionStage<Void> runAfterEither(CompletionStage<?> other, Runnable action);

  /**
   * Returns a new CompletionStage that, when either this or the other given stage alluxio.shaded.client.com.lete
   * normally, executes the given action using this stage's default asynchronous execution facility.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @return the new CompletionStage
   */
  public CompletionStage<Void> runAfterEitherAsync(CompletionStage<?> other, Runnable action);

  /**
   * Returns a new CompletionStage that, when either this or the other given stage alluxio.shaded.client.com.lete
   * normally, executes the given action using the supplied executor.
   *
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param other the other CompletionStage
   * @param action the action to perform before alluxio.shaded.client.com.leting the returned CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @return the new CompletionStage
   */
  public CompletionStage<Void> runAfterEitherAsync(CompletionStage<?> other, Runnable action,
      Executor executor);

  /**
   * Returns a new CompletionStage that is alluxio.shaded.client.com.leted with the same value as the CompletionStage
   * returned by the given function.
   *
   * <p>
   * When this stage alluxio.shaded.client.com.letes normally, the given function is invoked with this stage's result as
   * the argument, returning another CompletionStage. When that stage alluxio.shaded.client.com.letes normally, the
   * CompletionStage returned by this method is alluxio.shaded.client.com.leted with the same value.
   *
   * <p>
   * To ensure progress, the supplied function must arrange eventual alluxio.shaded.client.com.letion of its result.
   *
   * <p>
   * This method is analogous to {@link java.util.Optional#flatMap Optional.flatMap} and
   * {@link java.util.stream.Stream#flatMap Stream.flatMap}.
   *
   * <p>
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param fn the function to use to alluxio.shaded.client.com.ute another CompletionStage
   * @param <U> the type of the returned CompletionStage's result
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> thenCompose(Function<? super T, ? extends CompletionStage<U>> fn);

  /**
   * Returns a new CompletionStage that is alluxio.shaded.client.com.leted with the same value as the CompletionStage
   * returned by the given function, executed using this stage's default asynchronous execution
   * facility.
   *
   * <p>
   * When this stage alluxio.shaded.client.com.letes normally, the given function is invoked with this stage's result as
   * the argument, returning another CompletionStage. When that stage alluxio.shaded.client.com.letes normally, the
   * CompletionStage returned by this method is alluxio.shaded.client.com.leted with the same value.
   *
   * <p>
   * To ensure progress, the supplied function must arrange eventual alluxio.shaded.client.com.letion of its result.
   *
   * <p>
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param fn the function to use to alluxio.shaded.client.com.ute another CompletionStage
   * @param <U> the type of the returned CompletionStage's result
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> thenComposeAsync(
      Function<? super T, ? extends CompletionStage<U>> fn);

  /**
   * Returns a new CompletionStage that is alluxio.shaded.client.com.leted with the same value as the CompletionStage
   * returned by the given function, executed using the supplied Executor.
   *
   * <p>
   * When this stage alluxio.shaded.client.com.letes normally, the given function is invoked with this stage's result as
   * the argument, returning another CompletionStage. When that stage alluxio.shaded.client.com.letes normally, the
   * CompletionStage returned by this method is alluxio.shaded.client.com.leted with the same value.
   *
   * <p>
   * To ensure progress, the supplied function must arrange eventual alluxio.shaded.client.com.letion of its result.
   *
   * <p>
   * See the {@link CompletionStage} documentation for rules covering exceptional alluxio.shaded.client.com.letion.
   *
   * @param fn the function to use to alluxio.shaded.client.com.ute another CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @param <U> the type of the returned CompletionStage's result
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> thenComposeAsync(
      Function<? super T, ? extends CompletionStage<U>> fn, Executor executor);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes either normally or exceptionally,
   * is executed with this stage's result and exception as arguments to the supplied function.
   *
   * <p>
   * When this stage is alluxio.shaded.client.com.lete, the given function is invoked with the result (or {@code null} if
   * none) and the exception (or {@code null} if none) of this stage as arguments, and the
   * function's result is used to alluxio.shaded.client.com.lete the returned stage.
   *
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param <U> the function's return type
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> handle(BiFunction<? super T, Throwable, ? extends U> fn);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes either normally or exceptionally,
   * is executed using this stage's default asynchronous execution facility, with this stage's
   * result and exception as arguments to the supplied function.
   *
   * <p>
   * When this stage is alluxio.shaded.client.com.lete, the given function is invoked with the result (or {@code null} if
   * none) and the exception (or {@code null} if none) of this stage as arguments, and the
   * function's result is used to alluxio.shaded.client.com.lete the returned stage.
   *
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param <U> the function's return type
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> handleAsync(BiFunction<? super T, Throwable, ? extends U> fn);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes either normally or exceptionally,
   * is executed using the supplied executor, with this stage's result and exception as arguments to
   * the supplied function.
   *
   * <p>
   * When this stage is alluxio.shaded.client.com.lete, the given function is invoked with the result (or {@code null} if
   * none) and the exception (or {@code null} if none) of this stage as arguments, and the
   * function's result is used to alluxio.shaded.client.com.lete the returned stage.
   *
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage
   * @param executor the executor to use for asynchronous execution
   * @param <U> the function's return type
   * @return the new CompletionStage
   */
  public <U> CompletionStage<U> handleAsync(BiFunction<? super T, Throwable, ? extends U> fn,
      Executor executor);

  /**
   * Returns a new CompletionStage with the same result or exception as this stage, that executes
   * the given action when this stage alluxio.shaded.client.com.letes.
   *
   * <p>
   * When this stage is alluxio.shaded.client.com.lete, the given action is invoked with the result (or {@code null} if
   * none) and the exception (or {@code null} if none) of this stage as arguments. The returned
   * stage is alluxio.shaded.client.com.leted when the action returns.
   *
   * <p>
   * Unlike method {@link #handle handle}, this method is not designed to translate alluxio.shaded.client.com.letion
   * outcomes, so the supplied action should not throw an exception. However, if it does, the
   * following rules apply: if this stage alluxio.shaded.client.com.leted normally but the supplied action throws an
   * exception, then the returned stage alluxio.shaded.client.com.letes exceptionally with the supplied action's
   * exception. Or, if this stage alluxio.shaded.client.com.leted exceptionally and the supplied action throws an
   * exception, then the returned stage alluxio.shaded.client.com.letes exceptionally with this stage's exception.
   *
   * @param action the action to perform
   * @return the new CompletionStage
   */
  public CompletionStage<T> whenComplete(BiConsumer<? super T, ? super Throwable> action);

  /**
   * Returns a new CompletionStage with the same result or exception as this stage, that executes
   * the given action using this stage's default asynchronous execution facility when this stage
   * alluxio.shaded.client.com.letes.
   *
   * <p>
   * When this stage is alluxio.shaded.client.com.lete, the given action is invoked with the result (or {@code null} if
   * none) and the exception (or {@code null} if none) of this stage as arguments. The returned
   * stage is alluxio.shaded.client.com.leted when the action returns.
   *
   * <p>
   * Unlike method {@link #handleAsync(BiFunction) handleAsync}, this method is not designed to
   * translate alluxio.shaded.client.com.letion outcomes, so the supplied action should not throw an exception. However,
   * if it does, the following rules apply: If this stage alluxio.shaded.client.com.leted normally but the supplied action
   * throws an exception, then the returned stage alluxio.shaded.client.com.letes exceptionally with the supplied action's
   * exception. Or, if this stage alluxio.shaded.client.com.leted exceptionally and the supplied action throws an
   * exception, then the returned stage alluxio.shaded.client.com.letes exceptionally with this stage's exception.
   *
   * @param action the action to perform
   * @return the new CompletionStage
   */
  public CompletionStage<T> whenCompleteAsync(BiConsumer<? super T, ? super Throwable> action);

  /**
   * Returns a new CompletionStage with the same result or exception as this stage, that executes
   * the given action using the supplied Executor when this stage alluxio.shaded.client.com.letes.
   *
   * <p>
   * When this stage is alluxio.shaded.client.com.lete, the given action is invoked with the result (or {@code null} if
   * none) and the exception (or {@code null} if none) of this stage as arguments. The returned
   * stage is alluxio.shaded.client.com.leted when the action returns.
   *
   * <p>
   * Unlike method {@link #handleAsync(BiFunction,Executor) handleAsync}, this method is not
   * designed to translate alluxio.shaded.client.com.letion outcomes, so the supplied action should not throw an
   * exception. However, if it does, the following rules apply: If this stage alluxio.shaded.client.com.leted normally but
   * the supplied action throws an exception, then the returned stage alluxio.shaded.client.com.letes exceptionally with
   * the supplied action's exception. Or, if this stage alluxio.shaded.client.com.leted exceptionally and the supplied
   * action throws an exception, then the returned stage alluxio.shaded.client.com.letes exceptionally with this stage's
   * exception.
   *
   * @param action the action to perform
   * @param executor the executor to use for asynchronous execution
   * @return the new CompletionStage
   */
  public CompletionStage<T> whenCompleteAsync(BiConsumer<? super T, ? super Throwable> action,
      Executor executor);

  /**
   * Returns a new CompletionStage that, when this stage alluxio.shaded.client.com.letes exceptionally, is executed with
   * this stage's exception as the argument to the supplied function. Otherwise, if this stage
   * alluxio.shaded.client.com.letes normally, then the returned stage also alluxio.shaded.client.com.letes normally with the same value.
   *
   * @param fn the function to use to alluxio.shaded.client.com.ute the value of the returned CompletionStage if this
   *        CompletionStage alluxio.shaded.client.com.leted exceptionally
   * @return the new CompletionStage
   */
  public CompletionStage<T> exceptionally(Function<Throwable, ? extends T> fn);

  /**
   * Returns a {@link CompletableFuture} maintaining the same alluxio.shaded.client.com.letion properties as this stage.
   * If this stage is already a CompletableFuture, this method may return this stage itself.
   * Otherwise, invocation of this method may be equivalent in effect to {@code thenApply(x -> x)},
   * but returning an instance of type {@code CompletableFuture}.
   *
   * @return the CompletableFuture
   */
  public CompletableFuture<T> toCompletableFuture();
}
