/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file.cache;

import alluxio.client.file.CacheContext;
import alluxio.client.file.FileInStream;
import alluxio.client.file.URIStatus;
import alluxio.client.file.cache.CacheManager;
import alluxio.client.file.cache.PageId;
import alluxio.client.file.cache.store.ByteArrayTargetBuffer;
import alluxio.client.file.cache.store.ByteBufferTargetBuffer;
import alluxio.client.file.cache.store.PageReadTargetBuffer;
import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.exception.AlluxioException;
import alluxio.metrics.MetricKey;
import alluxio.metrics.MetricsSystem;
import alluxio.shaded.client.com.google.common.annotations.VisibleForTesting;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.com.google.common.base.Stopwatch;
import alluxio.shaded.client.com.google.common.base.Ticker;
import alluxio.shaded.client.com.google.common.io.Closer;
import alluxio.shaded.client.javax.annotation.concurrent.NotThreadSafe;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class LocalCacheFileInStream
extends FileInStream {
    private static final Logger LOG = LoggerFactory.getLogger(LocalCacheFileInStream.class);
    protected final long mPageSize;
    private final Closer mCloser = Closer.create();
    private final CacheManager mCacheManager;
    private final boolean mQuotaEnabled;
    private final CacheContext mCacheContext;
    private final URIStatus mStatus;
    private final FileInStreamOpener mExternalFileInStreamOpener;
    private final int mBufferSize;
    private byte[] mBuffer = null;
    private long mBufferStartOffset;
    private long mBufferEndOffset;
    private FileInStream mExternalFileInStream;
    private long mPosition = 0L;
    private boolean mClosed = false;
    private boolean mEOF = false;

    public static void registerMetrics() {
        Metrics.registerGauges();
    }

    public LocalCacheFileInStream(URIStatus status, FileInStreamOpener fileOpener, CacheManager cacheManager, AlluxioConfiguration conf) {
        this.mPageSize = conf.getBytes(PropertyKey.USER_CLIENT_CACHE_PAGE_SIZE);
        this.mExternalFileInStreamOpener = fileOpener;
        this.mCacheManager = cacheManager;
        this.mStatus = status;
        this.mQuotaEnabled = conf.getBoolean(PropertyKey.USER_CLIENT_CACHE_QUOTA_ENABLED);
        this.mCacheContext = this.mQuotaEnabled && status.getCacheContext() != null ? status.getCacheContext() : CacheContext.defaults();
        Metrics.registerGauges();
        this.mBufferSize = (int)conf.getBytes(PropertyKey.USER_CLIENT_CACHE_IN_STREAM_BUFFER_SIZE);
        Preconditions.checkArgument(this.mBufferSize >= 0, "Buffer size cannot be negative. %s", this.mPageSize);
        if (this.mBufferSize > 0) {
            this.mBuffer = new byte[this.mBufferSize];
        }
    }

    @Override
    public int read(byte[] bytesBuffer, int offset, int length) throws IOException {
        return this.readInternal(new ByteArrayTargetBuffer(bytesBuffer, offset), offset, length, ReadType.READ_INTO_BYTE_ARRAY, this.mPosition, false);
    }

    @Override
    public int read(ByteBuffer buffer, int offset, int length) throws IOException {
        int totalBytesRead = this.readInternal(new ByteBufferTargetBuffer(buffer), offset, length, ReadType.READ_INTO_BYTE_BUFFER, this.mPosition, false);
        if (totalBytesRead == -1) {
            return -1;
        }
        return totalBytesRead;
    }

    private int bufferedRead(PageReadTargetBuffer targetBuffer, int length, ReadType readType, long position, Stopwatch stopwatch) throws IOException {
        if (this.mBuffer == null) {
            return this.localCachedRead(targetBuffer, length, readType, position, stopwatch);
        }
        if (position > this.mBufferStartOffset && position < this.mBufferEndOffset) {
            int lengthToReadFromBuffer = (int)Math.min((long)length, this.mBufferEndOffset - position);
            targetBuffer.writeBytes(this.mBuffer, (int)(position - this.mBufferStartOffset), lengthToReadFromBuffer);
            return lengthToReadFromBuffer;
        }
        if (length >= this.mBufferSize) {
            return this.localCachedRead(targetBuffer, length, readType, position, stopwatch);
        }
        int bytesLoadToBuffer = (int)Math.min((long)this.mBufferSize, this.mStatus.getLength() - position);
        int bytesRead = this.localCachedRead(new ByteArrayTargetBuffer(this.mBuffer, 0), bytesLoadToBuffer, readType, position, stopwatch);
        this.mBufferStartOffset = position;
        this.mBufferEndOffset = position + (long)bytesRead;
        int dataReadFromBuffer = Math.min(bytesRead, length);
        targetBuffer.writeBytes(this.mBuffer, 0, dataReadFromBuffer);
        MetricsSystem.meter(MetricKey.CLIENT_CACHE_BYTES_READ_IN_STREAM_BUFFER.getName()).mark(dataReadFromBuffer);
        return dataReadFromBuffer;
    }

    private int localCachedRead(PageReadTargetBuffer bytesBuffer, int length, ReadType readType, long position, Stopwatch stopwatch) throws IOException {
        long currentPage = position / this.mPageSize;
        CacheContext cacheContext = this.mStatus.getCacheContext();
        PageId pageId = cacheContext != null && cacheContext.getCacheIdentifier() != null ? new PageId(cacheContext.getCacheIdentifier(), currentPage) : new PageId(Long.toString(this.mStatus.getFileId()), currentPage);
        int currentPageOffset = (int)(position % this.mPageSize);
        int bytesLeftInPage = (int)(this.mPageSize - (long)currentPageOffset);
        int bytesToReadInPage = Math.min(bytesLeftInPage, length);
        stopwatch.reset().start();
        int bytesRead = this.mCacheManager.get(pageId, currentPageOffset, bytesToReadInPage, bytesBuffer, this.mCacheContext);
        stopwatch.stop();
        if (bytesRead > 0) {
            MetricsSystem.meter(MetricKey.CLIENT_CACHE_BYTES_READ_CACHE.getName()).mark(bytesRead);
            if (cacheContext != null) {
                cacheContext.incrementCounter(MetricKey.CLIENT_CACHE_BYTES_READ_CACHE.getMetricName(), CacheContext.StatsUnit.BYTE, bytesRead);
                cacheContext.incrementCounter(MetricKey.CLIENT_CACHE_PAGE_READ_CACHE_TIME_NS.getMetricName(), CacheContext.StatsUnit.NANO, stopwatch.elapsed(TimeUnit.NANOSECONDS));
            }
            return bytesRead;
        }
        stopwatch.reset().start();
        byte[] page = this.readExternalPage(position, readType);
        stopwatch.stop();
        if (page.length > 0) {
            bytesBuffer.writeBytes(page, currentPageOffset, bytesToReadInPage);
            MetricsSystem.meter(MetricKey.CLIENT_CACHE_BYTES_REQUESTED_EXTERNAL.getName()).mark(bytesToReadInPage);
            if (cacheContext != null) {
                cacheContext.incrementCounter(MetricKey.CLIENT_CACHE_BYTES_REQUESTED_EXTERNAL.getMetricName(), CacheContext.StatsUnit.BYTE, bytesToReadInPage);
                cacheContext.incrementCounter(MetricKey.CLIENT_CACHE_PAGE_READ_EXTERNAL_TIME_NS.getMetricName(), CacheContext.StatsUnit.NANO, stopwatch.elapsed(TimeUnit.NANOSECONDS));
            }
            this.mCacheManager.put(pageId, page, this.mCacheContext);
        }
        return bytesToReadInPage;
    }

    private int readInternal(PageReadTargetBuffer targetBuffer, int offset, int length, ReadType readType, long position, boolean isPositionedRead) throws IOException {
        Preconditions.checkArgument(length >= 0, "length should be non-negative");
        Preconditions.checkArgument(offset >= 0, "offset should be non-negative");
        Preconditions.checkArgument(position >= 0L, "position should be non-negative");
        if (length == 0) {
            return 0;
        }
        if (position >= this.mStatus.getLength()) {
            return -1;
        }
        int totalBytesRead = 0;
        long currentPosition = position;
        long lengthToRead = Math.min((long)length, this.mStatus.getLength() - position);
        Stopwatch stopwatch = this.createUnstartedStopwatch();
        while ((long)totalBytesRead < lengthToRead) {
            int bytesRead = this.bufferedRead(targetBuffer, (int)(lengthToRead - (long)totalBytesRead), readType, currentPosition, stopwatch);
            totalBytesRead += bytesRead;
            currentPosition += (long)bytesRead;
            if (isPositionedRead) continue;
            this.mPosition = currentPosition;
        }
        if (totalBytesRead > length || totalBytesRead < length && currentPosition < this.mStatus.getLength()) {
            throw new IOException(String.format("Invalid number of bytes read - bytes to read = %d, actual bytes read = %d, bytes remains in file %d", length, totalBytesRead, this.remaining()));
        }
        return totalBytesRead;
    }

    @VisibleForTesting
    protected Stopwatch createUnstartedStopwatch() {
        return Stopwatch.createUnstarted(Ticker.systemTicker());
    }

    @Override
    public long skip(long n) {
        this.checkIfClosed();
        if (n <= 0L) {
            return 0L;
        }
        long toSkip = Math.min(this.remaining(), n);
        this.mPosition += toSkip;
        return toSkip;
    }

    @Override
    public void close() throws IOException {
        this.mCloser.close();
    }

    @Override
    public long remaining() {
        return this.mEOF ? 0L : this.mStatus.getLength() - this.mPosition;
    }

    @Override
    public int positionedRead(long pos, byte[] b, int off, int len) throws IOException {
        return this.readInternal(new ByteArrayTargetBuffer(b, off), off, len, ReadType.READ_INTO_BYTE_ARRAY, pos, true);
    }

    @Override
    public long getPos() {
        return this.mPosition;
    }

    @Override
    public void seek(long pos) {
        this.checkIfClosed();
        Preconditions.checkArgument(pos >= 0L, "Seek position is negative: %s", pos);
        Preconditions.checkArgument(pos <= this.mStatus.getLength(), "Seek position (%s) exceeds the length of the file (%s)", pos, this.mStatus.getLength());
        if (pos == this.mPosition) {
            return;
        }
        if (pos < this.mPosition) {
            this.mEOF = false;
        }
        this.mPosition = pos;
    }

    @Override
    public void unbuffer() {
        if (this.mExternalFileInStream != null) {
            this.mExternalFileInStream.unbuffer();
        }
    }

    private void checkIfClosed() {
        Preconditions.checkState(!this.mClosed, "Cannot operate on a closed stream");
    }

    private FileInStream getExternalFileInStream(long position) throws IOException {
        try {
            if (this.mExternalFileInStream == null) {
                this.mExternalFileInStream = this.mExternalFileInStreamOpener.open(this.mStatus);
                this.mCloser.register(this.mExternalFileInStream);
            }
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
        long pageStart = position - position % this.mPageSize;
        if (this.mExternalFileInStream.getPos() != pageStart) {
            this.mExternalFileInStream.seek(pageStart);
        }
        return this.mExternalFileInStream;
    }

    private synchronized byte[] readExternalPage(long position, ReadType readType) throws IOException {
        int totalBytesRead;
        int bytesRead;
        long pageStart = position - position % this.mPageSize;
        FileInStream stream = this.getExternalFileInStream(pageStart);
        int pageSize = (int)Math.min(this.mPageSize, this.mStatus.getLength() - pageStart);
        byte[] page = new byte[pageSize];
        ByteBuffer buffer = readType == ReadType.READ_INTO_BYTE_BUFFER ? ByteBuffer.wrap(page) : null;
        for (totalBytesRead = 0; totalBytesRead < pageSize; totalBytesRead += bytesRead) {
            switch (readType) {
                case READ_INTO_BYTE_ARRAY: {
                    bytesRead = stream.read(page, totalBytesRead, pageSize - totalBytesRead);
                    break;
                }
                case READ_INTO_BYTE_BUFFER: {
                    bytesRead = stream.read(buffer);
                    break;
                }
                default: {
                    throw new IOException("unsupported read type = " + (Object)((Object)readType));
                }
            }
            if (bytesRead <= 0) break;
        }
        MetricsSystem.meter(MetricKey.CLIENT_CACHE_BYTES_READ_EXTERNAL.getName()).mark(totalBytesRead);
        if (totalBytesRead != pageSize) {
            throw new IOException("Failed to read complete page from external storage. Bytes read: " + totalBytesRead + " Page size: " + pageSize);
        }
        return page;
    }

    static enum ReadType {
        READ_INTO_BYTE_ARRAY,
        READ_INTO_BYTE_BUFFER;

    }

    private static final class Metrics {
        private Metrics() {
        }

        private static void registerGauges() {
            MetricsSystem.registerGaugeIfAbsent(MetricsSystem.getMetricName(MetricKey.CLIENT_CACHE_HIT_RATE.getName()), () -> {
                long cacheMisses;
                long cacheHits = MetricsSystem.meter(MetricKey.CLIENT_CACHE_BYTES_READ_CACHE.getName()).getCount();
                long total = cacheHits + (cacheMisses = MetricsSystem.meter(MetricKey.CLIENT_CACHE_BYTES_REQUESTED_EXTERNAL.getName()).getCount());
                if (total > 0L) {
                    return (double)cacheHits / (1.0 * (double)total);
                }
                return 0;
            });
        }
    }

    public static interface FileInStreamOpener {
        public FileInStream open(URIStatus var1) throws IOException, AlluxioException;
    }
}

