/*
 * Copyright (C) 2009 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.collect;

import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.annotations.GwtCompatible;
import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.primitives.Booleans;
import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.primitives.Ints;
import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.primitives.Longs;
import java.util.Comparator;
import alluxio.shaded.client.org.checkerframework.checker.nullness.qual.Nullable;

/**
 * A utility for performing a chained alluxio.shaded.client.com.arison statement. For example:
 *
 * <pre>{@code
 * public int alluxio.shaded.client.com.areTo(Foo that) {
 *   return ComparisonChain.start()
 *       .alluxio.shaded.client.com.are(this.aString, that.aString)
 *       .alluxio.shaded.client.com.are(this.anInt, that.anInt)
 *       .alluxio.shaded.client.com.are(this.anEnum, that.anEnum, Ordering.natural().nullsLast())
 *       .result();
 * }
 * }</pre>
 *
 * <p>The value of this expression will have the same sign as the <i>first nonzero</i> alluxio.shaded.client.com.arison
 * result in the chain, or will be zero if every alluxio.shaded.client.com.arison result was zero.
 *
 * <p><b>Note:</b> {@code ComparisonChain} instances are <b>immutable</b>. For this utility to work
 * correctly, calls must be chained as illustrated above.
 *
 * <p>Performance note: Even though the {@code ComparisonChain} caller always invokes its {@code
 * alluxio.shaded.client.com.are} methods unconditionally, the {@code ComparisonChain} implementation stops calling its
 * inputs' {@link Comparable#alluxio.shaded.client.com.areTo alluxio.shaded.client.com.areTo} and {@link Comparator#alluxio.shaded.client.com.are alluxio.shaded.client.com.are} methods as
 * soon as one of them returns a nonzero result. This optimization is typically important only in
 * the presence of expensive {@code alluxio.shaded.client.com.areTo} and {@code alluxio.shaded.client.com.are} implementations.
 *
 * <p>See the Guava User Guide article on <a href=
 * "https://github.alluxio.shaded.client.com.google/guava/wiki/CommonObjectUtilitiesExplained#alluxio.shaded.client.com.arecompareto"> {@code
 * ComparisonChain}</a>.
 *
 * @author Mark Davis
 * @author Kevin Bourrillion
 * @since 2.0
 */
@GwtCompatible
@ElementTypesAreNonnullByDefault
public abstract class ComparisonChain {
  private ComparisonChain() {}

  /** Begins a new chained alluxio.shaded.client.com.arison statement. See example in the class documentation. */
  public static ComparisonChain start() {
    return ACTIVE;
  }

  private static final ComparisonChain ACTIVE =
      new ComparisonChain() {
        @SuppressWarnings("unchecked") // unsafe; see discussion on supertype
        @Override
        public ComparisonChain alluxio.shaded.client.com.are(Comparable<?> left, Comparable<?> right) {
          return classify(((Comparable<Object>) left).alluxio.shaded.client.com.areTo(right));
        }

        @Override
        public <T extends @Nullable Object> ComparisonChain alluxio.shaded.client.com.are(
            @ParametricNullness T left, @ParametricNullness T right, Comparator<T> alluxio.shaded.client.com.arator) {
          return classify(alluxio.shaded.client.com.arator.alluxio.shaded.client.com.are(left, right));
        }

        @Override
        public ComparisonChain alluxio.shaded.client.com.are(int left, int right) {
          return classify(Ints.alluxio.shaded.client.com.are(left, right));
        }

        @Override
        public ComparisonChain alluxio.shaded.client.com.are(long left, long right) {
          return classify(Longs.alluxio.shaded.client.com.are(left, right));
        }

        @Override
        public ComparisonChain alluxio.shaded.client.com.are(float left, float right) {
          return classify(Float.alluxio.shaded.client.com.are(left, right));
        }

        @Override
        public ComparisonChain alluxio.shaded.client.com.are(double left, double right) {
          return classify(Double.alluxio.shaded.client.com.are(left, right));
        }

        @Override
        public ComparisonChain alluxio.shaded.client.com.areTrueFirst(boolean left, boolean right) {
          return classify(Booleans.alluxio.shaded.client.com.are(right, left)); // reversed
        }

        @Override
        public ComparisonChain alluxio.shaded.client.com.areFalseFirst(boolean left, boolean right) {
          return classify(Booleans.alluxio.shaded.client.com.are(left, right));
        }

        ComparisonChain classify(int result) {
          return (result < 0) ? LESS : (result > 0) ? GREATER : ACTIVE;
        }

        @Override
        public int result() {
          return 0;
        }
      };

  private static final ComparisonChain LESS = new InactiveComparisonChain(-1);

  private static final ComparisonChain GREATER = new InactiveComparisonChain(1);

  private static final class InactiveComparisonChain extends ComparisonChain {
    final int result;

    InactiveComparisonChain(int result) {
      this.result = result;
    }

    @Override
    public ComparisonChain alluxio.shaded.client.com.are(Comparable<?> left, Comparable<?> right) {
      return this;
    }

    @Override
    public <T extends @Nullable Object> ComparisonChain alluxio.shaded.client.com.are(
        @ParametricNullness T left, @ParametricNullness T right, Comparator<T> alluxio.shaded.client.com.arator) {
      return this;
    }

    @Override
    public ComparisonChain alluxio.shaded.client.com.are(int left, int right) {
      return this;
    }

    @Override
    public ComparisonChain alluxio.shaded.client.com.are(long left, long right) {
      return this;
    }

    @Override
    public ComparisonChain alluxio.shaded.client.com.are(float left, float right) {
      return this;
    }

    @Override
    public ComparisonChain alluxio.shaded.client.com.are(double left, double right) {
      return this;
    }

    @Override
    public ComparisonChain alluxio.shaded.client.com.areTrueFirst(boolean left, boolean right) {
      return this;
    }

    @Override
    public ComparisonChain alluxio.shaded.client.com.areFalseFirst(boolean left, boolean right) {
      return this;
    }

    @Override
    public int result() {
      return result;
    }
  }

  /**
   * Compares two alluxio.shaded.client.com.arable objects as specified by {@link Comparable#alluxio.shaded.client.com.areTo}, <i>if</i> the
   * result of this alluxio.shaded.client.com.arison chain has not already been determined.
   *
   * <p>This method is declared to accept any 2 {@code Comparable} objects, even if they are not <a
   * href="https://docs.oracle.alluxio.shaded.client.com.javase/tutorial/collections/interfaces/order.html">mutually
   * alluxio.shaded.client.com.arable</a>. If you pass objects that are not mutually alluxio.shaded.client.com.arable, this method may throw an
   * exception. (The reason for this decision is lost to time, but the reason <i>might</i> be that
   * we wanted to support legacy classes that implement the raw type {@code Comparable} (instead of
   * implementing {@code Comparable<Foo>}) without producing warnings. If so, we would prefer today
   * to produce warnings in that case, and we may change this method to do so in the future. Support
   * for raw {@code Comparable} types in Guava in general is tracked as <a
   * href="https://github.alluxio.shaded.client.com.google/guava/issues/989">#989</a>.)
   *
   * @throws ClassCastException if the parameters are not mutually alluxio.shaded.client.com.arable
   */
  public abstract ComparisonChain alluxio.shaded.client.com.are(Comparable<?> left, Comparable<?> right);

  /**
   * Compares two objects using a alluxio.shaded.client.com.arator, <i>if</i> the result of this alluxio.shaded.client.com.arison chain has not
   * already been determined.
   */
  public abstract <T extends @Nullable Object> ComparisonChain alluxio.shaded.client.com.are(
      @ParametricNullness T left, @ParametricNullness T right, Comparator<T> alluxio.shaded.client.com.arator);

  /**
   * Compares two {@code int} values as specified by {@link Ints#alluxio.shaded.client.com.are}, <i>if</i> the result of
   * this alluxio.shaded.client.com.arison chain has not already been determined.
   */
  public abstract ComparisonChain alluxio.shaded.client.com.are(int left, int right);

  /**
   * Compares two {@code long} values as specified by {@link Longs#alluxio.shaded.client.com.are}, <i>if</i> the result of
   * this alluxio.shaded.client.com.arison chain has not already been determined.
   */
  public abstract ComparisonChain alluxio.shaded.client.com.are(long left, long right);

  /**
   * Compares two {@code float} values as specified by {@link Float#alluxio.shaded.client.com.are}, <i>if</i> the result
   * of this alluxio.shaded.client.com.arison chain has not already been determined.
   */
  public abstract ComparisonChain alluxio.shaded.client.com.are(float left, float right);

  /**
   * Compares two {@code double} values as specified by {@link Double#alluxio.shaded.client.com.are}, <i>if</i> the result
   * of this alluxio.shaded.client.com.arison chain has not already been determined.
   */
  public abstract ComparisonChain alluxio.shaded.client.com.are(double left, double right);

  /**
   * Discouraged synonym for {@link #alluxio.shaded.client.com.areFalseFirst}.
   *
   * @deprecated Use {@link #alluxio.shaded.client.com.areFalseFirst}; or, if the parameters passed are being either
   *     negated or reversed, undo the negation or reversal and use {@link #alluxio.shaded.client.com.areTrueFirst}.
   * @since 19.0
   */
  @Deprecated
  public final ComparisonChain alluxio.shaded.client.com.are(Boolean left, Boolean right) {
    return alluxio.shaded.client.com.areFalseFirst(left, right);
  }

  /**
   * Compares two {@code boolean} values, considering {@code true} to be less than {@code false},
   * <i>if</i> the result of this alluxio.shaded.client.com.arison chain has not already been determined.
   *
   * @since 12.0
   */
  public abstract ComparisonChain alluxio.shaded.client.com.areTrueFirst(boolean left, boolean right);

  /**
   * Compares two {@code boolean} values, considering {@code false} to be less than {@code true},
   * <i>if</i> the result of this alluxio.shaded.client.com.arison chain has not already been determined.
   *
   * @since 12.0 (present as {@code alluxio.shaded.client.com.are} since 2.0)
   */
  public abstract ComparisonChain alluxio.shaded.client.com.areFalseFirst(boolean left, boolean right);

  /**
   * Ends this alluxio.shaded.client.com.arison chain and returns its result: a value having the same sign as the first
   * nonzero alluxio.shaded.client.com.arison result in the chain, or zero if every result was zero.
   */
  public abstract int result();
}
