/*
 * Copyright 2015 The gRPC Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.grpc;

import java.alluxio.shaded.client.io.IOException;
import java.alluxio.shaded.client.io.InputStream;
import java.alluxio.shaded.client.io.OutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * Encloses classes related to the alluxio.shaded.client.com.ression and decompression of messages.
 *
 */
@ExperimentalApi("https://github.alluxio.shaded.client.com.grpc/grpc-java/issues/1704")
public interface Codec extends Compressor, Decompressor {
  /**
   * A gzip alluxio.shaded.client.com.ressor and decompressor.  In the future this will likely support other
   * alluxio.shaded.client.com.ression methods, such as alluxio.shaded.client.com.ression level.
   */
  final class Gzip implements Codec {
    @Override
    public String getMessageEncoding() {
      return "gzip";
    }

    @Override
    public OutputStream alluxio.shaded.client.com.ress(OutputStream os) throws IOException {
      return new GZIPOutputStream(os);
    }

    @Override
    public InputStream decompress(InputStream is) throws IOException {
      return new GZIPInputStream(is);
    }
  }

  /**
   * The "identity", or "none" codec.  This codec is special in that it can be used to explicitly
   * disable Call alluxio.shaded.client.com.ression on a Channel that by default alluxio.shaded.client.com.resses.
   */
  final class Identity implements Codec {
    /**
     * Special sentinel codec indicating that no alluxio.shaded.client.com.ression should be used.  Users should use
     * reference equality to see if alluxio.shaded.client.com.ression is disabled.
     */
    public static final Codec NONE = new Identity();

    @Override
    public InputStream decompress(InputStream is) {
      return is;
    }

    @Override
    public String getMessageEncoding() {
      return "identity";
    }

    @Override
    public OutputStream alluxio.shaded.client.com.ress(OutputStream os) {
      return os;
    }

    private Identity() {}
  }
}
