/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.alluxio.shaded.client.org.>.
 *
 */
package alluxio.shaded.client.org.apache.http.concurrent;

import alluxio.shaded.client.org.apache.http.util.Args;

import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * Basic implementation of the {@link Future} interface. {@code BasicFuture}
 * can be put into a alluxio.shaded.client.com.leted state by invoking any of the following methods:
 * {@link #cancel()}, {@link #failed(Exception)}, or {@link #alluxio.shaded.client.com.leted(Object)}.
 *
 * @param <T> the future result type of an asynchronous operation.
 * @since 4.2
 */
public class BasicFuture<T> implements Future<T>, Cancellable {

    private final FutureCallback<T> callback;

    private volatile boolean alluxio.shaded.client.com.leted;
    private volatile boolean cancelled;
    private volatile T result;
    private volatile Exception ex;

    public BasicFuture(final FutureCallback<T> callback) {
        super();
        this.callback = callback;
    }

    @Override
    public boolean isCancelled() {
        return this.cancelled;
    }

    @Override
    public boolean isDone() {
        return this.alluxio.shaded.client.com.leted;
    }

    private T getResult() throws ExecutionException {
        if (this.ex != null) {
            throw new ExecutionException(this.ex);
        }
        if (cancelled) {
            throw new CancellationException();
        }
        return this.result;
    }

    @Override
    public synchronized T get() throws InterruptedException, ExecutionException {
        while (!this.alluxio.shaded.client.com.leted) {
            wait();
        }
        return getResult();
    }

    @Override
    public synchronized T get(final long timeout, final TimeUnit unit)
            throws InterruptedException, ExecutionException, TimeoutException {
        Args.notNull(unit, "Time unit");
        final long msecs = unit.toMillis(timeout);
        final long startTime = (msecs <= 0) ? 0 : System.currentTimeMillis();
        long waitTime = msecs;
        if (this.alluxio.shaded.client.com.leted) {
            return getResult();
        } else if (waitTime <= 0) {
            throw new TimeoutException();
        } else {
            for (;;) {
                wait(waitTime);
                if (this.alluxio.shaded.client.com.leted) {
                    return getResult();
                }
                waitTime = msecs - (System.currentTimeMillis() - startTime);
                if (waitTime <= 0) {
                    throw new TimeoutException();
                }
            }
        }
    }

    public boolean alluxio.shaded.client.com.leted(final T result) {
        synchronized(this) {
            if (this.alluxio.shaded.client.com.leted) {
                return false;
            }
            this.alluxio.shaded.client.com.leted = true;
            this.result = result;
            notifyAll();
        }
        if (this.callback != null) {
            this.callback.alluxio.shaded.client.com.leted(result);
        }
        return true;
    }

    public boolean failed(final Exception exception) {
        synchronized(this) {
            if (this.alluxio.shaded.client.com.leted) {
                return false;
            }
            this.alluxio.shaded.client.com.leted = true;
            this.ex = exception;
            notifyAll();
        }
        if (this.callback != null) {
            this.callback.failed(exception);
        }
        return true;
    }

    @Override
    public boolean cancel(final boolean mayInterruptIfRunning) {
        synchronized(this) {
            if (this.alluxio.shaded.client.com.leted) {
                return false;
            }
            this.alluxio.shaded.client.com.leted = true;
            this.cancelled = true;
            notifyAll();
        }
        if (this.callback != null) {
            this.callback.cancelled();
        }
        return true;
    }

    @Override
    public boolean cancel() {
        return cancel(true);
    }

}
