/*
 * Copyright (C) 2011 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in alluxio.shaded.client.com.liance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.base;

import static alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.base.Preconditions.checkNotNull;

import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.annotations.Beta;
import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.annotations.GwtCompatible;
import java.alluxio.shaded.client.io.Serializable;
import alluxio.shaded.client.org.checkerframework.checker.nullness.qual.Nullable;

/**
 * Equivalence applied on functional result.
 *
 * @author Bob Lee
 * @since 10.0
 */
@Beta
@GwtCompatible
final class FunctionalEquivalence<F, T> extends Equivalence<F> implements Serializable {

  private static final long serialVersionUID = 0;

  private final Function<F, ? extends T> function;
  private final Equivalence<T> resultEquivalence;

  FunctionalEquivalence(Function<F, ? extends T> function, Equivalence<T> resultEquivalence) {
    this.function = checkNotNull(function);
    this.resultEquivalence = checkNotNull(resultEquivalence);
  }

  @Override
  protected boolean doEquivalent(F a, F b) {
    return resultEquivalence.equivalent(function.apply(a), function.apply(b));
  }

  @Override
  protected int doHash(F a) {
    return resultEquivalence.hash(function.apply(a));
  }

  @Override
  public boolean equals(@Nullable Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj instanceof FunctionalEquivalence) {
      FunctionalEquivalence<?, ?> that = (FunctionalEquivalence<?, ?>) obj;
      return function.equals(that.function) && resultEquivalence.equals(that.resultEquivalence);
    }
    return false;
  }

  @Override
  public int hashCode() {
    return Objects.hashCode(function, resultEquivalence);
  }

  @Override
  public String toString() {
    return resultEquivalence + ".onResultOf(" + function + ")";
  }
}
