/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.block.stream;

import alluxio.Seekable;
import alluxio.client.BoundedStream;
import alluxio.client.PositionedReadable;
import alluxio.client.ReadType;
import alluxio.client.block.stream.DataReader;
import alluxio.client.block.stream.GrpcDataReader;
import alluxio.client.block.stream.LocalFileDataReader;
import alluxio.client.file.FileSystemContext;
import alluxio.client.file.URIStatus;
import alluxio.client.file.options.InStreamOptions;
import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.exception.PreconditionMessage;
import alluxio.exception.status.NotFoundException;
import alluxio.grpc.ReadRequest;
import alluxio.network.protocol.databuffer.DataBuffer;
import alluxio.proto.dataserver.Protocol;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.javax.annotation.concurrent.NotThreadSafe;
import alluxio.util.io.BufferUtils;
import alluxio.util.network.NettyUtils;
import alluxio.util.network.NetworkAddressUtils;
import alluxio.wire.BlockInfo;
import alluxio.wire.WorkerNetAddress;
import java.io.IOException;
import java.io.InputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class BlockInStream
extends InputStream
implements BoundedStream,
Seekable,
PositionedReadable {
    private static final Logger LOG = LoggerFactory.getLogger(BlockInStream.class);
    private final WorkerNetAddress mAddress;
    private final BlockInStreamSource mInStreamSource;
    private final long mId;
    private final long mLength;
    private final byte[] mSingleByte = new byte[1];
    private long mPos = 0L;
    private DataBuffer mCurrentChunk;
    private DataReader mDataReader;
    private final DataReader.Factory mDataReaderFactory;
    private boolean mClosed = false;
    private boolean mEOF = false;

    public static BlockInStream create(FileSystemContext context, BlockInfo info, WorkerNetAddress dataSource, BlockInStreamSource dataSourceType, InStreamOptions options) throws IOException {
        boolean sourceIsLocal;
        URIStatus status = options.getStatus();
        ReadType readType = ReadType.fromProto(options.getOptions().getReadType());
        long blockId = info.getBlockId();
        long blockSize = info.getLength();
        ReadRequest.Builder builder = ReadRequest.newBuilder().setBlockId(blockId).setPromote(readType.isPromote());
        builder.setOpenUfsBlockOptions(options.getOpenUfsBlockOptions(blockId));
        builder.setPositionShort(options.getPositionShort());
        AlluxioConfiguration alluxioConf = context.getClusterConf();
        boolean shortCircuit = alluxioConf.getBoolean(PropertyKey.USER_SHORT_CIRCUIT_ENABLED);
        boolean shortCircuitPreferred = alluxioConf.getBoolean(PropertyKey.USER_SHORT_CIRCUIT_PREFERRED);
        boolean sourceSupportsDomainSocket = NettyUtils.isDomainSocketSupported(dataSource);
        boolean bl = sourceIsLocal = dataSourceType == BlockInStreamSource.LOCAL;
        if (sourceIsLocal && shortCircuit && (shortCircuitPreferred || !sourceSupportsDomainSocket)) {
            LOG.debug("Creating short circuit input stream for block {} @ {}", (Object)blockId, (Object)dataSource);
            try {
                return BlockInStream.createLocalBlockInStream(context, dataSource, blockId, blockSize, options);
            }
            catch (NotFoundException e) {
                LOG.warn("Failed to create short circuit input stream for block {} @ {}. Falling back to network transfer", (Object)blockId, (Object)dataSource);
            }
        }
        LOG.debug("Creating gRPC input stream for block {} @ {} from client {} reading through {}", new Object[]{blockId, dataSource, NetworkAddressUtils.getClientHostName(alluxioConf), dataSource});
        return BlockInStream.createGrpcBlockInStream(context, dataSource, dataSourceType, builder.buildPartial(), blockSize, options);
    }

    private static BlockInStream createLocalBlockInStream(FileSystemContext context, WorkerNetAddress address, long blockId, long length, InStreamOptions options) throws IOException {
        long chunkSize = context.getClusterConf().getBytes(PropertyKey.USER_LOCAL_READER_CHUNK_SIZE_BYTES);
        return new BlockInStream(new LocalFileDataReader.Factory(context, address, blockId, chunkSize, options), address, BlockInStreamSource.LOCAL, blockId, length);
    }

    private static BlockInStream createGrpcBlockInStream(FileSystemContext context, WorkerNetAddress address, BlockInStreamSource blockSource, ReadRequest readRequestPartial, long blockSize, InStreamOptions options) {
        ReadRequest.Builder readRequestBuilder = readRequestPartial.toBuilder();
        long chunkSize = context.getClusterConf().getBytes(PropertyKey.USER_STREAMING_READER_CHUNK_SIZE_BYTES);
        readRequestBuilder.setChunkSize(chunkSize);
        GrpcDataReader.Factory factory = new GrpcDataReader.Factory(context, address, readRequestBuilder.build());
        return new BlockInStream(factory, address, blockSource, readRequestPartial.getBlockId(), blockSize);
    }

    public static BlockInStream createRemoteBlockInStream(FileSystemContext context, long blockId, WorkerNetAddress address, BlockInStreamSource blockSource, long blockSize, Protocol.OpenUfsBlockOptions ufsOptions) {
        long chunkSize = context.getClusterConf().getBytes(PropertyKey.USER_STREAMING_READER_CHUNK_SIZE_BYTES);
        ReadRequest readRequest = ReadRequest.newBuilder().setBlockId(blockId).setOpenUfsBlockOptions(ufsOptions).setChunkSize(chunkSize).buildPartial();
        GrpcDataReader.Factory factory = new GrpcDataReader.Factory(context, address, readRequest.toBuilder().buildPartial());
        return new BlockInStream(factory, address, blockSource, blockId, blockSize);
    }

    protected BlockInStream(DataReader.Factory dataReaderFactory, WorkerNetAddress address, BlockInStreamSource blockSource, long id, long length) {
        this.mDataReaderFactory = dataReaderFactory;
        this.mAddress = address;
        this.mInStreamSource = blockSource;
        this.mId = id;
        this.mLength = length;
    }

    @Override
    public long getPos() {
        return this.mPos;
    }

    @Override
    public int read() throws IOException {
        int bytesRead = this.read(this.mSingleByte);
        if (bytesRead == -1) {
            return -1;
        }
        Preconditions.checkState(bytesRead == 1);
        return BufferUtils.byteToInt(this.mSingleByte[0]);
    }

    @Override
    public int read(byte[] b) throws IOException {
        return this.read(b, 0, b.length);
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.checkIfClosed();
        Preconditions.checkArgument(b != null, (Object)PreconditionMessage.ERR_READ_BUFFER_NULL);
        Preconditions.checkArgument(off >= 0 && len >= 0 && len + off <= b.length, PreconditionMessage.ERR_BUFFER_STATE.toString(), (Object)b.length, (Object)off, (Object)len);
        if (len == 0) {
            return 0;
        }
        this.readChunk();
        if (this.mCurrentChunk == null) {
            this.mEOF = true;
        }
        if (this.mEOF) {
            this.closeDataReader();
            Preconditions.checkState(this.mPos >= this.mLength, PreconditionMessage.BLOCK_LENGTH_INCONSISTENT.toString(), (Object)this.mId, (Object)this.mLength, (Object)this.mPos);
            return -1;
        }
        int toRead = Math.min(len, this.mCurrentChunk.readableBytes());
        this.mCurrentChunk.readBytes(b, off, toRead);
        this.mPos += (long)toRead;
        return toRead;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public int positionedRead(long pos, byte[] b, int off, int len) throws IOException {
        if (len == 0) {
            return 0;
        }
        if (pos < 0L || pos >= this.mLength) {
            return -1;
        }
        int lenCopy = len;
        try (DataReader reader = this.mDataReaderFactory.create(pos, len);){
            while (len > 0) {
                DataBuffer dataBuffer = null;
                try {
                    dataBuffer = reader.readChunk();
                    if (dataBuffer != null) {
                        Preconditions.checkState(dataBuffer.readableBytes() <= len);
                        int toRead = dataBuffer.readableBytes();
                        dataBuffer.readBytes(b, off, toRead);
                        len -= toRead;
                        off += toRead;
                        continue;
                    }
                    break;
                }
                finally {
                    if (dataBuffer == null) continue;
                    dataBuffer.release();
                }
            }
        }
        if (lenCopy == len) {
            return -1;
        }
        return lenCopy - len;
    }

    @Override
    public long remaining() {
        return this.mEOF ? 0L : this.mLength - this.mPos;
    }

    @Override
    public void seek(long pos) throws IOException {
        this.checkIfClosed();
        Preconditions.checkArgument(pos >= 0L, PreconditionMessage.ERR_SEEK_NEGATIVE.toString(), pos);
        Preconditions.checkArgument(pos <= this.mLength, PreconditionMessage.ERR_SEEK_PAST_END_OF_REGION.toString(), this.mId);
        if (pos == this.mPos) {
            return;
        }
        if (pos < this.mPos) {
            this.mEOF = false;
        }
        this.closeDataReader();
        this.mPos = pos;
    }

    @Override
    public long skip(long n) throws IOException {
        this.checkIfClosed();
        if (n <= 0L) {
            return 0L;
        }
        long toSkip = Math.min(this.remaining(), n);
        this.mPos += toSkip;
        this.closeDataReader();
        return toSkip;
    }

    @Override
    public void close() throws IOException {
        try {
            this.closeDataReader();
        }
        finally {
            this.mDataReaderFactory.close();
        }
        this.mClosed = true;
    }

    public boolean isShortCircuit() {
        return this.mDataReaderFactory.isShortCircuit();
    }

    private void readChunk() throws IOException {
        if (this.mDataReader == null) {
            this.mDataReader = this.mDataReaderFactory.create(this.mPos, this.mLength - this.mPos);
        }
        if (this.mCurrentChunk != null && this.mCurrentChunk.readableBytes() == 0) {
            this.mCurrentChunk.release();
            this.mCurrentChunk = null;
        }
        if (this.mCurrentChunk == null) {
            this.mCurrentChunk = this.mDataReader.readChunk();
        }
    }

    private void closeDataReader() throws IOException {
        if (this.mCurrentChunk != null) {
            this.mCurrentChunk.release();
            this.mCurrentChunk = null;
        }
        if (this.mDataReader != null) {
            this.mDataReader.close();
        }
        this.mDataReader = null;
    }

    private void checkIfClosed() {
        Preconditions.checkState(!this.mClosed, (Object)PreconditionMessage.ERR_CLOSED_BLOCK_IN_STREAM);
    }

    public WorkerNetAddress getAddress() {
        return this.mAddress;
    }

    public BlockInStreamSource getSource() {
        return this.mInStreamSource;
    }

    public long getId() {
        return this.mId;
    }

    public static enum BlockInStreamSource {
        LOCAL,
        REMOTE,
        UFS;

    }
}

