/*
 * Copyright 2002-2009 Andy Clark, Marc Guillemot
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.htmlunit.cyberneko;

/**
 * Parser for the Pre-defined named HTML entities.
 * <a href="https://html.spec.whatwg.org/multipage/parsing.html#character-reference-state">12.2.5.72 Character reference state</a>
 * <p>
 * From the spec:<br>
 * Consume the maximum number of characters possible, with the consumed characters matching
 * one of the identifiers in the first column of the named character references table
 * (in a case-sensitive manner).
 * Append each character to the temporary buffer when it's consumed.
 *
 * @author Ronald Brill
 */
public class HTMLEntitiesParser {
    public static final int STATE_START = 0;
    private static final int STATE_ENDS_WITH_SEMICOLON = -2;

    private static final int STATE_HEXADECIMAL_CHAR = -102;
    private static final int STATE_DECIMAL_CHAR = -104;

    private static final int STATE_NUMERIC_CHAR_END_SEMICOLON_MISSING = -105;

    private int state;
    private int consumedCount;
    private String match;
    private int code;
    private int matchLength;

    public String getMatch() {
        return match;
    }

    public int getMatchLength() {
        return matchLength;
    }

    public int getRewindCount() {
        return consumedCount - matchLength;
    }

    public boolean endsWithSemicolon() {
        return STATE_ENDS_WITH_SEMICOLON == state;
    }

    public HTMLEntitiesParser() {
        state = STATE_START;
    }

    public void setMatchFromCode() {
        // If the number is 0x00, then this is a null-character-reference parse error. Set the character reference code to 0xFFFD.
        // If the number is greater than 0x10FFFF, then this is a character-reference-outside-unicode-range parse error. Set the character reference code to 0xFFFD.
        if ((0x00 == code) || (code > 0x10FFFF)) {
            match = "\uFFFD";
            matchLength = consumedCount;
            return;
        }

        // If the number is a surrogate, then this is a surrogate-character-reference parse error. Set the character reference code to 0xFFFD
        if (Character.isSurrogate((char) code)) {
            match = "\uFFFD";
            return;
        }

        // If the number is a noncharacter, then this is a noncharacter-character-reference parse error.

        // If the number is 0x0D, or a control that's not ASCII whitespace, then this is a control-character-reference parse error.

        // If the number is one of the numbers in the first column of the following table, then find the row with that number in the first column,
        // and set the character reference code to the number in the second column of that row.
        switch (code) {
            case 0x80:
                match = "\u20AC";
                matchLength = consumedCount;
                return;

            case 0x82:
                match = "\u201A";
                matchLength = consumedCount;
                return;

            case 0x83:
                match = "\u0192";
                matchLength = consumedCount;
                return;

            case 0x84:
                match = "\u201E";
                matchLength = consumedCount;
                return;

            case 0x85:
                match = "\u2026";
                matchLength = consumedCount;
                return;

            case 0x86:
                match = "\u2020";
                matchLength = consumedCount;
                return;

            case 0x87:
                match = "\u2021";
                matchLength = consumedCount;
                return;

            case 0x88:
                match = "\u02C6";
                matchLength = consumedCount;
                return;

            case 0x89:
                match = "\u2030";
                matchLength = consumedCount;
                return;

            case 0x8A:
                match = "\u0160";
                matchLength = consumedCount;
                return;

            case 0x8B:
                match = "\u2039";
                matchLength = consumedCount;
                return;

            case 0x8C:
                match = "\u0152";
                matchLength = consumedCount;
                return;

            case 0x8E:
                match = "\u017D";
                matchLength = consumedCount;
                return;

            case 0x91:
                match = "\u2018";
                matchLength = consumedCount;
                return;

            case 0x92:
                match = "\u2019";
                matchLength = consumedCount;
                return;

            case 0x93:
                match = "\u201C";
                matchLength = consumedCount;
                return;

            case 0x94:
                match = "\u201D";
                matchLength = consumedCount;
                return;

            case 0x95:
                match = "\u2022";
                matchLength = consumedCount;
                return;

            case 0x96:
                match = "\u2013";
                matchLength = consumedCount;
                return;

            case 0x97:
                match = "\u2014";
                matchLength = consumedCount;
                return;

            case 0x98:
                match = "\u20DC";
                matchLength = consumedCount;
                return;

            case 0x99:
                match = "\u2122";
                matchLength = consumedCount;
                return;

            case 0x9A:
                match = "\u0161";
                matchLength = consumedCount;
                return;

            case 0x9B:
                match = "\u203A";
                matchLength = consumedCount;
                return;

            case 0x9C:
                match = "\u0153";
                matchLength = consumedCount;
                return;

            case 0x9E:
                match = "\u017E";
                matchLength = consumedCount;
                return;

            case 0x9F:
                match = "\u0178";
                matchLength = consumedCount;
                return;

            default:
                break;
        }
        match = new String(Character.toChars(code));
        matchLength = consumedCount;
    }

    public boolean parseNumeric(final int current) {
        consumedCount++;
        switch (state) {
            case STATE_START:
                if ('X' == current || 'x' == current) {
                    state = STATE_HEXADECIMAL_CHAR;
                    return true;
                }
                if ('0' <= current && current <= '9') {
                    state = STATE_DECIMAL_CHAR;
                    code = (code * 10) + current - 0x30;
                    return true;
                }
                break;
            case STATE_HEXADECIMAL_CHAR:
                if ('0' <= current && current <= '9') {
                    code = (code * 16) + current - 0x30;
                    return true;
                }
                if ('A' <= current && current <= 'F') {
                    code = (code * 16) + current - 0x37;
                    return true;
                }
                if ('a' <= current && current <= 'f') {
                    code = (code * 16) + current - 0x57;
                    return true;
                }
                if (';' == current) {
                    setMatchFromCode();
                    return false;
                }

                state = STATE_NUMERIC_CHAR_END_SEMICOLON_MISSING;
                setMatchFromCode();
                matchLength = consumedCount - 1;
                break;
            case STATE_DECIMAL_CHAR:
                if ('0' <= current && current <= '9') {
                    code = (code * 10) + current - 0x30;
                    return true;
                }
                if (';' == current) {
                    setMatchFromCode();
                    return false;
                }

                state = STATE_NUMERIC_CHAR_END_SEMICOLON_MISSING;
                setMatchFromCode();
                matchLength = consumedCount - 1;
                break;
        }
        return false;
    }

    private boolean parse1(final int current) {
        consumedCount++;
        switch (state) {
            case 0:
                switch (current) {
                    case 'A' :
                        state = 1;
                        return true;
                    case 'B' :
                        state = 77;
                        return true;
                    case 'C' :
                        state = 126;
                        return true;
                    case 'D' :
                        state = 342;
                        return true;
                    case 'E' :
                        state = 666;
                        return true;
                    case 'F' :
                        state = 788;
                        return true;
                    case 'G' :
                        state = 842;
                        return true;
                    case 'H' :
                        state = 929;
                        return true;
                    case 'I' :
                        state = 995;
                        return true;
                    case 'J' :
                        state = 1096;
                        return true;
                    case 'K' :
                        state = 1118;
                        return true;
                    case 'L' :
                        state = 1143;
                        return true;
                    case 'M' :
                        state = 1514;
                        return true;
                    case 'N' :
                        state = 1553;
                        return true;
                    case 'O' :
                        state = 2064;
                        return true;
                    case 'P' :
                        state = 2173;
                        return true;
                    case 'Q' :
                        state = 2258;
                        return true;
                    case 'R' :
                        state = 2270;
                        return true;
                    case 'S' :
                        state = 2525;
                        return true;
                    case 'T' :
                        state = 2722;
                        return true;
                    case 'U' :
                        state = 2817;
                        return true;
                    case 'V' :
                        state = 2997;
                        return true;
                    case 'W' :
                        state = 3067;
                        return true;
                    case 'X' :
                        state = 3084;
                        return true;
                    case 'Y' :
                        state = 3094;
                        return true;
                    case 'Z' :
                        state = 3125;
                        return true;
                    case 'a' :
                        state = 3166;
                        return true;
                    case 'b' :
                        state = 3311;
                        return true;
                    case 'c' :
                        state = 3623;
                        return true;
                    case 'd' :
                        state = 3915;
                        return true;
                    case 'e' :
                        state = 4156;
                        return true;
                    case 'f' :
                        state = 4342;
                        return true;
                    case 'g' :
                        state = 4442;
                        return true;
                    case 'h' :
                        state = 4588;
                        return true;
                    case 'i' :
                        state = 4704;
                        return true;
                    case 'j' :
                        state = 4847;
                        return true;
                    case 'k' :
                        state = 4873;
                        return true;
                    case 'l' :
                        state = 4904;
                        return true;
                    case 'm' :
                        state = 5397;
                        return true;
                    case 'n' :
                        state = 5510;
                        return true;
                    case 'o' :
                        state = 5945;
                        return true;
                    case 'p' :
                        state = 6092;
                        return true;
                    case 'q' :
                        state = 6272;
                        return true;
                    case 'r' :
                        state = 6309;
                        return true;
                    case 's' :
                        state = 6642;
                        return true;
                    case 't' :
                        state = 7043;
                        return true;
                    case 'u' :
                        state = 7238;
                        return true;
                    case 'v' :
                        state = 7408;
                        return true;
                    case 'w' :
                        state = 7562;
                        return true;
                    case 'x' :
                        state = 7593;
                        return true;
                    case 'y' :
                        state = 7672;
                        return true;
                    case 'z' :
                        state = 7702;
                        return true;
                }
                break;
            case 1:
                switch (current) {
                    case 'E' :
                        state = 2;
                        return true;
                    case 'M' :
                        state = 6;
                        return true;
                    case 'a' :
                        state = 8;
                        return true;
                    case 'b' :
                        state = 13;
                        return true;
                    case 'c' :
                        state = 18;
                        return true;
                    case 'f' :
                        state = 23;
                        return true;
                    case 'g' :
                        state = 25;
                        return true;
                    case 'l' :
                        state = 30;
                        return true;
                    case 'm' :
                        state = 34;
                        return true;
                    case 'n' :
                        state = 38;
                        return true;
                    case 'o' :
                        state = 40;
                        return true;
                    case 'p' :
                        state = 46;
                        return true;
                    case 'r' :
                        state = 58;
                        return true;
                    case 's' :
                        state = 62;
                        return true;
                    case 't' :
                        state = 69;
                        return true;
                    case 'u' :
                        state = 74;
                        return true;
                }
                break;
            case 2:
                switch (current) {
                    case 'l' :
                        state = 3;
                        return true;
                }
                break;
            case 3:
                switch (current) {
                    case 'i' :
                        state = 4;
                        return true;
                }
                break;
            case 4:
                switch (current) {
                    case 'g' : // AElig
                        match = "\u00C6";
                        matchLength = consumedCount;
                        state = 5;
                        return true;
                }
                break;
            case 5:
                switch (current) {
                    case ';' : // AElig;
                        match = "\u00C6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 6:
                switch (current) {
                    case 'P' : // AMP
                        match = "&";
                        matchLength = consumedCount;
                        state = 7;
                        return true;
                }
                break;
            case 7:
                switch (current) {
                    case ';' : // AMP;
                        match = "&";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 8:
                switch (current) {
                    case 'c' :
                        state = 9;
                        return true;
                }
                break;
            case 9:
                switch (current) {
                    case 'u' :
                        state = 10;
                        return true;
                }
                break;
            case 10:
                switch (current) {
                    case 't' :
                        state = 11;
                        return true;
                }
                break;
            case 11:
                switch (current) {
                    case 'e' : // Aacute
                        match = "\u00C1";
                        matchLength = consumedCount;
                        state = 12;
                        return true;
                }
                break;
            case 12:
                switch (current) {
                    case ';' : // Aacute;
                        match = "\u00C1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 13:
                switch (current) {
                    case 'r' :
                        state = 14;
                        return true;
                }
                break;
            case 14:
                switch (current) {
                    case 'e' :
                        state = 15;
                        return true;
                }
                break;
            case 15:
                switch (current) {
                    case 'v' :
                        state = 16;
                        return true;
                }
                break;
            case 16:
                switch (current) {
                    case 'e' :
                        state = 17;
                        return true;
                }
                break;
            case 17:
                switch (current) {
                    case ';' : // Abreve;
                        match = "\u0102";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 18:
                switch (current) {
                    case 'i' :
                        state = 19;
                        return true;
                    case 'y' :
                        state = 22;
                        return true;
                }
                break;
            case 19:
                switch (current) {
                    case 'r' :
                        state = 20;
                        return true;
                }
                break;
            case 20:
                switch (current) {
                    case 'c' : // Acirc
                        match = "\u00C2";
                        matchLength = consumedCount;
                        state = 21;
                        return true;
                }
                break;
            case 21:
                switch (current) {
                    case ';' : // Acirc;
                        match = "\u00C2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 22:
                switch (current) {
                    case ';' : // Acy;
                        match = "\u0410";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 23:
                switch (current) {
                    case 'r' :
                        state = 24;
                        return true;
                }
                break;
            case 24:
                switch (current) {
                    case ';' : // Afr;
                        match = "\uD835\uDD04";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 25:
                switch (current) {
                    case 'r' :
                        state = 26;
                        return true;
                }
                break;
            case 26:
                switch (current) {
                    case 'a' :
                        state = 27;
                        return true;
                }
                break;
            case 27:
                switch (current) {
                    case 'v' :
                        state = 28;
                        return true;
                }
                break;
            case 28:
                switch (current) {
                    case 'e' : // Agrave
                        match = "\u00C0";
                        matchLength = consumedCount;
                        state = 29;
                        return true;
                }
                break;
            case 29:
                switch (current) {
                    case ';' : // Agrave;
                        match = "\u00C0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 30:
                switch (current) {
                    case 'p' :
                        state = 31;
                        return true;
                }
                break;
            case 31:
                switch (current) {
                    case 'h' :
                        state = 32;
                        return true;
                }
                break;
            case 32:
                switch (current) {
                    case 'a' :
                        state = 33;
                        return true;
                }
                break;
            case 33:
                switch (current) {
                    case ';' : // Alpha;
                        match = "\u0391";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 34:
                switch (current) {
                    case 'a' :
                        state = 35;
                        return true;
                }
                break;
            case 35:
                switch (current) {
                    case 'c' :
                        state = 36;
                        return true;
                }
                break;
            case 36:
                switch (current) {
                    case 'r' :
                        state = 37;
                        return true;
                }
                break;
            case 37:
                switch (current) {
                    case ';' : // Amacr;
                        match = "\u0100";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 38:
                switch (current) {
                    case 'd' :
                        state = 39;
                        return true;
                }
                break;
            case 39:
                switch (current) {
                    case ';' : // And;
                        match = "\u2A53";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 40:
                switch (current) {
                    case 'g' :
                        state = 41;
                        return true;
                    case 'p' :
                        state = 44;
                        return true;
                }
                break;
            case 41:
                switch (current) {
                    case 'o' :
                        state = 42;
                        return true;
                }
                break;
            case 42:
                switch (current) {
                    case 'n' :
                        state = 43;
                        return true;
                }
                break;
            case 43:
                switch (current) {
                    case ';' : // Aogon;
                        match = "\u0104";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 44:
                switch (current) {
                    case 'f' :
                        state = 45;
                        return true;
                }
                break;
            case 45:
                switch (current) {
                    case ';' : // Aopf;
                        match = "\uD835\uDD38";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 46:
                switch (current) {
                    case 'p' :
                        state = 47;
                        return true;
                }
                break;
            case 47:
                switch (current) {
                    case 'l' :
                        state = 48;
                        return true;
                }
                break;
            case 48:
                switch (current) {
                    case 'y' :
                        state = 49;
                        return true;
                }
                break;
            case 49:
                switch (current) {
                    case 'F' :
                        state = 50;
                        return true;
                }
                break;
            case 50:
                switch (current) {
                    case 'u' :
                        state = 51;
                        return true;
                }
                break;
            case 51:
                switch (current) {
                    case 'n' :
                        state = 52;
                        return true;
                }
                break;
            case 52:
                switch (current) {
                    case 'c' :
                        state = 53;
                        return true;
                }
                break;
            case 53:
                switch (current) {
                    case 't' :
                        state = 54;
                        return true;
                }
                break;
            case 54:
                switch (current) {
                    case 'i' :
                        state = 55;
                        return true;
                }
                break;
            case 55:
                switch (current) {
                    case 'o' :
                        state = 56;
                        return true;
                }
                break;
            case 56:
                switch (current) {
                    case 'n' :
                        state = 57;
                        return true;
                }
                break;
            case 57:
                switch (current) {
                    case ';' : // ApplyFunction;
                        match = "\u2061";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 58:
                switch (current) {
                    case 'i' :
                        state = 59;
                        return true;
                }
                break;
            case 59:
                switch (current) {
                    case 'n' :
                        state = 60;
                        return true;
                }
                break;
            case 60:
                switch (current) {
                    case 'g' : // Aring
                        match = "\u00C5";
                        matchLength = consumedCount;
                        state = 61;
                        return true;
                }
                break;
            case 61:
                switch (current) {
                    case ';' : // Aring;
                        match = "\u00C5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 62:
                switch (current) {
                    case 'c' :
                        state = 63;
                        return true;
                    case 's' :
                        state = 65;
                        return true;
                }
                break;
            case 63:
                switch (current) {
                    case 'r' :
                        state = 64;
                        return true;
                }
                break;
            case 64:
                switch (current) {
                    case ';' : // Ascr;
                        match = "\uD835\uDC9C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 65:
                switch (current) {
                    case 'i' :
                        state = 66;
                        return true;
                }
                break;
            case 66:
                switch (current) {
                    case 'g' :
                        state = 67;
                        return true;
                }
                break;
            case 67:
                switch (current) {
                    case 'n' :
                        state = 68;
                        return true;
                }
                break;
            case 68:
                switch (current) {
                    case ';' : // Assign;
                        match = "\u2254";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 69:
                switch (current) {
                    case 'i' :
                        state = 70;
                        return true;
                }
                break;
            case 70:
                switch (current) {
                    case 'l' :
                        state = 71;
                        return true;
                }
                break;
            case 71:
                switch (current) {
                    case 'd' :
                        state = 72;
                        return true;
                }
                break;
            case 72:
                switch (current) {
                    case 'e' : // Atilde
                        match = "\u00C3";
                        matchLength = consumedCount;
                        state = 73;
                        return true;
                }
                break;
            case 73:
                switch (current) {
                    case ';' : // Atilde;
                        match = "\u00C3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 74:
                switch (current) {
                    case 'm' :
                        state = 75;
                        return true;
                }
                break;
            case 75:
                switch (current) {
                    case 'l' : // Auml
                        match = "\u00C4";
                        matchLength = consumedCount;
                        state = 76;
                        return true;
                }
                break;
            case 76:
                switch (current) {
                    case ';' : // Auml;
                        match = "\u00C4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 77:
                switch (current) {
                    case 'a' :
                        state = 78;
                        return true;
                    case 'c' :
                        state = 91;
                        return true;
                    case 'e' :
                        state = 93;
                        return true;
                    case 'f' :
                        state = 109;
                        return true;
                    case 'o' :
                        state = 111;
                        return true;
                    case 'r' :
                        state = 114;
                        return true;
                    case 's' :
                        state = 118;
                        return true;
                    case 'u' :
                        state = 121;
                        return true;
                }
                break;
            case 78:
                switch (current) {
                    case 'c' :
                        state = 79;
                        return true;
                    case 'r' :
                        state = 86;
                        return true;
                }
                break;
            case 79:
                switch (current) {
                    case 'k' :
                        state = 80;
                        return true;
                }
                break;
            case 80:
                switch (current) {
                    case 's' :
                        state = 81;
                        return true;
                }
                break;
            case 81:
                switch (current) {
                    case 'l' :
                        state = 82;
                        return true;
                }
                break;
            case 82:
                switch (current) {
                    case 'a' :
                        state = 83;
                        return true;
                }
                break;
            case 83:
                switch (current) {
                    case 's' :
                        state = 84;
                        return true;
                }
                break;
            case 84:
                switch (current) {
                    case 'h' :
                        state = 85;
                        return true;
                }
                break;
            case 85:
                switch (current) {
                    case ';' : // Backslash;
                        match = "\u2216";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 86:
                switch (current) {
                    case 'v' :
                        state = 87;
                        return true;
                    case 'w' :
                        state = 88;
                        return true;
                }
                break;
            case 87:
                switch (current) {
                    case ';' : // Barv;
                        match = "\u2AE7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 88:
                switch (current) {
                    case 'e' :
                        state = 89;
                        return true;
                }
                break;
            case 89:
                switch (current) {
                    case 'd' :
                        state = 90;
                        return true;
                }
                break;
            case 90:
                switch (current) {
                    case ';' : // Barwed;
                        match = "\u2306";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 91:
                switch (current) {
                    case 'y' :
                        state = 92;
                        return true;
                }
                break;
            case 92:
                switch (current) {
                    case ';' : // Bcy;
                        match = "\u0411";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 93:
                switch (current) {
                    case 'c' :
                        state = 94;
                        return true;
                    case 'r' :
                        state = 99;
                        return true;
                    case 't' :
                        state = 107;
                        return true;
                }
                break;
            case 94:
                switch (current) {
                    case 'a' :
                        state = 95;
                        return true;
                }
                break;
            case 95:
                switch (current) {
                    case 'u' :
                        state = 96;
                        return true;
                }
                break;
            case 96:
                switch (current) {
                    case 's' :
                        state = 97;
                        return true;
                }
                break;
            case 97:
                switch (current) {
                    case 'e' :
                        state = 98;
                        return true;
                }
                break;
            case 98:
                switch (current) {
                    case ';' : // Because;
                        match = "\u2235";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 99:
                switch (current) {
                    case 'n' :
                        state = 100;
                        return true;
                }
                break;
            case 100:
                switch (current) {
                    case 'o' :
                        state = 101;
                        return true;
                }
                break;
            case 101:
                switch (current) {
                    case 'u' :
                        state = 102;
                        return true;
                }
                break;
            case 102:
                switch (current) {
                    case 'l' :
                        state = 103;
                        return true;
                }
                break;
            case 103:
                switch (current) {
                    case 'l' :
                        state = 104;
                        return true;
                }
                break;
            case 104:
                switch (current) {
                    case 'i' :
                        state = 105;
                        return true;
                }
                break;
            case 105:
                switch (current) {
                    case 's' :
                        state = 106;
                        return true;
                }
                break;
            case 106:
                switch (current) {
                    case ';' : // Bernoullis;
                        match = "\u212C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 107:
                switch (current) {
                    case 'a' :
                        state = 108;
                        return true;
                }
                break;
            case 108:
                switch (current) {
                    case ';' : // Beta;
                        match = "\u0392";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 109:
                switch (current) {
                    case 'r' :
                        state = 110;
                        return true;
                }
                break;
            case 110:
                switch (current) {
                    case ';' : // Bfr;
                        match = "\uD835\uDD05";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 111:
                switch (current) {
                    case 'p' :
                        state = 112;
                        return true;
                }
                break;
            case 112:
                switch (current) {
                    case 'f' :
                        state = 113;
                        return true;
                }
                break;
            case 113:
                switch (current) {
                    case ';' : // Bopf;
                        match = "\uD835\uDD39";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 114:
                switch (current) {
                    case 'e' :
                        state = 115;
                        return true;
                }
                break;
            case 115:
                switch (current) {
                    case 'v' :
                        state = 116;
                        return true;
                }
                break;
            case 116:
                switch (current) {
                    case 'e' :
                        state = 117;
                        return true;
                }
                break;
            case 117:
                switch (current) {
                    case ';' : // Breve;
                        match = "\u02D8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 118:
                switch (current) {
                    case 'c' :
                        state = 119;
                        return true;
                }
                break;
            case 119:
                switch (current) {
                    case 'r' :
                        state = 120;
                        return true;
                }
                break;
            case 120:
                switch (current) {
                    case ';' : // Bscr;
                        match = "\u212C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 121:
                switch (current) {
                    case 'm' :
                        state = 122;
                        return true;
                }
                break;
            case 122:
                switch (current) {
                    case 'p' :
                        state = 123;
                        return true;
                }
                break;
            case 123:
                switch (current) {
                    case 'e' :
                        state = 124;
                        return true;
                }
                break;
            case 124:
                switch (current) {
                    case 'q' :
                        state = 125;
                        return true;
                }
                break;
            case 125:
                switch (current) {
                    case ';' : // Bumpeq;
                        match = "\u224E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 126:
                switch (current) {
                    case 'H' :
                        state = 127;
                        return true;
                    case 'O' :
                        state = 130;
                        return true;
                    case 'a' :
                        state = 133;
                        return true;
                    case 'c' :
                        state = 161;
                        return true;
                    case 'd' :
                        state = 178;
                        return true;
                    case 'e' :
                        state = 181;
                        return true;
                    case 'f' :
                        state = 194;
                        return true;
                    case 'h' :
                        state = 196;
                        return true;
                    case 'i' :
                        state = 198;
                        return true;
                    case 'l' :
                        state = 220;
                        return true;
                    case 'o' :
                        state = 266;
                        return true;
                    case 'r' :
                        state = 330;
                        return true;
                    case 's' :
                        state = 334;
                        return true;
                    case 'u' :
                        state = 337;
                        return true;
                }
                break;
            case 127:
                switch (current) {
                    case 'c' :
                        state = 128;
                        return true;
                }
                break;
            case 128:
                switch (current) {
                    case 'y' :
                        state = 129;
                        return true;
                }
                break;
            case 129:
                switch (current) {
                    case ';' : // CHcy;
                        match = "\u0427";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 130:
                switch (current) {
                    case 'P' :
                        state = 131;
                        return true;
                }
                break;
            case 131:
                switch (current) {
                    case 'Y' : // COPY
                        match = "\u00A9";
                        matchLength = consumedCount;
                        state = 132;
                        return true;
                }
                break;
            case 132:
                switch (current) {
                    case ';' : // COPY;
                        match = "\u00A9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 133:
                switch (current) {
                    case 'c' :
                        state = 134;
                        return true;
                    case 'p' :
                        state = 138;
                        return true;
                    case 'y' :
                        state = 156;
                        return true;
                }
                break;
            case 134:
                switch (current) {
                    case 'u' :
                        state = 135;
                        return true;
                }
                break;
            case 135:
                switch (current) {
                    case 't' :
                        state = 136;
                        return true;
                }
                break;
            case 136:
                switch (current) {
                    case 'e' :
                        state = 137;
                        return true;
                }
                break;
            case 137:
                switch (current) {
                    case ';' : // Cacute;
                        match = "\u0106";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 138:
                switch (current) {
                    case ';' : // Cap;
                        match = "\u22D2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'i' :
                        state = 139;
                        return true;
                }
                break;
            case 139:
                switch (current) {
                    case 't' :
                        state = 140;
                        return true;
                }
                break;
            case 140:
                switch (current) {
                    case 'a' :
                        state = 141;
                        return true;
                }
                break;
            case 141:
                switch (current) {
                    case 'l' :
                        state = 142;
                        return true;
                }
                break;
            case 142:
                switch (current) {
                    case 'D' :
                        state = 143;
                        return true;
                }
                break;
            case 143:
                switch (current) {
                    case 'i' :
                        state = 144;
                        return true;
                }
                break;
            case 144:
                switch (current) {
                    case 'f' :
                        state = 145;
                        return true;
                }
                break;
            case 145:
                switch (current) {
                    case 'f' :
                        state = 146;
                        return true;
                }
                break;
            case 146:
                switch (current) {
                    case 'e' :
                        state = 147;
                        return true;
                }
                break;
            case 147:
                switch (current) {
                    case 'r' :
                        state = 148;
                        return true;
                }
                break;
            case 148:
                switch (current) {
                    case 'e' :
                        state = 149;
                        return true;
                }
                break;
            case 149:
                switch (current) {
                    case 'n' :
                        state = 150;
                        return true;
                }
                break;
            case 150:
                switch (current) {
                    case 't' :
                        state = 151;
                        return true;
                }
                break;
            case 151:
                switch (current) {
                    case 'i' :
                        state = 152;
                        return true;
                }
                break;
            case 152:
                switch (current) {
                    case 'a' :
                        state = 153;
                        return true;
                }
                break;
            case 153:
                switch (current) {
                    case 'l' :
                        state = 154;
                        return true;
                }
                break;
            case 154:
                switch (current) {
                    case 'D' :
                        state = 155;
                        return true;
                }
                break;
            case 155:
                switch (current) {
                    case ';' : // CapitalDifferentialD;
                        match = "\u2145";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 156:
                switch (current) {
                    case 'l' :
                        state = 157;
                        return true;
                }
                break;
            case 157:
                switch (current) {
                    case 'e' :
                        state = 158;
                        return true;
                }
                break;
            case 158:
                switch (current) {
                    case 'y' :
                        state = 159;
                        return true;
                }
                break;
            case 159:
                switch (current) {
                    case 's' :
                        state = 160;
                        return true;
                }
                break;
            case 160:
                switch (current) {
                    case ';' : // Cayleys;
                        match = "\u212D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 161:
                switch (current) {
                    case 'a' :
                        state = 162;
                        return true;
                    case 'e' :
                        state = 166;
                        return true;
                    case 'i' :
                        state = 170;
                        return true;
                    case 'o' :
                        state = 173;
                        return true;
                }
                break;
            case 162:
                switch (current) {
                    case 'r' :
                        state = 163;
                        return true;
                }
                break;
            case 163:
                switch (current) {
                    case 'o' :
                        state = 164;
                        return true;
                }
                break;
            case 164:
                switch (current) {
                    case 'n' :
                        state = 165;
                        return true;
                }
                break;
            case 165:
                switch (current) {
                    case ';' : // Ccaron;
                        match = "\u010C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 166:
                switch (current) {
                    case 'd' :
                        state = 167;
                        return true;
                }
                break;
            case 167:
                switch (current) {
                    case 'i' :
                        state = 168;
                        return true;
                }
                break;
            case 168:
                switch (current) {
                    case 'l' : // Ccedil
                        match = "\u00C7";
                        matchLength = consumedCount;
                        state = 169;
                        return true;
                }
                break;
            case 169:
                switch (current) {
                    case ';' : // Ccedil;
                        match = "\u00C7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 170:
                switch (current) {
                    case 'r' :
                        state = 171;
                        return true;
                }
                break;
            case 171:
                switch (current) {
                    case 'c' :
                        state = 172;
                        return true;
                }
                break;
            case 172:
                switch (current) {
                    case ';' : // Ccirc;
                        match = "\u0108";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 173:
                switch (current) {
                    case 'n' :
                        state = 174;
                        return true;
                }
                break;
            case 174:
                switch (current) {
                    case 'i' :
                        state = 175;
                        return true;
                }
                break;
            case 175:
                switch (current) {
                    case 'n' :
                        state = 176;
                        return true;
                }
                break;
            case 176:
                switch (current) {
                    case 't' :
                        state = 177;
                        return true;
                }
                break;
            case 177:
                switch (current) {
                    case ';' : // Cconint;
                        match = "\u2230";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 178:
                switch (current) {
                    case 'o' :
                        state = 179;
                        return true;
                }
                break;
            case 179:
                switch (current) {
                    case 't' :
                        state = 180;
                        return true;
                }
                break;
            case 180:
                switch (current) {
                    case ';' : // Cdot;
                        match = "\u010A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 181:
                switch (current) {
                    case 'd' :
                        state = 182;
                        return true;
                    case 'n' :
                        state = 187;
                        return true;
                }
                break;
            case 182:
                switch (current) {
                    case 'i' :
                        state = 183;
                        return true;
                }
                break;
            case 183:
                switch (current) {
                    case 'l' :
                        state = 184;
                        return true;
                }
                break;
            case 184:
                switch (current) {
                    case 'l' :
                        state = 185;
                        return true;
                }
                break;
            case 185:
                switch (current) {
                    case 'a' :
                        state = 186;
                        return true;
                }
                break;
            case 186:
                switch (current) {
                    case ';' : // Cedilla;
                        match = "\u00B8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 187:
                switch (current) {
                    case 't' :
                        state = 188;
                        return true;
                }
                break;
            case 188:
                switch (current) {
                    case 'e' :
                        state = 189;
                        return true;
                }
                break;
            case 189:
                switch (current) {
                    case 'r' :
                        state = 190;
                        return true;
                }
                break;
            case 190:
                switch (current) {
                    case 'D' :
                        state = 191;
                        return true;
                }
                break;
            case 191:
                switch (current) {
                    case 'o' :
                        state = 192;
                        return true;
                }
                break;
            case 192:
                switch (current) {
                    case 't' :
                        state = 193;
                        return true;
                }
                break;
            case 193:
                switch (current) {
                    case ';' : // CenterDot;
                        match = "\u00B7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 194:
                switch (current) {
                    case 'r' :
                        state = 195;
                        return true;
                }
                break;
            case 195:
                switch (current) {
                    case ';' : // Cfr;
                        match = "\u212D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 196:
                switch (current) {
                    case 'i' :
                        state = 197;
                        return true;
                }
                break;
            case 197:
                switch (current) {
                    case ';' : // Chi;
                        match = "\u03A7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 198:
                switch (current) {
                    case 'r' :
                        state = 199;
                        return true;
                }
                break;
            case 199:
                switch (current) {
                    case 'c' :
                        state = 200;
                        return true;
                }
                break;
            case 200:
                switch (current) {
                    case 'l' :
                        state = 201;
                        return true;
                }
                break;
            case 201:
                switch (current) {
                    case 'e' :
                        state = 202;
                        return true;
                }
                break;
            case 202:
                switch (current) {
                    case 'D' :
                        state = 203;
                        return true;
                    case 'M' :
                        state = 206;
                        return true;
                    case 'P' :
                        state = 211;
                        return true;
                    case 'T' :
                        state = 215;
                        return true;
                }
                break;
            case 203:
                switch (current) {
                    case 'o' :
                        state = 204;
                        return true;
                }
                break;
            case 204:
                switch (current) {
                    case 't' :
                        state = 205;
                        return true;
                }
                break;
            case 205:
                switch (current) {
                    case ';' : // CircleDot;
                        match = "\u2299";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 206:
                switch (current) {
                    case 'i' :
                        state = 207;
                        return true;
                }
                break;
            case 207:
                switch (current) {
                    case 'n' :
                        state = 208;
                        return true;
                }
                break;
            case 208:
                switch (current) {
                    case 'u' :
                        state = 209;
                        return true;
                }
                break;
            case 209:
                switch (current) {
                    case 's' :
                        state = 210;
                        return true;
                }
                break;
            case 210:
                switch (current) {
                    case ';' : // CircleMinus;
                        match = "\u2296";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 211:
                switch (current) {
                    case 'l' :
                        state = 212;
                        return true;
                }
                break;
            case 212:
                switch (current) {
                    case 'u' :
                        state = 213;
                        return true;
                }
                break;
            case 213:
                switch (current) {
                    case 's' :
                        state = 214;
                        return true;
                }
                break;
            case 214:
                switch (current) {
                    case ';' : // CirclePlus;
                        match = "\u2295";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 215:
                switch (current) {
                    case 'i' :
                        state = 216;
                        return true;
                }
                break;
            case 216:
                switch (current) {
                    case 'm' :
                        state = 217;
                        return true;
                }
                break;
            case 217:
                switch (current) {
                    case 'e' :
                        state = 218;
                        return true;
                }
                break;
            case 218:
                switch (current) {
                    case 's' :
                        state = 219;
                        return true;
                }
                break;
            case 219:
                switch (current) {
                    case ';' : // CircleTimes;
                        match = "\u2297";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 220:
                switch (current) {
                    case 'o' :
                        state = 221;
                        return true;
                }
                break;
            case 221:
                switch (current) {
                    case 'c' :
                        state = 222;
                        return true;
                    case 's' :
                        state = 243;
                        return true;
                }
                break;
            case 222:
                switch (current) {
                    case 'k' :
                        state = 223;
                        return true;
                }
                break;
            case 223:
                switch (current) {
                    case 'w' :
                        state = 224;
                        return true;
                }
                break;
            case 224:
                switch (current) {
                    case 'i' :
                        state = 225;
                        return true;
                }
                break;
            case 225:
                switch (current) {
                    case 's' :
                        state = 226;
                        return true;
                }
                break;
            case 226:
                switch (current) {
                    case 'e' :
                        state = 227;
                        return true;
                }
                break;
            case 227:
                switch (current) {
                    case 'C' :
                        state = 228;
                        return true;
                }
                break;
            case 228:
                switch (current) {
                    case 'o' :
                        state = 229;
                        return true;
                }
                break;
            case 229:
                switch (current) {
                    case 'n' :
                        state = 230;
                        return true;
                }
                break;
            case 230:
                switch (current) {
                    case 't' :
                        state = 231;
                        return true;
                }
                break;
            case 231:
                switch (current) {
                    case 'o' :
                        state = 232;
                        return true;
                }
                break;
            case 232:
                switch (current) {
                    case 'u' :
                        state = 233;
                        return true;
                }
                break;
            case 233:
                switch (current) {
                    case 'r' :
                        state = 234;
                        return true;
                }
                break;
            case 234:
                switch (current) {
                    case 'I' :
                        state = 235;
                        return true;
                }
                break;
            case 235:
                switch (current) {
                    case 'n' :
                        state = 236;
                        return true;
                }
                break;
            case 236:
                switch (current) {
                    case 't' :
                        state = 237;
                        return true;
                }
                break;
            case 237:
                switch (current) {
                    case 'e' :
                        state = 238;
                        return true;
                }
                break;
            case 238:
                switch (current) {
                    case 'g' :
                        state = 239;
                        return true;
                }
                break;
            case 239:
                switch (current) {
                    case 'r' :
                        state = 240;
                        return true;
                }
                break;
            case 240:
                switch (current) {
                    case 'a' :
                        state = 241;
                        return true;
                }
                break;
            case 241:
                switch (current) {
                    case 'l' :
                        state = 242;
                        return true;
                }
                break;
            case 242:
                switch (current) {
                    case ';' : // ClockwiseContourIntegral;
                        match = "\u2232";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 243:
                switch (current) {
                    case 'e' :
                        state = 244;
                        return true;
                }
                break;
            case 244:
                switch (current) {
                    case 'C' :
                        state = 245;
                        return true;
                }
                break;
            case 245:
                switch (current) {
                    case 'u' :
                        state = 246;
                        return true;
                }
                break;
            case 246:
                switch (current) {
                    case 'r' :
                        state = 247;
                        return true;
                }
                break;
            case 247:
                switch (current) {
                    case 'l' :
                        state = 248;
                        return true;
                }
                break;
            case 248:
                switch (current) {
                    case 'y' :
                        state = 249;
                        return true;
                }
                break;
            case 249:
                switch (current) {
                    case 'D' :
                        state = 250;
                        return true;
                    case 'Q' :
                        state = 261;
                        return true;
                }
                break;
            case 250:
                switch (current) {
                    case 'o' :
                        state = 251;
                        return true;
                }
                break;
            case 251:
                switch (current) {
                    case 'u' :
                        state = 252;
                        return true;
                }
                break;
            case 252:
                switch (current) {
                    case 'b' :
                        state = 253;
                        return true;
                }
                break;
            case 253:
                switch (current) {
                    case 'l' :
                        state = 254;
                        return true;
                }
                break;
            case 254:
                switch (current) {
                    case 'e' :
                        state = 255;
                        return true;
                }
                break;
            case 255:
                switch (current) {
                    case 'Q' :
                        state = 256;
                        return true;
                }
                break;
            case 256:
                switch (current) {
                    case 'u' :
                        state = 257;
                        return true;
                }
                break;
            case 257:
                switch (current) {
                    case 'o' :
                        state = 258;
                        return true;
                }
                break;
            case 258:
                switch (current) {
                    case 't' :
                        state = 259;
                        return true;
                }
                break;
            case 259:
                switch (current) {
                    case 'e' :
                        state = 260;
                        return true;
                }
                break;
            case 260:
                switch (current) {
                    case ';' : // CloseCurlyDoubleQuote;
                        match = "\u201D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 261:
                switch (current) {
                    case 'u' :
                        state = 262;
                        return true;
                }
                break;
            case 262:
                switch (current) {
                    case 'o' :
                        state = 263;
                        return true;
                }
                break;
            case 263:
                switch (current) {
                    case 't' :
                        state = 264;
                        return true;
                }
                break;
            case 264:
                switch (current) {
                    case 'e' :
                        state = 265;
                        return true;
                }
                break;
            case 265:
                switch (current) {
                    case ';' : // CloseCurlyQuote;
                        match = "\u2019";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 266:
                switch (current) {
                    case 'l' :
                        state = 267;
                        return true;
                    case 'n' :
                        state = 271;
                        return true;
                    case 'p' :
                        state = 293;
                        return true;
                    case 'u' :
                        state = 301;
                        return true;
                }
                break;
            case 267:
                switch (current) {
                    case 'o' :
                        state = 268;
                        return true;
                }
                break;
            case 268:
                switch (current) {
                    case 'n' :
                        state = 269;
                        return true;
                }
                break;
            case 269:
                switch (current) {
                    case ';' : // Colon;
                        match = "\u2237";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 270;
                        return true;
                }
                break;
            case 270:
                switch (current) {
                    case ';' : // Colone;
                        match = "\u2A74";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 271:
                switch (current) {
                    case 'g' :
                        state = 272;
                        return true;
                    case 'i' :
                        state = 278;
                        return true;
                    case 't' :
                        state = 281;
                        return true;
                }
                break;
            case 272:
                switch (current) {
                    case 'r' :
                        state = 273;
                        return true;
                }
                break;
            case 273:
                switch (current) {
                    case 'u' :
                        state = 274;
                        return true;
                }
                break;
            case 274:
                switch (current) {
                    case 'e' :
                        state = 275;
                        return true;
                }
                break;
            case 275:
                switch (current) {
                    case 'n' :
                        state = 276;
                        return true;
                }
                break;
            case 276:
                switch (current) {
                    case 't' :
                        state = 277;
                        return true;
                }
                break;
            case 277:
                switch (current) {
                    case ';' : // Congruent;
                        match = "\u2261";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 278:
                switch (current) {
                    case 'n' :
                        state = 279;
                        return true;
                }
                break;
            case 279:
                switch (current) {
                    case 't' :
                        state = 280;
                        return true;
                }
                break;
            case 280:
                switch (current) {
                    case ';' : // Conint;
                        match = "\u222F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 281:
                switch (current) {
                    case 'o' :
                        state = 282;
                        return true;
                }
                break;
            case 282:
                switch (current) {
                    case 'u' :
                        state = 283;
                        return true;
                }
                break;
            case 283:
                switch (current) {
                    case 'r' :
                        state = 284;
                        return true;
                }
                break;
            case 284:
                switch (current) {
                    case 'I' :
                        state = 285;
                        return true;
                }
                break;
            case 285:
                switch (current) {
                    case 'n' :
                        state = 286;
                        return true;
                }
                break;
            case 286:
                switch (current) {
                    case 't' :
                        state = 287;
                        return true;
                }
                break;
            case 287:
                switch (current) {
                    case 'e' :
                        state = 288;
                        return true;
                }
                break;
            case 288:
                switch (current) {
                    case 'g' :
                        state = 289;
                        return true;
                }
                break;
            case 289:
                switch (current) {
                    case 'r' :
                        state = 290;
                        return true;
                }
                break;
            case 290:
                switch (current) {
                    case 'a' :
                        state = 291;
                        return true;
                }
                break;
            case 291:
                switch (current) {
                    case 'l' :
                        state = 292;
                        return true;
                }
                break;
            case 292:
                switch (current) {
                    case ';' : // ContourIntegral;
                        match = "\u222E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 293:
                switch (current) {
                    case 'f' :
                        state = 294;
                        return true;
                    case 'r' :
                        state = 295;
                        return true;
                }
                break;
            case 294:
                switch (current) {
                    case ';' : // Copf;
                        match = "\u2102";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 295:
                switch (current) {
                    case 'o' :
                        state = 296;
                        return true;
                }
                break;
            case 296:
                switch (current) {
                    case 'd' :
                        state = 297;
                        return true;
                }
                break;
            case 297:
                switch (current) {
                    case 'u' :
                        state = 298;
                        return true;
                }
                break;
            case 298:
                switch (current) {
                    case 'c' :
                        state = 299;
                        return true;
                }
                break;
            case 299:
                switch (current) {
                    case 't' :
                        state = 300;
                        return true;
                }
                break;
            case 300:
                switch (current) {
                    case ';' : // Coproduct;
                        match = "\u2210";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 301:
                switch (current) {
                    case 'n' :
                        state = 302;
                        return true;
                }
                break;
            case 302:
                switch (current) {
                    case 't' :
                        state = 303;
                        return true;
                }
                break;
            case 303:
                switch (current) {
                    case 'e' :
                        state = 304;
                        return true;
                }
                break;
            case 304:
                switch (current) {
                    case 'r' :
                        state = 305;
                        return true;
                }
                break;
            case 305:
                switch (current) {
                    case 'C' :
                        state = 306;
                        return true;
                }
                break;
            case 306:
                switch (current) {
                    case 'l' :
                        state = 307;
                        return true;
                }
                break;
            case 307:
                switch (current) {
                    case 'o' :
                        state = 308;
                        return true;
                }
                break;
            case 308:
                switch (current) {
                    case 'c' :
                        state = 309;
                        return true;
                }
                break;
            case 309:
                switch (current) {
                    case 'k' :
                        state = 310;
                        return true;
                }
                break;
            case 310:
                switch (current) {
                    case 'w' :
                        state = 311;
                        return true;
                }
                break;
            case 311:
                switch (current) {
                    case 'i' :
                        state = 312;
                        return true;
                }
                break;
            case 312:
                switch (current) {
                    case 's' :
                        state = 313;
                        return true;
                }
                break;
            case 313:
                switch (current) {
                    case 'e' :
                        state = 314;
                        return true;
                }
                break;
            case 314:
                switch (current) {
                    case 'C' :
                        state = 315;
                        return true;
                }
                break;
            case 315:
                switch (current) {
                    case 'o' :
                        state = 316;
                        return true;
                }
                break;
            case 316:
                switch (current) {
                    case 'n' :
                        state = 317;
                        return true;
                }
                break;
            case 317:
                switch (current) {
                    case 't' :
                        state = 318;
                        return true;
                }
                break;
            case 318:
                switch (current) {
                    case 'o' :
                        state = 319;
                        return true;
                }
                break;
            case 319:
                switch (current) {
                    case 'u' :
                        state = 320;
                        return true;
                }
                break;
            case 320:
                switch (current) {
                    case 'r' :
                        state = 321;
                        return true;
                }
                break;
            case 321:
                switch (current) {
                    case 'I' :
                        state = 322;
                        return true;
                }
                break;
            case 322:
                switch (current) {
                    case 'n' :
                        state = 323;
                        return true;
                }
                break;
            case 323:
                switch (current) {
                    case 't' :
                        state = 324;
                        return true;
                }
                break;
            case 324:
                switch (current) {
                    case 'e' :
                        state = 325;
                        return true;
                }
                break;
            case 325:
                switch (current) {
                    case 'g' :
                        state = 326;
                        return true;
                }
                break;
            case 326:
                switch (current) {
                    case 'r' :
                        state = 327;
                        return true;
                }
                break;
            case 327:
                switch (current) {
                    case 'a' :
                        state = 328;
                        return true;
                }
                break;
            case 328:
                switch (current) {
                    case 'l' :
                        state = 329;
                        return true;
                }
                break;
            case 329:
                switch (current) {
                    case ';' : // CounterClockwiseContourIntegral;
                        match = "\u2233";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 330:
                switch (current) {
                    case 'o' :
                        state = 331;
                        return true;
                }
                break;
            case 331:
                switch (current) {
                    case 's' :
                        state = 332;
                        return true;
                }
                break;
            case 332:
                switch (current) {
                    case 's' :
                        state = 333;
                        return true;
                }
                break;
            case 333:
                switch (current) {
                    case ';' : // Cross;
                        match = "\u2A2F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 334:
                switch (current) {
                    case 'c' :
                        state = 335;
                        return true;
                }
                break;
            case 335:
                switch (current) {
                    case 'r' :
                        state = 336;
                        return true;
                }
                break;
            case 336:
                switch (current) {
                    case ';' : // Cscr;
                        match = "\uD835\uDC9E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 337:
                switch (current) {
                    case 'p' :
                        state = 338;
                        return true;
                }
                break;
            case 338:
                switch (current) {
                    case ';' : // Cup;
                        match = "\u22D3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'C' :
                        state = 339;
                        return true;
                }
                break;
            case 339:
                switch (current) {
                    case 'a' :
                        state = 340;
                        return true;
                }
                break;
            case 340:
                switch (current) {
                    case 'p' :
                        state = 341;
                        return true;
                }
                break;
            case 341:
                switch (current) {
                    case ';' : // CupCap;
                        match = "\u224D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 342:
                switch (current) {
                    case 'D' :
                        state = 343;
                        return true;
                    case 'J' :
                        state = 350;
                        return true;
                    case 'S' :
                        state = 353;
                        return true;
                    case 'Z' :
                        state = 356;
                        return true;
                    case 'a' :
                        state = 359;
                        return true;
                    case 'c' :
                        state = 369;
                        return true;
                    case 'e' :
                        state = 375;
                        return true;
                    case 'f' :
                        state = 379;
                        return true;
                    case 'i' :
                        state = 381;
                        return true;
                    case 'o' :
                        state = 433;
                        return true;
                    case 's' :
                        state = 659;
                        return true;
                }
                break;
            case 343:
                switch (current) {
                    case ';' : // DD;
                        match = "\u2145";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'o' :
                        state = 344;
                        return true;
                }
                break;
            case 344:
                switch (current) {
                    case 't' :
                        state = 345;
                        return true;
                }
                break;
            case 345:
                switch (current) {
                    case 'r' :
                        state = 346;
                        return true;
                }
                break;
            case 346:
                switch (current) {
                    case 'a' :
                        state = 347;
                        return true;
                }
                break;
            case 347:
                switch (current) {
                    case 'h' :
                        state = 348;
                        return true;
                }
                break;
            case 348:
                switch (current) {
                    case 'd' :
                        state = 349;
                        return true;
                }
                break;
            case 349:
                switch (current) {
                    case ';' : // DDotrahd;
                        match = "\u2911";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 350:
                switch (current) {
                    case 'c' :
                        state = 351;
                        return true;
                }
                break;
            case 351:
                switch (current) {
                    case 'y' :
                        state = 352;
                        return true;
                }
                break;
            case 352:
                switch (current) {
                    case ';' : // DJcy;
                        match = "\u0402";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 353:
                switch (current) {
                    case 'c' :
                        state = 354;
                        return true;
                }
                break;
            case 354:
                switch (current) {
                    case 'y' :
                        state = 355;
                        return true;
                }
                break;
            case 355:
                switch (current) {
                    case ';' : // DScy;
                        match = "\u0405";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 356:
                switch (current) {
                    case 'c' :
                        state = 357;
                        return true;
                }
                break;
            case 357:
                switch (current) {
                    case 'y' :
                        state = 358;
                        return true;
                }
                break;
            case 358:
                switch (current) {
                    case ';' : // DZcy;
                        match = "\u040F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 359:
                switch (current) {
                    case 'g' :
                        state = 360;
                        return true;
                    case 'r' :
                        state = 364;
                        return true;
                    case 's' :
                        state = 366;
                        return true;
                }
                break;
            case 360:
                switch (current) {
                    case 'g' :
                        state = 361;
                        return true;
                }
                break;
            case 361:
                switch (current) {
                    case 'e' :
                        state = 362;
                        return true;
                }
                break;
            case 362:
                switch (current) {
                    case 'r' :
                        state = 363;
                        return true;
                }
                break;
            case 363:
                switch (current) {
                    case ';' : // Dagger;
                        match = "\u2021";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 364:
                switch (current) {
                    case 'r' :
                        state = 365;
                        return true;
                }
                break;
            case 365:
                switch (current) {
                    case ';' : // Darr;
                        match = "\u21A1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 366:
                switch (current) {
                    case 'h' :
                        state = 367;
                        return true;
                }
                break;
            case 367:
                switch (current) {
                    case 'v' :
                        state = 368;
                        return true;
                }
                break;
            case 368:
                switch (current) {
                    case ';' : // Dashv;
                        match = "\u2AE4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 369:
                switch (current) {
                    case 'a' :
                        state = 370;
                        return true;
                    case 'y' :
                        state = 374;
                        return true;
                }
                break;
            case 370:
                switch (current) {
                    case 'r' :
                        state = 371;
                        return true;
                }
                break;
            case 371:
                switch (current) {
                    case 'o' :
                        state = 372;
                        return true;
                }
                break;
            case 372:
                switch (current) {
                    case 'n' :
                        state = 373;
                        return true;
                }
                break;
            case 373:
                switch (current) {
                    case ';' : // Dcaron;
                        match = "\u010E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 374:
                switch (current) {
                    case ';' : // Dcy;
                        match = "\u0414";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 375:
                switch (current) {
                    case 'l' :
                        state = 376;
                        return true;
                }
                break;
            case 376:
                switch (current) {
                    case ';' : // Del;
                        match = "\u2207";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 't' :
                        state = 377;
                        return true;
                }
                break;
            case 377:
                switch (current) {
                    case 'a' :
                        state = 378;
                        return true;
                }
                break;
            case 378:
                switch (current) {
                    case ';' : // Delta;
                        match = "\u0394";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 379:
                switch (current) {
                    case 'r' :
                        state = 380;
                        return true;
                }
                break;
            case 380:
                switch (current) {
                    case ';' : // Dfr;
                        match = "\uD835\uDD07";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 381:
                switch (current) {
                    case 'a' :
                        state = 382;
                        return true;
                    case 'f' :
                        state = 422;
                        return true;
                }
                break;
            case 382:
                switch (current) {
                    case 'c' :
                        state = 383;
                        return true;
                    case 'm' :
                        state = 418;
                        return true;
                }
                break;
            case 383:
                switch (current) {
                    case 'r' :
                        state = 384;
                        return true;
                }
                break;
            case 384:
                switch (current) {
                    case 'i' :
                        state = 385;
                        return true;
                }
                break;
            case 385:
                switch (current) {
                    case 't' :
                        state = 386;
                        return true;
                }
                break;
            case 386:
                switch (current) {
                    case 'i' :
                        state = 387;
                        return true;
                }
                break;
            case 387:
                switch (current) {
                    case 'c' :
                        state = 388;
                        return true;
                }
                break;
            case 388:
                switch (current) {
                    case 'a' :
                        state = 389;
                        return true;
                }
                break;
            case 389:
                switch (current) {
                    case 'l' :
                        state = 390;
                        return true;
                }
                break;
            case 390:
                switch (current) {
                    case 'A' :
                        state = 391;
                        return true;
                    case 'D' :
                        state = 396;
                        return true;
                    case 'G' :
                        state = 408;
                        return true;
                    case 'T' :
                        state = 413;
                        return true;
                }
                break;
            case 391:
                switch (current) {
                    case 'c' :
                        state = 392;
                        return true;
                }
                break;
            case 392:
                switch (current) {
                    case 'u' :
                        state = 393;
                        return true;
                }
                break;
            case 393:
                switch (current) {
                    case 't' :
                        state = 394;
                        return true;
                }
                break;
            case 394:
                switch (current) {
                    case 'e' :
                        state = 395;
                        return true;
                }
                break;
            case 395:
                switch (current) {
                    case ';' : // DiacriticalAcute;
                        match = "\u00B4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 396:
                switch (current) {
                    case 'o' :
                        state = 397;
                        return true;
                }
                break;
            case 397:
                switch (current) {
                    case 't' :
                        state = 398;
                        return true;
                    case 'u' :
                        state = 399;
                        return true;
                }
                break;
            case 398:
                switch (current) {
                    case ';' : // DiacriticalDot;
                        match = "\u02D9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 399:
                switch (current) {
                    case 'b' :
                        state = 400;
                        return true;
                }
                break;
            case 400:
                switch (current) {
                    case 'l' :
                        state = 401;
                        return true;
                }
                break;
            case 401:
                switch (current) {
                    case 'e' :
                        state = 402;
                        return true;
                }
                break;
            case 402:
                switch (current) {
                    case 'A' :
                        state = 403;
                        return true;
                }
                break;
            case 403:
                switch (current) {
                    case 'c' :
                        state = 404;
                        return true;
                }
                break;
            case 404:
                switch (current) {
                    case 'u' :
                        state = 405;
                        return true;
                }
                break;
            case 405:
                switch (current) {
                    case 't' :
                        state = 406;
                        return true;
                }
                break;
            case 406:
                switch (current) {
                    case 'e' :
                        state = 407;
                        return true;
                }
                break;
            case 407:
                switch (current) {
                    case ';' : // DiacriticalDoubleAcute;
                        match = "\u02DD";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 408:
                switch (current) {
                    case 'r' :
                        state = 409;
                        return true;
                }
                break;
            case 409:
                switch (current) {
                    case 'a' :
                        state = 410;
                        return true;
                }
                break;
            case 410:
                switch (current) {
                    case 'v' :
                        state = 411;
                        return true;
                }
                break;
            case 411:
                switch (current) {
                    case 'e' :
                        state = 412;
                        return true;
                }
                break;
            case 412:
                switch (current) {
                    case ';' : // DiacriticalGrave;
                        match = "`";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 413:
                switch (current) {
                    case 'i' :
                        state = 414;
                        return true;
                }
                break;
            case 414:
                switch (current) {
                    case 'l' :
                        state = 415;
                        return true;
                }
                break;
            case 415:
                switch (current) {
                    case 'd' :
                        state = 416;
                        return true;
                }
                break;
            case 416:
                switch (current) {
                    case 'e' :
                        state = 417;
                        return true;
                }
                break;
            case 417:
                switch (current) {
                    case ';' : // DiacriticalTilde;
                        match = "\u02DC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 418:
                switch (current) {
                    case 'o' :
                        state = 419;
                        return true;
                }
                break;
            case 419:
                switch (current) {
                    case 'n' :
                        state = 420;
                        return true;
                }
                break;
            case 420:
                switch (current) {
                    case 'd' :
                        state = 421;
                        return true;
                }
                break;
            case 421:
                switch (current) {
                    case ';' : // Diamond;
                        match = "\u22C4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 422:
                switch (current) {
                    case 'f' :
                        state = 423;
                        return true;
                }
                break;
            case 423:
                switch (current) {
                    case 'e' :
                        state = 424;
                        return true;
                }
                break;
            case 424:
                switch (current) {
                    case 'r' :
                        state = 425;
                        return true;
                }
                break;
            case 425:
                switch (current) {
                    case 'e' :
                        state = 426;
                        return true;
                }
                break;
            case 426:
                switch (current) {
                    case 'n' :
                        state = 427;
                        return true;
                }
                break;
            case 427:
                switch (current) {
                    case 't' :
                        state = 428;
                        return true;
                }
                break;
            case 428:
                switch (current) {
                    case 'i' :
                        state = 429;
                        return true;
                }
                break;
            case 429:
                switch (current) {
                    case 'a' :
                        state = 430;
                        return true;
                }
                break;
            case 430:
                switch (current) {
                    case 'l' :
                        state = 431;
                        return true;
                }
                break;
            case 431:
                switch (current) {
                    case 'D' :
                        state = 432;
                        return true;
                }
                break;
            case 432:
                switch (current) {
                    case ';' : // DifferentialD;
                        match = "\u2146";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 433:
                switch (current) {
                    case 'p' :
                        state = 434;
                        return true;
                    case 't' :
                        state = 436;
                        return true;
                    case 'u' :
                        state = 445;
                        return true;
                    case 'w' :
                        state = 568;
                        return true;
                }
                break;
            case 434:
                switch (current) {
                    case 'f' :
                        state = 435;
                        return true;
                }
                break;
            case 435:
                switch (current) {
                    case ';' : // Dopf;
                        match = "\uD835\uDD3B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 436:
                switch (current) {
                    case ';' : // Dot;
                        match = "\u00A8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'D' :
                        state = 437;
                        return true;
                    case 'E' :
                        state = 440;
                        return true;
                }
                break;
            case 437:
                switch (current) {
                    case 'o' :
                        state = 438;
                        return true;
                }
                break;
            case 438:
                switch (current) {
                    case 't' :
                        state = 439;
                        return true;
                }
                break;
            case 439:
                switch (current) {
                    case ';' : // DotDot;
                        match = "\u20DC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 440:
                switch (current) {
                    case 'q' :
                        state = 441;
                        return true;
                }
                break;
            case 441:
                switch (current) {
                    case 'u' :
                        state = 442;
                        return true;
                }
                break;
            case 442:
                switch (current) {
                    case 'a' :
                        state = 443;
                        return true;
                }
                break;
            case 443:
                switch (current) {
                    case 'l' :
                        state = 444;
                        return true;
                }
                break;
            case 444:
                switch (current) {
                    case ';' : // DotEqual;
                        match = "\u2250";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 445:
                switch (current) {
                    case 'b' :
                        state = 446;
                        return true;
                }
                break;
            case 446:
                switch (current) {
                    case 'l' :
                        state = 447;
                        return true;
                }
                break;
            case 447:
                switch (current) {
                    case 'e' :
                        state = 448;
                        return true;
                }
                break;
            case 448:
                switch (current) {
                    case 'C' :
                        state = 449;
                        return true;
                    case 'D' :
                        state = 464;
                        return true;
                    case 'L' :
                        state = 474;
                        return true;
                    case 'R' :
                        state = 528;
                        return true;
                    case 'U' :
                        state = 541;
                        return true;
                    case 'V' :
                        state = 557;
                        return true;
                }
                break;
            case 449:
                switch (current) {
                    case 'o' :
                        state = 450;
                        return true;
                }
                break;
            case 450:
                switch (current) {
                    case 'n' :
                        state = 451;
                        return true;
                }
                break;
            case 451:
                switch (current) {
                    case 't' :
                        state = 452;
                        return true;
                }
                break;
            case 452:
                switch (current) {
                    case 'o' :
                        state = 453;
                        return true;
                }
                break;
            case 453:
                switch (current) {
                    case 'u' :
                        state = 454;
                        return true;
                }
                break;
            case 454:
                switch (current) {
                    case 'r' :
                        state = 455;
                        return true;
                }
                break;
            case 455:
                switch (current) {
                    case 'I' :
                        state = 456;
                        return true;
                }
                break;
            case 456:
                switch (current) {
                    case 'n' :
                        state = 457;
                        return true;
                }
                break;
            case 457:
                switch (current) {
                    case 't' :
                        state = 458;
                        return true;
                }
                break;
            case 458:
                switch (current) {
                    case 'e' :
                        state = 459;
                        return true;
                }
                break;
            case 459:
                switch (current) {
                    case 'g' :
                        state = 460;
                        return true;
                }
                break;
            case 460:
                switch (current) {
                    case 'r' :
                        state = 461;
                        return true;
                }
                break;
            case 461:
                switch (current) {
                    case 'a' :
                        state = 462;
                        return true;
                }
                break;
            case 462:
                switch (current) {
                    case 'l' :
                        state = 463;
                        return true;
                }
                break;
            case 463:
                switch (current) {
                    case ';' : // DoubleContourIntegral;
                        match = "\u222F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 464:
                switch (current) {
                    case 'o' :
                        state = 465;
                        return true;
                }
                break;
            case 465:
                switch (current) {
                    case 't' :
                        state = 466;
                        return true;
                    case 'w' :
                        state = 467;
                        return true;
                }
                break;
            case 466:
                switch (current) {
                    case ';' : // DoubleDot;
                        match = "\u00A8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 467:
                switch (current) {
                    case 'n' :
                        state = 468;
                        return true;
                }
                break;
            case 468:
                switch (current) {
                    case 'A' :
                        state = 469;
                        return true;
                }
                break;
            case 469:
                switch (current) {
                    case 'r' :
                        state = 470;
                        return true;
                }
                break;
            case 470:
                switch (current) {
                    case 'r' :
                        state = 471;
                        return true;
                }
                break;
            case 471:
                switch (current) {
                    case 'o' :
                        state = 472;
                        return true;
                }
                break;
            case 472:
                switch (current) {
                    case 'w' :
                        state = 473;
                        return true;
                }
                break;
            case 473:
                switch (current) {
                    case ';' : // DoubleDownArrow;
                        match = "\u21D3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 474:
                switch (current) {
                    case 'e' :
                        state = 475;
                        return true;
                    case 'o' :
                        state = 496;
                        return true;
                }
                break;
            case 475:
                switch (current) {
                    case 'f' :
                        state = 476;
                        return true;
                }
                break;
            case 476:
                switch (current) {
                    case 't' :
                        state = 477;
                        return true;
                }
                break;
            case 477:
                switch (current) {
                    case 'A' :
                        state = 478;
                        return true;
                    case 'R' :
                        state = 483;
                        return true;
                    case 'T' :
                        state = 493;
                        return true;
                }
                break;
            case 478:
                switch (current) {
                    case 'r' :
                        state = 479;
                        return true;
                }
                break;
            case 479:
                switch (current) {
                    case 'r' :
                        state = 480;
                        return true;
                }
                break;
            case 480:
                switch (current) {
                    case 'o' :
                        state = 481;
                        return true;
                }
                break;
            case 481:
                switch (current) {
                    case 'w' :
                        state = 482;
                        return true;
                }
                break;
            case 482:
                switch (current) {
                    case ';' : // DoubleLeftArrow;
                        match = "\u21D0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 483:
                switch (current) {
                    case 'i' :
                        state = 484;
                        return true;
                }
                break;
            case 484:
                switch (current) {
                    case 'g' :
                        state = 485;
                        return true;
                }
                break;
            case 485:
                switch (current) {
                    case 'h' :
                        state = 486;
                        return true;
                }
                break;
            case 486:
                switch (current) {
                    case 't' :
                        state = 487;
                        return true;
                }
                break;
            case 487:
                switch (current) {
                    case 'A' :
                        state = 488;
                        return true;
                }
                break;
            case 488:
                switch (current) {
                    case 'r' :
                        state = 489;
                        return true;
                }
                break;
            case 489:
                switch (current) {
                    case 'r' :
                        state = 490;
                        return true;
                }
                break;
            case 490:
                switch (current) {
                    case 'o' :
                        state = 491;
                        return true;
                }
                break;
            case 491:
                switch (current) {
                    case 'w' :
                        state = 492;
                        return true;
                }
                break;
            case 492:
                switch (current) {
                    case ';' : // DoubleLeftRightArrow;
                        match = "\u21D4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 493:
                switch (current) {
                    case 'e' :
                        state = 494;
                        return true;
                }
                break;
            case 494:
                switch (current) {
                    case 'e' :
                        state = 495;
                        return true;
                }
                break;
            case 495:
                switch (current) {
                    case ';' : // DoubleLeftTee;
                        match = "\u2AE4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 496:
                switch (current) {
                    case 'n' :
                        state = 497;
                        return true;
                }
                break;
            case 497:
                switch (current) {
                    case 'g' :
                        state = 498;
                        return true;
                }
                break;
            case 498:
                switch (current) {
                    case 'L' :
                        state = 499;
                        return true;
                    case 'R' :
                        state = 518;
                        return true;
                }
                break;
            case 499:
                switch (current) {
                    case 'e' :
                        state = 500;
                        return true;
                }
                break;
            case 500:
                switch (current) {
                    case 'f' :
                        state = 501;
                        return true;
                }
                break;
            case 501:
                switch (current) {
                    case 't' :
                        state = 502;
                        return true;
                }
                break;
            case 502:
                switch (current) {
                    case 'A' :
                        state = 503;
                        return true;
                    case 'R' :
                        state = 508;
                        return true;
                }
                break;
            case 503:
                switch (current) {
                    case 'r' :
                        state = 504;
                        return true;
                }
                break;
            case 504:
                switch (current) {
                    case 'r' :
                        state = 505;
                        return true;
                }
                break;
            case 505:
                switch (current) {
                    case 'o' :
                        state = 506;
                        return true;
                }
                break;
            case 506:
                switch (current) {
                    case 'w' :
                        state = 507;
                        return true;
                }
                break;
            case 507:
                switch (current) {
                    case ';' : // DoubleLongLeftArrow;
                        match = "\u27F8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 508:
                switch (current) {
                    case 'i' :
                        state = 509;
                        return true;
                }
                break;
            case 509:
                switch (current) {
                    case 'g' :
                        state = 510;
                        return true;
                }
                break;
            case 510:
                switch (current) {
                    case 'h' :
                        state = 511;
                        return true;
                }
                break;
            case 511:
                switch (current) {
                    case 't' :
                        state = 512;
                        return true;
                }
                break;
            case 512:
                switch (current) {
                    case 'A' :
                        state = 513;
                        return true;
                }
                break;
            case 513:
                switch (current) {
                    case 'r' :
                        state = 514;
                        return true;
                }
                break;
            case 514:
                switch (current) {
                    case 'r' :
                        state = 515;
                        return true;
                }
                break;
            case 515:
                switch (current) {
                    case 'o' :
                        state = 516;
                        return true;
                }
                break;
            case 516:
                switch (current) {
                    case 'w' :
                        state = 517;
                        return true;
                }
                break;
            case 517:
                switch (current) {
                    case ';' : // DoubleLongLeftRightArrow;
                        match = "\u27FA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 518:
                switch (current) {
                    case 'i' :
                        state = 519;
                        return true;
                }
                break;
            case 519:
                switch (current) {
                    case 'g' :
                        state = 520;
                        return true;
                }
                break;
            case 520:
                switch (current) {
                    case 'h' :
                        state = 521;
                        return true;
                }
                break;
            case 521:
                switch (current) {
                    case 't' :
                        state = 522;
                        return true;
                }
                break;
            case 522:
                switch (current) {
                    case 'A' :
                        state = 523;
                        return true;
                }
                break;
            case 523:
                switch (current) {
                    case 'r' :
                        state = 524;
                        return true;
                }
                break;
            case 524:
                switch (current) {
                    case 'r' :
                        state = 525;
                        return true;
                }
                break;
            case 525:
                switch (current) {
                    case 'o' :
                        state = 526;
                        return true;
                }
                break;
            case 526:
                switch (current) {
                    case 'w' :
                        state = 527;
                        return true;
                }
                break;
            case 527:
                switch (current) {
                    case ';' : // DoubleLongRightArrow;
                        match = "\u27F9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 528:
                switch (current) {
                    case 'i' :
                        state = 529;
                        return true;
                }
                break;
            case 529:
                switch (current) {
                    case 'g' :
                        state = 530;
                        return true;
                }
                break;
            case 530:
                switch (current) {
                    case 'h' :
                        state = 531;
                        return true;
                }
                break;
            case 531:
                switch (current) {
                    case 't' :
                        state = 532;
                        return true;
                }
                break;
            case 532:
                switch (current) {
                    case 'A' :
                        state = 533;
                        return true;
                    case 'T' :
                        state = 538;
                        return true;
                }
                break;
            case 533:
                switch (current) {
                    case 'r' :
                        state = 534;
                        return true;
                }
                break;
            case 534:
                switch (current) {
                    case 'r' :
                        state = 535;
                        return true;
                }
                break;
            case 535:
                switch (current) {
                    case 'o' :
                        state = 536;
                        return true;
                }
                break;
            case 536:
                switch (current) {
                    case 'w' :
                        state = 537;
                        return true;
                }
                break;
            case 537:
                switch (current) {
                    case ';' : // DoubleRightArrow;
                        match = "\u21D2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 538:
                switch (current) {
                    case 'e' :
                        state = 539;
                        return true;
                }
                break;
            case 539:
                switch (current) {
                    case 'e' :
                        state = 540;
                        return true;
                }
                break;
            case 540:
                switch (current) {
                    case ';' : // DoubleRightTee;
                        match = "\u22A8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 541:
                switch (current) {
                    case 'p' :
                        state = 542;
                        return true;
                }
                break;
            case 542:
                switch (current) {
                    case 'A' :
                        state = 543;
                        return true;
                    case 'D' :
                        state = 548;
                        return true;
                }
                break;
            case 543:
                switch (current) {
                    case 'r' :
                        state = 544;
                        return true;
                }
                break;
            case 544:
                switch (current) {
                    case 'r' :
                        state = 545;
                        return true;
                }
                break;
            case 545:
                switch (current) {
                    case 'o' :
                        state = 546;
                        return true;
                }
                break;
            case 546:
                switch (current) {
                    case 'w' :
                        state = 547;
                        return true;
                }
                break;
            case 547:
                switch (current) {
                    case ';' : // DoubleUpArrow;
                        match = "\u21D1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 548:
                switch (current) {
                    case 'o' :
                        state = 549;
                        return true;
                }
                break;
            case 549:
                switch (current) {
                    case 'w' :
                        state = 550;
                        return true;
                }
                break;
            case 550:
                switch (current) {
                    case 'n' :
                        state = 551;
                        return true;
                }
                break;
            case 551:
                switch (current) {
                    case 'A' :
                        state = 552;
                        return true;
                }
                break;
            case 552:
                switch (current) {
                    case 'r' :
                        state = 553;
                        return true;
                }
                break;
            case 553:
                switch (current) {
                    case 'r' :
                        state = 554;
                        return true;
                }
                break;
            case 554:
                switch (current) {
                    case 'o' :
                        state = 555;
                        return true;
                }
                break;
            case 555:
                switch (current) {
                    case 'w' :
                        state = 556;
                        return true;
                }
                break;
            case 556:
                switch (current) {
                    case ';' : // DoubleUpDownArrow;
                        match = "\u21D5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 557:
                switch (current) {
                    case 'e' :
                        state = 558;
                        return true;
                }
                break;
            case 558:
                switch (current) {
                    case 'r' :
                        state = 559;
                        return true;
                }
                break;
            case 559:
                switch (current) {
                    case 't' :
                        state = 560;
                        return true;
                }
                break;
            case 560:
                switch (current) {
                    case 'i' :
                        state = 561;
                        return true;
                }
                break;
            case 561:
                switch (current) {
                    case 'c' :
                        state = 562;
                        return true;
                }
                break;
            case 562:
                switch (current) {
                    case 'a' :
                        state = 563;
                        return true;
                }
                break;
            case 563:
                switch (current) {
                    case 'l' :
                        state = 564;
                        return true;
                }
                break;
            case 564:
                switch (current) {
                    case 'B' :
                        state = 565;
                        return true;
                }
                break;
            case 565:
                switch (current) {
                    case 'a' :
                        state = 566;
                        return true;
                }
                break;
            case 566:
                switch (current) {
                    case 'r' :
                        state = 567;
                        return true;
                }
                break;
            case 567:
                switch (current) {
                    case ';' : // DoubleVerticalBar;
                        match = "\u2225";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 568:
                switch (current) {
                    case 'n' :
                        state = 569;
                        return true;
                }
                break;
            case 569:
                switch (current) {
                    case 'A' :
                        state = 570;
                        return true;
                    case 'B' :
                        state = 585;
                        return true;
                    case 'L' :
                        state = 590;
                        return true;
                    case 'R' :
                        state = 623;
                        return true;
                    case 'T' :
                        state = 646;
                        return true;
                    case 'a' :
                        state = 654;
                        return true;
                }
                break;
            case 570:
                switch (current) {
                    case 'r' :
                        state = 571;
                        return true;
                }
                break;
            case 571:
                switch (current) {
                    case 'r' :
                        state = 572;
                        return true;
                }
                break;
            case 572:
                switch (current) {
                    case 'o' :
                        state = 573;
                        return true;
                }
                break;
            case 573:
                switch (current) {
                    case 'w' :
                        state = 574;
                        return true;
                }
                break;
            case 574:
                switch (current) {
                    case ';' : // DownArrow;
                        match = "\u2193";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 575;
                        return true;
                    case 'U' :
                        state = 578;
                        return true;
                }
                break;
            case 575:
                switch (current) {
                    case 'a' :
                        state = 576;
                        return true;
                }
                break;
            case 576:
                switch (current) {
                    case 'r' :
                        state = 577;
                        return true;
                }
                break;
            case 577:
                switch (current) {
                    case ';' : // DownArrowBar;
                        match = "\u2913";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 578:
                switch (current) {
                    case 'p' :
                        state = 579;
                        return true;
                }
                break;
            case 579:
                switch (current) {
                    case 'A' :
                        state = 580;
                        return true;
                }
                break;
            case 580:
                switch (current) {
                    case 'r' :
                        state = 581;
                        return true;
                }
                break;
            case 581:
                switch (current) {
                    case 'r' :
                        state = 582;
                        return true;
                }
                break;
            case 582:
                switch (current) {
                    case 'o' :
                        state = 583;
                        return true;
                }
                break;
            case 583:
                switch (current) {
                    case 'w' :
                        state = 584;
                        return true;
                }
                break;
            case 584:
                switch (current) {
                    case ';' : // DownArrowUpArrow;
                        match = "\u21F5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 585:
                switch (current) {
                    case 'r' :
                        state = 586;
                        return true;
                }
                break;
            case 586:
                switch (current) {
                    case 'e' :
                        state = 587;
                        return true;
                }
                break;
            case 587:
                switch (current) {
                    case 'v' :
                        state = 588;
                        return true;
                }
                break;
            case 588:
                switch (current) {
                    case 'e' :
                        state = 589;
                        return true;
                }
                break;
            case 589:
                switch (current) {
                    case ';' : // DownBreve;
                        match = "\u0311";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 590:
                switch (current) {
                    case 'e' :
                        state = 591;
                        return true;
                }
                break;
            case 591:
                switch (current) {
                    case 'f' :
                        state = 592;
                        return true;
                }
                break;
            case 592:
                switch (current) {
                    case 't' :
                        state = 593;
                        return true;
                }
                break;
            case 593:
                switch (current) {
                    case 'R' :
                        state = 594;
                        return true;
                    case 'T' :
                        state = 605;
                        return true;
                    case 'V' :
                        state = 614;
                        return true;
                }
                break;
            case 594:
                switch (current) {
                    case 'i' :
                        state = 595;
                        return true;
                }
                break;
            case 595:
                switch (current) {
                    case 'g' :
                        state = 596;
                        return true;
                }
                break;
            case 596:
                switch (current) {
                    case 'h' :
                        state = 597;
                        return true;
                }
                break;
            case 597:
                switch (current) {
                    case 't' :
                        state = 598;
                        return true;
                }
                break;
            case 598:
                switch (current) {
                    case 'V' :
                        state = 599;
                        return true;
                }
                break;
            case 599:
                switch (current) {
                    case 'e' :
                        state = 600;
                        return true;
                }
                break;
            case 600:
                switch (current) {
                    case 'c' :
                        state = 601;
                        return true;
                }
                break;
            case 601:
                switch (current) {
                    case 't' :
                        state = 602;
                        return true;
                }
                break;
            case 602:
                switch (current) {
                    case 'o' :
                        state = 603;
                        return true;
                }
                break;
            case 603:
                switch (current) {
                    case 'r' :
                        state = 604;
                        return true;
                }
                break;
            case 604:
                switch (current) {
                    case ';' : // DownLeftRightVector;
                        match = "\u2950";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 605:
                switch (current) {
                    case 'e' :
                        state = 606;
                        return true;
                }
                break;
            case 606:
                switch (current) {
                    case 'e' :
                        state = 607;
                        return true;
                }
                break;
            case 607:
                switch (current) {
                    case 'V' :
                        state = 608;
                        return true;
                }
                break;
            case 608:
                switch (current) {
                    case 'e' :
                        state = 609;
                        return true;
                }
                break;
            case 609:
                switch (current) {
                    case 'c' :
                        state = 610;
                        return true;
                }
                break;
            case 610:
                switch (current) {
                    case 't' :
                        state = 611;
                        return true;
                }
                break;
            case 611:
                switch (current) {
                    case 'o' :
                        state = 612;
                        return true;
                }
                break;
            case 612:
                switch (current) {
                    case 'r' :
                        state = 613;
                        return true;
                }
                break;
            case 613:
                switch (current) {
                    case ';' : // DownLeftTeeVector;
                        match = "\u295E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 614:
                switch (current) {
                    case 'e' :
                        state = 615;
                        return true;
                }
                break;
            case 615:
                switch (current) {
                    case 'c' :
                        state = 616;
                        return true;
                }
                break;
            case 616:
                switch (current) {
                    case 't' :
                        state = 617;
                        return true;
                }
                break;
            case 617:
                switch (current) {
                    case 'o' :
                        state = 618;
                        return true;
                }
                break;
            case 618:
                switch (current) {
                    case 'r' :
                        state = 619;
                        return true;
                }
                break;
            case 619:
                switch (current) {
                    case ';' : // DownLeftVector;
                        match = "\u21BD";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 620;
                        return true;
                }
                break;
            case 620:
                switch (current) {
                    case 'a' :
                        state = 621;
                        return true;
                }
                break;
            case 621:
                switch (current) {
                    case 'r' :
                        state = 622;
                        return true;
                }
                break;
            case 622:
                switch (current) {
                    case ';' : // DownLeftVectorBar;
                        match = "\u2956";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 623:
                switch (current) {
                    case 'i' :
                        state = 624;
                        return true;
                }
                break;
            case 624:
                switch (current) {
                    case 'g' :
                        state = 625;
                        return true;
                }
                break;
            case 625:
                switch (current) {
                    case 'h' :
                        state = 626;
                        return true;
                }
                break;
            case 626:
                switch (current) {
                    case 't' :
                        state = 627;
                        return true;
                }
                break;
            case 627:
                switch (current) {
                    case 'T' :
                        state = 628;
                        return true;
                    case 'V' :
                        state = 637;
                        return true;
                }
                break;
            case 628:
                switch (current) {
                    case 'e' :
                        state = 629;
                        return true;
                }
                break;
            case 629:
                switch (current) {
                    case 'e' :
                        state = 630;
                        return true;
                }
                break;
            case 630:
                switch (current) {
                    case 'V' :
                        state = 631;
                        return true;
                }
                break;
            case 631:
                switch (current) {
                    case 'e' :
                        state = 632;
                        return true;
                }
                break;
            case 632:
                switch (current) {
                    case 'c' :
                        state = 633;
                        return true;
                }
                break;
            case 633:
                switch (current) {
                    case 't' :
                        state = 634;
                        return true;
                }
                break;
            case 634:
                switch (current) {
                    case 'o' :
                        state = 635;
                        return true;
                }
                break;
            case 635:
                switch (current) {
                    case 'r' :
                        state = 636;
                        return true;
                }
                break;
            case 636:
                switch (current) {
                    case ';' : // DownRightTeeVector;
                        match = "\u295F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 637:
                switch (current) {
                    case 'e' :
                        state = 638;
                        return true;
                }
                break;
            case 638:
                switch (current) {
                    case 'c' :
                        state = 639;
                        return true;
                }
                break;
            case 639:
                switch (current) {
                    case 't' :
                        state = 640;
                        return true;
                }
                break;
            case 640:
                switch (current) {
                    case 'o' :
                        state = 641;
                        return true;
                }
                break;
            case 641:
                switch (current) {
                    case 'r' :
                        state = 642;
                        return true;
                }
                break;
            case 642:
                switch (current) {
                    case ';' : // DownRightVector;
                        match = "\u21C1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 643;
                        return true;
                }
                break;
            case 643:
                switch (current) {
                    case 'a' :
                        state = 644;
                        return true;
                }
                break;
            case 644:
                switch (current) {
                    case 'r' :
                        state = 645;
                        return true;
                }
                break;
            case 645:
                switch (current) {
                    case ';' : // DownRightVectorBar;
                        match = "\u2957";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 646:
                switch (current) {
                    case 'e' :
                        state = 647;
                        return true;
                }
                break;
            case 647:
                switch (current) {
                    case 'e' :
                        state = 648;
                        return true;
                }
                break;
            case 648:
                switch (current) {
                    case ';' : // DownTee;
                        match = "\u22A4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'A' :
                        state = 649;
                        return true;
                }
                break;
            case 649:
                switch (current) {
                    case 'r' :
                        state = 650;
                        return true;
                }
                break;
            case 650:
                switch (current) {
                    case 'r' :
                        state = 651;
                        return true;
                }
                break;
            case 651:
                switch (current) {
                    case 'o' :
                        state = 652;
                        return true;
                }
                break;
            case 652:
                switch (current) {
                    case 'w' :
                        state = 653;
                        return true;
                }
                break;
            case 653:
                switch (current) {
                    case ';' : // DownTeeArrow;
                        match = "\u21A7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 654:
                switch (current) {
                    case 'r' :
                        state = 655;
                        return true;
                }
                break;
            case 655:
                switch (current) {
                    case 'r' :
                        state = 656;
                        return true;
                }
                break;
            case 656:
                switch (current) {
                    case 'o' :
                        state = 657;
                        return true;
                }
                break;
            case 657:
                switch (current) {
                    case 'w' :
                        state = 658;
                        return true;
                }
                break;
            case 658:
                switch (current) {
                    case ';' : // Downarrow;
                        match = "\u21D3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 659:
                switch (current) {
                    case 'c' :
                        state = 660;
                        return true;
                    case 't' :
                        state = 662;
                        return true;
                }
                break;
            case 660:
                switch (current) {
                    case 'r' :
                        state = 661;
                        return true;
                }
                break;
            case 661:
                switch (current) {
                    case ';' : // Dscr;
                        match = "\uD835\uDC9F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 662:
                switch (current) {
                    case 'r' :
                        state = 663;
                        return true;
                }
                break;
            case 663:
                switch (current) {
                    case 'o' :
                        state = 664;
                        return true;
                }
                break;
            case 664:
                switch (current) {
                    case 'k' :
                        state = 665;
                        return true;
                }
                break;
            case 665:
                switch (current) {
                    case ';' : // Dstrok;
                        match = "\u0110";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 666:
                switch (current) {
                    case 'N' :
                        state = 667;
                        return true;
                    case 'T' :
                        state = 669;
                        return true;
                    case 'a' :
                        state = 671;
                        return true;
                    case 'c' :
                        state = 676;
                        return true;
                    case 'd' :
                        state = 685;
                        return true;
                    case 'f' :
                        state = 688;
                        return true;
                    case 'g' :
                        state = 690;
                        return true;
                    case 'l' :
                        state = 695;
                        return true;
                    case 'm' :
                        state = 701;
                        return true;
                    case 'o' :
                        state = 734;
                        return true;
                    case 'p' :
                        state = 740;
                        return true;
                    case 'q' :
                        state = 746;
                        return true;
                    case 's' :
                        state = 763;
                        return true;
                    case 't' :
                        state = 768;
                        return true;
                    case 'u' :
                        state = 770;
                        return true;
                    case 'x' :
                        state = 773;
                        return true;
                }
                break;
            case 667:
                switch (current) {
                    case 'G' :
                        state = 668;
                        return true;
                }
                break;
            case 668:
                switch (current) {
                    case ';' : // ENG;
                        match = "\u014A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 669:
                switch (current) {
                    case 'H' : // ETH
                        match = "\u00D0";
                        matchLength = consumedCount;
                        state = 670;
                        return true;
                }
                break;
            case 670:
                switch (current) {
                    case ';' : // ETH;
                        match = "\u00D0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 671:
                switch (current) {
                    case 'c' :
                        state = 672;
                        return true;
                }
                break;
            case 672:
                switch (current) {
                    case 'u' :
                        state = 673;
                        return true;
                }
                break;
            case 673:
                switch (current) {
                    case 't' :
                        state = 674;
                        return true;
                }
                break;
            case 674:
                switch (current) {
                    case 'e' : // Eacute
                        match = "\u00C9";
                        matchLength = consumedCount;
                        state = 675;
                        return true;
                }
                break;
            case 675:
                switch (current) {
                    case ';' : // Eacute;
                        match = "\u00C9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 676:
                switch (current) {
                    case 'a' :
                        state = 677;
                        return true;
                    case 'i' :
                        state = 681;
                        return true;
                    case 'y' :
                        state = 684;
                        return true;
                }
                break;
            case 677:
                switch (current) {
                    case 'r' :
                        state = 678;
                        return true;
                }
                break;
            case 678:
                switch (current) {
                    case 'o' :
                        state = 679;
                        return true;
                }
                break;
            case 679:
                switch (current) {
                    case 'n' :
                        state = 680;
                        return true;
                }
                break;
            case 680:
                switch (current) {
                    case ';' : // Ecaron;
                        match = "\u011A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 681:
                switch (current) {
                    case 'r' :
                        state = 682;
                        return true;
                }
                break;
            case 682:
                switch (current) {
                    case 'c' : // Ecirc
                        match = "\u00CA";
                        matchLength = consumedCount;
                        state = 683;
                        return true;
                }
                break;
            case 683:
                switch (current) {
                    case ';' : // Ecirc;
                        match = "\u00CA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 684:
                switch (current) {
                    case ';' : // Ecy;
                        match = "\u042D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 685:
                switch (current) {
                    case 'o' :
                        state = 686;
                        return true;
                }
                break;
            case 686:
                switch (current) {
                    case 't' :
                        state = 687;
                        return true;
                }
                break;
            case 687:
                switch (current) {
                    case ';' : // Edot;
                        match = "\u0116";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 688:
                switch (current) {
                    case 'r' :
                        state = 689;
                        return true;
                }
                break;
            case 689:
                switch (current) {
                    case ';' : // Efr;
                        match = "\uD835\uDD08";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 690:
                switch (current) {
                    case 'r' :
                        state = 691;
                        return true;
                }
                break;
            case 691:
                switch (current) {
                    case 'a' :
                        state = 692;
                        return true;
                }
                break;
            case 692:
                switch (current) {
                    case 'v' :
                        state = 693;
                        return true;
                }
                break;
            case 693:
                switch (current) {
                    case 'e' : // Egrave
                        match = "\u00C8";
                        matchLength = consumedCount;
                        state = 694;
                        return true;
                }
                break;
            case 694:
                switch (current) {
                    case ';' : // Egrave;
                        match = "\u00C8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 695:
                switch (current) {
                    case 'e' :
                        state = 696;
                        return true;
                }
                break;
            case 696:
                switch (current) {
                    case 'm' :
                        state = 697;
                        return true;
                }
                break;
            case 697:
                switch (current) {
                    case 'e' :
                        state = 698;
                        return true;
                }
                break;
            case 698:
                switch (current) {
                    case 'n' :
                        state = 699;
                        return true;
                }
                break;
            case 699:
                switch (current) {
                    case 't' :
                        state = 700;
                        return true;
                }
                break;
            case 700:
                switch (current) {
                    case ';' : // Element;
                        match = "\u2208";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 701:
                switch (current) {
                    case 'a' :
                        state = 702;
                        return true;
                    case 'p' :
                        state = 705;
                        return true;
                }
                break;
            case 702:
                switch (current) {
                    case 'c' :
                        state = 703;
                        return true;
                }
                break;
            case 703:
                switch (current) {
                    case 'r' :
                        state = 704;
                        return true;
                }
                break;
            case 704:
                switch (current) {
                    case ';' : // Emacr;
                        match = "\u0112";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 705:
                switch (current) {
                    case 't' :
                        state = 706;
                        return true;
                }
                break;
            case 706:
                switch (current) {
                    case 'y' :
                        state = 707;
                        return true;
                }
                break;
            case 707:
                switch (current) {
                    case 'S' :
                        state = 708;
                        return true;
                    case 'V' :
                        state = 719;
                        return true;
                }
                break;
            case 708:
                switch (current) {
                    case 'm' :
                        state = 709;
                        return true;
                }
                break;
            case 709:
                switch (current) {
                    case 'a' :
                        state = 710;
                        return true;
                }
                break;
            case 710:
                switch (current) {
                    case 'l' :
                        state = 711;
                        return true;
                }
                break;
            case 711:
                switch (current) {
                    case 'l' :
                        state = 712;
                        return true;
                }
                break;
            case 712:
                switch (current) {
                    case 'S' :
                        state = 713;
                        return true;
                }
                break;
            case 713:
                switch (current) {
                    case 'q' :
                        state = 714;
                        return true;
                }
                break;
            case 714:
                switch (current) {
                    case 'u' :
                        state = 715;
                        return true;
                }
                break;
            case 715:
                switch (current) {
                    case 'a' :
                        state = 716;
                        return true;
                }
                break;
            case 716:
                switch (current) {
                    case 'r' :
                        state = 717;
                        return true;
                }
                break;
            case 717:
                switch (current) {
                    case 'e' :
                        state = 718;
                        return true;
                }
                break;
            case 718:
                switch (current) {
                    case ';' : // EmptySmallSquare;
                        match = "\u25FB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 719:
                switch (current) {
                    case 'e' :
                        state = 720;
                        return true;
                }
                break;
            case 720:
                switch (current) {
                    case 'r' :
                        state = 721;
                        return true;
                }
                break;
            case 721:
                switch (current) {
                    case 'y' :
                        state = 722;
                        return true;
                }
                break;
            case 722:
                switch (current) {
                    case 'S' :
                        state = 723;
                        return true;
                }
                break;
            case 723:
                switch (current) {
                    case 'm' :
                        state = 724;
                        return true;
                }
                break;
            case 724:
                switch (current) {
                    case 'a' :
                        state = 725;
                        return true;
                }
                break;
            case 725:
                switch (current) {
                    case 'l' :
                        state = 726;
                        return true;
                }
                break;
            case 726:
                switch (current) {
                    case 'l' :
                        state = 727;
                        return true;
                }
                break;
            case 727:
                switch (current) {
                    case 'S' :
                        state = 728;
                        return true;
                }
                break;
            case 728:
                switch (current) {
                    case 'q' :
                        state = 729;
                        return true;
                }
                break;
            case 729:
                switch (current) {
                    case 'u' :
                        state = 730;
                        return true;
                }
                break;
            case 730:
                switch (current) {
                    case 'a' :
                        state = 731;
                        return true;
                }
                break;
            case 731:
                switch (current) {
                    case 'r' :
                        state = 732;
                        return true;
                }
                break;
            case 732:
                switch (current) {
                    case 'e' :
                        state = 733;
                        return true;
                }
                break;
            case 733:
                switch (current) {
                    case ';' : // EmptyVerySmallSquare;
                        match = "\u25AB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 734:
                switch (current) {
                    case 'g' :
                        state = 735;
                        return true;
                    case 'p' :
                        state = 738;
                        return true;
                }
                break;
            case 735:
                switch (current) {
                    case 'o' :
                        state = 736;
                        return true;
                }
                break;
            case 736:
                switch (current) {
                    case 'n' :
                        state = 737;
                        return true;
                }
                break;
            case 737:
                switch (current) {
                    case ';' : // Eogon;
                        match = "\u0118";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 738:
                switch (current) {
                    case 'f' :
                        state = 739;
                        return true;
                }
                break;
            case 739:
                switch (current) {
                    case ';' : // Eopf;
                        match = "\uD835\uDD3C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 740:
                switch (current) {
                    case 's' :
                        state = 741;
                        return true;
                }
                break;
            case 741:
                switch (current) {
                    case 'i' :
                        state = 742;
                        return true;
                }
                break;
            case 742:
                switch (current) {
                    case 'l' :
                        state = 743;
                        return true;
                }
                break;
            case 743:
                switch (current) {
                    case 'o' :
                        state = 744;
                        return true;
                }
                break;
            case 744:
                switch (current) {
                    case 'n' :
                        state = 745;
                        return true;
                }
                break;
            case 745:
                switch (current) {
                    case ';' : // Epsilon;
                        match = "\u0395";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 746:
                switch (current) {
                    case 'u' :
                        state = 747;
                        return true;
                }
                break;
            case 747:
                switch (current) {
                    case 'a' :
                        state = 748;
                        return true;
                    case 'i' :
                        state = 755;
                        return true;
                }
                break;
            case 748:
                switch (current) {
                    case 'l' :
                        state = 749;
                        return true;
                }
                break;
            case 749:
                switch (current) {
                    case ';' : // Equal;
                        match = "\u2A75";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'T' :
                        state = 750;
                        return true;
                }
                break;
            case 750:
                switch (current) {
                    case 'i' :
                        state = 751;
                        return true;
                }
                break;
            case 751:
                switch (current) {
                    case 'l' :
                        state = 752;
                        return true;
                }
                break;
            case 752:
                switch (current) {
                    case 'd' :
                        state = 753;
                        return true;
                }
                break;
            case 753:
                switch (current) {
                    case 'e' :
                        state = 754;
                        return true;
                }
                break;
            case 754:
                switch (current) {
                    case ';' : // EqualTilde;
                        match = "\u2242";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 755:
                switch (current) {
                    case 'l' :
                        state = 756;
                        return true;
                }
                break;
            case 756:
                switch (current) {
                    case 'i' :
                        state = 757;
                        return true;
                }
                break;
            case 757:
                switch (current) {
                    case 'b' :
                        state = 758;
                        return true;
                }
                break;
            case 758:
                switch (current) {
                    case 'r' :
                        state = 759;
                        return true;
                }
                break;
            case 759:
                switch (current) {
                    case 'i' :
                        state = 760;
                        return true;
                }
                break;
            case 760:
                switch (current) {
                    case 'u' :
                        state = 761;
                        return true;
                }
                break;
            case 761:
                switch (current) {
                    case 'm' :
                        state = 762;
                        return true;
                }
                break;
            case 762:
                switch (current) {
                    case ';' : // Equilibrium;
                        match = "\u21CC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 763:
                switch (current) {
                    case 'c' :
                        state = 764;
                        return true;
                    case 'i' :
                        state = 766;
                        return true;
                }
                break;
            case 764:
                switch (current) {
                    case 'r' :
                        state = 765;
                        return true;
                }
                break;
            case 765:
                switch (current) {
                    case ';' : // Escr;
                        match = "\u2130";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 766:
                switch (current) {
                    case 'm' :
                        state = 767;
                        return true;
                }
                break;
            case 767:
                switch (current) {
                    case ';' : // Esim;
                        match = "\u2A73";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 768:
                switch (current) {
                    case 'a' :
                        state = 769;
                        return true;
                }
                break;
            case 769:
                switch (current) {
                    case ';' : // Eta;
                        match = "\u0397";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 770:
                switch (current) {
                    case 'm' :
                        state = 771;
                        return true;
                }
                break;
            case 771:
                switch (current) {
                    case 'l' : // Euml
                        match = "\u00CB";
                        matchLength = consumedCount;
                        state = 772;
                        return true;
                }
                break;
            case 772:
                switch (current) {
                    case ';' : // Euml;
                        match = "\u00CB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 773:
                switch (current) {
                    case 'i' :
                        state = 774;
                        return true;
                    case 'p' :
                        state = 778;
                        return true;
                }
                break;
            case 774:
                switch (current) {
                    case 's' :
                        state = 775;
                        return true;
                }
                break;
            case 775:
                switch (current) {
                    case 't' :
                        state = 776;
                        return true;
                }
                break;
            case 776:
                switch (current) {
                    case 's' :
                        state = 777;
                        return true;
                }
                break;
            case 777:
                switch (current) {
                    case ';' : // Exists;
                        match = "\u2203";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 778:
                switch (current) {
                    case 'o' :
                        state = 779;
                        return true;
                }
                break;
            case 779:
                switch (current) {
                    case 'n' :
                        state = 780;
                        return true;
                }
                break;
            case 780:
                switch (current) {
                    case 'e' :
                        state = 781;
                        return true;
                }
                break;
            case 781:
                switch (current) {
                    case 'n' :
                        state = 782;
                        return true;
                }
                break;
            case 782:
                switch (current) {
                    case 't' :
                        state = 783;
                        return true;
                }
                break;
            case 783:
                switch (current) {
                    case 'i' :
                        state = 784;
                        return true;
                }
                break;
            case 784:
                switch (current) {
                    case 'a' :
                        state = 785;
                        return true;
                }
                break;
            case 785:
                switch (current) {
                    case 'l' :
                        state = 786;
                        return true;
                }
                break;
            case 786:
                switch (current) {
                    case 'E' :
                        state = 787;
                        return true;
                }
                break;
            case 787:
                switch (current) {
                    case ';' : // ExponentialE;
                        match = "\u2147";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 788:
                switch (current) {
                    case 'c' :
                        state = 789;
                        return true;
                    case 'f' :
                        state = 791;
                        return true;
                    case 'i' :
                        state = 793;
                        return true;
                    case 'o' :
                        state = 824;
                        return true;
                    case 's' :
                        state = 839;
                        return true;
                }
                break;
            case 789:
                switch (current) {
                    case 'y' :
                        state = 790;
                        return true;
                }
                break;
            case 790:
                switch (current) {
                    case ';' : // Fcy;
                        match = "\u0424";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 791:
                switch (current) {
                    case 'r' :
                        state = 792;
                        return true;
                }
                break;
            case 792:
                switch (current) {
                    case ';' : // Ffr;
                        match = "\uD835\uDD09";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 793:
                switch (current) {
                    case 'l' :
                        state = 794;
                        return true;
                }
                break;
            case 794:
                switch (current) {
                    case 'l' :
                        state = 795;
                        return true;
                }
                break;
            case 795:
                switch (current) {
                    case 'e' :
                        state = 796;
                        return true;
                }
                break;
            case 796:
                switch (current) {
                    case 'd' :
                        state = 797;
                        return true;
                }
                break;
            case 797:
                switch (current) {
                    case 'S' :
                        state = 798;
                        return true;
                    case 'V' :
                        state = 809;
                        return true;
                }
                break;
            case 798:
                switch (current) {
                    case 'm' :
                        state = 799;
                        return true;
                }
                break;
            case 799:
                switch (current) {
                    case 'a' :
                        state = 800;
                        return true;
                }
                break;
            case 800:
                switch (current) {
                    case 'l' :
                        state = 801;
                        return true;
                }
                break;
            case 801:
                switch (current) {
                    case 'l' :
                        state = 802;
                        return true;
                }
                break;
            case 802:
                switch (current) {
                    case 'S' :
                        state = 803;
                        return true;
                }
                break;
            case 803:
                switch (current) {
                    case 'q' :
                        state = 804;
                        return true;
                }
                break;
            case 804:
                switch (current) {
                    case 'u' :
                        state = 805;
                        return true;
                }
                break;
            case 805:
                switch (current) {
                    case 'a' :
                        state = 806;
                        return true;
                }
                break;
            case 806:
                switch (current) {
                    case 'r' :
                        state = 807;
                        return true;
                }
                break;
            case 807:
                switch (current) {
                    case 'e' :
                        state = 808;
                        return true;
                }
                break;
            case 808:
                switch (current) {
                    case ';' : // FilledSmallSquare;
                        match = "\u25FC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 809:
                switch (current) {
                    case 'e' :
                        state = 810;
                        return true;
                }
                break;
            case 810:
                switch (current) {
                    case 'r' :
                        state = 811;
                        return true;
                }
                break;
            case 811:
                switch (current) {
                    case 'y' :
                        state = 812;
                        return true;
                }
                break;
            case 812:
                switch (current) {
                    case 'S' :
                        state = 813;
                        return true;
                }
                break;
            case 813:
                switch (current) {
                    case 'm' :
                        state = 814;
                        return true;
                }
                break;
            case 814:
                switch (current) {
                    case 'a' :
                        state = 815;
                        return true;
                }
                break;
            case 815:
                switch (current) {
                    case 'l' :
                        state = 816;
                        return true;
                }
                break;
            case 816:
                switch (current) {
                    case 'l' :
                        state = 817;
                        return true;
                }
                break;
            case 817:
                switch (current) {
                    case 'S' :
                        state = 818;
                        return true;
                }
                break;
            case 818:
                switch (current) {
                    case 'q' :
                        state = 819;
                        return true;
                }
                break;
            case 819:
                switch (current) {
                    case 'u' :
                        state = 820;
                        return true;
                }
                break;
            case 820:
                switch (current) {
                    case 'a' :
                        state = 821;
                        return true;
                }
                break;
            case 821:
                switch (current) {
                    case 'r' :
                        state = 822;
                        return true;
                }
                break;
            case 822:
                switch (current) {
                    case 'e' :
                        state = 823;
                        return true;
                }
                break;
            case 823:
                switch (current) {
                    case ';' : // FilledVerySmallSquare;
                        match = "\u25AA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 824:
                switch (current) {
                    case 'p' :
                        state = 825;
                        return true;
                    case 'r' :
                        state = 827;
                        return true;
                    case 'u' :
                        state = 831;
                        return true;
                }
                break;
            case 825:
                switch (current) {
                    case 'f' :
                        state = 826;
                        return true;
                }
                break;
            case 826:
                switch (current) {
                    case ';' : // Fopf;
                        match = "\uD835\uDD3D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 827:
                switch (current) {
                    case 'A' :
                        state = 828;
                        return true;
                }
                break;
            case 828:
                switch (current) {
                    case 'l' :
                        state = 829;
                        return true;
                }
                break;
            case 829:
                switch (current) {
                    case 'l' :
                        state = 830;
                        return true;
                }
                break;
            case 830:
                switch (current) {
                    case ';' : // ForAll;
                        match = "\u2200";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 831:
                switch (current) {
                    case 'r' :
                        state = 832;
                        return true;
                }
                break;
            case 832:
                switch (current) {
                    case 'i' :
                        state = 833;
                        return true;
                }
                break;
            case 833:
                switch (current) {
                    case 'e' :
                        state = 834;
                        return true;
                }
                break;
            case 834:
                switch (current) {
                    case 'r' :
                        state = 835;
                        return true;
                }
                break;
            case 835:
                switch (current) {
                    case 't' :
                        state = 836;
                        return true;
                }
                break;
            case 836:
                switch (current) {
                    case 'r' :
                        state = 837;
                        return true;
                }
                break;
            case 837:
                switch (current) {
                    case 'f' :
                        state = 838;
                        return true;
                }
                break;
            case 838:
                switch (current) {
                    case ';' : // Fouriertrf;
                        match = "\u2131";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 839:
                switch (current) {
                    case 'c' :
                        state = 840;
                        return true;
                }
                break;
            case 840:
                switch (current) {
                    case 'r' :
                        state = 841;
                        return true;
                }
                break;
            case 841:
                switch (current) {
                    case ';' : // Fscr;
                        match = "\u2131";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 842:
                switch (current) {
                    case 'J' :
                        state = 843;
                        return true;
                    case 'T' : // GT
                        match = ">";
                        matchLength = consumedCount;
                        state = 846;
                        return true;
                    case 'a' :
                        state = 847;
                        return true;
                    case 'b' :
                        state = 852;
                        return true;
                    case 'c' :
                        state = 857;
                        return true;
                    case 'd' :
                        state = 866;
                        return true;
                    case 'f' :
                        state = 869;
                        return true;
                    case 'g' :
                        state = 871;
                        return true;
                    case 'o' :
                        state = 872;
                        return true;
                    case 'r' :
                        state = 875;
                        return true;
                    case 's' :
                        state = 925;
                        return true;
                    case 't' :
                        state = 928;
                        return true;
                }
                break;
            case 843:
                switch (current) {
                    case 'c' :
                        state = 844;
                        return true;
                }
                break;
            case 844:
                switch (current) {
                    case 'y' :
                        state = 845;
                        return true;
                }
                break;
            case 845:
                switch (current) {
                    case ';' : // GJcy;
                        match = "\u0403";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 846:
                switch (current) {
                    case ';' : // GT;
                        match = ">";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 847:
                switch (current) {
                    case 'm' :
                        state = 848;
                        return true;
                }
                break;
            case 848:
                switch (current) {
                    case 'm' :
                        state = 849;
                        return true;
                }
                break;
            case 849:
                switch (current) {
                    case 'a' :
                        state = 850;
                        return true;
                }
                break;
            case 850:
                switch (current) {
                    case ';' : // Gamma;
                        match = "\u0393";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'd' :
                        state = 851;
                        return true;
                }
                break;
            case 851:
                switch (current) {
                    case ';' : // Gammad;
                        match = "\u03DC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 852:
                switch (current) {
                    case 'r' :
                        state = 853;
                        return true;
                }
                break;
            case 853:
                switch (current) {
                    case 'e' :
                        state = 854;
                        return true;
                }
                break;
            case 854:
                switch (current) {
                    case 'v' :
                        state = 855;
                        return true;
                }
                break;
            case 855:
                switch (current) {
                    case 'e' :
                        state = 856;
                        return true;
                }
                break;
            case 856:
                switch (current) {
                    case ';' : // Gbreve;
                        match = "\u011E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 857:
                switch (current) {
                    case 'e' :
                        state = 858;
                        return true;
                    case 'i' :
                        state = 862;
                        return true;
                    case 'y' :
                        state = 865;
                        return true;
                }
                break;
            case 858:
                switch (current) {
                    case 'd' :
                        state = 859;
                        return true;
                }
                break;
            case 859:
                switch (current) {
                    case 'i' :
                        state = 860;
                        return true;
                }
                break;
            case 860:
                switch (current) {
                    case 'l' :
                        state = 861;
                        return true;
                }
                break;
            case 861:
                switch (current) {
                    case ';' : // Gcedil;
                        match = "\u0122";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 862:
                switch (current) {
                    case 'r' :
                        state = 863;
                        return true;
                }
                break;
            case 863:
                switch (current) {
                    case 'c' :
                        state = 864;
                        return true;
                }
                break;
            case 864:
                switch (current) {
                    case ';' : // Gcirc;
                        match = "\u011C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 865:
                switch (current) {
                    case ';' : // Gcy;
                        match = "\u0413";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 866:
                switch (current) {
                    case 'o' :
                        state = 867;
                        return true;
                }
                break;
            case 867:
                switch (current) {
                    case 't' :
                        state = 868;
                        return true;
                }
                break;
            case 868:
                switch (current) {
                    case ';' : // Gdot;
                        match = "\u0120";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 869:
                switch (current) {
                    case 'r' :
                        state = 870;
                        return true;
                }
                break;
            case 870:
                switch (current) {
                    case ';' : // Gfr;
                        match = "\uD835\uDD0A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 871:
                switch (current) {
                    case ';' : // Gg;
                        match = "\u22D9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 872:
                switch (current) {
                    case 'p' :
                        state = 873;
                        return true;
                }
                break;
            case 873:
                switch (current) {
                    case 'f' :
                        state = 874;
                        return true;
                }
                break;
            case 874:
                switch (current) {
                    case ';' : // Gopf;
                        match = "\uD835\uDD3E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 875:
                switch (current) {
                    case 'e' :
                        state = 876;
                        return true;
                }
                break;
            case 876:
                switch (current) {
                    case 'a' :
                        state = 877;
                        return true;
                }
                break;
            case 877:
                switch (current) {
                    case 't' :
                        state = 878;
                        return true;
                }
                break;
            case 878:
                switch (current) {
                    case 'e' :
                        state = 879;
                        return true;
                }
                break;
            case 879:
                switch (current) {
                    case 'r' :
                        state = 880;
                        return true;
                }
                break;
            case 880:
                switch (current) {
                    case 'E' :
                        state = 881;
                        return true;
                    case 'F' :
                        state = 890;
                        return true;
                    case 'G' :
                        state = 899;
                        return true;
                    case 'L' :
                        state = 906;
                        return true;
                    case 'S' :
                        state = 910;
                        return true;
                    case 'T' :
                        state = 920;
                        return true;
                }
                break;
            case 881:
                switch (current) {
                    case 'q' :
                        state = 882;
                        return true;
                }
                break;
            case 882:
                switch (current) {
                    case 'u' :
                        state = 883;
                        return true;
                }
                break;
            case 883:
                switch (current) {
                    case 'a' :
                        state = 884;
                        return true;
                }
                break;
            case 884:
                switch (current) {
                    case 'l' :
                        state = 885;
                        return true;
                }
                break;
            case 885:
                switch (current) {
                    case ';' : // GreaterEqual;
                        match = "\u2265";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'L' :
                        state = 886;
                        return true;
                }
                break;
            case 886:
                switch (current) {
                    case 'e' :
                        state = 887;
                        return true;
                }
                break;
            case 887:
                switch (current) {
                    case 's' :
                        state = 888;
                        return true;
                }
                break;
            case 888:
                switch (current) {
                    case 's' :
                        state = 889;
                        return true;
                }
                break;
            case 889:
                switch (current) {
                    case ';' : // GreaterEqualLess;
                        match = "\u22DB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 890:
                switch (current) {
                    case 'u' :
                        state = 891;
                        return true;
                }
                break;
            case 891:
                switch (current) {
                    case 'l' :
                        state = 892;
                        return true;
                }
                break;
            case 892:
                switch (current) {
                    case 'l' :
                        state = 893;
                        return true;
                }
                break;
            case 893:
                switch (current) {
                    case 'E' :
                        state = 894;
                        return true;
                }
                break;
            case 894:
                switch (current) {
                    case 'q' :
                        state = 895;
                        return true;
                }
                break;
            case 895:
                switch (current) {
                    case 'u' :
                        state = 896;
                        return true;
                }
                break;
            case 896:
                switch (current) {
                    case 'a' :
                        state = 897;
                        return true;
                }
                break;
            case 897:
                switch (current) {
                    case 'l' :
                        state = 898;
                        return true;
                }
                break;
            case 898:
                switch (current) {
                    case ';' : // GreaterFullEqual;
                        match = "\u2267";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 899:
                switch (current) {
                    case 'r' :
                        state = 900;
                        return true;
                }
                break;
            case 900:
                switch (current) {
                    case 'e' :
                        state = 901;
                        return true;
                }
                break;
            case 901:
                switch (current) {
                    case 'a' :
                        state = 902;
                        return true;
                }
                break;
            case 902:
                switch (current) {
                    case 't' :
                        state = 903;
                        return true;
                }
                break;
            case 903:
                switch (current) {
                    case 'e' :
                        state = 904;
                        return true;
                }
                break;
            case 904:
                switch (current) {
                    case 'r' :
                        state = 905;
                        return true;
                }
                break;
            case 905:
                switch (current) {
                    case ';' : // GreaterGreater;
                        match = "\u2AA2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 906:
                switch (current) {
                    case 'e' :
                        state = 907;
                        return true;
                }
                break;
            case 907:
                switch (current) {
                    case 's' :
                        state = 908;
                        return true;
                }
                break;
            case 908:
                switch (current) {
                    case 's' :
                        state = 909;
                        return true;
                }
                break;
            case 909:
                switch (current) {
                    case ';' : // GreaterLess;
                        match = "\u2277";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 910:
                switch (current) {
                    case 'l' :
                        state = 911;
                        return true;
                }
                break;
            case 911:
                switch (current) {
                    case 'a' :
                        state = 912;
                        return true;
                }
                break;
            case 912:
                switch (current) {
                    case 'n' :
                        state = 913;
                        return true;
                }
                break;
            case 913:
                switch (current) {
                    case 't' :
                        state = 914;
                        return true;
                }
                break;
            case 914:
                switch (current) {
                    case 'E' :
                        state = 915;
                        return true;
                }
                break;
            case 915:
                switch (current) {
                    case 'q' :
                        state = 916;
                        return true;
                }
                break;
            case 916:
                switch (current) {
                    case 'u' :
                        state = 917;
                        return true;
                }
                break;
            case 917:
                switch (current) {
                    case 'a' :
                        state = 918;
                        return true;
                }
                break;
            case 918:
                switch (current) {
                    case 'l' :
                        state = 919;
                        return true;
                }
                break;
            case 919:
                switch (current) {
                    case ';' : // GreaterSlantEqual;
                        match = "\u2A7E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 920:
                switch (current) {
                    case 'i' :
                        state = 921;
                        return true;
                }
                break;
            case 921:
                switch (current) {
                    case 'l' :
                        state = 922;
                        return true;
                }
                break;
            case 922:
                switch (current) {
                    case 'd' :
                        state = 923;
                        return true;
                }
                break;
            case 923:
                switch (current) {
                    case 'e' :
                        state = 924;
                        return true;
                }
                break;
            case 924:
                switch (current) {
                    case ';' : // GreaterTilde;
                        match = "\u2273";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 925:
                switch (current) {
                    case 'c' :
                        state = 926;
                        return true;
                }
                break;
            case 926:
                switch (current) {
                    case 'r' :
                        state = 927;
                        return true;
                }
                break;
            case 927:
                switch (current) {
                    case ';' : // Gscr;
                        match = "\uD835\uDCA2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 928:
                switch (current) {
                    case ';' : // Gt;
                        match = "\u226B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 929:
                switch (current) {
                    case 'A' :
                        state = 930;
                        return true;
                    case 'a' :
                        state = 935;
                        return true;
                    case 'c' :
                        state = 940;
                        return true;
                    case 'f' :
                        state = 944;
                        return true;
                    case 'i' :
                        state = 946;
                        return true;
                    case 'o' :
                        state = 957;
                        return true;
                    case 's' :
                        state = 972;
                        return true;
                    case 'u' :
                        state = 979;
                        return true;
                }
                break;
            case 930:
                switch (current) {
                    case 'R' :
                        state = 931;
                        return true;
                }
                break;
            case 931:
                switch (current) {
                    case 'D' :
                        state = 932;
                        return true;
                }
                break;
            case 932:
                switch (current) {
                    case 'c' :
                        state = 933;
                        return true;
                }
                break;
            case 933:
                switch (current) {
                    case 'y' :
                        state = 934;
                        return true;
                }
                break;
            case 934:
                switch (current) {
                    case ';' : // HARDcy;
                        match = "\u042A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 935:
                switch (current) {
                    case 'c' :
                        state = 936;
                        return true;
                    case 't' :
                        state = 939;
                        return true;
                }
                break;
            case 936:
                switch (current) {
                    case 'e' :
                        state = 937;
                        return true;
                }
                break;
            case 937:
                switch (current) {
                    case 'k' :
                        state = 938;
                        return true;
                }
                break;
            case 938:
                switch (current) {
                    case ';' : // Hacek;
                        match = "\u02C7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 939:
                switch (current) {
                    case ';' : // Hat;
                        match = "^";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 940:
                switch (current) {
                    case 'i' :
                        state = 941;
                        return true;
                }
                break;
            case 941:
                switch (current) {
                    case 'r' :
                        state = 942;
                        return true;
                }
                break;
            case 942:
                switch (current) {
                    case 'c' :
                        state = 943;
                        return true;
                }
                break;
            case 943:
                switch (current) {
                    case ';' : // Hcirc;
                        match = "\u0124";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 944:
                switch (current) {
                    case 'r' :
                        state = 945;
                        return true;
                }
                break;
            case 945:
                switch (current) {
                    case ';' : // Hfr;
                        match = "\u210C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 946:
                switch (current) {
                    case 'l' :
                        state = 947;
                        return true;
                }
                break;
            case 947:
                switch (current) {
                    case 'b' :
                        state = 948;
                        return true;
                }
                break;
            case 948:
                switch (current) {
                    case 'e' :
                        state = 949;
                        return true;
                }
                break;
            case 949:
                switch (current) {
                    case 'r' :
                        state = 950;
                        return true;
                }
                break;
            case 950:
                switch (current) {
                    case 't' :
                        state = 951;
                        return true;
                }
                break;
            case 951:
                switch (current) {
                    case 'S' :
                        state = 952;
                        return true;
                }
                break;
            case 952:
                switch (current) {
                    case 'p' :
                        state = 953;
                        return true;
                }
                break;
            case 953:
                switch (current) {
                    case 'a' :
                        state = 954;
                        return true;
                }
                break;
            case 954:
                switch (current) {
                    case 'c' :
                        state = 955;
                        return true;
                }
                break;
            case 955:
                switch (current) {
                    case 'e' :
                        state = 956;
                        return true;
                }
                break;
            case 956:
                switch (current) {
                    case ';' : // HilbertSpace;
                        match = "\u210B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 957:
                switch (current) {
                    case 'p' :
                        state = 958;
                        return true;
                    case 'r' :
                        state = 960;
                        return true;
                }
                break;
            case 958:
                switch (current) {
                    case 'f' :
                        state = 959;
                        return true;
                }
                break;
            case 959:
                switch (current) {
                    case ';' : // Hopf;
                        match = "\u210D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 960:
                switch (current) {
                    case 'i' :
                        state = 961;
                        return true;
                }
                break;
            case 961:
                switch (current) {
                    case 'z' :
                        state = 962;
                        return true;
                }
                break;
            case 962:
                switch (current) {
                    case 'o' :
                        state = 963;
                        return true;
                }
                break;
            case 963:
                switch (current) {
                    case 'n' :
                        state = 964;
                        return true;
                }
                break;
            case 964:
                switch (current) {
                    case 't' :
                        state = 965;
                        return true;
                }
                break;
            case 965:
                switch (current) {
                    case 'a' :
                        state = 966;
                        return true;
                }
                break;
            case 966:
                switch (current) {
                    case 'l' :
                        state = 967;
                        return true;
                }
                break;
            case 967:
                switch (current) {
                    case 'L' :
                        state = 968;
                        return true;
                }
                break;
            case 968:
                switch (current) {
                    case 'i' :
                        state = 969;
                        return true;
                }
                break;
            case 969:
                switch (current) {
                    case 'n' :
                        state = 970;
                        return true;
                }
                break;
            case 970:
                switch (current) {
                    case 'e' :
                        state = 971;
                        return true;
                }
                break;
            case 971:
                switch (current) {
                    case ';' : // HorizontalLine;
                        match = "\u2500";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 972:
                switch (current) {
                    case 'c' :
                        state = 973;
                        return true;
                    case 't' :
                        state = 975;
                        return true;
                }
                break;
            case 973:
                switch (current) {
                    case 'r' :
                        state = 974;
                        return true;
                }
                break;
            case 974:
                switch (current) {
                    case ';' : // Hscr;
                        match = "\u210B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 975:
                switch (current) {
                    case 'r' :
                        state = 976;
                        return true;
                }
                break;
            case 976:
                switch (current) {
                    case 'o' :
                        state = 977;
                        return true;
                }
                break;
            case 977:
                switch (current) {
                    case 'k' :
                        state = 978;
                        return true;
                }
                break;
            case 978:
                switch (current) {
                    case ';' : // Hstrok;
                        match = "\u0126";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 979:
                switch (current) {
                    case 'm' :
                        state = 980;
                        return true;
                }
                break;
            case 980:
                switch (current) {
                    case 'p' :
                        state = 981;
                        return true;
                }
                break;
            case 981:
                switch (current) {
                    case 'D' :
                        state = 982;
                        return true;
                    case 'E' :
                        state = 990;
                        return true;
                }
                break;
            case 982:
                switch (current) {
                    case 'o' :
                        state = 983;
                        return true;
                }
                break;
            case 983:
                switch (current) {
                    case 'w' :
                        state = 984;
                        return true;
                }
                break;
            case 984:
                switch (current) {
                    case 'n' :
                        state = 985;
                        return true;
                }
                break;
            case 985:
                switch (current) {
                    case 'H' :
                        state = 986;
                        return true;
                }
                break;
            case 986:
                switch (current) {
                    case 'u' :
                        state = 987;
                        return true;
                }
                break;
            case 987:
                switch (current) {
                    case 'm' :
                        state = 988;
                        return true;
                }
                break;
            case 988:
                switch (current) {
                    case 'p' :
                        state = 989;
                        return true;
                }
                break;
            case 989:
                switch (current) {
                    case ';' : // HumpDownHump;
                        match = "\u224E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 990:
                switch (current) {
                    case 'q' :
                        state = 991;
                        return true;
                }
                break;
            case 991:
                switch (current) {
                    case 'u' :
                        state = 992;
                        return true;
                }
                break;
            case 992:
                switch (current) {
                    case 'a' :
                        state = 993;
                        return true;
                }
                break;
            case 993:
                switch (current) {
                    case 'l' :
                        state = 994;
                        return true;
                }
                break;
            case 994:
                switch (current) {
                    case ';' : // HumpEqual;
                        match = "\u224F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 995:
                switch (current) {
                    case 'E' :
                        state = 996;
                        return true;
                    case 'J' :
                        state = 999;
                        return true;
                    case 'O' :
                        state = 1003;
                        return true;
                    case 'a' :
                        state = 1006;
                        return true;
                    case 'c' :
                        state = 1011;
                        return true;
                    case 'd' :
                        state = 1016;
                        return true;
                    case 'f' :
                        state = 1019;
                        return true;
                    case 'g' :
                        state = 1021;
                        return true;
                    case 'm' :
                        state = 1026;
                        return true;
                    case 'n' :
                        state = 1042;
                        return true;
                    case 'o' :
                        state = 1074;
                        return true;
                    case 's' :
                        state = 1082;
                        return true;
                    case 't' :
                        state = 1085;
                        return true;
                    case 'u' :
                        state = 1090;
                        return true;
                }
                break;
            case 996:
                switch (current) {
                    case 'c' :
                        state = 997;
                        return true;
                }
                break;
            case 997:
                switch (current) {
                    case 'y' :
                        state = 998;
                        return true;
                }
                break;
            case 998:
                switch (current) {
                    case ';' : // IEcy;
                        match = "\u0415";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 999:
                switch (current) {
                    case 'l' :
                        state = 1000;
                        return true;
                }
                break;
        }
        return false;
    }

    private boolean parse2(final int current) {
        consumedCount++;
        switch (state) {
            case 1000:
                switch (current) {
                    case 'i' :
                        state = 1001;
                        return true;
                }
                break;
            case 1001:
                switch (current) {
                    case 'g' :
                        state = 1002;
                        return true;
                }
                break;
            case 1002:
                switch (current) {
                    case ';' : // IJlig;
                        match = "\u0132";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1003:
                switch (current) {
                    case 'c' :
                        state = 1004;
                        return true;
                }
                break;
            case 1004:
                switch (current) {
                    case 'y' :
                        state = 1005;
                        return true;
                }
                break;
            case 1005:
                switch (current) {
                    case ';' : // IOcy;
                        match = "\u0401";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1006:
                switch (current) {
                    case 'c' :
                        state = 1007;
                        return true;
                }
                break;
            case 1007:
                switch (current) {
                    case 'u' :
                        state = 1008;
                        return true;
                }
                break;
            case 1008:
                switch (current) {
                    case 't' :
                        state = 1009;
                        return true;
                }
                break;
            case 1009:
                switch (current) {
                    case 'e' : // Iacute
                        match = "\u00CD";
                        matchLength = consumedCount;
                        state = 1010;
                        return true;
                }
                break;
            case 1010:
                switch (current) {
                    case ';' : // Iacute;
                        match = "\u00CD";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1011:
                switch (current) {
                    case 'i' :
                        state = 1012;
                        return true;
                    case 'y' :
                        state = 1015;
                        return true;
                }
                break;
            case 1012:
                switch (current) {
                    case 'r' :
                        state = 1013;
                        return true;
                }
                break;
            case 1013:
                switch (current) {
                    case 'c' : // Icirc
                        match = "\u00CE";
                        matchLength = consumedCount;
                        state = 1014;
                        return true;
                }
                break;
            case 1014:
                switch (current) {
                    case ';' : // Icirc;
                        match = "\u00CE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1015:
                switch (current) {
                    case ';' : // Icy;
                        match = "\u0418";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1016:
                switch (current) {
                    case 'o' :
                        state = 1017;
                        return true;
                }
                break;
            case 1017:
                switch (current) {
                    case 't' :
                        state = 1018;
                        return true;
                }
                break;
            case 1018:
                switch (current) {
                    case ';' : // Idot;
                        match = "\u0130";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1019:
                switch (current) {
                    case 'r' :
                        state = 1020;
                        return true;
                }
                break;
            case 1020:
                switch (current) {
                    case ';' : // Ifr;
                        match = "\u2111";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1021:
                switch (current) {
                    case 'r' :
                        state = 1022;
                        return true;
                }
                break;
            case 1022:
                switch (current) {
                    case 'a' :
                        state = 1023;
                        return true;
                }
                break;
            case 1023:
                switch (current) {
                    case 'v' :
                        state = 1024;
                        return true;
                }
                break;
            case 1024:
                switch (current) {
                    case 'e' : // Igrave
                        match = "\u00CC";
                        matchLength = consumedCount;
                        state = 1025;
                        return true;
                }
                break;
            case 1025:
                switch (current) {
                    case ';' : // Igrave;
                        match = "\u00CC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1026:
                switch (current) {
                    case ';' : // Im;
                        match = "\u2111";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 1027;
                        return true;
                    case 'p' :
                        state = 1037;
                        return true;
                }
                break;
            case 1027:
                switch (current) {
                    case 'c' :
                        state = 1028;
                        return true;
                    case 'g' :
                        state = 1030;
                        return true;
                }
                break;
            case 1028:
                switch (current) {
                    case 'r' :
                        state = 1029;
                        return true;
                }
                break;
            case 1029:
                switch (current) {
                    case ';' : // Imacr;
                        match = "\u012A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1030:
                switch (current) {
                    case 'i' :
                        state = 1031;
                        return true;
                }
                break;
            case 1031:
                switch (current) {
                    case 'n' :
                        state = 1032;
                        return true;
                }
                break;
            case 1032:
                switch (current) {
                    case 'a' :
                        state = 1033;
                        return true;
                }
                break;
            case 1033:
                switch (current) {
                    case 'r' :
                        state = 1034;
                        return true;
                }
                break;
            case 1034:
                switch (current) {
                    case 'y' :
                        state = 1035;
                        return true;
                }
                break;
            case 1035:
                switch (current) {
                    case 'I' :
                        state = 1036;
                        return true;
                }
                break;
            case 1036:
                switch (current) {
                    case ';' : // ImaginaryI;
                        match = "\u2148";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1037:
                switch (current) {
                    case 'l' :
                        state = 1038;
                        return true;
                }
                break;
            case 1038:
                switch (current) {
                    case 'i' :
                        state = 1039;
                        return true;
                }
                break;
            case 1039:
                switch (current) {
                    case 'e' :
                        state = 1040;
                        return true;
                }
                break;
            case 1040:
                switch (current) {
                    case 's' :
                        state = 1041;
                        return true;
                }
                break;
            case 1041:
                switch (current) {
                    case ';' : // Implies;
                        match = "\u21D2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1042:
                switch (current) {
                    case 't' :
                        state = 1043;
                        return true;
                    case 'v' :
                        state = 1057;
                        return true;
                }
                break;
            case 1043:
                switch (current) {
                    case ';' : // Int;
                        match = "\u222C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 1044;
                        return true;
                }
                break;
            case 1044:
                switch (current) {
                    case 'g' :
                        state = 1045;
                        return true;
                    case 'r' :
                        state = 1049;
                        return true;
                }
                break;
            case 1045:
                switch (current) {
                    case 'r' :
                        state = 1046;
                        return true;
                }
                break;
            case 1046:
                switch (current) {
                    case 'a' :
                        state = 1047;
                        return true;
                }
                break;
            case 1047:
                switch (current) {
                    case 'l' :
                        state = 1048;
                        return true;
                }
                break;
            case 1048:
                switch (current) {
                    case ';' : // Integral;
                        match = "\u222B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1049:
                switch (current) {
                    case 's' :
                        state = 1050;
                        return true;
                }
                break;
            case 1050:
                switch (current) {
                    case 'e' :
                        state = 1051;
                        return true;
                }
                break;
            case 1051:
                switch (current) {
                    case 'c' :
                        state = 1052;
                        return true;
                }
                break;
            case 1052:
                switch (current) {
                    case 't' :
                        state = 1053;
                        return true;
                }
                break;
            case 1053:
                switch (current) {
                    case 'i' :
                        state = 1054;
                        return true;
                }
                break;
            case 1054:
                switch (current) {
                    case 'o' :
                        state = 1055;
                        return true;
                }
                break;
            case 1055:
                switch (current) {
                    case 'n' :
                        state = 1056;
                        return true;
                }
                break;
            case 1056:
                switch (current) {
                    case ';' : // Intersection;
                        match = "\u22C2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1057:
                switch (current) {
                    case 'i' :
                        state = 1058;
                        return true;
                }
                break;
            case 1058:
                switch (current) {
                    case 's' :
                        state = 1059;
                        return true;
                }
                break;
            case 1059:
                switch (current) {
                    case 'i' :
                        state = 1060;
                        return true;
                }
                break;
            case 1060:
                switch (current) {
                    case 'b' :
                        state = 1061;
                        return true;
                }
                break;
            case 1061:
                switch (current) {
                    case 'l' :
                        state = 1062;
                        return true;
                }
                break;
            case 1062:
                switch (current) {
                    case 'e' :
                        state = 1063;
                        return true;
                }
                break;
            case 1063:
                switch (current) {
                    case 'C' :
                        state = 1064;
                        return true;
                    case 'T' :
                        state = 1069;
                        return true;
                }
                break;
            case 1064:
                switch (current) {
                    case 'o' :
                        state = 1065;
                        return true;
                }
                break;
            case 1065:
                switch (current) {
                    case 'm' :
                        state = 1066;
                        return true;
                }
                break;
            case 1066:
                switch (current) {
                    case 'm' :
                        state = 1067;
                        return true;
                }
                break;
            case 1067:
                switch (current) {
                    case 'a' :
                        state = 1068;
                        return true;
                }
                break;
            case 1068:
                switch (current) {
                    case ';' : // InvisibleComma;
                        match = "\u2063";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1069:
                switch (current) {
                    case 'i' :
                        state = 1070;
                        return true;
                }
                break;
            case 1070:
                switch (current) {
                    case 'm' :
                        state = 1071;
                        return true;
                }
                break;
            case 1071:
                switch (current) {
                    case 'e' :
                        state = 1072;
                        return true;
                }
                break;
            case 1072:
                switch (current) {
                    case 's' :
                        state = 1073;
                        return true;
                }
                break;
            case 1073:
                switch (current) {
                    case ';' : // InvisibleTimes;
                        match = "\u2062";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1074:
                switch (current) {
                    case 'g' :
                        state = 1075;
                        return true;
                    case 'p' :
                        state = 1078;
                        return true;
                    case 't' :
                        state = 1080;
                        return true;
                }
                break;
            case 1075:
                switch (current) {
                    case 'o' :
                        state = 1076;
                        return true;
                }
                break;
            case 1076:
                switch (current) {
                    case 'n' :
                        state = 1077;
                        return true;
                }
                break;
            case 1077:
                switch (current) {
                    case ';' : // Iogon;
                        match = "\u012E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1078:
                switch (current) {
                    case 'f' :
                        state = 1079;
                        return true;
                }
                break;
            case 1079:
                switch (current) {
                    case ';' : // Iopf;
                        match = "\uD835\uDD40";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1080:
                switch (current) {
                    case 'a' :
                        state = 1081;
                        return true;
                }
                break;
            case 1081:
                switch (current) {
                    case ';' : // Iota;
                        match = "\u0399";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1082:
                switch (current) {
                    case 'c' :
                        state = 1083;
                        return true;
                }
                break;
            case 1083:
                switch (current) {
                    case 'r' :
                        state = 1084;
                        return true;
                }
                break;
            case 1084:
                switch (current) {
                    case ';' : // Iscr;
                        match = "\u2110";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1085:
                switch (current) {
                    case 'i' :
                        state = 1086;
                        return true;
                }
                break;
            case 1086:
                switch (current) {
                    case 'l' :
                        state = 1087;
                        return true;
                }
                break;
            case 1087:
                switch (current) {
                    case 'd' :
                        state = 1088;
                        return true;
                }
                break;
            case 1088:
                switch (current) {
                    case 'e' :
                        state = 1089;
                        return true;
                }
                break;
            case 1089:
                switch (current) {
                    case ';' : // Itilde;
                        match = "\u0128";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1090:
                switch (current) {
                    case 'k' :
                        state = 1091;
                        return true;
                    case 'm' :
                        state = 1094;
                        return true;
                }
                break;
            case 1091:
                switch (current) {
                    case 'c' :
                        state = 1092;
                        return true;
                }
                break;
            case 1092:
                switch (current) {
                    case 'y' :
                        state = 1093;
                        return true;
                }
                break;
            case 1093:
                switch (current) {
                    case ';' : // Iukcy;
                        match = "\u0406";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1094:
                switch (current) {
                    case 'l' : // Iuml
                        match = "\u00CF";
                        matchLength = consumedCount;
                        state = 1095;
                        return true;
                }
                break;
            case 1095:
                switch (current) {
                    case ';' : // Iuml;
                        match = "\u00CF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1096:
                switch (current) {
                    case 'c' :
                        state = 1097;
                        return true;
                    case 'f' :
                        state = 1102;
                        return true;
                    case 'o' :
                        state = 1104;
                        return true;
                    case 's' :
                        state = 1107;
                        return true;
                    case 'u' :
                        state = 1114;
                        return true;
                }
                break;
            case 1097:
                switch (current) {
                    case 'i' :
                        state = 1098;
                        return true;
                    case 'y' :
                        state = 1101;
                        return true;
                }
                break;
            case 1098:
                switch (current) {
                    case 'r' :
                        state = 1099;
                        return true;
                }
                break;
            case 1099:
                switch (current) {
                    case 'c' :
                        state = 1100;
                        return true;
                }
                break;
            case 1100:
                switch (current) {
                    case ';' : // Jcirc;
                        match = "\u0134";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1101:
                switch (current) {
                    case ';' : // Jcy;
                        match = "\u0419";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1102:
                switch (current) {
                    case 'r' :
                        state = 1103;
                        return true;
                }
                break;
            case 1103:
                switch (current) {
                    case ';' : // Jfr;
                        match = "\uD835\uDD0D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1104:
                switch (current) {
                    case 'p' :
                        state = 1105;
                        return true;
                }
                break;
            case 1105:
                switch (current) {
                    case 'f' :
                        state = 1106;
                        return true;
                }
                break;
            case 1106:
                switch (current) {
                    case ';' : // Jopf;
                        match = "\uD835\uDD41";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1107:
                switch (current) {
                    case 'c' :
                        state = 1108;
                        return true;
                    case 'e' :
                        state = 1110;
                        return true;
                }
                break;
            case 1108:
                switch (current) {
                    case 'r' :
                        state = 1109;
                        return true;
                }
                break;
            case 1109:
                switch (current) {
                    case ';' : // Jscr;
                        match = "\uD835\uDCA5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1110:
                switch (current) {
                    case 'r' :
                        state = 1111;
                        return true;
                }
                break;
            case 1111:
                switch (current) {
                    case 'c' :
                        state = 1112;
                        return true;
                }
                break;
            case 1112:
                switch (current) {
                    case 'y' :
                        state = 1113;
                        return true;
                }
                break;
            case 1113:
                switch (current) {
                    case ';' : // Jsercy;
                        match = "\u0408";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1114:
                switch (current) {
                    case 'k' :
                        state = 1115;
                        return true;
                }
                break;
            case 1115:
                switch (current) {
                    case 'c' :
                        state = 1116;
                        return true;
                }
                break;
            case 1116:
                switch (current) {
                    case 'y' :
                        state = 1117;
                        return true;
                }
                break;
            case 1117:
                switch (current) {
                    case ';' : // Jukcy;
                        match = "\u0404";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1118:
                switch (current) {
                    case 'H' :
                        state = 1119;
                        return true;
                    case 'J' :
                        state = 1122;
                        return true;
                    case 'a' :
                        state = 1125;
                        return true;
                    case 'c' :
                        state = 1129;
                        return true;
                    case 'f' :
                        state = 1135;
                        return true;
                    case 'o' :
                        state = 1137;
                        return true;
                    case 's' :
                        state = 1140;
                        return true;
                }
                break;
            case 1119:
                switch (current) {
                    case 'c' :
                        state = 1120;
                        return true;
                }
                break;
            case 1120:
                switch (current) {
                    case 'y' :
                        state = 1121;
                        return true;
                }
                break;
            case 1121:
                switch (current) {
                    case ';' : // KHcy;
                        match = "\u0425";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1122:
                switch (current) {
                    case 'c' :
                        state = 1123;
                        return true;
                }
                break;
            case 1123:
                switch (current) {
                    case 'y' :
                        state = 1124;
                        return true;
                }
                break;
            case 1124:
                switch (current) {
                    case ';' : // KJcy;
                        match = "\u040C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1125:
                switch (current) {
                    case 'p' :
                        state = 1126;
                        return true;
                }
                break;
            case 1126:
                switch (current) {
                    case 'p' :
                        state = 1127;
                        return true;
                }
                break;
            case 1127:
                switch (current) {
                    case 'a' :
                        state = 1128;
                        return true;
                }
                break;
            case 1128:
                switch (current) {
                    case ';' : // Kappa;
                        match = "\u039A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1129:
                switch (current) {
                    case 'e' :
                        state = 1130;
                        return true;
                    case 'y' :
                        state = 1134;
                        return true;
                }
                break;
            case 1130:
                switch (current) {
                    case 'd' :
                        state = 1131;
                        return true;
                }
                break;
            case 1131:
                switch (current) {
                    case 'i' :
                        state = 1132;
                        return true;
                }
                break;
            case 1132:
                switch (current) {
                    case 'l' :
                        state = 1133;
                        return true;
                }
                break;
            case 1133:
                switch (current) {
                    case ';' : // Kcedil;
                        match = "\u0136";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1134:
                switch (current) {
                    case ';' : // Kcy;
                        match = "\u041A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1135:
                switch (current) {
                    case 'r' :
                        state = 1136;
                        return true;
                }
                break;
            case 1136:
                switch (current) {
                    case ';' : // Kfr;
                        match = "\uD835\uDD0E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1137:
                switch (current) {
                    case 'p' :
                        state = 1138;
                        return true;
                }
                break;
            case 1138:
                switch (current) {
                    case 'f' :
                        state = 1139;
                        return true;
                }
                break;
            case 1139:
                switch (current) {
                    case ';' : // Kopf;
                        match = "\uD835\uDD42";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1140:
                switch (current) {
                    case 'c' :
                        state = 1141;
                        return true;
                }
                break;
            case 1141:
                switch (current) {
                    case 'r' :
                        state = 1142;
                        return true;
                }
                break;
            case 1142:
                switch (current) {
                    case ';' : // Kscr;
                        match = "\uD835\uDCA6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1143:
                switch (current) {
                    case 'J' :
                        state = 1144;
                        return true;
                    case 'T' : // LT
                        match = "<";
                        matchLength = consumedCount;
                        state = 1147;
                        return true;
                    case 'a' :
                        state = 1148;
                        return true;
                    case 'c' :
                        state = 1169;
                        return true;
                    case 'e' :
                        state = 1179;
                        return true;
                    case 'f' :
                        state = 1404;
                        return true;
                    case 'l' :
                        state = 1406;
                        return true;
                    case 'm' :
                        state = 1415;
                        return true;
                    case 'o' :
                        state = 1420;
                        return true;
                    case 's' :
                        state = 1505;
                        return true;
                    case 't' :
                        state = 1513;
                        return true;
                }
                break;
            case 1144:
                switch (current) {
                    case 'c' :
                        state = 1145;
                        return true;
                }
                break;
            case 1145:
                switch (current) {
                    case 'y' :
                        state = 1146;
                        return true;
                }
                break;
            case 1146:
                switch (current) {
                    case ';' : // LJcy;
                        match = "\u0409";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1147:
                switch (current) {
                    case ';' : // LT;
                        match = "<";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1148:
                switch (current) {
                    case 'c' :
                        state = 1149;
                        return true;
                    case 'm' :
                        state = 1153;
                        return true;
                    case 'n' :
                        state = 1157;
                        return true;
                    case 'p' :
                        state = 1159;
                        return true;
                    case 'r' :
                        state = 1167;
                        return true;
                }
                break;
            case 1149:
                switch (current) {
                    case 'u' :
                        state = 1150;
                        return true;
                }
                break;
            case 1150:
                switch (current) {
                    case 't' :
                        state = 1151;
                        return true;
                }
                break;
            case 1151:
                switch (current) {
                    case 'e' :
                        state = 1152;
                        return true;
                }
                break;
            case 1152:
                switch (current) {
                    case ';' : // Lacute;
                        match = "\u0139";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1153:
                switch (current) {
                    case 'b' :
                        state = 1154;
                        return true;
                }
                break;
            case 1154:
                switch (current) {
                    case 'd' :
                        state = 1155;
                        return true;
                }
                break;
            case 1155:
                switch (current) {
                    case 'a' :
                        state = 1156;
                        return true;
                }
                break;
            case 1156:
                switch (current) {
                    case ';' : // Lambda;
                        match = "\u039B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1157:
                switch (current) {
                    case 'g' :
                        state = 1158;
                        return true;
                }
                break;
            case 1158:
                switch (current) {
                    case ';' : // Lang;
                        match = "\u27EA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1159:
                switch (current) {
                    case 'l' :
                        state = 1160;
                        return true;
                }
                break;
            case 1160:
                switch (current) {
                    case 'a' :
                        state = 1161;
                        return true;
                }
                break;
            case 1161:
                switch (current) {
                    case 'c' :
                        state = 1162;
                        return true;
                }
                break;
            case 1162:
                switch (current) {
                    case 'e' :
                        state = 1163;
                        return true;
                }
                break;
            case 1163:
                switch (current) {
                    case 't' :
                        state = 1164;
                        return true;
                }
                break;
            case 1164:
                switch (current) {
                    case 'r' :
                        state = 1165;
                        return true;
                }
                break;
            case 1165:
                switch (current) {
                    case 'f' :
                        state = 1166;
                        return true;
                }
                break;
            case 1166:
                switch (current) {
                    case ';' : // Laplacetrf;
                        match = "\u2112";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1167:
                switch (current) {
                    case 'r' :
                        state = 1168;
                        return true;
                }
                break;
            case 1168:
                switch (current) {
                    case ';' : // Larr;
                        match = "\u219E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1169:
                switch (current) {
                    case 'a' :
                        state = 1170;
                        return true;
                    case 'e' :
                        state = 1174;
                        return true;
                    case 'y' :
                        state = 1178;
                        return true;
                }
                break;
            case 1170:
                switch (current) {
                    case 'r' :
                        state = 1171;
                        return true;
                }
                break;
            case 1171:
                switch (current) {
                    case 'o' :
                        state = 1172;
                        return true;
                }
                break;
            case 1172:
                switch (current) {
                    case 'n' :
                        state = 1173;
                        return true;
                }
                break;
            case 1173:
                switch (current) {
                    case ';' : // Lcaron;
                        match = "\u013D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1174:
                switch (current) {
                    case 'd' :
                        state = 1175;
                        return true;
                }
                break;
            case 1175:
                switch (current) {
                    case 'i' :
                        state = 1176;
                        return true;
                }
                break;
            case 1176:
                switch (current) {
                    case 'l' :
                        state = 1177;
                        return true;
                }
                break;
            case 1177:
                switch (current) {
                    case ';' : // Lcedil;
                        match = "\u013B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1178:
                switch (current) {
                    case ';' : // Lcy;
                        match = "\u041B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1179:
                switch (current) {
                    case 'f' :
                        state = 1180;
                        return true;
                    case 's' :
                        state = 1355;
                        return true;
                }
                break;
            case 1180:
                switch (current) {
                    case 't' :
                        state = 1181;
                        return true;
                }
                break;
            case 1181:
                switch (current) {
                    case 'A' :
                        state = 1182;
                        return true;
                    case 'C' :
                        state = 1211;
                        return true;
                    case 'D' :
                        state = 1218;
                        return true;
                    case 'F' :
                        state = 1251;
                        return true;
                    case 'R' :
                        state = 1256;
                        return true;
                    case 'T' :
                        state = 1272;
                        return true;
                    case 'U' :
                        state = 1301;
                        return true;
                    case 'V' :
                        state = 1331;
                        return true;
                    case 'a' :
                        state = 1340;
                        return true;
                    case 'r' :
                        state = 1345;
                        return true;
                }
                break;
            case 1182:
                switch (current) {
                    case 'n' :
                        state = 1183;
                        return true;
                    case 'r' :
                        state = 1194;
                        return true;
                }
                break;
            case 1183:
                switch (current) {
                    case 'g' :
                        state = 1184;
                        return true;
                }
                break;
            case 1184:
                switch (current) {
                    case 'l' :
                        state = 1185;
                        return true;
                }
                break;
            case 1185:
                switch (current) {
                    case 'e' :
                        state = 1186;
                        return true;
                }
                break;
            case 1186:
                switch (current) {
                    case 'B' :
                        state = 1187;
                        return true;
                }
                break;
            case 1187:
                switch (current) {
                    case 'r' :
                        state = 1188;
                        return true;
                }
                break;
            case 1188:
                switch (current) {
                    case 'a' :
                        state = 1189;
                        return true;
                }
                break;
            case 1189:
                switch (current) {
                    case 'c' :
                        state = 1190;
                        return true;
                }
                break;
            case 1190:
                switch (current) {
                    case 'k' :
                        state = 1191;
                        return true;
                }
                break;
            case 1191:
                switch (current) {
                    case 'e' :
                        state = 1192;
                        return true;
                }
                break;
            case 1192:
                switch (current) {
                    case 't' :
                        state = 1193;
                        return true;
                }
                break;
            case 1193:
                switch (current) {
                    case ';' : // LeftAngleBracket;
                        match = "\u27E8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1194:
                switch (current) {
                    case 'r' :
                        state = 1195;
                        return true;
                }
                break;
            case 1195:
                switch (current) {
                    case 'o' :
                        state = 1196;
                        return true;
                }
                break;
            case 1196:
                switch (current) {
                    case 'w' :
                        state = 1197;
                        return true;
                }
                break;
            case 1197:
                switch (current) {
                    case ';' : // LeftArrow;
                        match = "\u2190";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1198;
                        return true;
                    case 'R' :
                        state = 1201;
                        return true;
                }
                break;
            case 1198:
                switch (current) {
                    case 'a' :
                        state = 1199;
                        return true;
                }
                break;
            case 1199:
                switch (current) {
                    case 'r' :
                        state = 1200;
                        return true;
                }
                break;
            case 1200:
                switch (current) {
                    case ';' : // LeftArrowBar;
                        match = "\u21E4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1201:
                switch (current) {
                    case 'i' :
                        state = 1202;
                        return true;
                }
                break;
            case 1202:
                switch (current) {
                    case 'g' :
                        state = 1203;
                        return true;
                }
                break;
            case 1203:
                switch (current) {
                    case 'h' :
                        state = 1204;
                        return true;
                }
                break;
            case 1204:
                switch (current) {
                    case 't' :
                        state = 1205;
                        return true;
                }
                break;
            case 1205:
                switch (current) {
                    case 'A' :
                        state = 1206;
                        return true;
                }
                break;
            case 1206:
                switch (current) {
                    case 'r' :
                        state = 1207;
                        return true;
                }
                break;
            case 1207:
                switch (current) {
                    case 'r' :
                        state = 1208;
                        return true;
                }
                break;
            case 1208:
                switch (current) {
                    case 'o' :
                        state = 1209;
                        return true;
                }
                break;
            case 1209:
                switch (current) {
                    case 'w' :
                        state = 1210;
                        return true;
                }
                break;
            case 1210:
                switch (current) {
                    case ';' : // LeftArrowRightArrow;
                        match = "\u21C6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1211:
                switch (current) {
                    case 'e' :
                        state = 1212;
                        return true;
                }
                break;
            case 1212:
                switch (current) {
                    case 'i' :
                        state = 1213;
                        return true;
                }
                break;
            case 1213:
                switch (current) {
                    case 'l' :
                        state = 1214;
                        return true;
                }
                break;
            case 1214:
                switch (current) {
                    case 'i' :
                        state = 1215;
                        return true;
                }
                break;
            case 1215:
                switch (current) {
                    case 'n' :
                        state = 1216;
                        return true;
                }
                break;
            case 1216:
                switch (current) {
                    case 'g' :
                        state = 1217;
                        return true;
                }
                break;
            case 1217:
                switch (current) {
                    case ';' : // LeftCeiling;
                        match = "\u2308";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1218:
                switch (current) {
                    case 'o' :
                        state = 1219;
                        return true;
                }
                break;
            case 1219:
                switch (current) {
                    case 'u' :
                        state = 1220;
                        return true;
                    case 'w' :
                        state = 1231;
                        return true;
                }
                break;
            case 1220:
                switch (current) {
                    case 'b' :
                        state = 1221;
                        return true;
                }
                break;
            case 1221:
                switch (current) {
                    case 'l' :
                        state = 1222;
                        return true;
                }
                break;
            case 1222:
                switch (current) {
                    case 'e' :
                        state = 1223;
                        return true;
                }
                break;
            case 1223:
                switch (current) {
                    case 'B' :
                        state = 1224;
                        return true;
                }
                break;
            case 1224:
                switch (current) {
                    case 'r' :
                        state = 1225;
                        return true;
                }
                break;
            case 1225:
                switch (current) {
                    case 'a' :
                        state = 1226;
                        return true;
                }
                break;
            case 1226:
                switch (current) {
                    case 'c' :
                        state = 1227;
                        return true;
                }
                break;
            case 1227:
                switch (current) {
                    case 'k' :
                        state = 1228;
                        return true;
                }
                break;
            case 1228:
                switch (current) {
                    case 'e' :
                        state = 1229;
                        return true;
                }
                break;
            case 1229:
                switch (current) {
                    case 't' :
                        state = 1230;
                        return true;
                }
                break;
            case 1230:
                switch (current) {
                    case ';' : // LeftDoubleBracket;
                        match = "\u27E6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1231:
                switch (current) {
                    case 'n' :
                        state = 1232;
                        return true;
                }
                break;
            case 1232:
                switch (current) {
                    case 'T' :
                        state = 1233;
                        return true;
                    case 'V' :
                        state = 1242;
                        return true;
                }
                break;
            case 1233:
                switch (current) {
                    case 'e' :
                        state = 1234;
                        return true;
                }
                break;
            case 1234:
                switch (current) {
                    case 'e' :
                        state = 1235;
                        return true;
                }
                break;
            case 1235:
                switch (current) {
                    case 'V' :
                        state = 1236;
                        return true;
                }
                break;
            case 1236:
                switch (current) {
                    case 'e' :
                        state = 1237;
                        return true;
                }
                break;
            case 1237:
                switch (current) {
                    case 'c' :
                        state = 1238;
                        return true;
                }
                break;
            case 1238:
                switch (current) {
                    case 't' :
                        state = 1239;
                        return true;
                }
                break;
            case 1239:
                switch (current) {
                    case 'o' :
                        state = 1240;
                        return true;
                }
                break;
            case 1240:
                switch (current) {
                    case 'r' :
                        state = 1241;
                        return true;
                }
                break;
            case 1241:
                switch (current) {
                    case ';' : // LeftDownTeeVector;
                        match = "\u2961";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1242:
                switch (current) {
                    case 'e' :
                        state = 1243;
                        return true;
                }
                break;
            case 1243:
                switch (current) {
                    case 'c' :
                        state = 1244;
                        return true;
                }
                break;
            case 1244:
                switch (current) {
                    case 't' :
                        state = 1245;
                        return true;
                }
                break;
            case 1245:
                switch (current) {
                    case 'o' :
                        state = 1246;
                        return true;
                }
                break;
            case 1246:
                switch (current) {
                    case 'r' :
                        state = 1247;
                        return true;
                }
                break;
            case 1247:
                switch (current) {
                    case ';' : // LeftDownVector;
                        match = "\u21C3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1248;
                        return true;
                }
                break;
            case 1248:
                switch (current) {
                    case 'a' :
                        state = 1249;
                        return true;
                }
                break;
            case 1249:
                switch (current) {
                    case 'r' :
                        state = 1250;
                        return true;
                }
                break;
            case 1250:
                switch (current) {
                    case ';' : // LeftDownVectorBar;
                        match = "\u2959";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1251:
                switch (current) {
                    case 'l' :
                        state = 1252;
                        return true;
                }
                break;
            case 1252:
                switch (current) {
                    case 'o' :
                        state = 1253;
                        return true;
                }
                break;
            case 1253:
                switch (current) {
                    case 'o' :
                        state = 1254;
                        return true;
                }
                break;
            case 1254:
                switch (current) {
                    case 'r' :
                        state = 1255;
                        return true;
                }
                break;
            case 1255:
                switch (current) {
                    case ';' : // LeftFloor;
                        match = "\u230A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1256:
                switch (current) {
                    case 'i' :
                        state = 1257;
                        return true;
                }
                break;
            case 1257:
                switch (current) {
                    case 'g' :
                        state = 1258;
                        return true;
                }
                break;
            case 1258:
                switch (current) {
                    case 'h' :
                        state = 1259;
                        return true;
                }
                break;
            case 1259:
                switch (current) {
                    case 't' :
                        state = 1260;
                        return true;
                }
                break;
            case 1260:
                switch (current) {
                    case 'A' :
                        state = 1261;
                        return true;
                    case 'V' :
                        state = 1266;
                        return true;
                }
                break;
            case 1261:
                switch (current) {
                    case 'r' :
                        state = 1262;
                        return true;
                }
                break;
            case 1262:
                switch (current) {
                    case 'r' :
                        state = 1263;
                        return true;
                }
                break;
            case 1263:
                switch (current) {
                    case 'o' :
                        state = 1264;
                        return true;
                }
                break;
            case 1264:
                switch (current) {
                    case 'w' :
                        state = 1265;
                        return true;
                }
                break;
            case 1265:
                switch (current) {
                    case ';' : // LeftRightArrow;
                        match = "\u2194";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1266:
                switch (current) {
                    case 'e' :
                        state = 1267;
                        return true;
                }
                break;
            case 1267:
                switch (current) {
                    case 'c' :
                        state = 1268;
                        return true;
                }
                break;
            case 1268:
                switch (current) {
                    case 't' :
                        state = 1269;
                        return true;
                }
                break;
            case 1269:
                switch (current) {
                    case 'o' :
                        state = 1270;
                        return true;
                }
                break;
            case 1270:
                switch (current) {
                    case 'r' :
                        state = 1271;
                        return true;
                }
                break;
            case 1271:
                switch (current) {
                    case ';' : // LeftRightVector;
                        match = "\u294E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1272:
                switch (current) {
                    case 'e' :
                        state = 1273;
                        return true;
                    case 'r' :
                        state = 1286;
                        return true;
                }
                break;
            case 1273:
                switch (current) {
                    case 'e' :
                        state = 1274;
                        return true;
                }
                break;
            case 1274:
                switch (current) {
                    case ';' : // LeftTee;
                        match = "\u22A3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'A' :
                        state = 1275;
                        return true;
                    case 'V' :
                        state = 1280;
                        return true;
                }
                break;
            case 1275:
                switch (current) {
                    case 'r' :
                        state = 1276;
                        return true;
                }
                break;
            case 1276:
                switch (current) {
                    case 'r' :
                        state = 1277;
                        return true;
                }
                break;
            case 1277:
                switch (current) {
                    case 'o' :
                        state = 1278;
                        return true;
                }
                break;
            case 1278:
                switch (current) {
                    case 'w' :
                        state = 1279;
                        return true;
                }
                break;
            case 1279:
                switch (current) {
                    case ';' : // LeftTeeArrow;
                        match = "\u21A4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1280:
                switch (current) {
                    case 'e' :
                        state = 1281;
                        return true;
                }
                break;
            case 1281:
                switch (current) {
                    case 'c' :
                        state = 1282;
                        return true;
                }
                break;
            case 1282:
                switch (current) {
                    case 't' :
                        state = 1283;
                        return true;
                }
                break;
            case 1283:
                switch (current) {
                    case 'o' :
                        state = 1284;
                        return true;
                }
                break;
            case 1284:
                switch (current) {
                    case 'r' :
                        state = 1285;
                        return true;
                }
                break;
            case 1285:
                switch (current) {
                    case ';' : // LeftTeeVector;
                        match = "\u295A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1286:
                switch (current) {
                    case 'i' :
                        state = 1287;
                        return true;
                }
                break;
            case 1287:
                switch (current) {
                    case 'a' :
                        state = 1288;
                        return true;
                }
                break;
            case 1288:
                switch (current) {
                    case 'n' :
                        state = 1289;
                        return true;
                }
                break;
            case 1289:
                switch (current) {
                    case 'g' :
                        state = 1290;
                        return true;
                }
                break;
            case 1290:
                switch (current) {
                    case 'l' :
                        state = 1291;
                        return true;
                }
                break;
            case 1291:
                switch (current) {
                    case 'e' :
                        state = 1292;
                        return true;
                }
                break;
            case 1292:
                switch (current) {
                    case ';' : // LeftTriangle;
                        match = "\u22B2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1293;
                        return true;
                    case 'E' :
                        state = 1296;
                        return true;
                }
                break;
            case 1293:
                switch (current) {
                    case 'a' :
                        state = 1294;
                        return true;
                }
                break;
            case 1294:
                switch (current) {
                    case 'r' :
                        state = 1295;
                        return true;
                }
                break;
            case 1295:
                switch (current) {
                    case ';' : // LeftTriangleBar;
                        match = "\u29CF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1296:
                switch (current) {
                    case 'q' :
                        state = 1297;
                        return true;
                }
                break;
            case 1297:
                switch (current) {
                    case 'u' :
                        state = 1298;
                        return true;
                }
                break;
            case 1298:
                switch (current) {
                    case 'a' :
                        state = 1299;
                        return true;
                }
                break;
            case 1299:
                switch (current) {
                    case 'l' :
                        state = 1300;
                        return true;
                }
                break;
            case 1300:
                switch (current) {
                    case ';' : // LeftTriangleEqual;
                        match = "\u22B4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1301:
                switch (current) {
                    case 'p' :
                        state = 1302;
                        return true;
                }
                break;
            case 1302:
                switch (current) {
                    case 'D' :
                        state = 1303;
                        return true;
                    case 'T' :
                        state = 1313;
                        return true;
                    case 'V' :
                        state = 1322;
                        return true;
                }
                break;
            case 1303:
                switch (current) {
                    case 'o' :
                        state = 1304;
                        return true;
                }
                break;
            case 1304:
                switch (current) {
                    case 'w' :
                        state = 1305;
                        return true;
                }
                break;
            case 1305:
                switch (current) {
                    case 'n' :
                        state = 1306;
                        return true;
                }
                break;
            case 1306:
                switch (current) {
                    case 'V' :
                        state = 1307;
                        return true;
                }
                break;
            case 1307:
                switch (current) {
                    case 'e' :
                        state = 1308;
                        return true;
                }
                break;
            case 1308:
                switch (current) {
                    case 'c' :
                        state = 1309;
                        return true;
                }
                break;
            case 1309:
                switch (current) {
                    case 't' :
                        state = 1310;
                        return true;
                }
                break;
            case 1310:
                switch (current) {
                    case 'o' :
                        state = 1311;
                        return true;
                }
                break;
            case 1311:
                switch (current) {
                    case 'r' :
                        state = 1312;
                        return true;
                }
                break;
            case 1312:
                switch (current) {
                    case ';' : // LeftUpDownVector;
                        match = "\u2951";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1313:
                switch (current) {
                    case 'e' :
                        state = 1314;
                        return true;
                }
                break;
            case 1314:
                switch (current) {
                    case 'e' :
                        state = 1315;
                        return true;
                }
                break;
            case 1315:
                switch (current) {
                    case 'V' :
                        state = 1316;
                        return true;
                }
                break;
            case 1316:
                switch (current) {
                    case 'e' :
                        state = 1317;
                        return true;
                }
                break;
            case 1317:
                switch (current) {
                    case 'c' :
                        state = 1318;
                        return true;
                }
                break;
            case 1318:
                switch (current) {
                    case 't' :
                        state = 1319;
                        return true;
                }
                break;
            case 1319:
                switch (current) {
                    case 'o' :
                        state = 1320;
                        return true;
                }
                break;
            case 1320:
                switch (current) {
                    case 'r' :
                        state = 1321;
                        return true;
                }
                break;
            case 1321:
                switch (current) {
                    case ';' : // LeftUpTeeVector;
                        match = "\u2960";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1322:
                switch (current) {
                    case 'e' :
                        state = 1323;
                        return true;
                }
                break;
            case 1323:
                switch (current) {
                    case 'c' :
                        state = 1324;
                        return true;
                }
                break;
            case 1324:
                switch (current) {
                    case 't' :
                        state = 1325;
                        return true;
                }
                break;
            case 1325:
                switch (current) {
                    case 'o' :
                        state = 1326;
                        return true;
                }
                break;
            case 1326:
                switch (current) {
                    case 'r' :
                        state = 1327;
                        return true;
                }
                break;
            case 1327:
                switch (current) {
                    case ';' : // LeftUpVector;
                        match = "\u21BF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1328;
                        return true;
                }
                break;
            case 1328:
                switch (current) {
                    case 'a' :
                        state = 1329;
                        return true;
                }
                break;
            case 1329:
                switch (current) {
                    case 'r' :
                        state = 1330;
                        return true;
                }
                break;
            case 1330:
                switch (current) {
                    case ';' : // LeftUpVectorBar;
                        match = "\u2958";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1331:
                switch (current) {
                    case 'e' :
                        state = 1332;
                        return true;
                }
                break;
            case 1332:
                switch (current) {
                    case 'c' :
                        state = 1333;
                        return true;
                }
                break;
            case 1333:
                switch (current) {
                    case 't' :
                        state = 1334;
                        return true;
                }
                break;
            case 1334:
                switch (current) {
                    case 'o' :
                        state = 1335;
                        return true;
                }
                break;
            case 1335:
                switch (current) {
                    case 'r' :
                        state = 1336;
                        return true;
                }
                break;
            case 1336:
                switch (current) {
                    case ';' : // LeftVector;
                        match = "\u21BC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1337;
                        return true;
                }
                break;
            case 1337:
                switch (current) {
                    case 'a' :
                        state = 1338;
                        return true;
                }
                break;
            case 1338:
                switch (current) {
                    case 'r' :
                        state = 1339;
                        return true;
                }
                break;
            case 1339:
                switch (current) {
                    case ';' : // LeftVectorBar;
                        match = "\u2952";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1340:
                switch (current) {
                    case 'r' :
                        state = 1341;
                        return true;
                }
                break;
            case 1341:
                switch (current) {
                    case 'r' :
                        state = 1342;
                        return true;
                }
                break;
            case 1342:
                switch (current) {
                    case 'o' :
                        state = 1343;
                        return true;
                }
                break;
            case 1343:
                switch (current) {
                    case 'w' :
                        state = 1344;
                        return true;
                }
                break;
            case 1344:
                switch (current) {
                    case ';' : // Leftarrow;
                        match = "\u21D0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1345:
                switch (current) {
                    case 'i' :
                        state = 1346;
                        return true;
                }
                break;
            case 1346:
                switch (current) {
                    case 'g' :
                        state = 1347;
                        return true;
                }
                break;
            case 1347:
                switch (current) {
                    case 'h' :
                        state = 1348;
                        return true;
                }
                break;
            case 1348:
                switch (current) {
                    case 't' :
                        state = 1349;
                        return true;
                }
                break;
            case 1349:
                switch (current) {
                    case 'a' :
                        state = 1350;
                        return true;
                }
                break;
            case 1350:
                switch (current) {
                    case 'r' :
                        state = 1351;
                        return true;
                }
                break;
            case 1351:
                switch (current) {
                    case 'r' :
                        state = 1352;
                        return true;
                }
                break;
            case 1352:
                switch (current) {
                    case 'o' :
                        state = 1353;
                        return true;
                }
                break;
            case 1353:
                switch (current) {
                    case 'w' :
                        state = 1354;
                        return true;
                }
                break;
            case 1354:
                switch (current) {
                    case ';' : // Leftrightarrow;
                        match = "\u21D4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1355:
                switch (current) {
                    case 's' :
                        state = 1356;
                        return true;
                }
                break;
            case 1356:
                switch (current) {
                    case 'E' :
                        state = 1357;
                        return true;
                    case 'F' :
                        state = 1369;
                        return true;
                    case 'G' :
                        state = 1378;
                        return true;
                    case 'L' :
                        state = 1385;
                        return true;
                    case 'S' :
                        state = 1389;
                        return true;
                    case 'T' :
                        state = 1399;
                        return true;
                }
                break;
            case 1357:
                switch (current) {
                    case 'q' :
                        state = 1358;
                        return true;
                }
                break;
            case 1358:
                switch (current) {
                    case 'u' :
                        state = 1359;
                        return true;
                }
                break;
            case 1359:
                switch (current) {
                    case 'a' :
                        state = 1360;
                        return true;
                }
                break;
            case 1360:
                switch (current) {
                    case 'l' :
                        state = 1361;
                        return true;
                }
                break;
            case 1361:
                switch (current) {
                    case 'G' :
                        state = 1362;
                        return true;
                }
                break;
            case 1362:
                switch (current) {
                    case 'r' :
                        state = 1363;
                        return true;
                }
                break;
            case 1363:
                switch (current) {
                    case 'e' :
                        state = 1364;
                        return true;
                }
                break;
            case 1364:
                switch (current) {
                    case 'a' :
                        state = 1365;
                        return true;
                }
                break;
            case 1365:
                switch (current) {
                    case 't' :
                        state = 1366;
                        return true;
                }
                break;
            case 1366:
                switch (current) {
                    case 'e' :
                        state = 1367;
                        return true;
                }
                break;
            case 1367:
                switch (current) {
                    case 'r' :
                        state = 1368;
                        return true;
                }
                break;
            case 1368:
                switch (current) {
                    case ';' : // LessEqualGreater;
                        match = "\u22DA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1369:
                switch (current) {
                    case 'u' :
                        state = 1370;
                        return true;
                }
                break;
            case 1370:
                switch (current) {
                    case 'l' :
                        state = 1371;
                        return true;
                }
                break;
            case 1371:
                switch (current) {
                    case 'l' :
                        state = 1372;
                        return true;
                }
                break;
            case 1372:
                switch (current) {
                    case 'E' :
                        state = 1373;
                        return true;
                }
                break;
            case 1373:
                switch (current) {
                    case 'q' :
                        state = 1374;
                        return true;
                }
                break;
            case 1374:
                switch (current) {
                    case 'u' :
                        state = 1375;
                        return true;
                }
                break;
            case 1375:
                switch (current) {
                    case 'a' :
                        state = 1376;
                        return true;
                }
                break;
            case 1376:
                switch (current) {
                    case 'l' :
                        state = 1377;
                        return true;
                }
                break;
            case 1377:
                switch (current) {
                    case ';' : // LessFullEqual;
                        match = "\u2266";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1378:
                switch (current) {
                    case 'r' :
                        state = 1379;
                        return true;
                }
                break;
            case 1379:
                switch (current) {
                    case 'e' :
                        state = 1380;
                        return true;
                }
                break;
            case 1380:
                switch (current) {
                    case 'a' :
                        state = 1381;
                        return true;
                }
                break;
            case 1381:
                switch (current) {
                    case 't' :
                        state = 1382;
                        return true;
                }
                break;
            case 1382:
                switch (current) {
                    case 'e' :
                        state = 1383;
                        return true;
                }
                break;
            case 1383:
                switch (current) {
                    case 'r' :
                        state = 1384;
                        return true;
                }
                break;
            case 1384:
                switch (current) {
                    case ';' : // LessGreater;
                        match = "\u2276";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1385:
                switch (current) {
                    case 'e' :
                        state = 1386;
                        return true;
                }
                break;
            case 1386:
                switch (current) {
                    case 's' :
                        state = 1387;
                        return true;
                }
                break;
            case 1387:
                switch (current) {
                    case 's' :
                        state = 1388;
                        return true;
                }
                break;
            case 1388:
                switch (current) {
                    case ';' : // LessLess;
                        match = "\u2AA1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1389:
                switch (current) {
                    case 'l' :
                        state = 1390;
                        return true;
                }
                break;
            case 1390:
                switch (current) {
                    case 'a' :
                        state = 1391;
                        return true;
                }
                break;
            case 1391:
                switch (current) {
                    case 'n' :
                        state = 1392;
                        return true;
                }
                break;
            case 1392:
                switch (current) {
                    case 't' :
                        state = 1393;
                        return true;
                }
                break;
            case 1393:
                switch (current) {
                    case 'E' :
                        state = 1394;
                        return true;
                }
                break;
            case 1394:
                switch (current) {
                    case 'q' :
                        state = 1395;
                        return true;
                }
                break;
            case 1395:
                switch (current) {
                    case 'u' :
                        state = 1396;
                        return true;
                }
                break;
            case 1396:
                switch (current) {
                    case 'a' :
                        state = 1397;
                        return true;
                }
                break;
            case 1397:
                switch (current) {
                    case 'l' :
                        state = 1398;
                        return true;
                }
                break;
            case 1398:
                switch (current) {
                    case ';' : // LessSlantEqual;
                        match = "\u2A7D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1399:
                switch (current) {
                    case 'i' :
                        state = 1400;
                        return true;
                }
                break;
            case 1400:
                switch (current) {
                    case 'l' :
                        state = 1401;
                        return true;
                }
                break;
            case 1401:
                switch (current) {
                    case 'd' :
                        state = 1402;
                        return true;
                }
                break;
            case 1402:
                switch (current) {
                    case 'e' :
                        state = 1403;
                        return true;
                }
                break;
            case 1403:
                switch (current) {
                    case ';' : // LessTilde;
                        match = "\u2272";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1404:
                switch (current) {
                    case 'r' :
                        state = 1405;
                        return true;
                }
                break;
            case 1405:
                switch (current) {
                    case ';' : // Lfr;
                        match = "\uD835\uDD0F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1406:
                switch (current) {
                    case ';' : // Ll;
                        match = "\u22D8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 1407;
                        return true;
                }
                break;
            case 1407:
                switch (current) {
                    case 'f' :
                        state = 1408;
                        return true;
                }
                break;
            case 1408:
                switch (current) {
                    case 't' :
                        state = 1409;
                        return true;
                }
                break;
            case 1409:
                switch (current) {
                    case 'a' :
                        state = 1410;
                        return true;
                }
                break;
            case 1410:
                switch (current) {
                    case 'r' :
                        state = 1411;
                        return true;
                }
                break;
            case 1411:
                switch (current) {
                    case 'r' :
                        state = 1412;
                        return true;
                }
                break;
            case 1412:
                switch (current) {
                    case 'o' :
                        state = 1413;
                        return true;
                }
                break;
            case 1413:
                switch (current) {
                    case 'w' :
                        state = 1414;
                        return true;
                }
                break;
            case 1414:
                switch (current) {
                    case ';' : // Lleftarrow;
                        match = "\u21DA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1415:
                switch (current) {
                    case 'i' :
                        state = 1416;
                        return true;
                }
                break;
            case 1416:
                switch (current) {
                    case 'd' :
                        state = 1417;
                        return true;
                }
                break;
            case 1417:
                switch (current) {
                    case 'o' :
                        state = 1418;
                        return true;
                }
                break;
            case 1418:
                switch (current) {
                    case 't' :
                        state = 1419;
                        return true;
                }
                break;
            case 1419:
                switch (current) {
                    case ';' : // Lmidot;
                        match = "\u013F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1420:
                switch (current) {
                    case 'n' :
                        state = 1421;
                        return true;
                    case 'p' :
                        state = 1481;
                        return true;
                    case 'w' :
                        state = 1483;
                        return true;
                }
                break;
            case 1421:
                switch (current) {
                    case 'g' :
                        state = 1422;
                        return true;
                }
                break;
            case 1422:
                switch (current) {
                    case 'L' :
                        state = 1423;
                        return true;
                    case 'R' :
                        state = 1442;
                        return true;
                    case 'l' :
                        state = 1452;
                        return true;
                    case 'r' :
                        state = 1471;
                        return true;
                }
                break;
            case 1423:
                switch (current) {
                    case 'e' :
                        state = 1424;
                        return true;
                }
                break;
            case 1424:
                switch (current) {
                    case 'f' :
                        state = 1425;
                        return true;
                }
                break;
            case 1425:
                switch (current) {
                    case 't' :
                        state = 1426;
                        return true;
                }
                break;
            case 1426:
                switch (current) {
                    case 'A' :
                        state = 1427;
                        return true;
                    case 'R' :
                        state = 1432;
                        return true;
                }
                break;
            case 1427:
                switch (current) {
                    case 'r' :
                        state = 1428;
                        return true;
                }
                break;
            case 1428:
                switch (current) {
                    case 'r' :
                        state = 1429;
                        return true;
                }
                break;
            case 1429:
                switch (current) {
                    case 'o' :
                        state = 1430;
                        return true;
                }
                break;
            case 1430:
                switch (current) {
                    case 'w' :
                        state = 1431;
                        return true;
                }
                break;
            case 1431:
                switch (current) {
                    case ';' : // LongLeftArrow;
                        match = "\u27F5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1432:
                switch (current) {
                    case 'i' :
                        state = 1433;
                        return true;
                }
                break;
            case 1433:
                switch (current) {
                    case 'g' :
                        state = 1434;
                        return true;
                }
                break;
            case 1434:
                switch (current) {
                    case 'h' :
                        state = 1435;
                        return true;
                }
                break;
            case 1435:
                switch (current) {
                    case 't' :
                        state = 1436;
                        return true;
                }
                break;
            case 1436:
                switch (current) {
                    case 'A' :
                        state = 1437;
                        return true;
                }
                break;
            case 1437:
                switch (current) {
                    case 'r' :
                        state = 1438;
                        return true;
                }
                break;
            case 1438:
                switch (current) {
                    case 'r' :
                        state = 1439;
                        return true;
                }
                break;
            case 1439:
                switch (current) {
                    case 'o' :
                        state = 1440;
                        return true;
                }
                break;
            case 1440:
                switch (current) {
                    case 'w' :
                        state = 1441;
                        return true;
                }
                break;
            case 1441:
                switch (current) {
                    case ';' : // LongLeftRightArrow;
                        match = "\u27F7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1442:
                switch (current) {
                    case 'i' :
                        state = 1443;
                        return true;
                }
                break;
            case 1443:
                switch (current) {
                    case 'g' :
                        state = 1444;
                        return true;
                }
                break;
            case 1444:
                switch (current) {
                    case 'h' :
                        state = 1445;
                        return true;
                }
                break;
            case 1445:
                switch (current) {
                    case 't' :
                        state = 1446;
                        return true;
                }
                break;
            case 1446:
                switch (current) {
                    case 'A' :
                        state = 1447;
                        return true;
                }
                break;
            case 1447:
                switch (current) {
                    case 'r' :
                        state = 1448;
                        return true;
                }
                break;
            case 1448:
                switch (current) {
                    case 'r' :
                        state = 1449;
                        return true;
                }
                break;
            case 1449:
                switch (current) {
                    case 'o' :
                        state = 1450;
                        return true;
                }
                break;
            case 1450:
                switch (current) {
                    case 'w' :
                        state = 1451;
                        return true;
                }
                break;
            case 1451:
                switch (current) {
                    case ';' : // LongRightArrow;
                        match = "\u27F6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1452:
                switch (current) {
                    case 'e' :
                        state = 1453;
                        return true;
                }
                break;
            case 1453:
                switch (current) {
                    case 'f' :
                        state = 1454;
                        return true;
                }
                break;
            case 1454:
                switch (current) {
                    case 't' :
                        state = 1455;
                        return true;
                }
                break;
            case 1455:
                switch (current) {
                    case 'a' :
                        state = 1456;
                        return true;
                    case 'r' :
                        state = 1461;
                        return true;
                }
                break;
            case 1456:
                switch (current) {
                    case 'r' :
                        state = 1457;
                        return true;
                }
                break;
            case 1457:
                switch (current) {
                    case 'r' :
                        state = 1458;
                        return true;
                }
                break;
            case 1458:
                switch (current) {
                    case 'o' :
                        state = 1459;
                        return true;
                }
                break;
            case 1459:
                switch (current) {
                    case 'w' :
                        state = 1460;
                        return true;
                }
                break;
            case 1460:
                switch (current) {
                    case ';' : // Longleftarrow;
                        match = "\u27F8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1461:
                switch (current) {
                    case 'i' :
                        state = 1462;
                        return true;
                }
                break;
            case 1462:
                switch (current) {
                    case 'g' :
                        state = 1463;
                        return true;
                }
                break;
            case 1463:
                switch (current) {
                    case 'h' :
                        state = 1464;
                        return true;
                }
                break;
            case 1464:
                switch (current) {
                    case 't' :
                        state = 1465;
                        return true;
                }
                break;
            case 1465:
                switch (current) {
                    case 'a' :
                        state = 1466;
                        return true;
                }
                break;
            case 1466:
                switch (current) {
                    case 'r' :
                        state = 1467;
                        return true;
                }
                break;
            case 1467:
                switch (current) {
                    case 'r' :
                        state = 1468;
                        return true;
                }
                break;
            case 1468:
                switch (current) {
                    case 'o' :
                        state = 1469;
                        return true;
                }
                break;
            case 1469:
                switch (current) {
                    case 'w' :
                        state = 1470;
                        return true;
                }
                break;
            case 1470:
                switch (current) {
                    case ';' : // Longleftrightarrow;
                        match = "\u27FA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1471:
                switch (current) {
                    case 'i' :
                        state = 1472;
                        return true;
                }
                break;
            case 1472:
                switch (current) {
                    case 'g' :
                        state = 1473;
                        return true;
                }
                break;
            case 1473:
                switch (current) {
                    case 'h' :
                        state = 1474;
                        return true;
                }
                break;
            case 1474:
                switch (current) {
                    case 't' :
                        state = 1475;
                        return true;
                }
                break;
            case 1475:
                switch (current) {
                    case 'a' :
                        state = 1476;
                        return true;
                }
                break;
            case 1476:
                switch (current) {
                    case 'r' :
                        state = 1477;
                        return true;
                }
                break;
            case 1477:
                switch (current) {
                    case 'r' :
                        state = 1478;
                        return true;
                }
                break;
            case 1478:
                switch (current) {
                    case 'o' :
                        state = 1479;
                        return true;
                }
                break;
            case 1479:
                switch (current) {
                    case 'w' :
                        state = 1480;
                        return true;
                }
                break;
            case 1480:
                switch (current) {
                    case ';' : // Longrightarrow;
                        match = "\u27F9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1481:
                switch (current) {
                    case 'f' :
                        state = 1482;
                        return true;
                }
                break;
            case 1482:
                switch (current) {
                    case ';' : // Lopf;
                        match = "\uD835\uDD43";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1483:
                switch (current) {
                    case 'e' :
                        state = 1484;
                        return true;
                }
                break;
            case 1484:
                switch (current) {
                    case 'r' :
                        state = 1485;
                        return true;
                }
                break;
            case 1485:
                switch (current) {
                    case 'L' :
                        state = 1486;
                        return true;
                    case 'R' :
                        state = 1495;
                        return true;
                }
                break;
            case 1486:
                switch (current) {
                    case 'e' :
                        state = 1487;
                        return true;
                }
                break;
            case 1487:
                switch (current) {
                    case 'f' :
                        state = 1488;
                        return true;
                }
                break;
            case 1488:
                switch (current) {
                    case 't' :
                        state = 1489;
                        return true;
                }
                break;
            case 1489:
                switch (current) {
                    case 'A' :
                        state = 1490;
                        return true;
                }
                break;
            case 1490:
                switch (current) {
                    case 'r' :
                        state = 1491;
                        return true;
                }
                break;
            case 1491:
                switch (current) {
                    case 'r' :
                        state = 1492;
                        return true;
                }
                break;
            case 1492:
                switch (current) {
                    case 'o' :
                        state = 1493;
                        return true;
                }
                break;
            case 1493:
                switch (current) {
                    case 'w' :
                        state = 1494;
                        return true;
                }
                break;
            case 1494:
                switch (current) {
                    case ';' : // LowerLeftArrow;
                        match = "\u2199";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1495:
                switch (current) {
                    case 'i' :
                        state = 1496;
                        return true;
                }
                break;
            case 1496:
                switch (current) {
                    case 'g' :
                        state = 1497;
                        return true;
                }
                break;
            case 1497:
                switch (current) {
                    case 'h' :
                        state = 1498;
                        return true;
                }
                break;
            case 1498:
                switch (current) {
                    case 't' :
                        state = 1499;
                        return true;
                }
                break;
            case 1499:
                switch (current) {
                    case 'A' :
                        state = 1500;
                        return true;
                }
                break;
            case 1500:
                switch (current) {
                    case 'r' :
                        state = 1501;
                        return true;
                }
                break;
            case 1501:
                switch (current) {
                    case 'r' :
                        state = 1502;
                        return true;
                }
                break;
            case 1502:
                switch (current) {
                    case 'o' :
                        state = 1503;
                        return true;
                }
                break;
            case 1503:
                switch (current) {
                    case 'w' :
                        state = 1504;
                        return true;
                }
                break;
            case 1504:
                switch (current) {
                    case ';' : // LowerRightArrow;
                        match = "\u2198";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1505:
                switch (current) {
                    case 'c' :
                        state = 1506;
                        return true;
                    case 'h' :
                        state = 1508;
                        return true;
                    case 't' :
                        state = 1509;
                        return true;
                }
                break;
            case 1506:
                switch (current) {
                    case 'r' :
                        state = 1507;
                        return true;
                }
                break;
            case 1507:
                switch (current) {
                    case ';' : // Lscr;
                        match = "\u2112";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1508:
                switch (current) {
                    case ';' : // Lsh;
                        match = "\u21B0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1509:
                switch (current) {
                    case 'r' :
                        state = 1510;
                        return true;
                }
                break;
            case 1510:
                switch (current) {
                    case 'o' :
                        state = 1511;
                        return true;
                }
                break;
            case 1511:
                switch (current) {
                    case 'k' :
                        state = 1512;
                        return true;
                }
                break;
            case 1512:
                switch (current) {
                    case ';' : // Lstrok;
                        match = "\u0141";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1513:
                switch (current) {
                    case ';' : // Lt;
                        match = "\u226A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1514:
                switch (current) {
                    case 'a' :
                        state = 1515;
                        return true;
                    case 'c' :
                        state = 1517;
                        return true;
                    case 'e' :
                        state = 1519;
                        return true;
                    case 'f' :
                        state = 1536;
                        return true;
                    case 'i' :
                        state = 1538;
                        return true;
                    case 'o' :
                        state = 1546;
                        return true;
                    case 's' :
                        state = 1549;
                        return true;
                    case 'u' :
                        state = 1552;
                        return true;
                }
                break;
            case 1515:
                switch (current) {
                    case 'p' :
                        state = 1516;
                        return true;
                }
                break;
            case 1516:
                switch (current) {
                    case ';' : // Map;
                        match = "\u2905";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1517:
                switch (current) {
                    case 'y' :
                        state = 1518;
                        return true;
                }
                break;
            case 1518:
                switch (current) {
                    case ';' : // Mcy;
                        match = "\u041C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1519:
                switch (current) {
                    case 'd' :
                        state = 1520;
                        return true;
                    case 'l' :
                        state = 1529;
                        return true;
                }
                break;
            case 1520:
                switch (current) {
                    case 'i' :
                        state = 1521;
                        return true;
                }
                break;
            case 1521:
                switch (current) {
                    case 'u' :
                        state = 1522;
                        return true;
                }
                break;
            case 1522:
                switch (current) {
                    case 'm' :
                        state = 1523;
                        return true;
                }
                break;
            case 1523:
                switch (current) {
                    case 'S' :
                        state = 1524;
                        return true;
                }
                break;
            case 1524:
                switch (current) {
                    case 'p' :
                        state = 1525;
                        return true;
                }
                break;
            case 1525:
                switch (current) {
                    case 'a' :
                        state = 1526;
                        return true;
                }
                break;
            case 1526:
                switch (current) {
                    case 'c' :
                        state = 1527;
                        return true;
                }
                break;
            case 1527:
                switch (current) {
                    case 'e' :
                        state = 1528;
                        return true;
                }
                break;
            case 1528:
                switch (current) {
                    case ';' : // MediumSpace;
                        match = "\u205F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1529:
                switch (current) {
                    case 'l' :
                        state = 1530;
                        return true;
                }
                break;
            case 1530:
                switch (current) {
                    case 'i' :
                        state = 1531;
                        return true;
                }
                break;
            case 1531:
                switch (current) {
                    case 'n' :
                        state = 1532;
                        return true;
                }
                break;
            case 1532:
                switch (current) {
                    case 't' :
                        state = 1533;
                        return true;
                }
                break;
            case 1533:
                switch (current) {
                    case 'r' :
                        state = 1534;
                        return true;
                }
                break;
            case 1534:
                switch (current) {
                    case 'f' :
                        state = 1535;
                        return true;
                }
                break;
            case 1535:
                switch (current) {
                    case ';' : // Mellintrf;
                        match = "\u2133";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1536:
                switch (current) {
                    case 'r' :
                        state = 1537;
                        return true;
                }
                break;
            case 1537:
                switch (current) {
                    case ';' : // Mfr;
                        match = "\uD835\uDD10";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1538:
                switch (current) {
                    case 'n' :
                        state = 1539;
                        return true;
                }
                break;
            case 1539:
                switch (current) {
                    case 'u' :
                        state = 1540;
                        return true;
                }
                break;
            case 1540:
                switch (current) {
                    case 's' :
                        state = 1541;
                        return true;
                }
                break;
            case 1541:
                switch (current) {
                    case 'P' :
                        state = 1542;
                        return true;
                }
                break;
            case 1542:
                switch (current) {
                    case 'l' :
                        state = 1543;
                        return true;
                }
                break;
            case 1543:
                switch (current) {
                    case 'u' :
                        state = 1544;
                        return true;
                }
                break;
            case 1544:
                switch (current) {
                    case 's' :
                        state = 1545;
                        return true;
                }
                break;
            case 1545:
                switch (current) {
                    case ';' : // MinusPlus;
                        match = "\u2213";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1546:
                switch (current) {
                    case 'p' :
                        state = 1547;
                        return true;
                }
                break;
            case 1547:
                switch (current) {
                    case 'f' :
                        state = 1548;
                        return true;
                }
                break;
            case 1548:
                switch (current) {
                    case ';' : // Mopf;
                        match = "\uD835\uDD44";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1549:
                switch (current) {
                    case 'c' :
                        state = 1550;
                        return true;
                }
                break;
            case 1550:
                switch (current) {
                    case 'r' :
                        state = 1551;
                        return true;
                }
                break;
            case 1551:
                switch (current) {
                    case ';' : // Mscr;
                        match = "\u2133";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1552:
                switch (current) {
                    case ';' : // Mu;
                        match = "\u039C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1553:
                switch (current) {
                    case 'J' :
                        state = 1554;
                        return true;
                    case 'a' :
                        state = 1557;
                        return true;
                    case 'c' :
                        state = 1562;
                        return true;
                    case 'e' :
                        state = 1572;
                        return true;
                    case 'f' :
                        state = 1650;
                        return true;
                    case 'o' :
                        state = 1652;
                        return true;
                    case 's' :
                        state = 2055;
                        return true;
                    case 't' :
                        state = 2058;
                        return true;
                    case 'u' :
                        state = 2063;
                        return true;
                }
                break;
            case 1554:
                switch (current) {
                    case 'c' :
                        state = 1555;
                        return true;
                }
                break;
            case 1555:
                switch (current) {
                    case 'y' :
                        state = 1556;
                        return true;
                }
                break;
            case 1556:
                switch (current) {
                    case ';' : // NJcy;
                        match = "\u040A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1557:
                switch (current) {
                    case 'c' :
                        state = 1558;
                        return true;
                }
                break;
            case 1558:
                switch (current) {
                    case 'u' :
                        state = 1559;
                        return true;
                }
                break;
            case 1559:
                switch (current) {
                    case 't' :
                        state = 1560;
                        return true;
                }
                break;
            case 1560:
                switch (current) {
                    case 'e' :
                        state = 1561;
                        return true;
                }
                break;
            case 1561:
                switch (current) {
                    case ';' : // Nacute;
                        match = "\u0143";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1562:
                switch (current) {
                    case 'a' :
                        state = 1563;
                        return true;
                    case 'e' :
                        state = 1567;
                        return true;
                    case 'y' :
                        state = 1571;
                        return true;
                }
                break;
            case 1563:
                switch (current) {
                    case 'r' :
                        state = 1564;
                        return true;
                }
                break;
            case 1564:
                switch (current) {
                    case 'o' :
                        state = 1565;
                        return true;
                }
                break;
            case 1565:
                switch (current) {
                    case 'n' :
                        state = 1566;
                        return true;
                }
                break;
            case 1566:
                switch (current) {
                    case ';' : // Ncaron;
                        match = "\u0147";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1567:
                switch (current) {
                    case 'd' :
                        state = 1568;
                        return true;
                }
                break;
            case 1568:
                switch (current) {
                    case 'i' :
                        state = 1569;
                        return true;
                }
                break;
            case 1569:
                switch (current) {
                    case 'l' :
                        state = 1570;
                        return true;
                }
                break;
            case 1570:
                switch (current) {
                    case ';' : // Ncedil;
                        match = "\u0145";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1571:
                switch (current) {
                    case ';' : // Ncy;
                        match = "\u041D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1572:
                switch (current) {
                    case 'g' :
                        state = 1573;
                        return true;
                    case 's' :
                        state = 1619;
                        return true;
                    case 'w' :
                        state = 1645;
                        return true;
                }
                break;
            case 1573:
                switch (current) {
                    case 'a' :
                        state = 1574;
                        return true;
                }
                break;
            case 1574:
                switch (current) {
                    case 't' :
                        state = 1575;
                        return true;
                }
                break;
            case 1575:
                switch (current) {
                    case 'i' :
                        state = 1576;
                        return true;
                }
                break;
            case 1576:
                switch (current) {
                    case 'v' :
                        state = 1577;
                        return true;
                }
                break;
            case 1577:
                switch (current) {
                    case 'e' :
                        state = 1578;
                        return true;
                }
                break;
            case 1578:
                switch (current) {
                    case 'M' :
                        state = 1579;
                        return true;
                    case 'T' :
                        state = 1590;
                        return true;
                    case 'V' :
                        state = 1606;
                        return true;
                }
                break;
            case 1579:
                switch (current) {
                    case 'e' :
                        state = 1580;
                        return true;
                }
                break;
            case 1580:
                switch (current) {
                    case 'd' :
                        state = 1581;
                        return true;
                }
                break;
            case 1581:
                switch (current) {
                    case 'i' :
                        state = 1582;
                        return true;
                }
                break;
            case 1582:
                switch (current) {
                    case 'u' :
                        state = 1583;
                        return true;
                }
                break;
            case 1583:
                switch (current) {
                    case 'm' :
                        state = 1584;
                        return true;
                }
                break;
            case 1584:
                switch (current) {
                    case 'S' :
                        state = 1585;
                        return true;
                }
                break;
            case 1585:
                switch (current) {
                    case 'p' :
                        state = 1586;
                        return true;
                }
                break;
            case 1586:
                switch (current) {
                    case 'a' :
                        state = 1587;
                        return true;
                }
                break;
            case 1587:
                switch (current) {
                    case 'c' :
                        state = 1588;
                        return true;
                }
                break;
            case 1588:
                switch (current) {
                    case 'e' :
                        state = 1589;
                        return true;
                }
                break;
            case 1589:
                switch (current) {
                    case ';' : // NegativeMediumSpace;
                        match = "\u200B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1590:
                switch (current) {
                    case 'h' :
                        state = 1591;
                        return true;
                }
                break;
            case 1591:
                switch (current) {
                    case 'i' :
                        state = 1592;
                        return true;
                }
                break;
            case 1592:
                switch (current) {
                    case 'c' :
                        state = 1593;
                        return true;
                    case 'n' :
                        state = 1600;
                        return true;
                }
                break;
            case 1593:
                switch (current) {
                    case 'k' :
                        state = 1594;
                        return true;
                }
                break;
            case 1594:
                switch (current) {
                    case 'S' :
                        state = 1595;
                        return true;
                }
                break;
            case 1595:
                switch (current) {
                    case 'p' :
                        state = 1596;
                        return true;
                }
                break;
            case 1596:
                switch (current) {
                    case 'a' :
                        state = 1597;
                        return true;
                }
                break;
            case 1597:
                switch (current) {
                    case 'c' :
                        state = 1598;
                        return true;
                }
                break;
            case 1598:
                switch (current) {
                    case 'e' :
                        state = 1599;
                        return true;
                }
                break;
            case 1599:
                switch (current) {
                    case ';' : // NegativeThickSpace;
                        match = "\u200B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1600:
                switch (current) {
                    case 'S' :
                        state = 1601;
                        return true;
                }
                break;
            case 1601:
                switch (current) {
                    case 'p' :
                        state = 1602;
                        return true;
                }
                break;
            case 1602:
                switch (current) {
                    case 'a' :
                        state = 1603;
                        return true;
                }
                break;
            case 1603:
                switch (current) {
                    case 'c' :
                        state = 1604;
                        return true;
                }
                break;
            case 1604:
                switch (current) {
                    case 'e' :
                        state = 1605;
                        return true;
                }
                break;
            case 1605:
                switch (current) {
                    case ';' : // NegativeThinSpace;
                        match = "\u200B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1606:
                switch (current) {
                    case 'e' :
                        state = 1607;
                        return true;
                }
                break;
            case 1607:
                switch (current) {
                    case 'r' :
                        state = 1608;
                        return true;
                }
                break;
            case 1608:
                switch (current) {
                    case 'y' :
                        state = 1609;
                        return true;
                }
                break;
            case 1609:
                switch (current) {
                    case 'T' :
                        state = 1610;
                        return true;
                }
                break;
            case 1610:
                switch (current) {
                    case 'h' :
                        state = 1611;
                        return true;
                }
                break;
            case 1611:
                switch (current) {
                    case 'i' :
                        state = 1612;
                        return true;
                }
                break;
            case 1612:
                switch (current) {
                    case 'n' :
                        state = 1613;
                        return true;
                }
                break;
            case 1613:
                switch (current) {
                    case 'S' :
                        state = 1614;
                        return true;
                }
                break;
            case 1614:
                switch (current) {
                    case 'p' :
                        state = 1615;
                        return true;
                }
                break;
            case 1615:
                switch (current) {
                    case 'a' :
                        state = 1616;
                        return true;
                }
                break;
            case 1616:
                switch (current) {
                    case 'c' :
                        state = 1617;
                        return true;
                }
                break;
            case 1617:
                switch (current) {
                    case 'e' :
                        state = 1618;
                        return true;
                }
                break;
            case 1618:
                switch (current) {
                    case ';' : // NegativeVeryThinSpace;
                        match = "\u200B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1619:
                switch (current) {
                    case 't' :
                        state = 1620;
                        return true;
                }
                break;
            case 1620:
                switch (current) {
                    case 'e' :
                        state = 1621;
                        return true;
                }
                break;
            case 1621:
                switch (current) {
                    case 'd' :
                        state = 1622;
                        return true;
                }
                break;
            case 1622:
                switch (current) {
                    case 'G' :
                        state = 1623;
                        return true;
                    case 'L' :
                        state = 1637;
                        return true;
                }
                break;
            case 1623:
                switch (current) {
                    case 'r' :
                        state = 1624;
                        return true;
                }
                break;
            case 1624:
                switch (current) {
                    case 'e' :
                        state = 1625;
                        return true;
                }
                break;
            case 1625:
                switch (current) {
                    case 'a' :
                        state = 1626;
                        return true;
                }
                break;
            case 1626:
                switch (current) {
                    case 't' :
                        state = 1627;
                        return true;
                }
                break;
            case 1627:
                switch (current) {
                    case 'e' :
                        state = 1628;
                        return true;
                }
                break;
            case 1628:
                switch (current) {
                    case 'r' :
                        state = 1629;
                        return true;
                }
                break;
            case 1629:
                switch (current) {
                    case 'G' :
                        state = 1630;
                        return true;
                }
                break;
            case 1630:
                switch (current) {
                    case 'r' :
                        state = 1631;
                        return true;
                }
                break;
            case 1631:
                switch (current) {
                    case 'e' :
                        state = 1632;
                        return true;
                }
                break;
            case 1632:
                switch (current) {
                    case 'a' :
                        state = 1633;
                        return true;
                }
                break;
            case 1633:
                switch (current) {
                    case 't' :
                        state = 1634;
                        return true;
                }
                break;
            case 1634:
                switch (current) {
                    case 'e' :
                        state = 1635;
                        return true;
                }
                break;
            case 1635:
                switch (current) {
                    case 'r' :
                        state = 1636;
                        return true;
                }
                break;
            case 1636:
                switch (current) {
                    case ';' : // NestedGreaterGreater;
                        match = "\u226B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1637:
                switch (current) {
                    case 'e' :
                        state = 1638;
                        return true;
                }
                break;
            case 1638:
                switch (current) {
                    case 's' :
                        state = 1639;
                        return true;
                }
                break;
            case 1639:
                switch (current) {
                    case 's' :
                        state = 1640;
                        return true;
                }
                break;
            case 1640:
                switch (current) {
                    case 'L' :
                        state = 1641;
                        return true;
                }
                break;
            case 1641:
                switch (current) {
                    case 'e' :
                        state = 1642;
                        return true;
                }
                break;
            case 1642:
                switch (current) {
                    case 's' :
                        state = 1643;
                        return true;
                }
                break;
            case 1643:
                switch (current) {
                    case 's' :
                        state = 1644;
                        return true;
                }
                break;
            case 1644:
                switch (current) {
                    case ';' : // NestedLessLess;
                        match = "\u226A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1645:
                switch (current) {
                    case 'L' :
                        state = 1646;
                        return true;
                }
                break;
            case 1646:
                switch (current) {
                    case 'i' :
                        state = 1647;
                        return true;
                }
                break;
            case 1647:
                switch (current) {
                    case 'n' :
                        state = 1648;
                        return true;
                }
                break;
            case 1648:
                switch (current) {
                    case 'e' :
                        state = 1649;
                        return true;
                }
                break;
            case 1649:
                switch (current) {
                    case ';' : // NewLine;
                        match = "\n";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1650:
                switch (current) {
                    case 'r' :
                        state = 1651;
                        return true;
                }
                break;
            case 1651:
                switch (current) {
                    case ';' : // Nfr;
                        match = "\uD835\uDD11";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1652:
                switch (current) {
                    case 'B' :
                        state = 1653;
                        return true;
                    case 'n' :
                        state = 1658;
                        return true;
                    case 'p' :
                        state = 1672;
                        return true;
                    case 't' :
                        state = 1674;
                        return true;
                }
                break;
            case 1653:
                switch (current) {
                    case 'r' :
                        state = 1654;
                        return true;
                }
                break;
            case 1654:
                switch (current) {
                    case 'e' :
                        state = 1655;
                        return true;
                }
                break;
            case 1655:
                switch (current) {
                    case 'a' :
                        state = 1656;
                        return true;
                }
                break;
            case 1656:
                switch (current) {
                    case 'k' :
                        state = 1657;
                        return true;
                }
                break;
            case 1657:
                switch (current) {
                    case ';' : // NoBreak;
                        match = "\u2060";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1658:
                switch (current) {
                    case 'B' :
                        state = 1659;
                        return true;
                }
                break;
            case 1659:
                switch (current) {
                    case 'r' :
                        state = 1660;
                        return true;
                }
                break;
            case 1660:
                switch (current) {
                    case 'e' :
                        state = 1661;
                        return true;
                }
                break;
            case 1661:
                switch (current) {
                    case 'a' :
                        state = 1662;
                        return true;
                }
                break;
            case 1662:
                switch (current) {
                    case 'k' :
                        state = 1663;
                        return true;
                }
                break;
            case 1663:
                switch (current) {
                    case 'i' :
                        state = 1664;
                        return true;
                }
                break;
            case 1664:
                switch (current) {
                    case 'n' :
                        state = 1665;
                        return true;
                }
                break;
            case 1665:
                switch (current) {
                    case 'g' :
                        state = 1666;
                        return true;
                }
                break;
            case 1666:
                switch (current) {
                    case 'S' :
                        state = 1667;
                        return true;
                }
                break;
            case 1667:
                switch (current) {
                    case 'p' :
                        state = 1668;
                        return true;
                }
                break;
            case 1668:
                switch (current) {
                    case 'a' :
                        state = 1669;
                        return true;
                }
                break;
            case 1669:
                switch (current) {
                    case 'c' :
                        state = 1670;
                        return true;
                }
                break;
            case 1670:
                switch (current) {
                    case 'e' :
                        state = 1671;
                        return true;
                }
                break;
            case 1671:
                switch (current) {
                    case ';' : // NonBreakingSpace;
                        match = "\u00A0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1672:
                switch (current) {
                    case 'f' :
                        state = 1673;
                        return true;
                }
                break;
            case 1673:
                switch (current) {
                    case ';' : // Nopf;
                        match = "\u2115";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1674:
                switch (current) {
                    case ';' : // Not;
                        match = "\u2AEC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'C' :
                        state = 1675;
                        return true;
                    case 'D' :
                        state = 1689;
                        return true;
                    case 'E' :
                        state = 1706;
                        return true;
                    case 'G' :
                        state = 1727;
                        return true;
                    case 'H' :
                        state = 1780;
                        return true;
                    case 'L' :
                        state = 1801;
                        return true;
                    case 'N' :
                        state = 1860;
                        return true;
                    case 'P' :
                        state = 1888;
                        return true;
                    case 'R' :
                        state = 1911;
                        return true;
                    case 'S' :
                        state = 1945;
                        return true;
                    case 'T' :
                        state = 2020;
                        return true;
                    case 'V' :
                        state = 2044;
                        return true;
                }
                break;
            case 1675:
                switch (current) {
                    case 'o' :
                        state = 1676;
                        return true;
                    case 'u' :
                        state = 1684;
                        return true;
                }
                break;
            case 1676:
                switch (current) {
                    case 'n' :
                        state = 1677;
                        return true;
                }
                break;
            case 1677:
                switch (current) {
                    case 'g' :
                        state = 1678;
                        return true;
                }
                break;
            case 1678:
                switch (current) {
                    case 'r' :
                        state = 1679;
                        return true;
                }
                break;
            case 1679:
                switch (current) {
                    case 'u' :
                        state = 1680;
                        return true;
                }
                break;
            case 1680:
                switch (current) {
                    case 'e' :
                        state = 1681;
                        return true;
                }
                break;
            case 1681:
                switch (current) {
                    case 'n' :
                        state = 1682;
                        return true;
                }
                break;
            case 1682:
                switch (current) {
                    case 't' :
                        state = 1683;
                        return true;
                }
                break;
            case 1683:
                switch (current) {
                    case ';' : // NotCongruent;
                        match = "\u2262";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1684:
                switch (current) {
                    case 'p' :
                        state = 1685;
                        return true;
                }
                break;
            case 1685:
                switch (current) {
                    case 'C' :
                        state = 1686;
                        return true;
                }
                break;
            case 1686:
                switch (current) {
                    case 'a' :
                        state = 1687;
                        return true;
                }
                break;
            case 1687:
                switch (current) {
                    case 'p' :
                        state = 1688;
                        return true;
                }
                break;
            case 1688:
                switch (current) {
                    case ';' : // NotCupCap;
                        match = "\u226D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1689:
                switch (current) {
                    case 'o' :
                        state = 1690;
                        return true;
                }
                break;
            case 1690:
                switch (current) {
                    case 'u' :
                        state = 1691;
                        return true;
                }
                break;
            case 1691:
                switch (current) {
                    case 'b' :
                        state = 1692;
                        return true;
                }
                break;
            case 1692:
                switch (current) {
                    case 'l' :
                        state = 1693;
                        return true;
                }
                break;
            case 1693:
                switch (current) {
                    case 'e' :
                        state = 1694;
                        return true;
                }
                break;
            case 1694:
                switch (current) {
                    case 'V' :
                        state = 1695;
                        return true;
                }
                break;
            case 1695:
                switch (current) {
                    case 'e' :
                        state = 1696;
                        return true;
                }
                break;
            case 1696:
                switch (current) {
                    case 'r' :
                        state = 1697;
                        return true;
                }
                break;
            case 1697:
                switch (current) {
                    case 't' :
                        state = 1698;
                        return true;
                }
                break;
            case 1698:
                switch (current) {
                    case 'i' :
                        state = 1699;
                        return true;
                }
                break;
            case 1699:
                switch (current) {
                    case 'c' :
                        state = 1700;
                        return true;
                }
                break;
            case 1700:
                switch (current) {
                    case 'a' :
                        state = 1701;
                        return true;
                }
                break;
            case 1701:
                switch (current) {
                    case 'l' :
                        state = 1702;
                        return true;
                }
                break;
            case 1702:
                switch (current) {
                    case 'B' :
                        state = 1703;
                        return true;
                }
                break;
            case 1703:
                switch (current) {
                    case 'a' :
                        state = 1704;
                        return true;
                }
                break;
            case 1704:
                switch (current) {
                    case 'r' :
                        state = 1705;
                        return true;
                }
                break;
            case 1705:
                switch (current) {
                    case ';' : // NotDoubleVerticalBar;
                        match = "\u2226";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1706:
                switch (current) {
                    case 'l' :
                        state = 1707;
                        return true;
                    case 'q' :
                        state = 1713;
                        return true;
                    case 'x' :
                        state = 1722;
                        return true;
                }
                break;
            case 1707:
                switch (current) {
                    case 'e' :
                        state = 1708;
                        return true;
                }
                break;
            case 1708:
                switch (current) {
                    case 'm' :
                        state = 1709;
                        return true;
                }
                break;
            case 1709:
                switch (current) {
                    case 'e' :
                        state = 1710;
                        return true;
                }
                break;
            case 1710:
                switch (current) {
                    case 'n' :
                        state = 1711;
                        return true;
                }
                break;
            case 1711:
                switch (current) {
                    case 't' :
                        state = 1712;
                        return true;
                }
                break;
            case 1712:
                switch (current) {
                    case ';' : // NotElement;
                        match = "\u2209";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1713:
                switch (current) {
                    case 'u' :
                        state = 1714;
                        return true;
                }
                break;
            case 1714:
                switch (current) {
                    case 'a' :
                        state = 1715;
                        return true;
                }
                break;
            case 1715:
                switch (current) {
                    case 'l' :
                        state = 1716;
                        return true;
                }
                break;
            case 1716:
                switch (current) {
                    case ';' : // NotEqual;
                        match = "\u2260";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'T' :
                        state = 1717;
                        return true;
                }
                break;
            case 1717:
                switch (current) {
                    case 'i' :
                        state = 1718;
                        return true;
                }
                break;
            case 1718:
                switch (current) {
                    case 'l' :
                        state = 1719;
                        return true;
                }
                break;
            case 1719:
                switch (current) {
                    case 'd' :
                        state = 1720;
                        return true;
                }
                break;
            case 1720:
                switch (current) {
                    case 'e' :
                        state = 1721;
                        return true;
                }
                break;
            case 1721:
                switch (current) {
                    case ';' : // NotEqualTilde;
                        match = "\u2242\u0338";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1722:
                switch (current) {
                    case 'i' :
                        state = 1723;
                        return true;
                }
                break;
            case 1723:
                switch (current) {
                    case 's' :
                        state = 1724;
                        return true;
                }
                break;
            case 1724:
                switch (current) {
                    case 't' :
                        state = 1725;
                        return true;
                }
                break;
            case 1725:
                switch (current) {
                    case 's' :
                        state = 1726;
                        return true;
                }
                break;
            case 1726:
                switch (current) {
                    case ';' : // NotExists;
                        match = "\u2204";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1727:
                switch (current) {
                    case 'r' :
                        state = 1728;
                        return true;
                }
                break;
            case 1728:
                switch (current) {
                    case 'e' :
                        state = 1729;
                        return true;
                }
                break;
            case 1729:
                switch (current) {
                    case 'a' :
                        state = 1730;
                        return true;
                }
                break;
            case 1730:
                switch (current) {
                    case 't' :
                        state = 1731;
                        return true;
                }
                break;
            case 1731:
                switch (current) {
                    case 'e' :
                        state = 1732;
                        return true;
                }
                break;
            case 1732:
                switch (current) {
                    case 'r' :
                        state = 1733;
                        return true;
                }
                break;
            case 1733:
                switch (current) {
                    case ';' : // NotGreater;
                        match = "\u226F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1734;
                        return true;
                    case 'F' :
                        state = 1739;
                        return true;
                    case 'G' :
                        state = 1750;
                        return true;
                    case 'L' :
                        state = 1759;
                        return true;
                    case 'S' :
                        state = 1763;
                        return true;
                    case 'T' :
                        state = 1775;
                        return true;
                }
                break;
            case 1734:
                switch (current) {
                    case 'q' :
                        state = 1735;
                        return true;
                }
                break;
            case 1735:
                switch (current) {
                    case 'u' :
                        state = 1736;
                        return true;
                }
                break;
            case 1736:
                switch (current) {
                    case 'a' :
                        state = 1737;
                        return true;
                }
                break;
            case 1737:
                switch (current) {
                    case 'l' :
                        state = 1738;
                        return true;
                }
                break;
            case 1738:
                switch (current) {
                    case ';' : // NotGreaterEqual;
                        match = "\u2271";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1739:
                switch (current) {
                    case 'u' :
                        state = 1740;
                        return true;
                }
                break;
            case 1740:
                switch (current) {
                    case 'l' :
                        state = 1741;
                        return true;
                }
                break;
            case 1741:
                switch (current) {
                    case 'l' :
                        state = 1742;
                        return true;
                }
                break;
            case 1742:
                switch (current) {
                    case 'E' :
                        state = 1743;
                        return true;
                }
                break;
            case 1743:
                switch (current) {
                    case 'q' :
                        state = 1744;
                        return true;
                }
                break;
            case 1744:
                switch (current) {
                    case 'u' :
                        state = 1745;
                        return true;
                }
                break;
            case 1745:
                switch (current) {
                    case 'a' :
                        state = 1746;
                        return true;
                }
                break;
            case 1746:
                switch (current) {
                    case 'l' :
                        state = 1747;
                        return true;
                }
                break;
            case 1747:
                switch (current) {
                    case ';' :
                        state = 1748;
                        return true;
                }
                break;
            case 1748:
                switch (current) {
                    case '≧' :
                        state = 1749;
                        return true;
                }
                break;
            case 1749:
                switch (current) {
                    case '̸' : // NotGreaterFullEqual;≧̸
                        match = "";
                        matchLength = consumedCount;
                        return false;
                }
                break;
            case 1750:
                switch (current) {
                    case 'r' :
                        state = 1751;
                        return true;
                }
                break;
            case 1751:
                switch (current) {
                    case 'e' :
                        state = 1752;
                        return true;
                }
                break;
            case 1752:
                switch (current) {
                    case 'a' :
                        state = 1753;
                        return true;
                }
                break;
            case 1753:
                switch (current) {
                    case 't' :
                        state = 1754;
                        return true;
                }
                break;
            case 1754:
                switch (current) {
                    case 'e' :
                        state = 1755;
                        return true;
                }
                break;
            case 1755:
                switch (current) {
                    case 'r' :
                        state = 1756;
                        return true;
                }
                break;
            case 1756:
                switch (current) {
                    case ';' :
                        state = 1757;
                        return true;
                }
                break;
            case 1757:
                switch (current) {
                    case '≫' :
                        state = 1758;
                        return true;
                }
                break;
            case 1758:
                switch (current) {
                    case '̸' : // NotGreaterGreater;≫̸
                        match = "";
                        matchLength = consumedCount;
                        return false;
                }
                break;
            case 1759:
                switch (current) {
                    case 'e' :
                        state = 1760;
                        return true;
                }
                break;
            case 1760:
                switch (current) {
                    case 's' :
                        state = 1761;
                        return true;
                }
                break;
            case 1761:
                switch (current) {
                    case 's' :
                        state = 1762;
                        return true;
                }
                break;
            case 1762:
                switch (current) {
                    case ';' : // NotGreaterLess;
                        match = "\u2279";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1763:
                switch (current) {
                    case 'l' :
                        state = 1764;
                        return true;
                }
                break;
            case 1764:
                switch (current) {
                    case 'a' :
                        state = 1765;
                        return true;
                }
                break;
            case 1765:
                switch (current) {
                    case 'n' :
                        state = 1766;
                        return true;
                }
                break;
            case 1766:
                switch (current) {
                    case 't' :
                        state = 1767;
                        return true;
                }
                break;
            case 1767:
                switch (current) {
                    case 'E' :
                        state = 1768;
                        return true;
                }
                break;
            case 1768:
                switch (current) {
                    case 'q' :
                        state = 1769;
                        return true;
                }
                break;
            case 1769:
                switch (current) {
                    case 'u' :
                        state = 1770;
                        return true;
                }
                break;
            case 1770:
                switch (current) {
                    case 'a' :
                        state = 1771;
                        return true;
                }
                break;
            case 1771:
                switch (current) {
                    case 'l' :
                        state = 1772;
                        return true;
                }
                break;
            case 1772:
                switch (current) {
                    case ';' :
                        state = 1773;
                        return true;
                }
                break;
            case 1773:
                switch (current) {
                    case '⩾' :
                        state = 1774;
                        return true;
                }
                break;
            case 1774:
                switch (current) {
                    case '̸' : // NotGreaterSlantEqual;⩾̸
                        match = "";
                        matchLength = consumedCount;
                        return false;
                }
                break;
            case 1775:
                switch (current) {
                    case 'i' :
                        state = 1776;
                        return true;
                }
                break;
            case 1776:
                switch (current) {
                    case 'l' :
                        state = 1777;
                        return true;
                }
                break;
            case 1777:
                switch (current) {
                    case 'd' :
                        state = 1778;
                        return true;
                }
                break;
            case 1778:
                switch (current) {
                    case 'e' :
                        state = 1779;
                        return true;
                }
                break;
            case 1779:
                switch (current) {
                    case ';' : // NotGreaterTilde;
                        match = "\u2275";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1780:
                switch (current) {
                    case 'u' :
                        state = 1781;
                        return true;
                }
                break;
            case 1781:
                switch (current) {
                    case 'm' :
                        state = 1782;
                        return true;
                }
                break;
            case 1782:
                switch (current) {
                    case 'p' :
                        state = 1783;
                        return true;
                }
                break;
            case 1783:
                switch (current) {
                    case 'D' :
                        state = 1784;
                        return true;
                    case 'E' :
                        state = 1794;
                        return true;
                }
                break;
            case 1784:
                switch (current) {
                    case 'o' :
                        state = 1785;
                        return true;
                }
                break;
            case 1785:
                switch (current) {
                    case 'w' :
                        state = 1786;
                        return true;
                }
                break;
            case 1786:
                switch (current) {
                    case 'n' :
                        state = 1787;
                        return true;
                }
                break;
            case 1787:
                switch (current) {
                    case 'H' :
                        state = 1788;
                        return true;
                }
                break;
            case 1788:
                switch (current) {
                    case 'u' :
                        state = 1789;
                        return true;
                }
                break;
            case 1789:
                switch (current) {
                    case 'm' :
                        state = 1790;
                        return true;
                }
                break;
            case 1790:
                switch (current) {
                    case 'p' :
                        state = 1791;
                        return true;
                }
                break;
            case 1791:
                switch (current) {
                    case ';' :
                        state = 1792;
                        return true;
                }
                break;
            case 1792:
                switch (current) {
                    case '≎' :
                        state = 1793;
                        return true;
                }
                break;
            case 1793:
                switch (current) {
                    case '̸' : // NotHumpDownHump;≎̸
                        match = "";
                        matchLength = consumedCount;
                        return false;
                }
                break;
            case 1794:
                switch (current) {
                    case 'q' :
                        state = 1795;
                        return true;
                }
                break;
            case 1795:
                switch (current) {
                    case 'u' :
                        state = 1796;
                        return true;
                }
                break;
            case 1796:
                switch (current) {
                    case 'a' :
                        state = 1797;
                        return true;
                }
                break;
            case 1797:
                switch (current) {
                    case 'l' :
                        state = 1798;
                        return true;
                }
                break;
            case 1798:
                switch (current) {
                    case ';' :
                        state = 1799;
                        return true;
                }
                break;
            case 1799:
                switch (current) {
                    case '≏' :
                        state = 1800;
                        return true;
                }
                break;
            case 1800:
                switch (current) {
                    case '̸' : // NotHumpEqual;≏̸
                        match = "";
                        matchLength = consumedCount;
                        return false;
                }
                break;
            case 1801:
                switch (current) {
                    case 'e' :
                        state = 1802;
                        return true;
                }
                break;
            case 1802:
                switch (current) {
                    case 'f' :
                        state = 1803;
                        return true;
                    case 's' :
                        state = 1823;
                        return true;
                }
                break;
            case 1803:
                switch (current) {
                    case 't' :
                        state = 1804;
                        return true;
                }
                break;
            case 1804:
                switch (current) {
                    case 'T' :
                        state = 1805;
                        return true;
                }
                break;
            case 1805:
                switch (current) {
                    case 'r' :
                        state = 1806;
                        return true;
                }
                break;
            case 1806:
                switch (current) {
                    case 'i' :
                        state = 1807;
                        return true;
                }
                break;
            case 1807:
                switch (current) {
                    case 'a' :
                        state = 1808;
                        return true;
                }
                break;
            case 1808:
                switch (current) {
                    case 'n' :
                        state = 1809;
                        return true;
                }
                break;
            case 1809:
                switch (current) {
                    case 'g' :
                        state = 1810;
                        return true;
                }
                break;
            case 1810:
                switch (current) {
                    case 'l' :
                        state = 1811;
                        return true;
                }
                break;
            case 1811:
                switch (current) {
                    case 'e' :
                        state = 1812;
                        return true;
                }
                break;
            case 1812:
                switch (current) {
                    case ';' : // NotLeftTriangle;
                        match = "\u22EA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1813;
                        return true;
                    case 'E' :
                        state = 1818;
                        return true;
                }
                break;
            case 1813:
                switch (current) {
                    case 'a' :
                        state = 1814;
                        return true;
                }
                break;
            case 1814:
                switch (current) {
                    case 'r' :
                        state = 1815;
                        return true;
                }
                break;
            case 1815:
                switch (current) {
                    case ';' :
                        state = 1816;
                        return true;
                }
                break;
            case 1816:
                switch (current) {
                    case '⧏' :
                        state = 1817;
                        return true;
                }
                break;
            case 1817:
                switch (current) {
                    case '̸' : // NotLeftTriangleBar;⧏̸
                        match = "";
                        matchLength = consumedCount;
                        return false;
                }
                break;
            case 1818:
                switch (current) {
                    case 'q' :
                        state = 1819;
                        return true;
                }
                break;
            case 1819:
                switch (current) {
                    case 'u' :
                        state = 1820;
                        return true;
                }
                break;
            case 1820:
                switch (current) {
                    case 'a' :
                        state = 1821;
                        return true;
                }
                break;
            case 1821:
                switch (current) {
                    case 'l' :
                        state = 1822;
                        return true;
                }
                break;
            case 1822:
                switch (current) {
                    case ';' : // NotLeftTriangleEqual;
                        match = "\u22EC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1823:
                switch (current) {
                    case 's' :
                        state = 1824;
                        return true;
                }
                break;
            case 1824:
                switch (current) {
                    case ';' : // NotLess;
                        match = "\u226E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1825;
                        return true;
                    case 'G' :
                        state = 1830;
                        return true;
                    case 'L' :
                        state = 1837;
                        return true;
                    case 'S' :
                        state = 1843;
                        return true;
                    case 'T' :
                        state = 1855;
                        return true;
                }
                break;
            case 1825:
                switch (current) {
                    case 'q' :
                        state = 1826;
                        return true;
                }
                break;
            case 1826:
                switch (current) {
                    case 'u' :
                        state = 1827;
                        return true;
                }
                break;
            case 1827:
                switch (current) {
                    case 'a' :
                        state = 1828;
                        return true;
                }
                break;
            case 1828:
                switch (current) {
                    case 'l' :
                        state = 1829;
                        return true;
                }
                break;
            case 1829:
                switch (current) {
                    case ';' : // NotLessEqual;
                        match = "\u2270";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1830:
                switch (current) {
                    case 'r' :
                        state = 1831;
                        return true;
                }
                break;
            case 1831:
                switch (current) {
                    case 'e' :
                        state = 1832;
                        return true;
                }
                break;
            case 1832:
                switch (current) {
                    case 'a' :
                        state = 1833;
                        return true;
                }
                break;
            case 1833:
                switch (current) {
                    case 't' :
                        state = 1834;
                        return true;
                }
                break;
            case 1834:
                switch (current) {
                    case 'e' :
                        state = 1835;
                        return true;
                }
                break;
            case 1835:
                switch (current) {
                    case 'r' :
                        state = 1836;
                        return true;
                }
                break;
            case 1836:
                switch (current) {
                    case ';' : // NotLessGreater;
                        match = "\u2278";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1837:
                switch (current) {
                    case 'e' :
                        state = 1838;
                        return true;
                }
                break;
            case 1838:
                switch (current) {
                    case 's' :
                        state = 1839;
                        return true;
                }
                break;
            case 1839:
                switch (current) {
                    case 's' :
                        state = 1840;
                        return true;
                }
                break;
            case 1840:
                switch (current) {
                    case ';' :
                        state = 1841;
                        return true;
                }
                break;
            case 1841:
                switch (current) {
                    case '≪' :
                        state = 1842;
                        return true;
                }
                break;
            case 1842:
                switch (current) {
                    case '̸' : // NotLessLess;≪̸
                        match = "";
                        matchLength = consumedCount;
                        return false;
                }
                break;
            case 1843:
                switch (current) {
                    case 'l' :
                        state = 1844;
                        return true;
                }
                break;
            case 1844:
                switch (current) {
                    case 'a' :
                        state = 1845;
                        return true;
                }
                break;
            case 1845:
                switch (current) {
                    case 'n' :
                        state = 1846;
                        return true;
                }
                break;
            case 1846:
                switch (current) {
                    case 't' :
                        state = 1847;
                        return true;
                }
                break;
            case 1847:
                switch (current) {
                    case 'E' :
                        state = 1848;
                        return true;
                }
                break;
            case 1848:
                switch (current) {
                    case 'q' :
                        state = 1849;
                        return true;
                }
                break;
            case 1849:
                switch (current) {
                    case 'u' :
                        state = 1850;
                        return true;
                }
                break;
            case 1850:
                switch (current) {
                    case 'a' :
                        state = 1851;
                        return true;
                }
                break;
            case 1851:
                switch (current) {
                    case 'l' :
                        state = 1852;
                        return true;
                }
                break;
            case 1852:
                switch (current) {
                    case ';' :
                        state = 1853;
                        return true;
                }
                break;
            case 1853:
                switch (current) {
                    case '⩽' :
                        state = 1854;
                        return true;
                }
                break;
            case 1854:
                switch (current) {
                    case '̸' : // NotLessSlantEqual;⩽̸
                        match = "";
                        matchLength = consumedCount;
                        return false;
                }
                break;
            case 1855:
                switch (current) {
                    case 'i' :
                        state = 1856;
                        return true;
                }
                break;
            case 1856:
                switch (current) {
                    case 'l' :
                        state = 1857;
                        return true;
                }
                break;
            case 1857:
                switch (current) {
                    case 'd' :
                        state = 1858;
                        return true;
                }
                break;
            case 1858:
                switch (current) {
                    case 'e' :
                        state = 1859;
                        return true;
                }
                break;
            case 1859:
                switch (current) {
                    case ';' : // NotLessTilde;
                        match = "\u2274";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1860:
                switch (current) {
                    case 'e' :
                        state = 1861;
                        return true;
                }
                break;
            case 1861:
                switch (current) {
                    case 's' :
                        state = 1862;
                        return true;
                }
                break;
            case 1862:
                switch (current) {
                    case 't' :
                        state = 1863;
                        return true;
                }
                break;
            case 1863:
                switch (current) {
                    case 'e' :
                        state = 1864;
                        return true;
                }
                break;
            case 1864:
                switch (current) {
                    case 'd' :
                        state = 1865;
                        return true;
                }
                break;
            case 1865:
                switch (current) {
                    case 'G' :
                        state = 1866;
                        return true;
                    case 'L' :
                        state = 1880;
                        return true;
                }
                break;
            case 1866:
                switch (current) {
                    case 'r' :
                        state = 1867;
                        return true;
                }
                break;
            case 1867:
                switch (current) {
                    case 'e' :
                        state = 1868;
                        return true;
                }
                break;
            case 1868:
                switch (current) {
                    case 'a' :
                        state = 1869;
                        return true;
                }
                break;
            case 1869:
                switch (current) {
                    case 't' :
                        state = 1870;
                        return true;
                }
                break;
            case 1870:
                switch (current) {
                    case 'e' :
                        state = 1871;
                        return true;
                }
                break;
            case 1871:
                switch (current) {
                    case 'r' :
                        state = 1872;
                        return true;
                }
                break;
            case 1872:
                switch (current) {
                    case 'G' :
                        state = 1873;
                        return true;
                }
                break;
            case 1873:
                switch (current) {
                    case 'r' :
                        state = 1874;
                        return true;
                }
                break;
            case 1874:
                switch (current) {
                    case 'e' :
                        state = 1875;
                        return true;
                }
                break;
            case 1875:
                switch (current) {
                    case 'a' :
                        state = 1876;
                        return true;
                }
                break;
            case 1876:
                switch (current) {
                    case 't' :
                        state = 1877;
                        return true;
                }
                break;
            case 1877:
                switch (current) {
                    case 'e' :
                        state = 1878;
                        return true;
                }
                break;
            case 1878:
                switch (current) {
                    case 'r' :
                        state = 1879;
                        return true;
                }
                break;
            case 1879:
                switch (current) {
                    case ';' : // NotNestedGreaterGreater;
                        match = "\u2AA2\u0338";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1880:
                switch (current) {
                    case 'e' :
                        state = 1881;
                        return true;
                }
                break;
            case 1881:
                switch (current) {
                    case 's' :
                        state = 1882;
                        return true;
                }
                break;
            case 1882:
                switch (current) {
                    case 's' :
                        state = 1883;
                        return true;
                }
                break;
            case 1883:
                switch (current) {
                    case 'L' :
                        state = 1884;
                        return true;
                }
                break;
            case 1884:
                switch (current) {
                    case 'e' :
                        state = 1885;
                        return true;
                }
                break;
            case 1885:
                switch (current) {
                    case 's' :
                        state = 1886;
                        return true;
                }
                break;
            case 1886:
                switch (current) {
                    case 's' :
                        state = 1887;
                        return true;
                }
                break;
            case 1887:
                switch (current) {
                    case ';' : // NotNestedLessLess;
                        match = "\u2AA1\u0338";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1888:
                switch (current) {
                    case 'r' :
                        state = 1889;
                        return true;
                }
                break;
            case 1889:
                switch (current) {
                    case 'e' :
                        state = 1890;
                        return true;
                }
                break;
            case 1890:
                switch (current) {
                    case 'c' :
                        state = 1891;
                        return true;
                }
                break;
            case 1891:
                switch (current) {
                    case 'e' :
                        state = 1892;
                        return true;
                }
                break;
            case 1892:
                switch (current) {
                    case 'd' :
                        state = 1893;
                        return true;
                }
                break;
            case 1893:
                switch (current) {
                    case 'e' :
                        state = 1894;
                        return true;
                }
                break;
            case 1894:
                switch (current) {
                    case 's' :
                        state = 1895;
                        return true;
                }
                break;
            case 1895:
                switch (current) {
                    case ';' : // NotPrecedes;
                        match = "\u2280";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1896;
                        return true;
                    case 'S' :
                        state = 1901;
                        return true;
                }
                break;
            case 1896:
                switch (current) {
                    case 'q' :
                        state = 1897;
                        return true;
                }
                break;
            case 1897:
                switch (current) {
                    case 'u' :
                        state = 1898;
                        return true;
                }
                break;
            case 1898:
                switch (current) {
                    case 'a' :
                        state = 1899;
                        return true;
                }
                break;
            case 1899:
                switch (current) {
                    case 'l' :
                        state = 1900;
                        return true;
                }
                break;
            case 1900:
                switch (current) {
                    case ';' : // NotPrecedesEqual;
                        match = "\u2AAF\u0338";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1901:
                switch (current) {
                    case 'l' :
                        state = 1902;
                        return true;
                }
                break;
            case 1902:
                switch (current) {
                    case 'a' :
                        state = 1903;
                        return true;
                }
                break;
            case 1903:
                switch (current) {
                    case 'n' :
                        state = 1904;
                        return true;
                }
                break;
            case 1904:
                switch (current) {
                    case 't' :
                        state = 1905;
                        return true;
                }
                break;
            case 1905:
                switch (current) {
                    case 'E' :
                        state = 1906;
                        return true;
                }
                break;
            case 1906:
                switch (current) {
                    case 'q' :
                        state = 1907;
                        return true;
                }
                break;
            case 1907:
                switch (current) {
                    case 'u' :
                        state = 1908;
                        return true;
                }
                break;
            case 1908:
                switch (current) {
                    case 'a' :
                        state = 1909;
                        return true;
                }
                break;
            case 1909:
                switch (current) {
                    case 'l' :
                        state = 1910;
                        return true;
                }
                break;
            case 1910:
                switch (current) {
                    case ';' : // NotPrecedesSlantEqual;
                        match = "\u22E0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1911:
                switch (current) {
                    case 'e' :
                        state = 1912;
                        return true;
                    case 'i' :
                        state = 1925;
                        return true;
                }
                break;
            case 1912:
                switch (current) {
                    case 'v' :
                        state = 1913;
                        return true;
                }
                break;
            case 1913:
                switch (current) {
                    case 'e' :
                        state = 1914;
                        return true;
                }
                break;
            case 1914:
                switch (current) {
                    case 'r' :
                        state = 1915;
                        return true;
                }
                break;
            case 1915:
                switch (current) {
                    case 's' :
                        state = 1916;
                        return true;
                }
                break;
            case 1916:
                switch (current) {
                    case 'e' :
                        state = 1917;
                        return true;
                }
                break;
            case 1917:
                switch (current) {
                    case 'E' :
                        state = 1918;
                        return true;
                }
                break;
            case 1918:
                switch (current) {
                    case 'l' :
                        state = 1919;
                        return true;
                }
                break;
            case 1919:
                switch (current) {
                    case 'e' :
                        state = 1920;
                        return true;
                }
                break;
            case 1920:
                switch (current) {
                    case 'm' :
                        state = 1921;
                        return true;
                }
                break;
            case 1921:
                switch (current) {
                    case 'e' :
                        state = 1922;
                        return true;
                }
                break;
            case 1922:
                switch (current) {
                    case 'n' :
                        state = 1923;
                        return true;
                }
                break;
            case 1923:
                switch (current) {
                    case 't' :
                        state = 1924;
                        return true;
                }
                break;
            case 1924:
                switch (current) {
                    case ';' : // NotReverseElement;
                        match = "\u220C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1925:
                switch (current) {
                    case 'g' :
                        state = 1926;
                        return true;
                }
                break;
            case 1926:
                switch (current) {
                    case 'h' :
                        state = 1927;
                        return true;
                }
                break;
            case 1927:
                switch (current) {
                    case 't' :
                        state = 1928;
                        return true;
                }
                break;
            case 1928:
                switch (current) {
                    case 'T' :
                        state = 1929;
                        return true;
                }
                break;
            case 1929:
                switch (current) {
                    case 'r' :
                        state = 1930;
                        return true;
                }
                break;
            case 1930:
                switch (current) {
                    case 'i' :
                        state = 1931;
                        return true;
                }
                break;
            case 1931:
                switch (current) {
                    case 'a' :
                        state = 1932;
                        return true;
                }
                break;
            case 1932:
                switch (current) {
                    case 'n' :
                        state = 1933;
                        return true;
                }
                break;
            case 1933:
                switch (current) {
                    case 'g' :
                        state = 1934;
                        return true;
                }
                break;
            case 1934:
                switch (current) {
                    case 'l' :
                        state = 1935;
                        return true;
                }
                break;
            case 1935:
                switch (current) {
                    case 'e' :
                        state = 1936;
                        return true;
                }
                break;
            case 1936:
                switch (current) {
                    case ';' : // NotRightTriangle;
                        match = "\u22EB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1937;
                        return true;
                    case 'E' :
                        state = 1940;
                        return true;
                }
                break;
            case 1937:
                switch (current) {
                    case 'a' :
                        state = 1938;
                        return true;
                }
                break;
            case 1938:
                switch (current) {
                    case 'r' :
                        state = 1939;
                        return true;
                }
                break;
            case 1939:
                switch (current) {
                    case ';' : // NotRightTriangleBar;
                        match = "\u29D0\u0338";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1940:
                switch (current) {
                    case 'q' :
                        state = 1941;
                        return true;
                }
                break;
            case 1941:
                switch (current) {
                    case 'u' :
                        state = 1942;
                        return true;
                }
                break;
            case 1942:
                switch (current) {
                    case 'a' :
                        state = 1943;
                        return true;
                }
                break;
            case 1943:
                switch (current) {
                    case 'l' :
                        state = 1944;
                        return true;
                }
                break;
            case 1944:
                switch (current) {
                    case ';' : // NotRightTriangleEqual;
                        match = "\u22ED";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1945:
                switch (current) {
                    case 'q' :
                        state = 1946;
                        return true;
                    case 'u' :
                        state = 1973;
                        return true;
                }
                break;
            case 1946:
                switch (current) {
                    case 'u' :
                        state = 1947;
                        return true;
                }
                break;
            case 1947:
                switch (current) {
                    case 'a' :
                        state = 1948;
                        return true;
                }
                break;
            case 1948:
                switch (current) {
                    case 'r' :
                        state = 1949;
                        return true;
                }
                break;
            case 1949:
                switch (current) {
                    case 'e' :
                        state = 1950;
                        return true;
                }
                break;
            case 1950:
                switch (current) {
                    case 'S' :
                        state = 1951;
                        return true;
                }
                break;
            case 1951:
                switch (current) {
                    case 'u' :
                        state = 1952;
                        return true;
                }
                break;
            case 1952:
                switch (current) {
                    case 'b' :
                        state = 1953;
                        return true;
                    case 'p' :
                        state = 1962;
                        return true;
                }
                break;
            case 1953:
                switch (current) {
                    case 's' :
                        state = 1954;
                        return true;
                }
                break;
            case 1954:
                switch (current) {
                    case 'e' :
                        state = 1955;
                        return true;
                }
                break;
            case 1955:
                switch (current) {
                    case 't' :
                        state = 1956;
                        return true;
                }
                break;
            case 1956:
                switch (current) {
                    case ';' : // NotSquareSubset;
                        match = "\u228F\u0338";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1957;
                        return true;
                }
                break;
            case 1957:
                switch (current) {
                    case 'q' :
                        state = 1958;
                        return true;
                }
                break;
            case 1958:
                switch (current) {
                    case 'u' :
                        state = 1959;
                        return true;
                }
                break;
            case 1959:
                switch (current) {
                    case 'a' :
                        state = 1960;
                        return true;
                }
                break;
            case 1960:
                switch (current) {
                    case 'l' :
                        state = 1961;
                        return true;
                }
                break;
            case 1961:
                switch (current) {
                    case ';' : // NotSquareSubsetEqual;
                        match = "\u22E2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1962:
                switch (current) {
                    case 'e' :
                        state = 1963;
                        return true;
                }
                break;
            case 1963:
                switch (current) {
                    case 'r' :
                        state = 1964;
                        return true;
                }
                break;
            case 1964:
                switch (current) {
                    case 's' :
                        state = 1965;
                        return true;
                }
                break;
            case 1965:
                switch (current) {
                    case 'e' :
                        state = 1966;
                        return true;
                }
                break;
            case 1966:
                switch (current) {
                    case 't' :
                        state = 1967;
                        return true;
                }
                break;
            case 1967:
                switch (current) {
                    case ';' : // NotSquareSuperset;
                        match = "\u2290\u0338";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1968;
                        return true;
                }
                break;
            case 1968:
                switch (current) {
                    case 'q' :
                        state = 1969;
                        return true;
                }
                break;
            case 1969:
                switch (current) {
                    case 'u' :
                        state = 1970;
                        return true;
                }
                break;
            case 1970:
                switch (current) {
                    case 'a' :
                        state = 1971;
                        return true;
                }
                break;
            case 1971:
                switch (current) {
                    case 'l' :
                        state = 1972;
                        return true;
                }
                break;
            case 1972:
                switch (current) {
                    case ';' : // NotSquareSupersetEqual;
                        match = "\u22E3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1973:
                switch (current) {
                    case 'b' :
                        state = 1974;
                        return true;
                    case 'c' :
                        state = 1983;
                        return true;
                    case 'p' :
                        state = 2009;
                        return true;
                }
                break;
            case 1974:
                switch (current) {
                    case 's' :
                        state = 1975;
                        return true;
                }
                break;
            case 1975:
                switch (current) {
                    case 'e' :
                        state = 1976;
                        return true;
                }
                break;
            case 1976:
                switch (current) {
                    case 't' :
                        state = 1977;
                        return true;
                }
                break;
            case 1977:
                switch (current) {
                    case ';' : // NotSubset;
                        match = "\u2282\u20D2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1978;
                        return true;
                }
                break;
            case 1978:
                switch (current) {
                    case 'q' :
                        state = 1979;
                        return true;
                }
                break;
            case 1979:
                switch (current) {
                    case 'u' :
                        state = 1980;
                        return true;
                }
                break;
            case 1980:
                switch (current) {
                    case 'a' :
                        state = 1981;
                        return true;
                }
                break;
            case 1981:
                switch (current) {
                    case 'l' :
                        state = 1982;
                        return true;
                }
                break;
            case 1982:
                switch (current) {
                    case ';' : // NotSubsetEqual;
                        match = "\u2288";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1983:
                switch (current) {
                    case 'c' :
                        state = 1984;
                        return true;
                }
                break;
            case 1984:
                switch (current) {
                    case 'e' :
                        state = 1985;
                        return true;
                }
                break;
            case 1985:
                switch (current) {
                    case 'e' :
                        state = 1986;
                        return true;
                }
                break;
            case 1986:
                switch (current) {
                    case 'd' :
                        state = 1987;
                        return true;
                }
                break;
            case 1987:
                switch (current) {
                    case 's' :
                        state = 1988;
                        return true;
                }
                break;
            case 1988:
                switch (current) {
                    case ';' : // NotSucceeds;
                        match = "\u2281";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1989;
                        return true;
                    case 'S' :
                        state = 1994;
                        return true;
                    case 'T' :
                        state = 2004;
                        return true;
                }
                break;
            case 1989:
                switch (current) {
                    case 'q' :
                        state = 1990;
                        return true;
                }
                break;
            case 1990:
                switch (current) {
                    case 'u' :
                        state = 1991;
                        return true;
                }
                break;
            case 1991:
                switch (current) {
                    case 'a' :
                        state = 1992;
                        return true;
                }
                break;
            case 1992:
                switch (current) {
                    case 'l' :
                        state = 1993;
                        return true;
                }
                break;
            case 1993:
                switch (current) {
                    case ';' : // NotSucceedsEqual;
                        match = "\u2AB0\u0338";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 1994:
                switch (current) {
                    case 'l' :
                        state = 1995;
                        return true;
                }
                break;
            case 1995:
                switch (current) {
                    case 'a' :
                        state = 1996;
                        return true;
                }
                break;
            case 1996:
                switch (current) {
                    case 'n' :
                        state = 1997;
                        return true;
                }
                break;
            case 1997:
                switch (current) {
                    case 't' :
                        state = 1998;
                        return true;
                }
                break;
            case 1998:
                switch (current) {
                    case 'E' :
                        state = 1999;
                        return true;
                }
                break;
            case 1999:
                switch (current) {
                    case 'q' :
                        state = 2000;
                        return true;
                }
                break;
        }
        return false;
    }

    private boolean parse3(final int current) {
        consumedCount++;
        switch (state) {
            case 2000:
                switch (current) {
                    case 'u' :
                        state = 2001;
                        return true;
                }
                break;
            case 2001:
                switch (current) {
                    case 'a' :
                        state = 2002;
                        return true;
                }
                break;
            case 2002:
                switch (current) {
                    case 'l' :
                        state = 2003;
                        return true;
                }
                break;
            case 2003:
                switch (current) {
                    case ';' : // NotSucceedsSlantEqual;
                        match = "\u22E1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2004:
                switch (current) {
                    case 'i' :
                        state = 2005;
                        return true;
                }
                break;
            case 2005:
                switch (current) {
                    case 'l' :
                        state = 2006;
                        return true;
                }
                break;
            case 2006:
                switch (current) {
                    case 'd' :
                        state = 2007;
                        return true;
                }
                break;
            case 2007:
                switch (current) {
                    case 'e' :
                        state = 2008;
                        return true;
                }
                break;
            case 2008:
                switch (current) {
                    case ';' : // NotSucceedsTilde;
                        match = "\u227F\u0338";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2009:
                switch (current) {
                    case 'e' :
                        state = 2010;
                        return true;
                }
                break;
            case 2010:
                switch (current) {
                    case 'r' :
                        state = 2011;
                        return true;
                }
                break;
            case 2011:
                switch (current) {
                    case 's' :
                        state = 2012;
                        return true;
                }
                break;
            case 2012:
                switch (current) {
                    case 'e' :
                        state = 2013;
                        return true;
                }
                break;
            case 2013:
                switch (current) {
                    case 't' :
                        state = 2014;
                        return true;
                }
                break;
            case 2014:
                switch (current) {
                    case ';' : // NotSuperset;
                        match = "\u2283\u20D2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2015;
                        return true;
                }
                break;
            case 2015:
                switch (current) {
                    case 'q' :
                        state = 2016;
                        return true;
                }
                break;
            case 2016:
                switch (current) {
                    case 'u' :
                        state = 2017;
                        return true;
                }
                break;
            case 2017:
                switch (current) {
                    case 'a' :
                        state = 2018;
                        return true;
                }
                break;
            case 2018:
                switch (current) {
                    case 'l' :
                        state = 2019;
                        return true;
                }
                break;
            case 2019:
                switch (current) {
                    case ';' : // NotSupersetEqual;
                        match = "\u2289";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2020:
                switch (current) {
                    case 'i' :
                        state = 2021;
                        return true;
                }
                break;
            case 2021:
                switch (current) {
                    case 'l' :
                        state = 2022;
                        return true;
                }
                break;
            case 2022:
                switch (current) {
                    case 'd' :
                        state = 2023;
                        return true;
                }
                break;
            case 2023:
                switch (current) {
                    case 'e' :
                        state = 2024;
                        return true;
                }
                break;
            case 2024:
                switch (current) {
                    case ';' : // NotTilde;
                        match = "\u2241";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2025;
                        return true;
                    case 'F' :
                        state = 2030;
                        return true;
                    case 'T' :
                        state = 2039;
                        return true;
                }
                break;
            case 2025:
                switch (current) {
                    case 'q' :
                        state = 2026;
                        return true;
                }
                break;
            case 2026:
                switch (current) {
                    case 'u' :
                        state = 2027;
                        return true;
                }
                break;
            case 2027:
                switch (current) {
                    case 'a' :
                        state = 2028;
                        return true;
                }
                break;
            case 2028:
                switch (current) {
                    case 'l' :
                        state = 2029;
                        return true;
                }
                break;
            case 2029:
                switch (current) {
                    case ';' : // NotTildeEqual;
                        match = "\u2244";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2030:
                switch (current) {
                    case 'u' :
                        state = 2031;
                        return true;
                }
                break;
            case 2031:
                switch (current) {
                    case 'l' :
                        state = 2032;
                        return true;
                }
                break;
            case 2032:
                switch (current) {
                    case 'l' :
                        state = 2033;
                        return true;
                }
                break;
            case 2033:
                switch (current) {
                    case 'E' :
                        state = 2034;
                        return true;
                }
                break;
            case 2034:
                switch (current) {
                    case 'q' :
                        state = 2035;
                        return true;
                }
                break;
            case 2035:
                switch (current) {
                    case 'u' :
                        state = 2036;
                        return true;
                }
                break;
            case 2036:
                switch (current) {
                    case 'a' :
                        state = 2037;
                        return true;
                }
                break;
            case 2037:
                switch (current) {
                    case 'l' :
                        state = 2038;
                        return true;
                }
                break;
            case 2038:
                switch (current) {
                    case ';' : // NotTildeFullEqual;
                        match = "\u2247";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2039:
                switch (current) {
                    case 'i' :
                        state = 2040;
                        return true;
                }
                break;
            case 2040:
                switch (current) {
                    case 'l' :
                        state = 2041;
                        return true;
                }
                break;
            case 2041:
                switch (current) {
                    case 'd' :
                        state = 2042;
                        return true;
                }
                break;
            case 2042:
                switch (current) {
                    case 'e' :
                        state = 2043;
                        return true;
                }
                break;
            case 2043:
                switch (current) {
                    case ';' : // NotTildeTilde;
                        match = "\u2249";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2044:
                switch (current) {
                    case 'e' :
                        state = 2045;
                        return true;
                }
                break;
            case 2045:
                switch (current) {
                    case 'r' :
                        state = 2046;
                        return true;
                }
                break;
            case 2046:
                switch (current) {
                    case 't' :
                        state = 2047;
                        return true;
                }
                break;
            case 2047:
                switch (current) {
                    case 'i' :
                        state = 2048;
                        return true;
                }
                break;
            case 2048:
                switch (current) {
                    case 'c' :
                        state = 2049;
                        return true;
                }
                break;
            case 2049:
                switch (current) {
                    case 'a' :
                        state = 2050;
                        return true;
                }
                break;
            case 2050:
                switch (current) {
                    case 'l' :
                        state = 2051;
                        return true;
                }
                break;
            case 2051:
                switch (current) {
                    case 'B' :
                        state = 2052;
                        return true;
                }
                break;
            case 2052:
                switch (current) {
                    case 'a' :
                        state = 2053;
                        return true;
                }
                break;
            case 2053:
                switch (current) {
                    case 'r' :
                        state = 2054;
                        return true;
                }
                break;
            case 2054:
                switch (current) {
                    case ';' : // NotVerticalBar;
                        match = "\u2224";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2055:
                switch (current) {
                    case 'c' :
                        state = 2056;
                        return true;
                }
                break;
            case 2056:
                switch (current) {
                    case 'r' :
                        state = 2057;
                        return true;
                }
                break;
            case 2057:
                switch (current) {
                    case ';' : // Nscr;
                        match = "\uD835\uDCA9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2058:
                switch (current) {
                    case 'i' :
                        state = 2059;
                        return true;
                }
                break;
            case 2059:
                switch (current) {
                    case 'l' :
                        state = 2060;
                        return true;
                }
                break;
            case 2060:
                switch (current) {
                    case 'd' :
                        state = 2061;
                        return true;
                }
                break;
            case 2061:
                switch (current) {
                    case 'e' : // Ntilde
                        match = "\u00D1";
                        matchLength = consumedCount;
                        state = 2062;
                        return true;
                }
                break;
            case 2062:
                switch (current) {
                    case ';' : // Ntilde;
                        match = "\u00D1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2063:
                switch (current) {
                    case ';' : // Nu;
                        match = "\u039D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2064:
                switch (current) {
                    case 'E' :
                        state = 2065;
                        return true;
                    case 'a' :
                        state = 2069;
                        return true;
                    case 'c' :
                        state = 2074;
                        return true;
                    case 'd' :
                        state = 2079;
                        return true;
                    case 'f' :
                        state = 2084;
                        return true;
                    case 'g' :
                        state = 2086;
                        return true;
                    case 'm' :
                        state = 2091;
                        return true;
                    case 'o' :
                        state = 2103;
                        return true;
                    case 'p' :
                        state = 2106;
                        return true;
                    case 'r' :
                        state = 2130;
                        return true;
                    case 's' :
                        state = 2131;
                        return true;
                    case 't' :
                        state = 2138;
                        return true;
                    case 'u' :
                        state = 2146;
                        return true;
                    case 'v' :
                        state = 2149;
                        return true;
                }
                break;
            case 2065:
                switch (current) {
                    case 'l' :
                        state = 2066;
                        return true;
                }
                break;
            case 2066:
                switch (current) {
                    case 'i' :
                        state = 2067;
                        return true;
                }
                break;
            case 2067:
                switch (current) {
                    case 'g' :
                        state = 2068;
                        return true;
                }
                break;
            case 2068:
                switch (current) {
                    case ';' : // OElig;
                        match = "\u0152";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2069:
                switch (current) {
                    case 'c' :
                        state = 2070;
                        return true;
                }
                break;
            case 2070:
                switch (current) {
                    case 'u' :
                        state = 2071;
                        return true;
                }
                break;
            case 2071:
                switch (current) {
                    case 't' :
                        state = 2072;
                        return true;
                }
                break;
            case 2072:
                switch (current) {
                    case 'e' : // Oacute
                        match = "\u00D3";
                        matchLength = consumedCount;
                        state = 2073;
                        return true;
                }
                break;
            case 2073:
                switch (current) {
                    case ';' : // Oacute;
                        match = "\u00D3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2074:
                switch (current) {
                    case 'i' :
                        state = 2075;
                        return true;
                    case 'y' :
                        state = 2078;
                        return true;
                }
                break;
            case 2075:
                switch (current) {
                    case 'r' :
                        state = 2076;
                        return true;
                }
                break;
            case 2076:
                switch (current) {
                    case 'c' : // Ocirc
                        match = "\u00D4";
                        matchLength = consumedCount;
                        state = 2077;
                        return true;
                }
                break;
            case 2077:
                switch (current) {
                    case ';' : // Ocirc;
                        match = "\u00D4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2078:
                switch (current) {
                    case ';' : // Ocy;
                        match = "\u041E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2079:
                switch (current) {
                    case 'b' :
                        state = 2080;
                        return true;
                }
                break;
            case 2080:
                switch (current) {
                    case 'l' :
                        state = 2081;
                        return true;
                }
                break;
            case 2081:
                switch (current) {
                    case 'a' :
                        state = 2082;
                        return true;
                }
                break;
            case 2082:
                switch (current) {
                    case 'c' :
                        state = 2083;
                        return true;
                }
                break;
            case 2083:
                switch (current) {
                    case ';' : // Odblac;
                        match = "\u0150";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2084:
                switch (current) {
                    case 'r' :
                        state = 2085;
                        return true;
                }
                break;
            case 2085:
                switch (current) {
                    case ';' : // Ofr;
                        match = "\uD835\uDD12";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2086:
                switch (current) {
                    case 'r' :
                        state = 2087;
                        return true;
                }
                break;
            case 2087:
                switch (current) {
                    case 'a' :
                        state = 2088;
                        return true;
                }
                break;
            case 2088:
                switch (current) {
                    case 'v' :
                        state = 2089;
                        return true;
                }
                break;
            case 2089:
                switch (current) {
                    case 'e' : // Ograve
                        match = "\u00D2";
                        matchLength = consumedCount;
                        state = 2090;
                        return true;
                }
                break;
            case 2090:
                switch (current) {
                    case ';' : // Ograve;
                        match = "\u00D2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2091:
                switch (current) {
                    case 'a' :
                        state = 2092;
                        return true;
                    case 'e' :
                        state = 2095;
                        return true;
                    case 'i' :
                        state = 2098;
                        return true;
                }
                break;
            case 2092:
                switch (current) {
                    case 'c' :
                        state = 2093;
                        return true;
                }
                break;
            case 2093:
                switch (current) {
                    case 'r' :
                        state = 2094;
                        return true;
                }
                break;
            case 2094:
                switch (current) {
                    case ';' : // Omacr;
                        match = "\u014C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2095:
                switch (current) {
                    case 'g' :
                        state = 2096;
                        return true;
                }
                break;
            case 2096:
                switch (current) {
                    case 'a' :
                        state = 2097;
                        return true;
                }
                break;
            case 2097:
                switch (current) {
                    case ';' : // Omega;
                        match = "\u03A9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2098:
                switch (current) {
                    case 'c' :
                        state = 2099;
                        return true;
                }
                break;
            case 2099:
                switch (current) {
                    case 'r' :
                        state = 2100;
                        return true;
                }
                break;
            case 2100:
                switch (current) {
                    case 'o' :
                        state = 2101;
                        return true;
                }
                break;
            case 2101:
                switch (current) {
                    case 'n' :
                        state = 2102;
                        return true;
                }
                break;
            case 2102:
                switch (current) {
                    case ';' : // Omicron;
                        match = "\u039F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2103:
                switch (current) {
                    case 'p' :
                        state = 2104;
                        return true;
                }
                break;
            case 2104:
                switch (current) {
                    case 'f' :
                        state = 2105;
                        return true;
                }
                break;
            case 2105:
                switch (current) {
                    case ';' : // Oopf;
                        match = "\uD835\uDD46";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2106:
                switch (current) {
                    case 'e' :
                        state = 2107;
                        return true;
                }
                break;
            case 2107:
                switch (current) {
                    case 'n' :
                        state = 2108;
                        return true;
                }
                break;
            case 2108:
                switch (current) {
                    case 'C' :
                        state = 2109;
                        return true;
                }
                break;
            case 2109:
                switch (current) {
                    case 'u' :
                        state = 2110;
                        return true;
                }
                break;
            case 2110:
                switch (current) {
                    case 'r' :
                        state = 2111;
                        return true;
                }
                break;
            case 2111:
                switch (current) {
                    case 'l' :
                        state = 2112;
                        return true;
                }
                break;
            case 2112:
                switch (current) {
                    case 'y' :
                        state = 2113;
                        return true;
                }
                break;
            case 2113:
                switch (current) {
                    case 'D' :
                        state = 2114;
                        return true;
                    case 'Q' :
                        state = 2125;
                        return true;
                }
                break;
            case 2114:
                switch (current) {
                    case 'o' :
                        state = 2115;
                        return true;
                }
                break;
            case 2115:
                switch (current) {
                    case 'u' :
                        state = 2116;
                        return true;
                }
                break;
            case 2116:
                switch (current) {
                    case 'b' :
                        state = 2117;
                        return true;
                }
                break;
            case 2117:
                switch (current) {
                    case 'l' :
                        state = 2118;
                        return true;
                }
                break;
            case 2118:
                switch (current) {
                    case 'e' :
                        state = 2119;
                        return true;
                }
                break;
            case 2119:
                switch (current) {
                    case 'Q' :
                        state = 2120;
                        return true;
                }
                break;
            case 2120:
                switch (current) {
                    case 'u' :
                        state = 2121;
                        return true;
                }
                break;
            case 2121:
                switch (current) {
                    case 'o' :
                        state = 2122;
                        return true;
                }
                break;
            case 2122:
                switch (current) {
                    case 't' :
                        state = 2123;
                        return true;
                }
                break;
            case 2123:
                switch (current) {
                    case 'e' :
                        state = 2124;
                        return true;
                }
                break;
            case 2124:
                switch (current) {
                    case ';' : // OpenCurlyDoubleQuote;
                        match = "\u201C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2125:
                switch (current) {
                    case 'u' :
                        state = 2126;
                        return true;
                }
                break;
            case 2126:
                switch (current) {
                    case 'o' :
                        state = 2127;
                        return true;
                }
                break;
            case 2127:
                switch (current) {
                    case 't' :
                        state = 2128;
                        return true;
                }
                break;
            case 2128:
                switch (current) {
                    case 'e' :
                        state = 2129;
                        return true;
                }
                break;
            case 2129:
                switch (current) {
                    case ';' : // OpenCurlyQuote;
                        match = "\u2018";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2130:
                switch (current) {
                    case ';' : // Or;
                        match = "\u2A54";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2131:
                switch (current) {
                    case 'c' :
                        state = 2132;
                        return true;
                    case 'l' :
                        state = 2134;
                        return true;
                }
                break;
            case 2132:
                switch (current) {
                    case 'r' :
                        state = 2133;
                        return true;
                }
                break;
            case 2133:
                switch (current) {
                    case ';' : // Oscr;
                        match = "\uD835\uDCAA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2134:
                switch (current) {
                    case 'a' :
                        state = 2135;
                        return true;
                }
                break;
            case 2135:
                switch (current) {
                    case 's' :
                        state = 2136;
                        return true;
                }
                break;
            case 2136:
                switch (current) {
                    case 'h' : // Oslash
                        match = "\u00D8";
                        matchLength = consumedCount;
                        state = 2137;
                        return true;
                }
                break;
            case 2137:
                switch (current) {
                    case ';' : // Oslash;
                        match = "\u00D8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2138:
                switch (current) {
                    case 'i' :
                        state = 2139;
                        return true;
                }
                break;
            case 2139:
                switch (current) {
                    case 'l' :
                        state = 2140;
                        return true;
                    case 'm' :
                        state = 2143;
                        return true;
                }
                break;
            case 2140:
                switch (current) {
                    case 'd' :
                        state = 2141;
                        return true;
                }
                break;
            case 2141:
                switch (current) {
                    case 'e' : // Otilde
                        match = "\u00D5";
                        matchLength = consumedCount;
                        state = 2142;
                        return true;
                }
                break;
            case 2142:
                switch (current) {
                    case ';' : // Otilde;
                        match = "\u00D5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2143:
                switch (current) {
                    case 'e' :
                        state = 2144;
                        return true;
                }
                break;
            case 2144:
                switch (current) {
                    case 's' :
                        state = 2145;
                        return true;
                }
                break;
            case 2145:
                switch (current) {
                    case ';' : // Otimes;
                        match = "\u2A37";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2146:
                switch (current) {
                    case 'm' :
                        state = 2147;
                        return true;
                }
                break;
            case 2147:
                switch (current) {
                    case 'l' : // Ouml
                        match = "\u00D6";
                        matchLength = consumedCount;
                        state = 2148;
                        return true;
                }
                break;
            case 2148:
                switch (current) {
                    case ';' : // Ouml;
                        match = "\u00D6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2149:
                switch (current) {
                    case 'e' :
                        state = 2150;
                        return true;
                }
                break;
            case 2150:
                switch (current) {
                    case 'r' :
                        state = 2151;
                        return true;
                }
                break;
            case 2151:
                switch (current) {
                    case 'B' :
                        state = 2152;
                        return true;
                    case 'P' :
                        state = 2162;
                        return true;
                }
                break;
            case 2152:
                switch (current) {
                    case 'a' :
                        state = 2153;
                        return true;
                    case 'r' :
                        state = 2155;
                        return true;
                }
                break;
            case 2153:
                switch (current) {
                    case 'r' :
                        state = 2154;
                        return true;
                }
                break;
            case 2154:
                switch (current) {
                    case ';' : // OverBar;
                        match = "\u203E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2155:
                switch (current) {
                    case 'a' :
                        state = 2156;
                        return true;
                }
                break;
            case 2156:
                switch (current) {
                    case 'c' :
                        state = 2157;
                        return true;
                }
                break;
            case 2157:
                switch (current) {
                    case 'e' :
                        state = 2158;
                        return true;
                    case 'k' :
                        state = 2159;
                        return true;
                }
                break;
            case 2158:
                switch (current) {
                    case ';' : // OverBrace;
                        match = "\u23DE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2159:
                switch (current) {
                    case 'e' :
                        state = 2160;
                        return true;
                }
                break;
            case 2160:
                switch (current) {
                    case 't' :
                        state = 2161;
                        return true;
                }
                break;
            case 2161:
                switch (current) {
                    case ';' : // OverBracket;
                        match = "\u23B4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2162:
                switch (current) {
                    case 'a' :
                        state = 2163;
                        return true;
                }
                break;
            case 2163:
                switch (current) {
                    case 'r' :
                        state = 2164;
                        return true;
                }
                break;
            case 2164:
                switch (current) {
                    case 'e' :
                        state = 2165;
                        return true;
                }
                break;
            case 2165:
                switch (current) {
                    case 'n' :
                        state = 2166;
                        return true;
                }
                break;
            case 2166:
                switch (current) {
                    case 't' :
                        state = 2167;
                        return true;
                }
                break;
            case 2167:
                switch (current) {
                    case 'h' :
                        state = 2168;
                        return true;
                }
                break;
            case 2168:
                switch (current) {
                    case 'e' :
                        state = 2169;
                        return true;
                }
                break;
            case 2169:
                switch (current) {
                    case 's' :
                        state = 2170;
                        return true;
                }
                break;
            case 2170:
                switch (current) {
                    case 'i' :
                        state = 2171;
                        return true;
                }
                break;
            case 2171:
                switch (current) {
                    case 's' :
                        state = 2172;
                        return true;
                }
                break;
            case 2172:
                switch (current) {
                    case ';' : // OverParenthesis;
                        match = "\u23DC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2173:
                switch (current) {
                    case 'a' :
                        state = 2174;
                        return true;
                    case 'c' :
                        state = 2181;
                        return true;
                    case 'f' :
                        state = 2183;
                        return true;
                    case 'h' :
                        state = 2185;
                        return true;
                    case 'i' :
                        state = 2187;
                        return true;
                    case 'l' :
                        state = 2188;
                        return true;
                    case 'o' :
                        state = 2196;
                        return true;
                    case 'r' :
                        state = 2210;
                        return true;
                    case 's' :
                        state = 2254;
                        return true;
                }
                break;
            case 2174:
                switch (current) {
                    case 'r' :
                        state = 2175;
                        return true;
                }
                break;
            case 2175:
                switch (current) {
                    case 't' :
                        state = 2176;
                        return true;
                }
                break;
            case 2176:
                switch (current) {
                    case 'i' :
                        state = 2177;
                        return true;
                }
                break;
            case 2177:
                switch (current) {
                    case 'a' :
                        state = 2178;
                        return true;
                }
                break;
            case 2178:
                switch (current) {
                    case 'l' :
                        state = 2179;
                        return true;
                }
                break;
            case 2179:
                switch (current) {
                    case 'D' :
                        state = 2180;
                        return true;
                }
                break;
            case 2180:
                switch (current) {
                    case ';' : // PartialD;
                        match = "\u2202";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2181:
                switch (current) {
                    case 'y' :
                        state = 2182;
                        return true;
                }
                break;
            case 2182:
                switch (current) {
                    case ';' : // Pcy;
                        match = "\u041F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2183:
                switch (current) {
                    case 'r' :
                        state = 2184;
                        return true;
                }
                break;
            case 2184:
                switch (current) {
                    case ';' : // Pfr;
                        match = "\uD835\uDD13";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2185:
                switch (current) {
                    case 'i' :
                        state = 2186;
                        return true;
                }
                break;
            case 2186:
                switch (current) {
                    case ';' : // Phi;
                        match = "\u03A6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2187:
                switch (current) {
                    case ';' : // Pi;
                        match = "\u03A0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2188:
                switch (current) {
                    case 'u' :
                        state = 2189;
                        return true;
                }
                break;
            case 2189:
                switch (current) {
                    case 's' :
                        state = 2190;
                        return true;
                }
                break;
            case 2190:
                switch (current) {
                    case 'M' :
                        state = 2191;
                        return true;
                }
                break;
            case 2191:
                switch (current) {
                    case 'i' :
                        state = 2192;
                        return true;
                }
                break;
            case 2192:
                switch (current) {
                    case 'n' :
                        state = 2193;
                        return true;
                }
                break;
            case 2193:
                switch (current) {
                    case 'u' :
                        state = 2194;
                        return true;
                }
                break;
            case 2194:
                switch (current) {
                    case 's' :
                        state = 2195;
                        return true;
                }
                break;
            case 2195:
                switch (current) {
                    case ';' : // PlusMinus;
                        match = "\u00B1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2196:
                switch (current) {
                    case 'i' :
                        state = 2197;
                        return true;
                    case 'p' :
                        state = 2208;
                        return true;
                }
                break;
            case 2197:
                switch (current) {
                    case 'n' :
                        state = 2198;
                        return true;
                }
                break;
            case 2198:
                switch (current) {
                    case 'c' :
                        state = 2199;
                        return true;
                }
                break;
            case 2199:
                switch (current) {
                    case 'a' :
                        state = 2200;
                        return true;
                }
                break;
            case 2200:
                switch (current) {
                    case 'r' :
                        state = 2201;
                        return true;
                }
                break;
            case 2201:
                switch (current) {
                    case 'e' :
                        state = 2202;
                        return true;
                }
                break;
            case 2202:
                switch (current) {
                    case 'p' :
                        state = 2203;
                        return true;
                }
                break;
            case 2203:
                switch (current) {
                    case 'l' :
                        state = 2204;
                        return true;
                }
                break;
            case 2204:
                switch (current) {
                    case 'a' :
                        state = 2205;
                        return true;
                }
                break;
            case 2205:
                switch (current) {
                    case 'n' :
                        state = 2206;
                        return true;
                }
                break;
            case 2206:
                switch (current) {
                    case 'e' :
                        state = 2207;
                        return true;
                }
                break;
            case 2207:
                switch (current) {
                    case ';' : // Poincareplane;
                        match = "\u210C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2208:
                switch (current) {
                    case 'f' :
                        state = 2209;
                        return true;
                }
                break;
            case 2209:
                switch (current) {
                    case ';' : // Popf;
                        match = "\u2119";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2210:
                switch (current) {
                    case ';' : // Pr;
                        match = "\u2ABB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 2211;
                        return true;
                    case 'i' :
                        state = 2237;
                        return true;
                    case 'o' :
                        state = 2240;
                        return true;
                }
                break;
            case 2211:
                switch (current) {
                    case 'c' :
                        state = 2212;
                        return true;
                }
                break;
            case 2212:
                switch (current) {
                    case 'e' :
                        state = 2213;
                        return true;
                }
                break;
            case 2213:
                switch (current) {
                    case 'd' :
                        state = 2214;
                        return true;
                }
                break;
            case 2214:
                switch (current) {
                    case 'e' :
                        state = 2215;
                        return true;
                }
                break;
            case 2215:
                switch (current) {
                    case 's' :
                        state = 2216;
                        return true;
                }
                break;
            case 2216:
                switch (current) {
                    case ';' : // Precedes;
                        match = "\u227A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2217;
                        return true;
                    case 'S' :
                        state = 2222;
                        return true;
                    case 'T' :
                        state = 2232;
                        return true;
                }
                break;
            case 2217:
                switch (current) {
                    case 'q' :
                        state = 2218;
                        return true;
                }
                break;
            case 2218:
                switch (current) {
                    case 'u' :
                        state = 2219;
                        return true;
                }
                break;
            case 2219:
                switch (current) {
                    case 'a' :
                        state = 2220;
                        return true;
                }
                break;
            case 2220:
                switch (current) {
                    case 'l' :
                        state = 2221;
                        return true;
                }
                break;
            case 2221:
                switch (current) {
                    case ';' : // PrecedesEqual;
                        match = "\u2AAF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2222:
                switch (current) {
                    case 'l' :
                        state = 2223;
                        return true;
                }
                break;
            case 2223:
                switch (current) {
                    case 'a' :
                        state = 2224;
                        return true;
                }
                break;
            case 2224:
                switch (current) {
                    case 'n' :
                        state = 2225;
                        return true;
                }
                break;
            case 2225:
                switch (current) {
                    case 't' :
                        state = 2226;
                        return true;
                }
                break;
            case 2226:
                switch (current) {
                    case 'E' :
                        state = 2227;
                        return true;
                }
                break;
            case 2227:
                switch (current) {
                    case 'q' :
                        state = 2228;
                        return true;
                }
                break;
            case 2228:
                switch (current) {
                    case 'u' :
                        state = 2229;
                        return true;
                }
                break;
            case 2229:
                switch (current) {
                    case 'a' :
                        state = 2230;
                        return true;
                }
                break;
            case 2230:
                switch (current) {
                    case 'l' :
                        state = 2231;
                        return true;
                }
                break;
            case 2231:
                switch (current) {
                    case ';' : // PrecedesSlantEqual;
                        match = "\u227C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2232:
                switch (current) {
                    case 'i' :
                        state = 2233;
                        return true;
                }
                break;
            case 2233:
                switch (current) {
                    case 'l' :
                        state = 2234;
                        return true;
                }
                break;
            case 2234:
                switch (current) {
                    case 'd' :
                        state = 2235;
                        return true;
                }
                break;
            case 2235:
                switch (current) {
                    case 'e' :
                        state = 2236;
                        return true;
                }
                break;
            case 2236:
                switch (current) {
                    case ';' : // PrecedesTilde;
                        match = "\u227E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2237:
                switch (current) {
                    case 'm' :
                        state = 2238;
                        return true;
                }
                break;
            case 2238:
                switch (current) {
                    case 'e' :
                        state = 2239;
                        return true;
                }
                break;
            case 2239:
                switch (current) {
                    case ';' : // Prime;
                        match = "\u2033";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2240:
                switch (current) {
                    case 'd' :
                        state = 2241;
                        return true;
                    case 'p' :
                        state = 2245;
                        return true;
                }
                break;
            case 2241:
                switch (current) {
                    case 'u' :
                        state = 2242;
                        return true;
                }
                break;
            case 2242:
                switch (current) {
                    case 'c' :
                        state = 2243;
                        return true;
                }
                break;
            case 2243:
                switch (current) {
                    case 't' :
                        state = 2244;
                        return true;
                }
                break;
            case 2244:
                switch (current) {
                    case ';' : // Product;
                        match = "\u220F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2245:
                switch (current) {
                    case 'o' :
                        state = 2246;
                        return true;
                }
                break;
            case 2246:
                switch (current) {
                    case 'r' :
                        state = 2247;
                        return true;
                }
                break;
            case 2247:
                switch (current) {
                    case 't' :
                        state = 2248;
                        return true;
                }
                break;
            case 2248:
                switch (current) {
                    case 'i' :
                        state = 2249;
                        return true;
                }
                break;
            case 2249:
                switch (current) {
                    case 'o' :
                        state = 2250;
                        return true;
                }
                break;
            case 2250:
                switch (current) {
                    case 'n' :
                        state = 2251;
                        return true;
                }
                break;
            case 2251:
                switch (current) {
                    case ';' : // Proportion;
                        match = "\u2237";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 2252;
                        return true;
                }
                break;
            case 2252:
                switch (current) {
                    case 'l' :
                        state = 2253;
                        return true;
                }
                break;
            case 2253:
                switch (current) {
                    case ';' : // Proportional;
                        match = "\u221D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2254:
                switch (current) {
                    case 'c' :
                        state = 2255;
                        return true;
                    case 'i' :
                        state = 2257;
                        return true;
                }
                break;
            case 2255:
                switch (current) {
                    case 'r' :
                        state = 2256;
                        return true;
                }
                break;
            case 2256:
                switch (current) {
                    case ';' : // Pscr;
                        match = "\uD835\uDCAB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2257:
                switch (current) {
                    case ';' : // Psi;
                        match = "\u03A8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2258:
                switch (current) {
                    case 'U' :
                        state = 2259;
                        return true;
                    case 'f' :
                        state = 2262;
                        return true;
                    case 'o' :
                        state = 2264;
                        return true;
                    case 's' :
                        state = 2267;
                        return true;
                }
                break;
            case 2259:
                switch (current) {
                    case 'O' :
                        state = 2260;
                        return true;
                }
                break;
            case 2260:
                switch (current) {
                    case 'T' : // QUOT
                        match = "\"";
                        matchLength = consumedCount;
                        state = 2261;
                        return true;
                }
                break;
            case 2261:
                switch (current) {
                    case ';' : // QUOT;
                        match = "\"";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2262:
                switch (current) {
                    case 'r' :
                        state = 2263;
                        return true;
                }
                break;
            case 2263:
                switch (current) {
                    case ';' : // Qfr;
                        match = "\uD835\uDD14";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2264:
                switch (current) {
                    case 'p' :
                        state = 2265;
                        return true;
                }
                break;
            case 2265:
                switch (current) {
                    case 'f' :
                        state = 2266;
                        return true;
                }
                break;
            case 2266:
                switch (current) {
                    case ';' : // Qopf;
                        match = "\u211A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2267:
                switch (current) {
                    case 'c' :
                        state = 2268;
                        return true;
                }
                break;
            case 2268:
                switch (current) {
                    case 'r' :
                        state = 2269;
                        return true;
                }
                break;
            case 2269:
                switch (current) {
                    case ';' : // Qscr;
                        match = "\uD835\uDCAC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2270:
                switch (current) {
                    case 'B' :
                        state = 2271;
                        return true;
                    case 'E' :
                        state = 2275;
                        return true;
                    case 'a' :
                        state = 2277;
                        return true;
                    case 'c' :
                        state = 2288;
                        return true;
                    case 'e' :
                        state = 2298;
                        return true;
                    case 'f' :
                        state = 2334;
                        return true;
                    case 'h' :
                        state = 2336;
                        return true;
                    case 'i' :
                        state = 2338;
                        return true;
                    case 'o' :
                        state = 2488;
                        return true;
                    case 'r' :
                        state = 2501;
                        return true;
                    case 's' :
                        state = 2511;
                        return true;
                    case 'u' :
                        state = 2515;
                        return true;
                }
                break;
            case 2271:
                switch (current) {
                    case 'a' :
                        state = 2272;
                        return true;
                }
                break;
            case 2272:
                switch (current) {
                    case 'r' :
                        state = 2273;
                        return true;
                }
                break;
            case 2273:
                switch (current) {
                    case 'r' :
                        state = 2274;
                        return true;
                }
                break;
            case 2274:
                switch (current) {
                    case ';' : // RBarr;
                        match = "\u2910";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2275:
                switch (current) {
                    case 'G' : // REG
                        match = "\u00AE";
                        matchLength = consumedCount;
                        state = 2276;
                        return true;
                }
                break;
            case 2276:
                switch (current) {
                    case ';' : // REG;
                        match = "\u00AE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2277:
                switch (current) {
                    case 'c' :
                        state = 2278;
                        return true;
                    case 'n' :
                        state = 2282;
                        return true;
                    case 'r' :
                        state = 2284;
                        return true;
                }
                break;
            case 2278:
                switch (current) {
                    case 'u' :
                        state = 2279;
                        return true;
                }
                break;
            case 2279:
                switch (current) {
                    case 't' :
                        state = 2280;
                        return true;
                }
                break;
            case 2280:
                switch (current) {
                    case 'e' :
                        state = 2281;
                        return true;
                }
                break;
            case 2281:
                switch (current) {
                    case ';' : // Racute;
                        match = "\u0154";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2282:
                switch (current) {
                    case 'g' :
                        state = 2283;
                        return true;
                }
                break;
            case 2283:
                switch (current) {
                    case ';' : // Rang;
                        match = "\u27EB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2284:
                switch (current) {
                    case 'r' :
                        state = 2285;
                        return true;
                }
                break;
            case 2285:
                switch (current) {
                    case ';' : // Rarr;
                        match = "\u21A0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 't' :
                        state = 2286;
                        return true;
                }
                break;
            case 2286:
                switch (current) {
                    case 'l' :
                        state = 2287;
                        return true;
                }
                break;
            case 2287:
                switch (current) {
                    case ';' : // Rarrtl;
                        match = "\u2916";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2288:
                switch (current) {
                    case 'a' :
                        state = 2289;
                        return true;
                    case 'e' :
                        state = 2293;
                        return true;
                    case 'y' :
                        state = 2297;
                        return true;
                }
                break;
            case 2289:
                switch (current) {
                    case 'r' :
                        state = 2290;
                        return true;
                }
                break;
            case 2290:
                switch (current) {
                    case 'o' :
                        state = 2291;
                        return true;
                }
                break;
            case 2291:
                switch (current) {
                    case 'n' :
                        state = 2292;
                        return true;
                }
                break;
            case 2292:
                switch (current) {
                    case ';' : // Rcaron;
                        match = "\u0158";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2293:
                switch (current) {
                    case 'd' :
                        state = 2294;
                        return true;
                }
                break;
            case 2294:
                switch (current) {
                    case 'i' :
                        state = 2295;
                        return true;
                }
                break;
            case 2295:
                switch (current) {
                    case 'l' :
                        state = 2296;
                        return true;
                }
                break;
            case 2296:
                switch (current) {
                    case ';' : // Rcedil;
                        match = "\u0156";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2297:
                switch (current) {
                    case ';' : // Rcy;
                        match = "\u0420";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2298:
                switch (current) {
                    case ';' : // Re;
                        match = "\u211C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'v' :
                        state = 2299;
                        return true;
                }
                break;
            case 2299:
                switch (current) {
                    case 'e' :
                        state = 2300;
                        return true;
                }
                break;
            case 2300:
                switch (current) {
                    case 'r' :
                        state = 2301;
                        return true;
                }
                break;
            case 2301:
                switch (current) {
                    case 's' :
                        state = 2302;
                        return true;
                }
                break;
            case 2302:
                switch (current) {
                    case 'e' :
                        state = 2303;
                        return true;
                }
                break;
            case 2303:
                switch (current) {
                    case 'E' :
                        state = 2304;
                        return true;
                    case 'U' :
                        state = 2321;
                        return true;
                }
                break;
            case 2304:
                switch (current) {
                    case 'l' :
                        state = 2305;
                        return true;
                    case 'q' :
                        state = 2311;
                        return true;
                }
                break;
            case 2305:
                switch (current) {
                    case 'e' :
                        state = 2306;
                        return true;
                }
                break;
            case 2306:
                switch (current) {
                    case 'm' :
                        state = 2307;
                        return true;
                }
                break;
            case 2307:
                switch (current) {
                    case 'e' :
                        state = 2308;
                        return true;
                }
                break;
            case 2308:
                switch (current) {
                    case 'n' :
                        state = 2309;
                        return true;
                }
                break;
            case 2309:
                switch (current) {
                    case 't' :
                        state = 2310;
                        return true;
                }
                break;
            case 2310:
                switch (current) {
                    case ';' : // ReverseElement;
                        match = "\u220B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2311:
                switch (current) {
                    case 'u' :
                        state = 2312;
                        return true;
                }
                break;
            case 2312:
                switch (current) {
                    case 'i' :
                        state = 2313;
                        return true;
                }
                break;
            case 2313:
                switch (current) {
                    case 'l' :
                        state = 2314;
                        return true;
                }
                break;
            case 2314:
                switch (current) {
                    case 'i' :
                        state = 2315;
                        return true;
                }
                break;
            case 2315:
                switch (current) {
                    case 'b' :
                        state = 2316;
                        return true;
                }
                break;
            case 2316:
                switch (current) {
                    case 'r' :
                        state = 2317;
                        return true;
                }
                break;
            case 2317:
                switch (current) {
                    case 'i' :
                        state = 2318;
                        return true;
                }
                break;
            case 2318:
                switch (current) {
                    case 'u' :
                        state = 2319;
                        return true;
                }
                break;
            case 2319:
                switch (current) {
                    case 'm' :
                        state = 2320;
                        return true;
                }
                break;
            case 2320:
                switch (current) {
                    case ';' : // ReverseEquilibrium;
                        match = "\u21CB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2321:
                switch (current) {
                    case 'p' :
                        state = 2322;
                        return true;
                }
                break;
            case 2322:
                switch (current) {
                    case 'E' :
                        state = 2323;
                        return true;
                }
                break;
            case 2323:
                switch (current) {
                    case 'q' :
                        state = 2324;
                        return true;
                }
                break;
            case 2324:
                switch (current) {
                    case 'u' :
                        state = 2325;
                        return true;
                }
                break;
            case 2325:
                switch (current) {
                    case 'i' :
                        state = 2326;
                        return true;
                }
                break;
            case 2326:
                switch (current) {
                    case 'l' :
                        state = 2327;
                        return true;
                }
                break;
            case 2327:
                switch (current) {
                    case 'i' :
                        state = 2328;
                        return true;
                }
                break;
            case 2328:
                switch (current) {
                    case 'b' :
                        state = 2329;
                        return true;
                }
                break;
            case 2329:
                switch (current) {
                    case 'r' :
                        state = 2330;
                        return true;
                }
                break;
            case 2330:
                switch (current) {
                    case 'i' :
                        state = 2331;
                        return true;
                }
                break;
            case 2331:
                switch (current) {
                    case 'u' :
                        state = 2332;
                        return true;
                }
                break;
            case 2332:
                switch (current) {
                    case 'm' :
                        state = 2333;
                        return true;
                }
                break;
            case 2333:
                switch (current) {
                    case ';' : // ReverseUpEquilibrium;
                        match = "\u296F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2334:
                switch (current) {
                    case 'r' :
                        state = 2335;
                        return true;
                }
                break;
            case 2335:
                switch (current) {
                    case ';' : // Rfr;
                        match = "\u211C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2336:
                switch (current) {
                    case 'o' :
                        state = 2337;
                        return true;
                }
                break;
            case 2337:
                switch (current) {
                    case ';' : // Rho;
                        match = "\u03A1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2338:
                switch (current) {
                    case 'g' :
                        state = 2339;
                        return true;
                }
                break;
            case 2339:
                switch (current) {
                    case 'h' :
                        state = 2340;
                        return true;
                }
                break;
            case 2340:
                switch (current) {
                    case 't' :
                        state = 2341;
                        return true;
                }
                break;
            case 2341:
                switch (current) {
                    case 'A' :
                        state = 2342;
                        return true;
                    case 'C' :
                        state = 2370;
                        return true;
                    case 'D' :
                        state = 2377;
                        return true;
                    case 'F' :
                        state = 2410;
                        return true;
                    case 'T' :
                        state = 2415;
                        return true;
                    case 'U' :
                        state = 2444;
                        return true;
                    case 'V' :
                        state = 2474;
                        return true;
                    case 'a' :
                        state = 2483;
                        return true;
                }
                break;
            case 2342:
                switch (current) {
                    case 'n' :
                        state = 2343;
                        return true;
                    case 'r' :
                        state = 2354;
                        return true;
                }
                break;
            case 2343:
                switch (current) {
                    case 'g' :
                        state = 2344;
                        return true;
                }
                break;
            case 2344:
                switch (current) {
                    case 'l' :
                        state = 2345;
                        return true;
                }
                break;
            case 2345:
                switch (current) {
                    case 'e' :
                        state = 2346;
                        return true;
                }
                break;
            case 2346:
                switch (current) {
                    case 'B' :
                        state = 2347;
                        return true;
                }
                break;
            case 2347:
                switch (current) {
                    case 'r' :
                        state = 2348;
                        return true;
                }
                break;
            case 2348:
                switch (current) {
                    case 'a' :
                        state = 2349;
                        return true;
                }
                break;
            case 2349:
                switch (current) {
                    case 'c' :
                        state = 2350;
                        return true;
                }
                break;
            case 2350:
                switch (current) {
                    case 'k' :
                        state = 2351;
                        return true;
                }
                break;
            case 2351:
                switch (current) {
                    case 'e' :
                        state = 2352;
                        return true;
                }
                break;
            case 2352:
                switch (current) {
                    case 't' :
                        state = 2353;
                        return true;
                }
                break;
            case 2353:
                switch (current) {
                    case ';' : // RightAngleBracket;
                        match = "\u27E9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2354:
                switch (current) {
                    case 'r' :
                        state = 2355;
                        return true;
                }
                break;
            case 2355:
                switch (current) {
                    case 'o' :
                        state = 2356;
                        return true;
                }
                break;
            case 2356:
                switch (current) {
                    case 'w' :
                        state = 2357;
                        return true;
                }
                break;
            case 2357:
                switch (current) {
                    case ';' : // RightArrow;
                        match = "\u2192";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 2358;
                        return true;
                    case 'L' :
                        state = 2361;
                        return true;
                }
                break;
            case 2358:
                switch (current) {
                    case 'a' :
                        state = 2359;
                        return true;
                }
                break;
            case 2359:
                switch (current) {
                    case 'r' :
                        state = 2360;
                        return true;
                }
                break;
            case 2360:
                switch (current) {
                    case ';' : // RightArrowBar;
                        match = "\u21E5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2361:
                switch (current) {
                    case 'e' :
                        state = 2362;
                        return true;
                }
                break;
            case 2362:
                switch (current) {
                    case 'f' :
                        state = 2363;
                        return true;
                }
                break;
            case 2363:
                switch (current) {
                    case 't' :
                        state = 2364;
                        return true;
                }
                break;
            case 2364:
                switch (current) {
                    case 'A' :
                        state = 2365;
                        return true;
                }
                break;
            case 2365:
                switch (current) {
                    case 'r' :
                        state = 2366;
                        return true;
                }
                break;
            case 2366:
                switch (current) {
                    case 'r' :
                        state = 2367;
                        return true;
                }
                break;
            case 2367:
                switch (current) {
                    case 'o' :
                        state = 2368;
                        return true;
                }
                break;
            case 2368:
                switch (current) {
                    case 'w' :
                        state = 2369;
                        return true;
                }
                break;
            case 2369:
                switch (current) {
                    case ';' : // RightArrowLeftArrow;
                        match = "\u21C4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2370:
                switch (current) {
                    case 'e' :
                        state = 2371;
                        return true;
                }
                break;
            case 2371:
                switch (current) {
                    case 'i' :
                        state = 2372;
                        return true;
                }
                break;
            case 2372:
                switch (current) {
                    case 'l' :
                        state = 2373;
                        return true;
                }
                break;
            case 2373:
                switch (current) {
                    case 'i' :
                        state = 2374;
                        return true;
                }
                break;
            case 2374:
                switch (current) {
                    case 'n' :
                        state = 2375;
                        return true;
                }
                break;
            case 2375:
                switch (current) {
                    case 'g' :
                        state = 2376;
                        return true;
                }
                break;
            case 2376:
                switch (current) {
                    case ';' : // RightCeiling;
                        match = "\u2309";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2377:
                switch (current) {
                    case 'o' :
                        state = 2378;
                        return true;
                }
                break;
            case 2378:
                switch (current) {
                    case 'u' :
                        state = 2379;
                        return true;
                    case 'w' :
                        state = 2390;
                        return true;
                }
                break;
            case 2379:
                switch (current) {
                    case 'b' :
                        state = 2380;
                        return true;
                }
                break;
            case 2380:
                switch (current) {
                    case 'l' :
                        state = 2381;
                        return true;
                }
                break;
            case 2381:
                switch (current) {
                    case 'e' :
                        state = 2382;
                        return true;
                }
                break;
            case 2382:
                switch (current) {
                    case 'B' :
                        state = 2383;
                        return true;
                }
                break;
            case 2383:
                switch (current) {
                    case 'r' :
                        state = 2384;
                        return true;
                }
                break;
            case 2384:
                switch (current) {
                    case 'a' :
                        state = 2385;
                        return true;
                }
                break;
            case 2385:
                switch (current) {
                    case 'c' :
                        state = 2386;
                        return true;
                }
                break;
            case 2386:
                switch (current) {
                    case 'k' :
                        state = 2387;
                        return true;
                }
                break;
            case 2387:
                switch (current) {
                    case 'e' :
                        state = 2388;
                        return true;
                }
                break;
            case 2388:
                switch (current) {
                    case 't' :
                        state = 2389;
                        return true;
                }
                break;
            case 2389:
                switch (current) {
                    case ';' : // RightDoubleBracket;
                        match = "\u27E7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2390:
                switch (current) {
                    case 'n' :
                        state = 2391;
                        return true;
                }
                break;
            case 2391:
                switch (current) {
                    case 'T' :
                        state = 2392;
                        return true;
                    case 'V' :
                        state = 2401;
                        return true;
                }
                break;
            case 2392:
                switch (current) {
                    case 'e' :
                        state = 2393;
                        return true;
                }
                break;
            case 2393:
                switch (current) {
                    case 'e' :
                        state = 2394;
                        return true;
                }
                break;
            case 2394:
                switch (current) {
                    case 'V' :
                        state = 2395;
                        return true;
                }
                break;
            case 2395:
                switch (current) {
                    case 'e' :
                        state = 2396;
                        return true;
                }
                break;
            case 2396:
                switch (current) {
                    case 'c' :
                        state = 2397;
                        return true;
                }
                break;
            case 2397:
                switch (current) {
                    case 't' :
                        state = 2398;
                        return true;
                }
                break;
            case 2398:
                switch (current) {
                    case 'o' :
                        state = 2399;
                        return true;
                }
                break;
            case 2399:
                switch (current) {
                    case 'r' :
                        state = 2400;
                        return true;
                }
                break;
            case 2400:
                switch (current) {
                    case ';' : // RightDownTeeVector;
                        match = "\u295D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2401:
                switch (current) {
                    case 'e' :
                        state = 2402;
                        return true;
                }
                break;
            case 2402:
                switch (current) {
                    case 'c' :
                        state = 2403;
                        return true;
                }
                break;
            case 2403:
                switch (current) {
                    case 't' :
                        state = 2404;
                        return true;
                }
                break;
            case 2404:
                switch (current) {
                    case 'o' :
                        state = 2405;
                        return true;
                }
                break;
            case 2405:
                switch (current) {
                    case 'r' :
                        state = 2406;
                        return true;
                }
                break;
            case 2406:
                switch (current) {
                    case ';' : // RightDownVector;
                        match = "\u21C2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 2407;
                        return true;
                }
                break;
            case 2407:
                switch (current) {
                    case 'a' :
                        state = 2408;
                        return true;
                }
                break;
            case 2408:
                switch (current) {
                    case 'r' :
                        state = 2409;
                        return true;
                }
                break;
            case 2409:
                switch (current) {
                    case ';' : // RightDownVectorBar;
                        match = "\u2955";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2410:
                switch (current) {
                    case 'l' :
                        state = 2411;
                        return true;
                }
                break;
            case 2411:
                switch (current) {
                    case 'o' :
                        state = 2412;
                        return true;
                }
                break;
            case 2412:
                switch (current) {
                    case 'o' :
                        state = 2413;
                        return true;
                }
                break;
            case 2413:
                switch (current) {
                    case 'r' :
                        state = 2414;
                        return true;
                }
                break;
            case 2414:
                switch (current) {
                    case ';' : // RightFloor;
                        match = "\u230B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2415:
                switch (current) {
                    case 'e' :
                        state = 2416;
                        return true;
                    case 'r' :
                        state = 2429;
                        return true;
                }
                break;
            case 2416:
                switch (current) {
                    case 'e' :
                        state = 2417;
                        return true;
                }
                break;
            case 2417:
                switch (current) {
                    case ';' : // RightTee;
                        match = "\u22A2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'A' :
                        state = 2418;
                        return true;
                    case 'V' :
                        state = 2423;
                        return true;
                }
                break;
            case 2418:
                switch (current) {
                    case 'r' :
                        state = 2419;
                        return true;
                }
                break;
            case 2419:
                switch (current) {
                    case 'r' :
                        state = 2420;
                        return true;
                }
                break;
            case 2420:
                switch (current) {
                    case 'o' :
                        state = 2421;
                        return true;
                }
                break;
            case 2421:
                switch (current) {
                    case 'w' :
                        state = 2422;
                        return true;
                }
                break;
            case 2422:
                switch (current) {
                    case ';' : // RightTeeArrow;
                        match = "\u21A6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2423:
                switch (current) {
                    case 'e' :
                        state = 2424;
                        return true;
                }
                break;
            case 2424:
                switch (current) {
                    case 'c' :
                        state = 2425;
                        return true;
                }
                break;
            case 2425:
                switch (current) {
                    case 't' :
                        state = 2426;
                        return true;
                }
                break;
            case 2426:
                switch (current) {
                    case 'o' :
                        state = 2427;
                        return true;
                }
                break;
            case 2427:
                switch (current) {
                    case 'r' :
                        state = 2428;
                        return true;
                }
                break;
            case 2428:
                switch (current) {
                    case ';' : // RightTeeVector;
                        match = "\u295B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2429:
                switch (current) {
                    case 'i' :
                        state = 2430;
                        return true;
                }
                break;
            case 2430:
                switch (current) {
                    case 'a' :
                        state = 2431;
                        return true;
                }
                break;
            case 2431:
                switch (current) {
                    case 'n' :
                        state = 2432;
                        return true;
                }
                break;
            case 2432:
                switch (current) {
                    case 'g' :
                        state = 2433;
                        return true;
                }
                break;
            case 2433:
                switch (current) {
                    case 'l' :
                        state = 2434;
                        return true;
                }
                break;
            case 2434:
                switch (current) {
                    case 'e' :
                        state = 2435;
                        return true;
                }
                break;
            case 2435:
                switch (current) {
                    case ';' : // RightTriangle;
                        match = "\u22B3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 2436;
                        return true;
                    case 'E' :
                        state = 2439;
                        return true;
                }
                break;
            case 2436:
                switch (current) {
                    case 'a' :
                        state = 2437;
                        return true;
                }
                break;
            case 2437:
                switch (current) {
                    case 'r' :
                        state = 2438;
                        return true;
                }
                break;
            case 2438:
                switch (current) {
                    case ';' : // RightTriangleBar;
                        match = "\u29D0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2439:
                switch (current) {
                    case 'q' :
                        state = 2440;
                        return true;
                }
                break;
            case 2440:
                switch (current) {
                    case 'u' :
                        state = 2441;
                        return true;
                }
                break;
            case 2441:
                switch (current) {
                    case 'a' :
                        state = 2442;
                        return true;
                }
                break;
            case 2442:
                switch (current) {
                    case 'l' :
                        state = 2443;
                        return true;
                }
                break;
            case 2443:
                switch (current) {
                    case ';' : // RightTriangleEqual;
                        match = "\u22B5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2444:
                switch (current) {
                    case 'p' :
                        state = 2445;
                        return true;
                }
                break;
            case 2445:
                switch (current) {
                    case 'D' :
                        state = 2446;
                        return true;
                    case 'T' :
                        state = 2456;
                        return true;
                    case 'V' :
                        state = 2465;
                        return true;
                }
                break;
            case 2446:
                switch (current) {
                    case 'o' :
                        state = 2447;
                        return true;
                }
                break;
            case 2447:
                switch (current) {
                    case 'w' :
                        state = 2448;
                        return true;
                }
                break;
            case 2448:
                switch (current) {
                    case 'n' :
                        state = 2449;
                        return true;
                }
                break;
            case 2449:
                switch (current) {
                    case 'V' :
                        state = 2450;
                        return true;
                }
                break;
            case 2450:
                switch (current) {
                    case 'e' :
                        state = 2451;
                        return true;
                }
                break;
            case 2451:
                switch (current) {
                    case 'c' :
                        state = 2452;
                        return true;
                }
                break;
            case 2452:
                switch (current) {
                    case 't' :
                        state = 2453;
                        return true;
                }
                break;
            case 2453:
                switch (current) {
                    case 'o' :
                        state = 2454;
                        return true;
                }
                break;
            case 2454:
                switch (current) {
                    case 'r' :
                        state = 2455;
                        return true;
                }
                break;
            case 2455:
                switch (current) {
                    case ';' : // RightUpDownVector;
                        match = "\u294F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2456:
                switch (current) {
                    case 'e' :
                        state = 2457;
                        return true;
                }
                break;
            case 2457:
                switch (current) {
                    case 'e' :
                        state = 2458;
                        return true;
                }
                break;
            case 2458:
                switch (current) {
                    case 'V' :
                        state = 2459;
                        return true;
                }
                break;
            case 2459:
                switch (current) {
                    case 'e' :
                        state = 2460;
                        return true;
                }
                break;
            case 2460:
                switch (current) {
                    case 'c' :
                        state = 2461;
                        return true;
                }
                break;
            case 2461:
                switch (current) {
                    case 't' :
                        state = 2462;
                        return true;
                }
                break;
            case 2462:
                switch (current) {
                    case 'o' :
                        state = 2463;
                        return true;
                }
                break;
            case 2463:
                switch (current) {
                    case 'r' :
                        state = 2464;
                        return true;
                }
                break;
            case 2464:
                switch (current) {
                    case ';' : // RightUpTeeVector;
                        match = "\u295C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2465:
                switch (current) {
                    case 'e' :
                        state = 2466;
                        return true;
                }
                break;
            case 2466:
                switch (current) {
                    case 'c' :
                        state = 2467;
                        return true;
                }
                break;
            case 2467:
                switch (current) {
                    case 't' :
                        state = 2468;
                        return true;
                }
                break;
            case 2468:
                switch (current) {
                    case 'o' :
                        state = 2469;
                        return true;
                }
                break;
            case 2469:
                switch (current) {
                    case 'r' :
                        state = 2470;
                        return true;
                }
                break;
            case 2470:
                switch (current) {
                    case ';' : // RightUpVector;
                        match = "\u21BE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 2471;
                        return true;
                }
                break;
            case 2471:
                switch (current) {
                    case 'a' :
                        state = 2472;
                        return true;
                }
                break;
            case 2472:
                switch (current) {
                    case 'r' :
                        state = 2473;
                        return true;
                }
                break;
            case 2473:
                switch (current) {
                    case ';' : // RightUpVectorBar;
                        match = "\u2954";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2474:
                switch (current) {
                    case 'e' :
                        state = 2475;
                        return true;
                }
                break;
            case 2475:
                switch (current) {
                    case 'c' :
                        state = 2476;
                        return true;
                }
                break;
            case 2476:
                switch (current) {
                    case 't' :
                        state = 2477;
                        return true;
                }
                break;
            case 2477:
                switch (current) {
                    case 'o' :
                        state = 2478;
                        return true;
                }
                break;
            case 2478:
                switch (current) {
                    case 'r' :
                        state = 2479;
                        return true;
                }
                break;
            case 2479:
                switch (current) {
                    case ';' : // RightVector;
                        match = "\u21C0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 2480;
                        return true;
                }
                break;
            case 2480:
                switch (current) {
                    case 'a' :
                        state = 2481;
                        return true;
                }
                break;
            case 2481:
                switch (current) {
                    case 'r' :
                        state = 2482;
                        return true;
                }
                break;
            case 2482:
                switch (current) {
                    case ';' : // RightVectorBar;
                        match = "\u2953";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2483:
                switch (current) {
                    case 'r' :
                        state = 2484;
                        return true;
                }
                break;
            case 2484:
                switch (current) {
                    case 'r' :
                        state = 2485;
                        return true;
                }
                break;
            case 2485:
                switch (current) {
                    case 'o' :
                        state = 2486;
                        return true;
                }
                break;
            case 2486:
                switch (current) {
                    case 'w' :
                        state = 2487;
                        return true;
                }
                break;
            case 2487:
                switch (current) {
                    case ';' : // Rightarrow;
                        match = "\u21D2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2488:
                switch (current) {
                    case 'p' :
                        state = 2489;
                        return true;
                    case 'u' :
                        state = 2491;
                        return true;
                }
                break;
            case 2489:
                switch (current) {
                    case 'f' :
                        state = 2490;
                        return true;
                }
                break;
            case 2490:
                switch (current) {
                    case ';' : // Ropf;
                        match = "\u211D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2491:
                switch (current) {
                    case 'n' :
                        state = 2492;
                        return true;
                }
                break;
            case 2492:
                switch (current) {
                    case 'd' :
                        state = 2493;
                        return true;
                }
                break;
            case 2493:
                switch (current) {
                    case 'I' :
                        state = 2494;
                        return true;
                }
                break;
            case 2494:
                switch (current) {
                    case 'm' :
                        state = 2495;
                        return true;
                }
                break;
            case 2495:
                switch (current) {
                    case 'p' :
                        state = 2496;
                        return true;
                }
                break;
            case 2496:
                switch (current) {
                    case 'l' :
                        state = 2497;
                        return true;
                }
                break;
            case 2497:
                switch (current) {
                    case 'i' :
                        state = 2498;
                        return true;
                }
                break;
            case 2498:
                switch (current) {
                    case 'e' :
                        state = 2499;
                        return true;
                }
                break;
            case 2499:
                switch (current) {
                    case 's' :
                        state = 2500;
                        return true;
                }
                break;
            case 2500:
                switch (current) {
                    case ';' : // RoundImplies;
                        match = "\u2970";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2501:
                switch (current) {
                    case 'i' :
                        state = 2502;
                        return true;
                }
                break;
            case 2502:
                switch (current) {
                    case 'g' :
                        state = 2503;
                        return true;
                }
                break;
            case 2503:
                switch (current) {
                    case 'h' :
                        state = 2504;
                        return true;
                }
                break;
            case 2504:
                switch (current) {
                    case 't' :
                        state = 2505;
                        return true;
                }
                break;
            case 2505:
                switch (current) {
                    case 'a' :
                        state = 2506;
                        return true;
                }
                break;
            case 2506:
                switch (current) {
                    case 'r' :
                        state = 2507;
                        return true;
                }
                break;
            case 2507:
                switch (current) {
                    case 'r' :
                        state = 2508;
                        return true;
                }
                break;
            case 2508:
                switch (current) {
                    case 'o' :
                        state = 2509;
                        return true;
                }
                break;
            case 2509:
                switch (current) {
                    case 'w' :
                        state = 2510;
                        return true;
                }
                break;
            case 2510:
                switch (current) {
                    case ';' : // Rrightarrow;
                        match = "\u21DB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2511:
                switch (current) {
                    case 'c' :
                        state = 2512;
                        return true;
                    case 'h' :
                        state = 2514;
                        return true;
                }
                break;
            case 2512:
                switch (current) {
                    case 'r' :
                        state = 2513;
                        return true;
                }
                break;
            case 2513:
                switch (current) {
                    case ';' : // Rscr;
                        match = "\u211B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2514:
                switch (current) {
                    case ';' : // Rsh;
                        match = "\u21B1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2515:
                switch (current) {
                    case 'l' :
                        state = 2516;
                        return true;
                }
                break;
            case 2516:
                switch (current) {
                    case 'e' :
                        state = 2517;
                        return true;
                }
                break;
            case 2517:
                switch (current) {
                    case 'D' :
                        state = 2518;
                        return true;
                }
                break;
            case 2518:
                switch (current) {
                    case 'e' :
                        state = 2519;
                        return true;
                }
                break;
            case 2519:
                switch (current) {
                    case 'l' :
                        state = 2520;
                        return true;
                }
                break;
            case 2520:
                switch (current) {
                    case 'a' :
                        state = 2521;
                        return true;
                }
                break;
            case 2521:
                switch (current) {
                    case 'y' :
                        state = 2522;
                        return true;
                }
                break;
            case 2522:
                switch (current) {
                    case 'e' :
                        state = 2523;
                        return true;
                }
                break;
            case 2523:
                switch (current) {
                    case 'd' :
                        state = 2524;
                        return true;
                }
                break;
            case 2524:
                switch (current) {
                    case ';' : // RuleDelayed;
                        match = "\u29F4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2525:
                switch (current) {
                    case 'H' :
                        state = 2526;
                        return true;
                    case 'O' :
                        state = 2533;
                        return true;
                    case 'a' :
                        state = 2538;
                        return true;
                    case 'c' :
                        state = 2543;
                        return true;
                    case 'f' :
                        state = 2556;
                        return true;
                    case 'h' :
                        state = 2558;
                        return true;
                    case 'i' :
                        state = 2597;
                        return true;
                    case 'm' :
                        state = 2601;
                        return true;
                    case 'o' :
                        state = 2611;
                        return true;
                    case 'q' :
                        state = 2614;
                        return true;
                    case 's' :
                        state = 2660;
                        return true;
                    case 't' :
                        state = 2663;
                        return true;
                    case 'u' :
                        state = 2666;
                        return true;
                }
                break;
            case 2526:
                switch (current) {
                    case 'C' :
                        state = 2527;
                        return true;
                    case 'c' :
                        state = 2531;
                        return true;
                }
                break;
            case 2527:
                switch (current) {
                    case 'H' :
                        state = 2528;
                        return true;
                }
                break;
            case 2528:
                switch (current) {
                    case 'c' :
                        state = 2529;
                        return true;
                }
                break;
            case 2529:
                switch (current) {
                    case 'y' :
                        state = 2530;
                        return true;
                }
                break;
            case 2530:
                switch (current) {
                    case ';' : // SHCHcy;
                        match = "\u0429";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2531:
                switch (current) {
                    case 'y' :
                        state = 2532;
                        return true;
                }
                break;
            case 2532:
                switch (current) {
                    case ';' : // SHcy;
                        match = "\u0428";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2533:
                switch (current) {
                    case 'F' :
                        state = 2534;
                        return true;
                }
                break;
            case 2534:
                switch (current) {
                    case 'T' :
                        state = 2535;
                        return true;
                }
                break;
            case 2535:
                switch (current) {
                    case 'c' :
                        state = 2536;
                        return true;
                }
                break;
            case 2536:
                switch (current) {
                    case 'y' :
                        state = 2537;
                        return true;
                }
                break;
            case 2537:
                switch (current) {
                    case ';' : // SOFTcy;
                        match = "\u042C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2538:
                switch (current) {
                    case 'c' :
                        state = 2539;
                        return true;
                }
                break;
            case 2539:
                switch (current) {
                    case 'u' :
                        state = 2540;
                        return true;
                }
                break;
            case 2540:
                switch (current) {
                    case 't' :
                        state = 2541;
                        return true;
                }
                break;
            case 2541:
                switch (current) {
                    case 'e' :
                        state = 2542;
                        return true;
                }
                break;
            case 2542:
                switch (current) {
                    case ';' : // Sacute;
                        match = "\u015A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2543:
                switch (current) {
                    case ';' : // Sc;
                        match = "\u2ABC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 2544;
                        return true;
                    case 'e' :
                        state = 2548;
                        return true;
                    case 'i' :
                        state = 2552;
                        return true;
                    case 'y' :
                        state = 2555;
                        return true;
                }
                break;
            case 2544:
                switch (current) {
                    case 'r' :
                        state = 2545;
                        return true;
                }
                break;
            case 2545:
                switch (current) {
                    case 'o' :
                        state = 2546;
                        return true;
                }
                break;
            case 2546:
                switch (current) {
                    case 'n' :
                        state = 2547;
                        return true;
                }
                break;
            case 2547:
                switch (current) {
                    case ';' : // Scaron;
                        match = "\u0160";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2548:
                switch (current) {
                    case 'd' :
                        state = 2549;
                        return true;
                }
                break;
            case 2549:
                switch (current) {
                    case 'i' :
                        state = 2550;
                        return true;
                }
                break;
            case 2550:
                switch (current) {
                    case 'l' :
                        state = 2551;
                        return true;
                }
                break;
            case 2551:
                switch (current) {
                    case ';' : // Scedil;
                        match = "\u015E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2552:
                switch (current) {
                    case 'r' :
                        state = 2553;
                        return true;
                }
                break;
            case 2553:
                switch (current) {
                    case 'c' :
                        state = 2554;
                        return true;
                }
                break;
            case 2554:
                switch (current) {
                    case ';' : // Scirc;
                        match = "\u015C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2555:
                switch (current) {
                    case ';' : // Scy;
                        match = "\u0421";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2556:
                switch (current) {
                    case 'r' :
                        state = 2557;
                        return true;
                }
                break;
            case 2557:
                switch (current) {
                    case ';' : // Sfr;
                        match = "\uD835\uDD16";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2558:
                switch (current) {
                    case 'o' :
                        state = 2559;
                        return true;
                }
                break;
            case 2559:
                switch (current) {
                    case 'r' :
                        state = 2560;
                        return true;
                }
                break;
            case 2560:
                switch (current) {
                    case 't' :
                        state = 2561;
                        return true;
                }
                break;
            case 2561:
                switch (current) {
                    case 'D' :
                        state = 2562;
                        return true;
                    case 'L' :
                        state = 2571;
                        return true;
                    case 'R' :
                        state = 2580;
                        return true;
                    case 'U' :
                        state = 2590;
                        return true;
                }
                break;
            case 2562:
                switch (current) {
                    case 'o' :
                        state = 2563;
                        return true;
                }
                break;
            case 2563:
                switch (current) {
                    case 'w' :
                        state = 2564;
                        return true;
                }
                break;
            case 2564:
                switch (current) {
                    case 'n' :
                        state = 2565;
                        return true;
                }
                break;
            case 2565:
                switch (current) {
                    case 'A' :
                        state = 2566;
                        return true;
                }
                break;
            case 2566:
                switch (current) {
                    case 'r' :
                        state = 2567;
                        return true;
                }
                break;
            case 2567:
                switch (current) {
                    case 'r' :
                        state = 2568;
                        return true;
                }
                break;
            case 2568:
                switch (current) {
                    case 'o' :
                        state = 2569;
                        return true;
                }
                break;
            case 2569:
                switch (current) {
                    case 'w' :
                        state = 2570;
                        return true;
                }
                break;
            case 2570:
                switch (current) {
                    case ';' : // ShortDownArrow;
                        match = "\u2193";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2571:
                switch (current) {
                    case 'e' :
                        state = 2572;
                        return true;
                }
                break;
            case 2572:
                switch (current) {
                    case 'f' :
                        state = 2573;
                        return true;
                }
                break;
            case 2573:
                switch (current) {
                    case 't' :
                        state = 2574;
                        return true;
                }
                break;
            case 2574:
                switch (current) {
                    case 'A' :
                        state = 2575;
                        return true;
                }
                break;
            case 2575:
                switch (current) {
                    case 'r' :
                        state = 2576;
                        return true;
                }
                break;
            case 2576:
                switch (current) {
                    case 'r' :
                        state = 2577;
                        return true;
                }
                break;
            case 2577:
                switch (current) {
                    case 'o' :
                        state = 2578;
                        return true;
                }
                break;
            case 2578:
                switch (current) {
                    case 'w' :
                        state = 2579;
                        return true;
                }
                break;
            case 2579:
                switch (current) {
                    case ';' : // ShortLeftArrow;
                        match = "\u2190";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2580:
                switch (current) {
                    case 'i' :
                        state = 2581;
                        return true;
                }
                break;
            case 2581:
                switch (current) {
                    case 'g' :
                        state = 2582;
                        return true;
                }
                break;
            case 2582:
                switch (current) {
                    case 'h' :
                        state = 2583;
                        return true;
                }
                break;
            case 2583:
                switch (current) {
                    case 't' :
                        state = 2584;
                        return true;
                }
                break;
            case 2584:
                switch (current) {
                    case 'A' :
                        state = 2585;
                        return true;
                }
                break;
            case 2585:
                switch (current) {
                    case 'r' :
                        state = 2586;
                        return true;
                }
                break;
            case 2586:
                switch (current) {
                    case 'r' :
                        state = 2587;
                        return true;
                }
                break;
            case 2587:
                switch (current) {
                    case 'o' :
                        state = 2588;
                        return true;
                }
                break;
            case 2588:
                switch (current) {
                    case 'w' :
                        state = 2589;
                        return true;
                }
                break;
            case 2589:
                switch (current) {
                    case ';' : // ShortRightArrow;
                        match = "\u2192";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2590:
                switch (current) {
                    case 'p' :
                        state = 2591;
                        return true;
                }
                break;
            case 2591:
                switch (current) {
                    case 'A' :
                        state = 2592;
                        return true;
                }
                break;
            case 2592:
                switch (current) {
                    case 'r' :
                        state = 2593;
                        return true;
                }
                break;
            case 2593:
                switch (current) {
                    case 'r' :
                        state = 2594;
                        return true;
                }
                break;
            case 2594:
                switch (current) {
                    case 'o' :
                        state = 2595;
                        return true;
                }
                break;
            case 2595:
                switch (current) {
                    case 'w' :
                        state = 2596;
                        return true;
                }
                break;
            case 2596:
                switch (current) {
                    case ';' : // ShortUpArrow;
                        match = "\u2191";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2597:
                switch (current) {
                    case 'g' :
                        state = 2598;
                        return true;
                }
                break;
            case 2598:
                switch (current) {
                    case 'm' :
                        state = 2599;
                        return true;
                }
                break;
            case 2599:
                switch (current) {
                    case 'a' :
                        state = 2600;
                        return true;
                }
                break;
            case 2600:
                switch (current) {
                    case ';' : // Sigma;
                        match = "\u03A3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2601:
                switch (current) {
                    case 'a' :
                        state = 2602;
                        return true;
                }
                break;
            case 2602:
                switch (current) {
                    case 'l' :
                        state = 2603;
                        return true;
                }
                break;
            case 2603:
                switch (current) {
                    case 'l' :
                        state = 2604;
                        return true;
                }
                break;
            case 2604:
                switch (current) {
                    case 'C' :
                        state = 2605;
                        return true;
                }
                break;
            case 2605:
                switch (current) {
                    case 'i' :
                        state = 2606;
                        return true;
                }
                break;
            case 2606:
                switch (current) {
                    case 'r' :
                        state = 2607;
                        return true;
                }
                break;
            case 2607:
                switch (current) {
                    case 'c' :
                        state = 2608;
                        return true;
                }
                break;
            case 2608:
                switch (current) {
                    case 'l' :
                        state = 2609;
                        return true;
                }
                break;
            case 2609:
                switch (current) {
                    case 'e' :
                        state = 2610;
                        return true;
                }
                break;
            case 2610:
                switch (current) {
                    case ';' : // SmallCircle;
                        match = "\u2218";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2611:
                switch (current) {
                    case 'p' :
                        state = 2612;
                        return true;
                }
                break;
            case 2612:
                switch (current) {
                    case 'f' :
                        state = 2613;
                        return true;
                }
                break;
            case 2613:
                switch (current) {
                    case ';' : // Sopf;
                        match = "\uD835\uDD4A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2614:
                switch (current) {
                    case 'r' :
                        state = 2615;
                        return true;
                    case 'u' :
                        state = 2617;
                        return true;
                }
                break;
            case 2615:
                switch (current) {
                    case 't' :
                        state = 2616;
                        return true;
                }
                break;
            case 2616:
                switch (current) {
                    case ';' : // Sqrt;
                        match = "\u221A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2617:
                switch (current) {
                    case 'a' :
                        state = 2618;
                        return true;
                }
                break;
            case 2618:
                switch (current) {
                    case 'r' :
                        state = 2619;
                        return true;
                }
                break;
            case 2619:
                switch (current) {
                    case 'e' :
                        state = 2620;
                        return true;
                }
                break;
            case 2620:
                switch (current) {
                    case ';' : // Square;
                        match = "\u25A1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'I' :
                        state = 2621;
                        return true;
                    case 'S' :
                        state = 2633;
                        return true;
                    case 'U' :
                        state = 2655;
                        return true;
                }
                break;
            case 2621:
                switch (current) {
                    case 'n' :
                        state = 2622;
                        return true;
                }
                break;
            case 2622:
                switch (current) {
                    case 't' :
                        state = 2623;
                        return true;
                }
                break;
            case 2623:
                switch (current) {
                    case 'e' :
                        state = 2624;
                        return true;
                }
                break;
            case 2624:
                switch (current) {
                    case 'r' :
                        state = 2625;
                        return true;
                }
                break;
            case 2625:
                switch (current) {
                    case 's' :
                        state = 2626;
                        return true;
                }
                break;
            case 2626:
                switch (current) {
                    case 'e' :
                        state = 2627;
                        return true;
                }
                break;
            case 2627:
                switch (current) {
                    case 'c' :
                        state = 2628;
                        return true;
                }
                break;
            case 2628:
                switch (current) {
                    case 't' :
                        state = 2629;
                        return true;
                }
                break;
            case 2629:
                switch (current) {
                    case 'i' :
                        state = 2630;
                        return true;
                }
                break;
            case 2630:
                switch (current) {
                    case 'o' :
                        state = 2631;
                        return true;
                }
                break;
            case 2631:
                switch (current) {
                    case 'n' :
                        state = 2632;
                        return true;
                }
                break;
            case 2632:
                switch (current) {
                    case ';' : // SquareIntersection;
                        match = "\u2293";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2633:
                switch (current) {
                    case 'u' :
                        state = 2634;
                        return true;
                }
                break;
            case 2634:
                switch (current) {
                    case 'b' :
                        state = 2635;
                        return true;
                    case 'p' :
                        state = 2644;
                        return true;
                }
                break;
            case 2635:
                switch (current) {
                    case 's' :
                        state = 2636;
                        return true;
                }
                break;
            case 2636:
                switch (current) {
                    case 'e' :
                        state = 2637;
                        return true;
                }
                break;
            case 2637:
                switch (current) {
                    case 't' :
                        state = 2638;
                        return true;
                }
                break;
            case 2638:
                switch (current) {
                    case ';' : // SquareSubset;
                        match = "\u228F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2639;
                        return true;
                }
                break;
            case 2639:
                switch (current) {
                    case 'q' :
                        state = 2640;
                        return true;
                }
                break;
            case 2640:
                switch (current) {
                    case 'u' :
                        state = 2641;
                        return true;
                }
                break;
            case 2641:
                switch (current) {
                    case 'a' :
                        state = 2642;
                        return true;
                }
                break;
            case 2642:
                switch (current) {
                    case 'l' :
                        state = 2643;
                        return true;
                }
                break;
            case 2643:
                switch (current) {
                    case ';' : // SquareSubsetEqual;
                        match = "\u2291";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2644:
                switch (current) {
                    case 'e' :
                        state = 2645;
                        return true;
                }
                break;
            case 2645:
                switch (current) {
                    case 'r' :
                        state = 2646;
                        return true;
                }
                break;
            case 2646:
                switch (current) {
                    case 's' :
                        state = 2647;
                        return true;
                }
                break;
            case 2647:
                switch (current) {
                    case 'e' :
                        state = 2648;
                        return true;
                }
                break;
            case 2648:
                switch (current) {
                    case 't' :
                        state = 2649;
                        return true;
                }
                break;
            case 2649:
                switch (current) {
                    case ';' : // SquareSuperset;
                        match = "\u2290";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2650;
                        return true;
                }
                break;
            case 2650:
                switch (current) {
                    case 'q' :
                        state = 2651;
                        return true;
                }
                break;
            case 2651:
                switch (current) {
                    case 'u' :
                        state = 2652;
                        return true;
                }
                break;
            case 2652:
                switch (current) {
                    case 'a' :
                        state = 2653;
                        return true;
                }
                break;
            case 2653:
                switch (current) {
                    case 'l' :
                        state = 2654;
                        return true;
                }
                break;
            case 2654:
                switch (current) {
                    case ';' : // SquareSupersetEqual;
                        match = "\u2292";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2655:
                switch (current) {
                    case 'n' :
                        state = 2656;
                        return true;
                }
                break;
            case 2656:
                switch (current) {
                    case 'i' :
                        state = 2657;
                        return true;
                }
                break;
            case 2657:
                switch (current) {
                    case 'o' :
                        state = 2658;
                        return true;
                }
                break;
            case 2658:
                switch (current) {
                    case 'n' :
                        state = 2659;
                        return true;
                }
                break;
            case 2659:
                switch (current) {
                    case ';' : // SquareUnion;
                        match = "\u2294";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2660:
                switch (current) {
                    case 'c' :
                        state = 2661;
                        return true;
                }
                break;
            case 2661:
                switch (current) {
                    case 'r' :
                        state = 2662;
                        return true;
                }
                break;
            case 2662:
                switch (current) {
                    case ';' : // Sscr;
                        match = "\uD835\uDCAE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2663:
                switch (current) {
                    case 'a' :
                        state = 2664;
                        return true;
                }
                break;
            case 2664:
                switch (current) {
                    case 'r' :
                        state = 2665;
                        return true;
                }
                break;
            case 2665:
                switch (current) {
                    case ';' : // Star;
                        match = "\u22C6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2666:
                switch (current) {
                    case 'b' :
                        state = 2667;
                        return true;
                    case 'c' :
                        state = 2676;
                        return true;
                    case 'm' :
                        state = 2707;
                        return true;
                    case 'p' :
                        state = 2708;
                        return true;
                }
                break;
            case 2667:
                switch (current) {
                    case ';' : // Sub;
                        match = "\u22D0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 's' :
                        state = 2668;
                        return true;
                }
                break;
            case 2668:
                switch (current) {
                    case 'e' :
                        state = 2669;
                        return true;
                }
                break;
            case 2669:
                switch (current) {
                    case 't' :
                        state = 2670;
                        return true;
                }
                break;
            case 2670:
                switch (current) {
                    case ';' : // Subset;
                        match = "\u22D0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2671;
                        return true;
                }
                break;
            case 2671:
                switch (current) {
                    case 'q' :
                        state = 2672;
                        return true;
                }
                break;
            case 2672:
                switch (current) {
                    case 'u' :
                        state = 2673;
                        return true;
                }
                break;
            case 2673:
                switch (current) {
                    case 'a' :
                        state = 2674;
                        return true;
                }
                break;
            case 2674:
                switch (current) {
                    case 'l' :
                        state = 2675;
                        return true;
                }
                break;
            case 2675:
                switch (current) {
                    case ';' : // SubsetEqual;
                        match = "\u2286";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2676:
                switch (current) {
                    case 'c' :
                        state = 2677;
                        return true;
                    case 'h' :
                        state = 2702;
                        return true;
                }
                break;
            case 2677:
                switch (current) {
                    case 'e' :
                        state = 2678;
                        return true;
                }
                break;
            case 2678:
                switch (current) {
                    case 'e' :
                        state = 2679;
                        return true;
                }
                break;
            case 2679:
                switch (current) {
                    case 'd' :
                        state = 2680;
                        return true;
                }
                break;
            case 2680:
                switch (current) {
                    case 's' :
                        state = 2681;
                        return true;
                }
                break;
            case 2681:
                switch (current) {
                    case ';' : // Succeeds;
                        match = "\u227B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2682;
                        return true;
                    case 'S' :
                        state = 2687;
                        return true;
                    case 'T' :
                        state = 2697;
                        return true;
                }
                break;
            case 2682:
                switch (current) {
                    case 'q' :
                        state = 2683;
                        return true;
                }
                break;
            case 2683:
                switch (current) {
                    case 'u' :
                        state = 2684;
                        return true;
                }
                break;
            case 2684:
                switch (current) {
                    case 'a' :
                        state = 2685;
                        return true;
                }
                break;
            case 2685:
                switch (current) {
                    case 'l' :
                        state = 2686;
                        return true;
                }
                break;
            case 2686:
                switch (current) {
                    case ';' : // SucceedsEqual;
                        match = "\u2AB0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2687:
                switch (current) {
                    case 'l' :
                        state = 2688;
                        return true;
                }
                break;
            case 2688:
                switch (current) {
                    case 'a' :
                        state = 2689;
                        return true;
                }
                break;
            case 2689:
                switch (current) {
                    case 'n' :
                        state = 2690;
                        return true;
                }
                break;
            case 2690:
                switch (current) {
                    case 't' :
                        state = 2691;
                        return true;
                }
                break;
            case 2691:
                switch (current) {
                    case 'E' :
                        state = 2692;
                        return true;
                }
                break;
            case 2692:
                switch (current) {
                    case 'q' :
                        state = 2693;
                        return true;
                }
                break;
            case 2693:
                switch (current) {
                    case 'u' :
                        state = 2694;
                        return true;
                }
                break;
            case 2694:
                switch (current) {
                    case 'a' :
                        state = 2695;
                        return true;
                }
                break;
            case 2695:
                switch (current) {
                    case 'l' :
                        state = 2696;
                        return true;
                }
                break;
            case 2696:
                switch (current) {
                    case ';' : // SucceedsSlantEqual;
                        match = "\u227D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2697:
                switch (current) {
                    case 'i' :
                        state = 2698;
                        return true;
                }
                break;
            case 2698:
                switch (current) {
                    case 'l' :
                        state = 2699;
                        return true;
                }
                break;
            case 2699:
                switch (current) {
                    case 'd' :
                        state = 2700;
                        return true;
                }
                break;
            case 2700:
                switch (current) {
                    case 'e' :
                        state = 2701;
                        return true;
                }
                break;
            case 2701:
                switch (current) {
                    case ';' : // SucceedsTilde;
                        match = "\u227F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2702:
                switch (current) {
                    case 'T' :
                        state = 2703;
                        return true;
                }
                break;
            case 2703:
                switch (current) {
                    case 'h' :
                        state = 2704;
                        return true;
                }
                break;
            case 2704:
                switch (current) {
                    case 'a' :
                        state = 2705;
                        return true;
                }
                break;
            case 2705:
                switch (current) {
                    case 't' :
                        state = 2706;
                        return true;
                }
                break;
            case 2706:
                switch (current) {
                    case ';' : // SuchThat;
                        match = "\u220B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2707:
                switch (current) {
                    case ';' : // Sum;
                        match = "\u2211";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2708:
                switch (current) {
                    case ';' : // Sup;
                        match = "\u22D1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 2709;
                        return true;
                    case 's' :
                        state = 2719;
                        return true;
                }
                break;
            case 2709:
                switch (current) {
                    case 'r' :
                        state = 2710;
                        return true;
                }
                break;
            case 2710:
                switch (current) {
                    case 's' :
                        state = 2711;
                        return true;
                }
                break;
            case 2711:
                switch (current) {
                    case 'e' :
                        state = 2712;
                        return true;
                }
                break;
            case 2712:
                switch (current) {
                    case 't' :
                        state = 2713;
                        return true;
                }
                break;
            case 2713:
                switch (current) {
                    case ';' : // Superset;
                        match = "\u2283";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2714;
                        return true;
                }
                break;
            case 2714:
                switch (current) {
                    case 'q' :
                        state = 2715;
                        return true;
                }
                break;
            case 2715:
                switch (current) {
                    case 'u' :
                        state = 2716;
                        return true;
                }
                break;
            case 2716:
                switch (current) {
                    case 'a' :
                        state = 2717;
                        return true;
                }
                break;
            case 2717:
                switch (current) {
                    case 'l' :
                        state = 2718;
                        return true;
                }
                break;
            case 2718:
                switch (current) {
                    case ';' : // SupersetEqual;
                        match = "\u2287";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2719:
                switch (current) {
                    case 'e' :
                        state = 2720;
                        return true;
                }
                break;
            case 2720:
                switch (current) {
                    case 't' :
                        state = 2721;
                        return true;
                }
                break;
            case 2721:
                switch (current) {
                    case ';' : // Supset;
                        match = "\u22D1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2722:
                switch (current) {
                    case 'H' :
                        state = 2723;
                        return true;
                    case 'R' :
                        state = 2727;
                        return true;
                    case 'S' :
                        state = 2731;
                        return true;
                    case 'a' :
                        state = 2737;
                        return true;
                    case 'c' :
                        state = 2740;
                        return true;
                    case 'f' :
                        state = 2750;
                        return true;
                    case 'h' :
                        state = 2752;
                        return true;
                    case 'i' :
                        state = 2776;
                        return true;
                    case 'o' :
                        state = 2799;
                        return true;
                    case 'r' :
                        state = 2802;
                        return true;
                    case 's' :
                        state = 2810;
                        return true;
                }
                break;
            case 2723:
                switch (current) {
                    case 'O' :
                        state = 2724;
                        return true;
                }
                break;
            case 2724:
                switch (current) {
                    case 'R' :
                        state = 2725;
                        return true;
                }
                break;
            case 2725:
                switch (current) {
                    case 'N' : // THORN
                        match = "\u00DE";
                        matchLength = consumedCount;
                        state = 2726;
                        return true;
                }
                break;
            case 2726:
                switch (current) {
                    case ';' : // THORN;
                        match = "\u00DE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2727:
                switch (current) {
                    case 'A' :
                        state = 2728;
                        return true;
                }
                break;
            case 2728:
                switch (current) {
                    case 'D' :
                        state = 2729;
                        return true;
                }
                break;
            case 2729:
                switch (current) {
                    case 'E' :
                        state = 2730;
                        return true;
                }
                break;
            case 2730:
                switch (current) {
                    case ';' : // TRADE;
                        match = "\u2122";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2731:
                switch (current) {
                    case 'H' :
                        state = 2732;
                        return true;
                    case 'c' :
                        state = 2735;
                        return true;
                }
                break;
            case 2732:
                switch (current) {
                    case 'c' :
                        state = 2733;
                        return true;
                }
                break;
            case 2733:
                switch (current) {
                    case 'y' :
                        state = 2734;
                        return true;
                }
                break;
            case 2734:
                switch (current) {
                    case ';' : // TSHcy;
                        match = "\u040B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2735:
                switch (current) {
                    case 'y' :
                        state = 2736;
                        return true;
                }
                break;
            case 2736:
                switch (current) {
                    case ';' : // TScy;
                        match = "\u0426";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2737:
                switch (current) {
                    case 'b' :
                        state = 2738;
                        return true;
                    case 'u' :
                        state = 2739;
                        return true;
                }
                break;
            case 2738:
                switch (current) {
                    case ';' : // Tab;
                        match = "\u0009";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2739:
                switch (current) {
                    case ';' : // Tau;
                        match = "\u03A4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2740:
                switch (current) {
                    case 'a' :
                        state = 2741;
                        return true;
                    case 'e' :
                        state = 2745;
                        return true;
                    case 'y' :
                        state = 2749;
                        return true;
                }
                break;
            case 2741:
                switch (current) {
                    case 'r' :
                        state = 2742;
                        return true;
                }
                break;
            case 2742:
                switch (current) {
                    case 'o' :
                        state = 2743;
                        return true;
                }
                break;
            case 2743:
                switch (current) {
                    case 'n' :
                        state = 2744;
                        return true;
                }
                break;
            case 2744:
                switch (current) {
                    case ';' : // Tcaron;
                        match = "\u0164";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2745:
                switch (current) {
                    case 'd' :
                        state = 2746;
                        return true;
                }
                break;
            case 2746:
                switch (current) {
                    case 'i' :
                        state = 2747;
                        return true;
                }
                break;
            case 2747:
                switch (current) {
                    case 'l' :
                        state = 2748;
                        return true;
                }
                break;
            case 2748:
                switch (current) {
                    case ';' : // Tcedil;
                        match = "\u0162";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2749:
                switch (current) {
                    case ';' : // Tcy;
                        match = "\u0422";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2750:
                switch (current) {
                    case 'r' :
                        state = 2751;
                        return true;
                }
                break;
            case 2751:
                switch (current) {
                    case ';' : // Tfr;
                        match = "\uD835\uDD17";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2752:
                switch (current) {
                    case 'e' :
                        state = 2753;
                        return true;
                    case 'i' :
                        state = 2762;
                        return true;
                }
                break;
            case 2753:
                switch (current) {
                    case 'r' :
                        state = 2754;
                        return true;
                    case 't' :
                        state = 2760;
                        return true;
                }
                break;
            case 2754:
                switch (current) {
                    case 'e' :
                        state = 2755;
                        return true;
                }
                break;
            case 2755:
                switch (current) {
                    case 'f' :
                        state = 2756;
                        return true;
                }
                break;
            case 2756:
                switch (current) {
                    case 'o' :
                        state = 2757;
                        return true;
                }
                break;
            case 2757:
                switch (current) {
                    case 'r' :
                        state = 2758;
                        return true;
                }
                break;
            case 2758:
                switch (current) {
                    case 'e' :
                        state = 2759;
                        return true;
                }
                break;
            case 2759:
                switch (current) {
                    case ';' : // Therefore;
                        match = "\u2234";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2760:
                switch (current) {
                    case 'a' :
                        state = 2761;
                        return true;
                }
                break;
            case 2761:
                switch (current) {
                    case ';' : // Theta;
                        match = "\u0398";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2762:
                switch (current) {
                    case 'c' :
                        state = 2763;
                        return true;
                    case 'n' :
                        state = 2770;
                        return true;
                }
                break;
            case 2763:
                switch (current) {
                    case 'k' :
                        state = 2764;
                        return true;
                }
                break;
            case 2764:
                switch (current) {
                    case 'S' :
                        state = 2765;
                        return true;
                }
                break;
            case 2765:
                switch (current) {
                    case 'p' :
                        state = 2766;
                        return true;
                }
                break;
            case 2766:
                switch (current) {
                    case 'a' :
                        state = 2767;
                        return true;
                }
                break;
            case 2767:
                switch (current) {
                    case 'c' :
                        state = 2768;
                        return true;
                }
                break;
            case 2768:
                switch (current) {
                    case 'e' :
                        state = 2769;
                        return true;
                }
                break;
            case 2769:
                switch (current) {
                    case ';' : // ThickSpace;
                        match = "\u205F\u200A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2770:
                switch (current) {
                    case 'S' :
                        state = 2771;
                        return true;
                }
                break;
            case 2771:
                switch (current) {
                    case 'p' :
                        state = 2772;
                        return true;
                }
                break;
            case 2772:
                switch (current) {
                    case 'a' :
                        state = 2773;
                        return true;
                }
                break;
            case 2773:
                switch (current) {
                    case 'c' :
                        state = 2774;
                        return true;
                }
                break;
            case 2774:
                switch (current) {
                    case 'e' :
                        state = 2775;
                        return true;
                }
                break;
            case 2775:
                switch (current) {
                    case ';' : // ThinSpace;
                        match = "\u2009";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2776:
                switch (current) {
                    case 'l' :
                        state = 2777;
                        return true;
                }
                break;
            case 2777:
                switch (current) {
                    case 'd' :
                        state = 2778;
                        return true;
                }
                break;
            case 2778:
                switch (current) {
                    case 'e' :
                        state = 2779;
                        return true;
                }
                break;
            case 2779:
                switch (current) {
                    case ';' : // Tilde;
                        match = "\u223C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2780;
                        return true;
                    case 'F' :
                        state = 2785;
                        return true;
                    case 'T' :
                        state = 2794;
                        return true;
                }
                break;
            case 2780:
                switch (current) {
                    case 'q' :
                        state = 2781;
                        return true;
                }
                break;
            case 2781:
                switch (current) {
                    case 'u' :
                        state = 2782;
                        return true;
                }
                break;
            case 2782:
                switch (current) {
                    case 'a' :
                        state = 2783;
                        return true;
                }
                break;
            case 2783:
                switch (current) {
                    case 'l' :
                        state = 2784;
                        return true;
                }
                break;
            case 2784:
                switch (current) {
                    case ';' : // TildeEqual;
                        match = "\u2243";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2785:
                switch (current) {
                    case 'u' :
                        state = 2786;
                        return true;
                }
                break;
            case 2786:
                switch (current) {
                    case 'l' :
                        state = 2787;
                        return true;
                }
                break;
            case 2787:
                switch (current) {
                    case 'l' :
                        state = 2788;
                        return true;
                }
                break;
            case 2788:
                switch (current) {
                    case 'E' :
                        state = 2789;
                        return true;
                }
                break;
            case 2789:
                switch (current) {
                    case 'q' :
                        state = 2790;
                        return true;
                }
                break;
            case 2790:
                switch (current) {
                    case 'u' :
                        state = 2791;
                        return true;
                }
                break;
            case 2791:
                switch (current) {
                    case 'a' :
                        state = 2792;
                        return true;
                }
                break;
            case 2792:
                switch (current) {
                    case 'l' :
                        state = 2793;
                        return true;
                }
                break;
            case 2793:
                switch (current) {
                    case ';' : // TildeFullEqual;
                        match = "\u2245";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2794:
                switch (current) {
                    case 'i' :
                        state = 2795;
                        return true;
                }
                break;
            case 2795:
                switch (current) {
                    case 'l' :
                        state = 2796;
                        return true;
                }
                break;
            case 2796:
                switch (current) {
                    case 'd' :
                        state = 2797;
                        return true;
                }
                break;
            case 2797:
                switch (current) {
                    case 'e' :
                        state = 2798;
                        return true;
                }
                break;
            case 2798:
                switch (current) {
                    case ';' : // TildeTilde;
                        match = "\u2248";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2799:
                switch (current) {
                    case 'p' :
                        state = 2800;
                        return true;
                }
                break;
            case 2800:
                switch (current) {
                    case 'f' :
                        state = 2801;
                        return true;
                }
                break;
            case 2801:
                switch (current) {
                    case ';' : // Topf;
                        match = "\uD835\uDD4B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2802:
                switch (current) {
                    case 'i' :
                        state = 2803;
                        return true;
                }
                break;
            case 2803:
                switch (current) {
                    case 'p' :
                        state = 2804;
                        return true;
                }
                break;
            case 2804:
                switch (current) {
                    case 'l' :
                        state = 2805;
                        return true;
                }
                break;
            case 2805:
                switch (current) {
                    case 'e' :
                        state = 2806;
                        return true;
                }
                break;
            case 2806:
                switch (current) {
                    case 'D' :
                        state = 2807;
                        return true;
                }
                break;
            case 2807:
                switch (current) {
                    case 'o' :
                        state = 2808;
                        return true;
                }
                break;
            case 2808:
                switch (current) {
                    case 't' :
                        state = 2809;
                        return true;
                }
                break;
            case 2809:
                switch (current) {
                    case ';' : // TripleDot;
                        match = "\u20DB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2810:
                switch (current) {
                    case 'c' :
                        state = 2811;
                        return true;
                    case 't' :
                        state = 2813;
                        return true;
                }
                break;
            case 2811:
                switch (current) {
                    case 'r' :
                        state = 2812;
                        return true;
                }
                break;
            case 2812:
                switch (current) {
                    case ';' : // Tscr;
                        match = "\uD835\uDCAF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2813:
                switch (current) {
                    case 'r' :
                        state = 2814;
                        return true;
                }
                break;
            case 2814:
                switch (current) {
                    case 'o' :
                        state = 2815;
                        return true;
                }
                break;
            case 2815:
                switch (current) {
                    case 'k' :
                        state = 2816;
                        return true;
                }
                break;
            case 2816:
                switch (current) {
                    case ';' : // Tstrok;
                        match = "\u0166";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2817:
                switch (current) {
                    case 'a' :
                        state = 2818;
                        return true;
                    case 'b' :
                        state = 2829;
                        return true;
                    case 'c' :
                        state = 2836;
                        return true;
                    case 'd' :
                        state = 2841;
                        return true;
                    case 'f' :
                        state = 2846;
                        return true;
                    case 'g' :
                        state = 2848;
                        return true;
                    case 'm' :
                        state = 2853;
                        return true;
                    case 'n' :
                        state = 2857;
                        return true;
                    case 'o' :
                        state = 2889;
                        return true;
                    case 'p' :
                        state = 2895;
                        return true;
                    case 'r' :
                        state = 2982;
                        return true;
                    case 's' :
                        state = 2986;
                        return true;
                    case 't' :
                        state = 2989;
                        return true;
                    case 'u' :
                        state = 2994;
                        return true;
                }
                break;
            case 2818:
                switch (current) {
                    case 'c' :
                        state = 2819;
                        return true;
                    case 'r' :
                        state = 2823;
                        return true;
                }
                break;
            case 2819:
                switch (current) {
                    case 'u' :
                        state = 2820;
                        return true;
                }
                break;
            case 2820:
                switch (current) {
                    case 't' :
                        state = 2821;
                        return true;
                }
                break;
            case 2821:
                switch (current) {
                    case 'e' : // Uacute
                        match = "\u00DA";
                        matchLength = consumedCount;
                        state = 2822;
                        return true;
                }
                break;
            case 2822:
                switch (current) {
                    case ';' : // Uacute;
                        match = "\u00DA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2823:
                switch (current) {
                    case 'r' :
                        state = 2824;
                        return true;
                }
                break;
            case 2824:
                switch (current) {
                    case ';' : // Uarr;
                        match = "\u219F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'o' :
                        state = 2825;
                        return true;
                }
                break;
            case 2825:
                switch (current) {
                    case 'c' :
                        state = 2826;
                        return true;
                }
                break;
            case 2826:
                switch (current) {
                    case 'i' :
                        state = 2827;
                        return true;
                }
                break;
            case 2827:
                switch (current) {
                    case 'r' :
                        state = 2828;
                        return true;
                }
                break;
            case 2828:
                switch (current) {
                    case ';' : // Uarrocir;
                        match = "\u2949";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2829:
                switch (current) {
                    case 'r' :
                        state = 2830;
                        return true;
                }
                break;
            case 2830:
                switch (current) {
                    case 'c' :
                        state = 2831;
                        return true;
                    case 'e' :
                        state = 2833;
                        return true;
                }
                break;
            case 2831:
                switch (current) {
                    case 'y' :
                        state = 2832;
                        return true;
                }
                break;
            case 2832:
                switch (current) {
                    case ';' : // Ubrcy;
                        match = "\u040E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2833:
                switch (current) {
                    case 'v' :
                        state = 2834;
                        return true;
                }
                break;
            case 2834:
                switch (current) {
                    case 'e' :
                        state = 2835;
                        return true;
                }
                break;
            case 2835:
                switch (current) {
                    case ';' : // Ubreve;
                        match = "\u016C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2836:
                switch (current) {
                    case 'i' :
                        state = 2837;
                        return true;
                    case 'y' :
                        state = 2840;
                        return true;
                }
                break;
            case 2837:
                switch (current) {
                    case 'r' :
                        state = 2838;
                        return true;
                }
                break;
            case 2838:
                switch (current) {
                    case 'c' : // Ucirc
                        match = "\u00DB";
                        matchLength = consumedCount;
                        state = 2839;
                        return true;
                }
                break;
            case 2839:
                switch (current) {
                    case ';' : // Ucirc;
                        match = "\u00DB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2840:
                switch (current) {
                    case ';' : // Ucy;
                        match = "\u0423";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2841:
                switch (current) {
                    case 'b' :
                        state = 2842;
                        return true;
                }
                break;
            case 2842:
                switch (current) {
                    case 'l' :
                        state = 2843;
                        return true;
                }
                break;
            case 2843:
                switch (current) {
                    case 'a' :
                        state = 2844;
                        return true;
                }
                break;
            case 2844:
                switch (current) {
                    case 'c' :
                        state = 2845;
                        return true;
                }
                break;
            case 2845:
                switch (current) {
                    case ';' : // Udblac;
                        match = "\u0170";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2846:
                switch (current) {
                    case 'r' :
                        state = 2847;
                        return true;
                }
                break;
            case 2847:
                switch (current) {
                    case ';' : // Ufr;
                        match = "\uD835\uDD18";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2848:
                switch (current) {
                    case 'r' :
                        state = 2849;
                        return true;
                }
                break;
            case 2849:
                switch (current) {
                    case 'a' :
                        state = 2850;
                        return true;
                }
                break;
            case 2850:
                switch (current) {
                    case 'v' :
                        state = 2851;
                        return true;
                }
                break;
            case 2851:
                switch (current) {
                    case 'e' : // Ugrave
                        match = "\u00D9";
                        matchLength = consumedCount;
                        state = 2852;
                        return true;
                }
                break;
            case 2852:
                switch (current) {
                    case ';' : // Ugrave;
                        match = "\u00D9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2853:
                switch (current) {
                    case 'a' :
                        state = 2854;
                        return true;
                }
                break;
            case 2854:
                switch (current) {
                    case 'c' :
                        state = 2855;
                        return true;
                }
                break;
            case 2855:
                switch (current) {
                    case 'r' :
                        state = 2856;
                        return true;
                }
                break;
            case 2856:
                switch (current) {
                    case ';' : // Umacr;
                        match = "\u016A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2857:
                switch (current) {
                    case 'd' :
                        state = 2858;
                        return true;
                    case 'i' :
                        state = 2882;
                        return true;
                }
                break;
            case 2858:
                switch (current) {
                    case 'e' :
                        state = 2859;
                        return true;
                }
                break;
            case 2859:
                switch (current) {
                    case 'r' :
                        state = 2860;
                        return true;
                }
                break;
            case 2860:
                switch (current) {
                    case 'B' :
                        state = 2861;
                        return true;
                    case 'P' :
                        state = 2871;
                        return true;
                }
                break;
            case 2861:
                switch (current) {
                    case 'a' :
                        state = 2862;
                        return true;
                    case 'r' :
                        state = 2864;
                        return true;
                }
                break;
            case 2862:
                switch (current) {
                    case 'r' :
                        state = 2863;
                        return true;
                }
                break;
            case 2863:
                switch (current) {
                    case ';' : // UnderBar;
                        match = "_";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2864:
                switch (current) {
                    case 'a' :
                        state = 2865;
                        return true;
                }
                break;
            case 2865:
                switch (current) {
                    case 'c' :
                        state = 2866;
                        return true;
                }
                break;
            case 2866:
                switch (current) {
                    case 'e' :
                        state = 2867;
                        return true;
                    case 'k' :
                        state = 2868;
                        return true;
                }
                break;
            case 2867:
                switch (current) {
                    case ';' : // UnderBrace;
                        match = "\u23DF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2868:
                switch (current) {
                    case 'e' :
                        state = 2869;
                        return true;
                }
                break;
            case 2869:
                switch (current) {
                    case 't' :
                        state = 2870;
                        return true;
                }
                break;
            case 2870:
                switch (current) {
                    case ';' : // UnderBracket;
                        match = "\u23B5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2871:
                switch (current) {
                    case 'a' :
                        state = 2872;
                        return true;
                }
                break;
            case 2872:
                switch (current) {
                    case 'r' :
                        state = 2873;
                        return true;
                }
                break;
            case 2873:
                switch (current) {
                    case 'e' :
                        state = 2874;
                        return true;
                }
                break;
            case 2874:
                switch (current) {
                    case 'n' :
                        state = 2875;
                        return true;
                }
                break;
            case 2875:
                switch (current) {
                    case 't' :
                        state = 2876;
                        return true;
                }
                break;
            case 2876:
                switch (current) {
                    case 'h' :
                        state = 2877;
                        return true;
                }
                break;
            case 2877:
                switch (current) {
                    case 'e' :
                        state = 2878;
                        return true;
                }
                break;
            case 2878:
                switch (current) {
                    case 's' :
                        state = 2879;
                        return true;
                }
                break;
            case 2879:
                switch (current) {
                    case 'i' :
                        state = 2880;
                        return true;
                }
                break;
            case 2880:
                switch (current) {
                    case 's' :
                        state = 2881;
                        return true;
                }
                break;
            case 2881:
                switch (current) {
                    case ';' : // UnderParenthesis;
                        match = "\u23DD";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2882:
                switch (current) {
                    case 'o' :
                        state = 2883;
                        return true;
                }
                break;
            case 2883:
                switch (current) {
                    case 'n' :
                        state = 2884;
                        return true;
                }
                break;
            case 2884:
                switch (current) {
                    case ';' : // Union;
                        match = "\u22C3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'P' :
                        state = 2885;
                        return true;
                }
                break;
            case 2885:
                switch (current) {
                    case 'l' :
                        state = 2886;
                        return true;
                }
                break;
            case 2886:
                switch (current) {
                    case 'u' :
                        state = 2887;
                        return true;
                }
                break;
            case 2887:
                switch (current) {
                    case 's' :
                        state = 2888;
                        return true;
                }
                break;
            case 2888:
                switch (current) {
                    case ';' : // UnionPlus;
                        match = "\u228E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2889:
                switch (current) {
                    case 'g' :
                        state = 2890;
                        return true;
                    case 'p' :
                        state = 2893;
                        return true;
                }
                break;
            case 2890:
                switch (current) {
                    case 'o' :
                        state = 2891;
                        return true;
                }
                break;
            case 2891:
                switch (current) {
                    case 'n' :
                        state = 2892;
                        return true;
                }
                break;
            case 2892:
                switch (current) {
                    case ';' : // Uogon;
                        match = "\u0172";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2893:
                switch (current) {
                    case 'f' :
                        state = 2894;
                        return true;
                }
                break;
            case 2894:
                switch (current) {
                    case ';' : // Uopf;
                        match = "\uD835\uDD4C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2895:
                switch (current) {
                    case 'A' :
                        state = 2896;
                        return true;
                    case 'D' :
                        state = 2913;
                        return true;
                    case 'E' :
                        state = 2922;
                        return true;
                    case 'T' :
                        state = 2933;
                        return true;
                    case 'a' :
                        state = 2941;
                        return true;
                    case 'd' :
                        state = 2946;
                        return true;
                    case 'p' :
                        state = 2955;
                        return true;
                    case 's' :
                        state = 2977;
                        return true;
                }
                break;
            case 2896:
                switch (current) {
                    case 'r' :
                        state = 2897;
                        return true;
                }
                break;
            case 2897:
                switch (current) {
                    case 'r' :
                        state = 2898;
                        return true;
                }
                break;
            case 2898:
                switch (current) {
                    case 'o' :
                        state = 2899;
                        return true;
                }
                break;
            case 2899:
                switch (current) {
                    case 'w' :
                        state = 2900;
                        return true;
                }
                break;
            case 2900:
                switch (current) {
                    case ';' : // UpArrow;
                        match = "\u2191";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 2901;
                        return true;
                    case 'D' :
                        state = 2904;
                        return true;
                }
                break;
            case 2901:
                switch (current) {
                    case 'a' :
                        state = 2902;
                        return true;
                }
                break;
            case 2902:
                switch (current) {
                    case 'r' :
                        state = 2903;
                        return true;
                }
                break;
            case 2903:
                switch (current) {
                    case ';' : // UpArrowBar;
                        match = "\u2912";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2904:
                switch (current) {
                    case 'o' :
                        state = 2905;
                        return true;
                }
                break;
            case 2905:
                switch (current) {
                    case 'w' :
                        state = 2906;
                        return true;
                }
                break;
            case 2906:
                switch (current) {
                    case 'n' :
                        state = 2907;
                        return true;
                }
                break;
            case 2907:
                switch (current) {
                    case 'A' :
                        state = 2908;
                        return true;
                }
                break;
            case 2908:
                switch (current) {
                    case 'r' :
                        state = 2909;
                        return true;
                }
                break;
            case 2909:
                switch (current) {
                    case 'r' :
                        state = 2910;
                        return true;
                }
                break;
            case 2910:
                switch (current) {
                    case 'o' :
                        state = 2911;
                        return true;
                }
                break;
            case 2911:
                switch (current) {
                    case 'w' :
                        state = 2912;
                        return true;
                }
                break;
            case 2912:
                switch (current) {
                    case ';' : // UpArrowDownArrow;
                        match = "\u21C5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2913:
                switch (current) {
                    case 'o' :
                        state = 2914;
                        return true;
                }
                break;
            case 2914:
                switch (current) {
                    case 'w' :
                        state = 2915;
                        return true;
                }
                break;
            case 2915:
                switch (current) {
                    case 'n' :
                        state = 2916;
                        return true;
                }
                break;
            case 2916:
                switch (current) {
                    case 'A' :
                        state = 2917;
                        return true;
                }
                break;
            case 2917:
                switch (current) {
                    case 'r' :
                        state = 2918;
                        return true;
                }
                break;
            case 2918:
                switch (current) {
                    case 'r' :
                        state = 2919;
                        return true;
                }
                break;
            case 2919:
                switch (current) {
                    case 'o' :
                        state = 2920;
                        return true;
                }
                break;
            case 2920:
                switch (current) {
                    case 'w' :
                        state = 2921;
                        return true;
                }
                break;
            case 2921:
                switch (current) {
                    case ';' : // UpDownArrow;
                        match = "\u2195";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2922:
                switch (current) {
                    case 'q' :
                        state = 2923;
                        return true;
                }
                break;
            case 2923:
                switch (current) {
                    case 'u' :
                        state = 2924;
                        return true;
                }
                break;
            case 2924:
                switch (current) {
                    case 'i' :
                        state = 2925;
                        return true;
                }
                break;
            case 2925:
                switch (current) {
                    case 'l' :
                        state = 2926;
                        return true;
                }
                break;
            case 2926:
                switch (current) {
                    case 'i' :
                        state = 2927;
                        return true;
                }
                break;
            case 2927:
                switch (current) {
                    case 'b' :
                        state = 2928;
                        return true;
                }
                break;
            case 2928:
                switch (current) {
                    case 'r' :
                        state = 2929;
                        return true;
                }
                break;
            case 2929:
                switch (current) {
                    case 'i' :
                        state = 2930;
                        return true;
                }
                break;
            case 2930:
                switch (current) {
                    case 'u' :
                        state = 2931;
                        return true;
                }
                break;
            case 2931:
                switch (current) {
                    case 'm' :
                        state = 2932;
                        return true;
                }
                break;
            case 2932:
                switch (current) {
                    case ';' : // UpEquilibrium;
                        match = "\u296E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2933:
                switch (current) {
                    case 'e' :
                        state = 2934;
                        return true;
                }
                break;
            case 2934:
                switch (current) {
                    case 'e' :
                        state = 2935;
                        return true;
                }
                break;
            case 2935:
                switch (current) {
                    case ';' : // UpTee;
                        match = "\u22A5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'A' :
                        state = 2936;
                        return true;
                }
                break;
            case 2936:
                switch (current) {
                    case 'r' :
                        state = 2937;
                        return true;
                }
                break;
            case 2937:
                switch (current) {
                    case 'r' :
                        state = 2938;
                        return true;
                }
                break;
            case 2938:
                switch (current) {
                    case 'o' :
                        state = 2939;
                        return true;
                }
                break;
            case 2939:
                switch (current) {
                    case 'w' :
                        state = 2940;
                        return true;
                }
                break;
            case 2940:
                switch (current) {
                    case ';' : // UpTeeArrow;
                        match = "\u21A5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2941:
                switch (current) {
                    case 'r' :
                        state = 2942;
                        return true;
                }
                break;
            case 2942:
                switch (current) {
                    case 'r' :
                        state = 2943;
                        return true;
                }
                break;
            case 2943:
                switch (current) {
                    case 'o' :
                        state = 2944;
                        return true;
                }
                break;
            case 2944:
                switch (current) {
                    case 'w' :
                        state = 2945;
                        return true;
                }
                break;
            case 2945:
                switch (current) {
                    case ';' : // Uparrow;
                        match = "\u21D1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2946:
                switch (current) {
                    case 'o' :
                        state = 2947;
                        return true;
                }
                break;
            case 2947:
                switch (current) {
                    case 'w' :
                        state = 2948;
                        return true;
                }
                break;
            case 2948:
                switch (current) {
                    case 'n' :
                        state = 2949;
                        return true;
                }
                break;
            case 2949:
                switch (current) {
                    case 'a' :
                        state = 2950;
                        return true;
                }
                break;
            case 2950:
                switch (current) {
                    case 'r' :
                        state = 2951;
                        return true;
                }
                break;
            case 2951:
                switch (current) {
                    case 'r' :
                        state = 2952;
                        return true;
                }
                break;
            case 2952:
                switch (current) {
                    case 'o' :
                        state = 2953;
                        return true;
                }
                break;
            case 2953:
                switch (current) {
                    case 'w' :
                        state = 2954;
                        return true;
                }
                break;
            case 2954:
                switch (current) {
                    case ';' : // Updownarrow;
                        match = "\u21D5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2955:
                switch (current) {
                    case 'e' :
                        state = 2956;
                        return true;
                }
                break;
            case 2956:
                switch (current) {
                    case 'r' :
                        state = 2957;
                        return true;
                }
                break;
            case 2957:
                switch (current) {
                    case 'L' :
                        state = 2958;
                        return true;
                    case 'R' :
                        state = 2967;
                        return true;
                }
                break;
            case 2958:
                switch (current) {
                    case 'e' :
                        state = 2959;
                        return true;
                }
                break;
            case 2959:
                switch (current) {
                    case 'f' :
                        state = 2960;
                        return true;
                }
                break;
            case 2960:
                switch (current) {
                    case 't' :
                        state = 2961;
                        return true;
                }
                break;
            case 2961:
                switch (current) {
                    case 'A' :
                        state = 2962;
                        return true;
                }
                break;
            case 2962:
                switch (current) {
                    case 'r' :
                        state = 2963;
                        return true;
                }
                break;
            case 2963:
                switch (current) {
                    case 'r' :
                        state = 2964;
                        return true;
                }
                break;
            case 2964:
                switch (current) {
                    case 'o' :
                        state = 2965;
                        return true;
                }
                break;
            case 2965:
                switch (current) {
                    case 'w' :
                        state = 2966;
                        return true;
                }
                break;
            case 2966:
                switch (current) {
                    case ';' : // UpperLeftArrow;
                        match = "\u2196";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2967:
                switch (current) {
                    case 'i' :
                        state = 2968;
                        return true;
                }
                break;
            case 2968:
                switch (current) {
                    case 'g' :
                        state = 2969;
                        return true;
                }
                break;
            case 2969:
                switch (current) {
                    case 'h' :
                        state = 2970;
                        return true;
                }
                break;
            case 2970:
                switch (current) {
                    case 't' :
                        state = 2971;
                        return true;
                }
                break;
            case 2971:
                switch (current) {
                    case 'A' :
                        state = 2972;
                        return true;
                }
                break;
            case 2972:
                switch (current) {
                    case 'r' :
                        state = 2973;
                        return true;
                }
                break;
            case 2973:
                switch (current) {
                    case 'r' :
                        state = 2974;
                        return true;
                }
                break;
            case 2974:
                switch (current) {
                    case 'o' :
                        state = 2975;
                        return true;
                }
                break;
            case 2975:
                switch (current) {
                    case 'w' :
                        state = 2976;
                        return true;
                }
                break;
            case 2976:
                switch (current) {
                    case ';' : // UpperRightArrow;
                        match = "\u2197";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2977:
                switch (current) {
                    case 'i' :
                        state = 2978;
                        return true;
                }
                break;
            case 2978:
                switch (current) {
                    case ';' : // Upsi;
                        match = "\u03D2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'l' :
                        state = 2979;
                        return true;
                }
                break;
            case 2979:
                switch (current) {
                    case 'o' :
                        state = 2980;
                        return true;
                }
                break;
            case 2980:
                switch (current) {
                    case 'n' :
                        state = 2981;
                        return true;
                }
                break;
            case 2981:
                switch (current) {
                    case ';' : // Upsilon;
                        match = "\u03A5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2982:
                switch (current) {
                    case 'i' :
                        state = 2983;
                        return true;
                }
                break;
            case 2983:
                switch (current) {
                    case 'n' :
                        state = 2984;
                        return true;
                }
                break;
            case 2984:
                switch (current) {
                    case 'g' :
                        state = 2985;
                        return true;
                }
                break;
            case 2985:
                switch (current) {
                    case ';' : // Uring;
                        match = "\u016E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2986:
                switch (current) {
                    case 'c' :
                        state = 2987;
                        return true;
                }
                break;
            case 2987:
                switch (current) {
                    case 'r' :
                        state = 2988;
                        return true;
                }
                break;
            case 2988:
                switch (current) {
                    case ';' : // Uscr;
                        match = "\uD835\uDCB0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2989:
                switch (current) {
                    case 'i' :
                        state = 2990;
                        return true;
                }
                break;
            case 2990:
                switch (current) {
                    case 'l' :
                        state = 2991;
                        return true;
                }
                break;
            case 2991:
                switch (current) {
                    case 'd' :
                        state = 2992;
                        return true;
                }
                break;
            case 2992:
                switch (current) {
                    case 'e' :
                        state = 2993;
                        return true;
                }
                break;
            case 2993:
                switch (current) {
                    case ';' : // Utilde;
                        match = "\u0168";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2994:
                switch (current) {
                    case 'm' :
                        state = 2995;
                        return true;
                }
                break;
            case 2995:
                switch (current) {
                    case 'l' : // Uuml
                        match = "\u00DC";
                        matchLength = consumedCount;
                        state = 2996;
                        return true;
                }
                break;
            case 2996:
                switch (current) {
                    case ';' : // Uuml;
                        match = "\u00DC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 2997:
                switch (current) {
                    case 'D' :
                        state = 2998;
                        return true;
                    case 'b' :
                        state = 3002;
                        return true;
                    case 'c' :
                        state = 3005;
                        return true;
                    case 'd' :
                        state = 3007;
                        return true;
                    case 'e' :
                        state = 3012;
                        return true;
                    case 'f' :
                        state = 3054;
                        return true;
                    case 'o' :
                        state = 3056;
                        return true;
                    case 's' :
                        state = 3059;
                        return true;
                    case 'v' :
                        state = 3062;
                        return true;
                }
                break;
            case 2998:
                switch (current) {
                    case 'a' :
                        state = 2999;
                        return true;
                }
                break;
            case 2999:
                switch (current) {
                    case 's' :
                        state = 3000;
                        return true;
                }
                break;
        }
        return false;
    }

    private boolean parse4(final int current) {
        consumedCount++;
        switch (state) {
            case 3000:
                switch (current) {
                    case 'h' :
                        state = 3001;
                        return true;
                }
                break;
            case 3001:
                switch (current) {
                    case ';' : // VDash;
                        match = "\u22AB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3002:
                switch (current) {
                    case 'a' :
                        state = 3003;
                        return true;
                }
                break;
            case 3003:
                switch (current) {
                    case 'r' :
                        state = 3004;
                        return true;
                }
                break;
            case 3004:
                switch (current) {
                    case ';' : // Vbar;
                        match = "\u2AEB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3005:
                switch (current) {
                    case 'y' :
                        state = 3006;
                        return true;
                }
                break;
            case 3006:
                switch (current) {
                    case ';' : // Vcy;
                        match = "\u0412";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3007:
                switch (current) {
                    case 'a' :
                        state = 3008;
                        return true;
                }
                break;
            case 3008:
                switch (current) {
                    case 's' :
                        state = 3009;
                        return true;
                }
                break;
            case 3009:
                switch (current) {
                    case 'h' :
                        state = 3010;
                        return true;
                }
                break;
            case 3010:
                switch (current) {
                    case ';' : // Vdash;
                        match = "\u22A9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'l' :
                        state = 3011;
                        return true;
                }
                break;
            case 3011:
                switch (current) {
                    case ';' : // Vdashl;
                        match = "\u2AE6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3012:
                switch (current) {
                    case 'e' :
                        state = 3013;
                        return true;
                    case 'r' :
                        state = 3014;
                        return true;
                }
                break;
            case 3013:
                switch (current) {
                    case ';' : // Vee;
                        match = "\u22C1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3014:
                switch (current) {
                    case 'b' :
                        state = 3015;
                        return true;
                    case 't' :
                        state = 3018;
                        return true;
                    case 'y' :
                        state = 3044;
                        return true;
                }
                break;
            case 3015:
                switch (current) {
                    case 'a' :
                        state = 3016;
                        return true;
                }
                break;
            case 3016:
                switch (current) {
                    case 'r' :
                        state = 3017;
                        return true;
                }
                break;
            case 3017:
                switch (current) {
                    case ';' : // Verbar;
                        match = "\u2016";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3018:
                switch (current) {
                    case ';' : // Vert;
                        match = "\u2016";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'i' :
                        state = 3019;
                        return true;
                }
                break;
            case 3019:
                switch (current) {
                    case 'c' :
                        state = 3020;
                        return true;
                }
                break;
            case 3020:
                switch (current) {
                    case 'a' :
                        state = 3021;
                        return true;
                }
                break;
            case 3021:
                switch (current) {
                    case 'l' :
                        state = 3022;
                        return true;
                }
                break;
            case 3022:
                switch (current) {
                    case 'B' :
                        state = 3023;
                        return true;
                    case 'L' :
                        state = 3026;
                        return true;
                    case 'S' :
                        state = 3030;
                        return true;
                    case 'T' :
                        state = 3039;
                        return true;
                }
                break;
            case 3023:
                switch (current) {
                    case 'a' :
                        state = 3024;
                        return true;
                }
                break;
            case 3024:
                switch (current) {
                    case 'r' :
                        state = 3025;
                        return true;
                }
                break;
            case 3025:
                switch (current) {
                    case ';' : // VerticalBar;
                        match = "\u2223";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3026:
                switch (current) {
                    case 'i' :
                        state = 3027;
                        return true;
                }
                break;
            case 3027:
                switch (current) {
                    case 'n' :
                        state = 3028;
                        return true;
                }
                break;
            case 3028:
                switch (current) {
                    case 'e' :
                        state = 3029;
                        return true;
                }
                break;
            case 3029:
                switch (current) {
                    case ';' : // VerticalLine;
                        match = "|";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3030:
                switch (current) {
                    case 'e' :
                        state = 3031;
                        return true;
                }
                break;
            case 3031:
                switch (current) {
                    case 'p' :
                        state = 3032;
                        return true;
                }
                break;
            case 3032:
                switch (current) {
                    case 'a' :
                        state = 3033;
                        return true;
                }
                break;
            case 3033:
                switch (current) {
                    case 'r' :
                        state = 3034;
                        return true;
                }
                break;
            case 3034:
                switch (current) {
                    case 'a' :
                        state = 3035;
                        return true;
                }
                break;
            case 3035:
                switch (current) {
                    case 't' :
                        state = 3036;
                        return true;
                }
                break;
            case 3036:
                switch (current) {
                    case 'o' :
                        state = 3037;
                        return true;
                }
                break;
            case 3037:
                switch (current) {
                    case 'r' :
                        state = 3038;
                        return true;
                }
                break;
            case 3038:
                switch (current) {
                    case ';' : // VerticalSeparator;
                        match = "\u2758";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3039:
                switch (current) {
                    case 'i' :
                        state = 3040;
                        return true;
                }
                break;
            case 3040:
                switch (current) {
                    case 'l' :
                        state = 3041;
                        return true;
                }
                break;
            case 3041:
                switch (current) {
                    case 'd' :
                        state = 3042;
                        return true;
                }
                break;
            case 3042:
                switch (current) {
                    case 'e' :
                        state = 3043;
                        return true;
                }
                break;
            case 3043:
                switch (current) {
                    case ';' : // VerticalTilde;
                        match = "\u2240";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3044:
                switch (current) {
                    case 'T' :
                        state = 3045;
                        return true;
                }
                break;
            case 3045:
                switch (current) {
                    case 'h' :
                        state = 3046;
                        return true;
                }
                break;
            case 3046:
                switch (current) {
                    case 'i' :
                        state = 3047;
                        return true;
                }
                break;
            case 3047:
                switch (current) {
                    case 'n' :
                        state = 3048;
                        return true;
                }
                break;
            case 3048:
                switch (current) {
                    case 'S' :
                        state = 3049;
                        return true;
                }
                break;
            case 3049:
                switch (current) {
                    case 'p' :
                        state = 3050;
                        return true;
                }
                break;
            case 3050:
                switch (current) {
                    case 'a' :
                        state = 3051;
                        return true;
                }
                break;
            case 3051:
                switch (current) {
                    case 'c' :
                        state = 3052;
                        return true;
                }
                break;
            case 3052:
                switch (current) {
                    case 'e' :
                        state = 3053;
                        return true;
                }
                break;
            case 3053:
                switch (current) {
                    case ';' : // VeryThinSpace;
                        match = "\u200A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3054:
                switch (current) {
                    case 'r' :
                        state = 3055;
                        return true;
                }
                break;
            case 3055:
                switch (current) {
                    case ';' : // Vfr;
                        match = "\uD835\uDD19";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3056:
                switch (current) {
                    case 'p' :
                        state = 3057;
                        return true;
                }
                break;
            case 3057:
                switch (current) {
                    case 'f' :
                        state = 3058;
                        return true;
                }
                break;
            case 3058:
                switch (current) {
                    case ';' : // Vopf;
                        match = "\uD835\uDD4D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3059:
                switch (current) {
                    case 'c' :
                        state = 3060;
                        return true;
                }
                break;
            case 3060:
                switch (current) {
                    case 'r' :
                        state = 3061;
                        return true;
                }
                break;
            case 3061:
                switch (current) {
                    case ';' : // Vscr;
                        match = "\uD835\uDCB1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3062:
                switch (current) {
                    case 'd' :
                        state = 3063;
                        return true;
                }
                break;
            case 3063:
                switch (current) {
                    case 'a' :
                        state = 3064;
                        return true;
                }
                break;
            case 3064:
                switch (current) {
                    case 's' :
                        state = 3065;
                        return true;
                }
                break;
            case 3065:
                switch (current) {
                    case 'h' :
                        state = 3066;
                        return true;
                }
                break;
            case 3066:
                switch (current) {
                    case ';' : // Vvdash;
                        match = "\u22AA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3067:
                switch (current) {
                    case 'c' :
                        state = 3068;
                        return true;
                    case 'e' :
                        state = 3072;
                        return true;
                    case 'f' :
                        state = 3076;
                        return true;
                    case 'o' :
                        state = 3078;
                        return true;
                    case 's' :
                        state = 3081;
                        return true;
                }
                break;
            case 3068:
                switch (current) {
                    case 'i' :
                        state = 3069;
                        return true;
                }
                break;
            case 3069:
                switch (current) {
                    case 'r' :
                        state = 3070;
                        return true;
                }
                break;
            case 3070:
                switch (current) {
                    case 'c' :
                        state = 3071;
                        return true;
                }
                break;
            case 3071:
                switch (current) {
                    case ';' : // Wcirc;
                        match = "\u0174";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3072:
                switch (current) {
                    case 'd' :
                        state = 3073;
                        return true;
                }
                break;
            case 3073:
                switch (current) {
                    case 'g' :
                        state = 3074;
                        return true;
                }
                break;
            case 3074:
                switch (current) {
                    case 'e' :
                        state = 3075;
                        return true;
                }
                break;
            case 3075:
                switch (current) {
                    case ';' : // Wedge;
                        match = "\u22C0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3076:
                switch (current) {
                    case 'r' :
                        state = 3077;
                        return true;
                }
                break;
            case 3077:
                switch (current) {
                    case ';' : // Wfr;
                        match = "\uD835\uDD1A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3078:
                switch (current) {
                    case 'p' :
                        state = 3079;
                        return true;
                }
                break;
            case 3079:
                switch (current) {
                    case 'f' :
                        state = 3080;
                        return true;
                }
                break;
            case 3080:
                switch (current) {
                    case ';' : // Wopf;
                        match = "\uD835\uDD4E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3081:
                switch (current) {
                    case 'c' :
                        state = 3082;
                        return true;
                }
                break;
            case 3082:
                switch (current) {
                    case 'r' :
                        state = 3083;
                        return true;
                }
                break;
            case 3083:
                switch (current) {
                    case ';' : // Wscr;
                        match = "\uD835\uDCB2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3084:
                switch (current) {
                    case 'f' :
                        state = 3085;
                        return true;
                    case 'i' :
                        state = 3087;
                        return true;
                    case 'o' :
                        state = 3088;
                        return true;
                    case 's' :
                        state = 3091;
                        return true;
                }
                break;
            case 3085:
                switch (current) {
                    case 'r' :
                        state = 3086;
                        return true;
                }
                break;
            case 3086:
                switch (current) {
                    case ';' : // Xfr;
                        match = "\uD835\uDD1B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3087:
                switch (current) {
                    case ';' : // Xi;
                        match = "\u039E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3088:
                switch (current) {
                    case 'p' :
                        state = 3089;
                        return true;
                }
                break;
            case 3089:
                switch (current) {
                    case 'f' :
                        state = 3090;
                        return true;
                }
                break;
            case 3090:
                switch (current) {
                    case ';' : // Xopf;
                        match = "\uD835\uDD4F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3091:
                switch (current) {
                    case 'c' :
                        state = 3092;
                        return true;
                }
                break;
            case 3092:
                switch (current) {
                    case 'r' :
                        state = 3093;
                        return true;
                }
                break;
            case 3093:
                switch (current) {
                    case ';' : // Xscr;
                        match = "\uD835\uDCB3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3094:
                switch (current) {
                    case 'A' :
                        state = 3095;
                        return true;
                    case 'I' :
                        state = 3098;
                        return true;
                    case 'U' :
                        state = 3101;
                        return true;
                    case 'a' :
                        state = 3104;
                        return true;
                    case 'c' :
                        state = 3109;
                        return true;
                    case 'f' :
                        state = 3114;
                        return true;
                    case 'o' :
                        state = 3116;
                        return true;
                    case 's' :
                        state = 3119;
                        return true;
                    case 'u' :
                        state = 3122;
                        return true;
                }
                break;
            case 3095:
                switch (current) {
                    case 'c' :
                        state = 3096;
                        return true;
                }
                break;
            case 3096:
                switch (current) {
                    case 'y' :
                        state = 3097;
                        return true;
                }
                break;
            case 3097:
                switch (current) {
                    case ';' : // YAcy;
                        match = "\u042F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3098:
                switch (current) {
                    case 'c' :
                        state = 3099;
                        return true;
                }
                break;
            case 3099:
                switch (current) {
                    case 'y' :
                        state = 3100;
                        return true;
                }
                break;
            case 3100:
                switch (current) {
                    case ';' : // YIcy;
                        match = "\u0407";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3101:
                switch (current) {
                    case 'c' :
                        state = 3102;
                        return true;
                }
                break;
            case 3102:
                switch (current) {
                    case 'y' :
                        state = 3103;
                        return true;
                }
                break;
            case 3103:
                switch (current) {
                    case ';' : // YUcy;
                        match = "\u042E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3104:
                switch (current) {
                    case 'c' :
                        state = 3105;
                        return true;
                }
                break;
            case 3105:
                switch (current) {
                    case 'u' :
                        state = 3106;
                        return true;
                }
                break;
            case 3106:
                switch (current) {
                    case 't' :
                        state = 3107;
                        return true;
                }
                break;
            case 3107:
                switch (current) {
                    case 'e' : // Yacute
                        match = "\u00DD";
                        matchLength = consumedCount;
                        state = 3108;
                        return true;
                }
                break;
            case 3108:
                switch (current) {
                    case ';' : // Yacute;
                        match = "\u00DD";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3109:
                switch (current) {
                    case 'i' :
                        state = 3110;
                        return true;
                    case 'y' :
                        state = 3113;
                        return true;
                }
                break;
            case 3110:
                switch (current) {
                    case 'r' :
                        state = 3111;
                        return true;
                }
                break;
            case 3111:
                switch (current) {
                    case 'c' :
                        state = 3112;
                        return true;
                }
                break;
            case 3112:
                switch (current) {
                    case ';' : // Ycirc;
                        match = "\u0176";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3113:
                switch (current) {
                    case ';' : // Ycy;
                        match = "\u042B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3114:
                switch (current) {
                    case 'r' :
                        state = 3115;
                        return true;
                }
                break;
            case 3115:
                switch (current) {
                    case ';' : // Yfr;
                        match = "\uD835\uDD1C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3116:
                switch (current) {
                    case 'p' :
                        state = 3117;
                        return true;
                }
                break;
            case 3117:
                switch (current) {
                    case 'f' :
                        state = 3118;
                        return true;
                }
                break;
            case 3118:
                switch (current) {
                    case ';' : // Yopf;
                        match = "\uD835\uDD50";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3119:
                switch (current) {
                    case 'c' :
                        state = 3120;
                        return true;
                }
                break;
            case 3120:
                switch (current) {
                    case 'r' :
                        state = 3121;
                        return true;
                }
                break;
            case 3121:
                switch (current) {
                    case ';' : // Yscr;
                        match = "\uD835\uDCB4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3122:
                switch (current) {
                    case 'm' :
                        state = 3123;
                        return true;
                }
                break;
            case 3123:
                switch (current) {
                    case 'l' :
                        state = 3124;
                        return true;
                }
                break;
            case 3124:
                switch (current) {
                    case ';' : // Yuml;
                        match = "\u0178";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3125:
                switch (current) {
                    case 'H' :
                        state = 3126;
                        return true;
                    case 'a' :
                        state = 3129;
                        return true;
                    case 'c' :
                        state = 3134;
                        return true;
                    case 'd' :
                        state = 3140;
                        return true;
                    case 'e' :
                        state = 3143;
                        return true;
                    case 'f' :
                        state = 3158;
                        return true;
                    case 'o' :
                        state = 3160;
                        return true;
                    case 's' :
                        state = 3163;
                        return true;
                }
                break;
            case 3126:
                switch (current) {
                    case 'c' :
                        state = 3127;
                        return true;
                }
                break;
            case 3127:
                switch (current) {
                    case 'y' :
                        state = 3128;
                        return true;
                }
                break;
            case 3128:
                switch (current) {
                    case ';' : // ZHcy;
                        match = "\u0416";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3129:
                switch (current) {
                    case 'c' :
                        state = 3130;
                        return true;
                }
                break;
            case 3130:
                switch (current) {
                    case 'u' :
                        state = 3131;
                        return true;
                }
                break;
            case 3131:
                switch (current) {
                    case 't' :
                        state = 3132;
                        return true;
                }
                break;
            case 3132:
                switch (current) {
                    case 'e' :
                        state = 3133;
                        return true;
                }
                break;
            case 3133:
                switch (current) {
                    case ';' : // Zacute;
                        match = "\u0179";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3134:
                switch (current) {
                    case 'a' :
                        state = 3135;
                        return true;
                    case 'y' :
                        state = 3139;
                        return true;
                }
                break;
            case 3135:
                switch (current) {
                    case 'r' :
                        state = 3136;
                        return true;
                }
                break;
            case 3136:
                switch (current) {
                    case 'o' :
                        state = 3137;
                        return true;
                }
                break;
            case 3137:
                switch (current) {
                    case 'n' :
                        state = 3138;
                        return true;
                }
                break;
            case 3138:
                switch (current) {
                    case ';' : // Zcaron;
                        match = "\u017D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3139:
                switch (current) {
                    case ';' : // Zcy;
                        match = "\u0417";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3140:
                switch (current) {
                    case 'o' :
                        state = 3141;
                        return true;
                }
                break;
            case 3141:
                switch (current) {
                    case 't' :
                        state = 3142;
                        return true;
                }
                break;
            case 3142:
                switch (current) {
                    case ';' : // Zdot;
                        match = "\u017B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3143:
                switch (current) {
                    case 'r' :
                        state = 3144;
                        return true;
                    case 't' :
                        state = 3156;
                        return true;
                }
                break;
            case 3144:
                switch (current) {
                    case 'o' :
                        state = 3145;
                        return true;
                }
                break;
            case 3145:
                switch (current) {
                    case 'W' :
                        state = 3146;
                        return true;
                }
                break;
            case 3146:
                switch (current) {
                    case 'i' :
                        state = 3147;
                        return true;
                }
                break;
            case 3147:
                switch (current) {
                    case 'd' :
                        state = 3148;
                        return true;
                }
                break;
            case 3148:
                switch (current) {
                    case 't' :
                        state = 3149;
                        return true;
                }
                break;
            case 3149:
                switch (current) {
                    case 'h' :
                        state = 3150;
                        return true;
                }
                break;
            case 3150:
                switch (current) {
                    case 'S' :
                        state = 3151;
                        return true;
                }
                break;
            case 3151:
                switch (current) {
                    case 'p' :
                        state = 3152;
                        return true;
                }
                break;
            case 3152:
                switch (current) {
                    case 'a' :
                        state = 3153;
                        return true;
                }
                break;
            case 3153:
                switch (current) {
                    case 'c' :
                        state = 3154;
                        return true;
                }
                break;
            case 3154:
                switch (current) {
                    case 'e' :
                        state = 3155;
                        return true;
                }
                break;
            case 3155:
                switch (current) {
                    case ';' : // ZeroWidthSpace;
                        match = "\u200B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3156:
                switch (current) {
                    case 'a' :
                        state = 3157;
                        return true;
                }
                break;
            case 3157:
                switch (current) {
                    case ';' : // Zeta;
                        match = "\u0396";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3158:
                switch (current) {
                    case 'r' :
                        state = 3159;
                        return true;
                }
                break;
            case 3159:
                switch (current) {
                    case ';' : // Zfr;
                        match = "\u2128";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3160:
                switch (current) {
                    case 'p' :
                        state = 3161;
                        return true;
                }
                break;
            case 3161:
                switch (current) {
                    case 'f' :
                        state = 3162;
                        return true;
                }
                break;
            case 3162:
                switch (current) {
                    case ';' : // Zopf;
                        match = "\u2124";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3163:
                switch (current) {
                    case 'c' :
                        state = 3164;
                        return true;
                }
                break;
            case 3164:
                switch (current) {
                    case 'r' :
                        state = 3165;
                        return true;
                }
                break;
            case 3165:
                switch (current) {
                    case ';' : // Zscr;
                        match = "\uD835\uDCB5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3166:
                switch (current) {
                    case 'a' :
                        state = 3167;
                        return true;
                    case 'b' :
                        state = 3172;
                        return true;
                    case 'c' :
                        state = 3177;
                        return true;
                    case 'e' :
                        state = 3187;
                        return true;
                    case 'f' :
                        state = 3191;
                        return true;
                    case 'g' :
                        state = 3193;
                        return true;
                    case 'l' :
                        state = 3198;
                        return true;
                    case 'm' :
                        state = 3209;
                        return true;
                    case 'n' :
                        state = 3216;
                        return true;
                    case 'o' :
                        state = 3257;
                        return true;
                    case 'p' :
                        state = 3263;
                        return true;
                    case 'r' :
                        state = 3280;
                        return true;
                    case 's' :
                        state = 3284;
                        return true;
                    case 't' :
                        state = 3293;
                        return true;
                    case 'u' :
                        state = 3298;
                        return true;
                    case 'w' :
                        state = 3301;
                        return true;
                }
                break;
            case 3167:
                switch (current) {
                    case 'c' :
                        state = 3168;
                        return true;
                }
                break;
            case 3168:
                switch (current) {
                    case 'u' :
                        state = 3169;
                        return true;
                }
                break;
            case 3169:
                switch (current) {
                    case 't' :
                        state = 3170;
                        return true;
                }
                break;
            case 3170:
                switch (current) {
                    case 'e' : // aacute
                        match = "\u00E1";
                        matchLength = consumedCount;
                        state = 3171;
                        return true;
                }
                break;
            case 3171:
                switch (current) {
                    case ';' : // aacute;
                        match = "\u00E1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3172:
                switch (current) {
                    case 'r' :
                        state = 3173;
                        return true;
                }
                break;
            case 3173:
                switch (current) {
                    case 'e' :
                        state = 3174;
                        return true;
                }
                break;
            case 3174:
                switch (current) {
                    case 'v' :
                        state = 3175;
                        return true;
                }
                break;
            case 3175:
                switch (current) {
                    case 'e' :
                        state = 3176;
                        return true;
                }
                break;
            case 3176:
                switch (current) {
                    case ';' : // abreve;
                        match = "\u0103";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3177:
                switch (current) {
                    case ';' : // ac;
                        match = "\u223E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 3178;
                        return true;
                    case 'd' :
                        state = 3179;
                        return true;
                    case 'i' :
                        state = 3180;
                        return true;
                    case 'u' :
                        state = 3183;
                        return true;
                    case 'y' :
                        state = 3186;
                        return true;
                }
                break;
            case 3178:
                switch (current) {
                    case ';' : // acE;
                        match = "\u223E\u0333";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3179:
                switch (current) {
                    case ';' : // acd;
                        match = "\u223F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3180:
                switch (current) {
                    case 'r' :
                        state = 3181;
                        return true;
                }
                break;
            case 3181:
                switch (current) {
                    case 'c' : // acirc
                        match = "\u00E2";
                        matchLength = consumedCount;
                        state = 3182;
                        return true;
                }
                break;
            case 3182:
                switch (current) {
                    case ';' : // acirc;
                        match = "\u00E2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3183:
                switch (current) {
                    case 't' :
                        state = 3184;
                        return true;
                }
                break;
            case 3184:
                switch (current) {
                    case 'e' : // acute
                        match = "\u00B4";
                        matchLength = consumedCount;
                        state = 3185;
                        return true;
                }
                break;
            case 3185:
                switch (current) {
                    case ';' : // acute;
                        match = "\u00B4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3186:
                switch (current) {
                    case ';' : // acy;
                        match = "\u0430";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3187:
                switch (current) {
                    case 'l' :
                        state = 3188;
                        return true;
                }
                break;
            case 3188:
                switch (current) {
                    case 'i' :
                        state = 3189;
                        return true;
                }
                break;
            case 3189:
                switch (current) {
                    case 'g' : // aelig
                        match = "\u00E6";
                        matchLength = consumedCount;
                        state = 3190;
                        return true;
                }
                break;
            case 3190:
                switch (current) {
                    case ';' : // aelig;
                        match = "\u00E6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3191:
                switch (current) {
                    case ';' : // af;
                        match = "\u2061";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'r' :
                        state = 3192;
                        return true;
                }
                break;
            case 3192:
                switch (current) {
                    case ';' : // afr;
                        match = "\uD835\uDD1E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3193:
                switch (current) {
                    case 'r' :
                        state = 3194;
                        return true;
                }
                break;
            case 3194:
                switch (current) {
                    case 'a' :
                        state = 3195;
                        return true;
                }
                break;
            case 3195:
                switch (current) {
                    case 'v' :
                        state = 3196;
                        return true;
                }
                break;
            case 3196:
                switch (current) {
                    case 'e' : // agrave
                        match = "\u00E0";
                        matchLength = consumedCount;
                        state = 3197;
                        return true;
                }
                break;
            case 3197:
                switch (current) {
                    case ';' : // agrave;
                        match = "\u00E0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3198:
                switch (current) {
                    case 'e' :
                        state = 3199;
                        return true;
                    case 'p' :
                        state = 3206;
                        return true;
                }
                break;
            case 3199:
                switch (current) {
                    case 'f' :
                        state = 3200;
                        return true;
                    case 'p' :
                        state = 3204;
                        return true;
                }
                break;
            case 3200:
                switch (current) {
                    case 's' :
                        state = 3201;
                        return true;
                }
                break;
            case 3201:
                switch (current) {
                    case 'y' :
                        state = 3202;
                        return true;
                }
                break;
            case 3202:
                switch (current) {
                    case 'm' :
                        state = 3203;
                        return true;
                }
                break;
            case 3203:
                switch (current) {
                    case ';' : // alefsym;
                        match = "\u2135";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3204:
                switch (current) {
                    case 'h' :
                        state = 3205;
                        return true;
                }
                break;
            case 3205:
                switch (current) {
                    case ';' : // aleph;
                        match = "\u2135";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3206:
                switch (current) {
                    case 'h' :
                        state = 3207;
                        return true;
                }
                break;
            case 3207:
                switch (current) {
                    case 'a' :
                        state = 3208;
                        return true;
                }
                break;
            case 3208:
                switch (current) {
                    case ';' : // alpha;
                        match = "\u03B1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3209:
                switch (current) {
                    case 'a' :
                        state = 3210;
                        return true;
                    case 'p' : // amp
                        match = "&";
                        matchLength = consumedCount;
                        state = 3215;
                        return true;
                }
                break;
            case 3210:
                switch (current) {
                    case 'c' :
                        state = 3211;
                        return true;
                    case 'l' :
                        state = 3213;
                        return true;
                }
                break;
            case 3211:
                switch (current) {
                    case 'r' :
                        state = 3212;
                        return true;
                }
                break;
            case 3212:
                switch (current) {
                    case ';' : // amacr;
                        match = "\u0101";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3213:
                switch (current) {
                    case 'g' :
                        state = 3214;
                        return true;
                }
                break;
            case 3214:
                switch (current) {
                    case ';' : // amalg;
                        match = "\u2A3F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3215:
                switch (current) {
                    case ';' : // amp;
                        match = "&";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3216:
                switch (current) {
                    case 'd' :
                        state = 3217;
                        return true;
                    case 'g' :
                        state = 3228;
                        return true;
                }
                break;
            case 3217:
                switch (current) {
                    case ';' : // and;
                        match = "\u2227";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 3218;
                        return true;
                    case 'd' :
                        state = 3221;
                        return true;
                    case 's' :
                        state = 3222;
                        return true;
                    case 'v' :
                        state = 3227;
                        return true;
                }
                break;
            case 3218:
                switch (current) {
                    case 'n' :
                        state = 3219;
                        return true;
                }
                break;
            case 3219:
                switch (current) {
                    case 'd' :
                        state = 3220;
                        return true;
                }
                break;
            case 3220:
                switch (current) {
                    case ';' : // andand;
                        match = "\u2A55";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3221:
                switch (current) {
                    case ';' : // andd;
                        match = "\u2A5C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3222:
                switch (current) {
                    case 'l' :
                        state = 3223;
                        return true;
                }
                break;
            case 3223:
                switch (current) {
                    case 'o' :
                        state = 3224;
                        return true;
                }
                break;
            case 3224:
                switch (current) {
                    case 'p' :
                        state = 3225;
                        return true;
                }
                break;
            case 3225:
                switch (current) {
                    case 'e' :
                        state = 3226;
                        return true;
                }
                break;
            case 3226:
                switch (current) {
                    case ';' : // andslope;
                        match = "\u2A58";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3227:
                switch (current) {
                    case ';' : // andv;
                        match = "\u2A5A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3228:
                switch (current) {
                    case ';' : // ang;
                        match = "\u2220";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3229;
                        return true;
                    case 'l' :
                        state = 3230;
                        return true;
                    case 'm' :
                        state = 3232;
                        return true;
                    case 'r' :
                        state = 3244;
                        return true;
                    case 's' :
                        state = 3249;
                        return true;
                    case 'z' :
                        state = 3253;
                        return true;
                }
                break;
            case 3229:
                switch (current) {
                    case ';' : // ange;
                        match = "\u29A4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3230:
                switch (current) {
                    case 'e' :
                        state = 3231;
                        return true;
                }
                break;
            case 3231:
                switch (current) {
                    case ';' : // angle;
                        match = "\u2220";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3232:
                switch (current) {
                    case 's' :
                        state = 3233;
                        return true;
                }
                break;
            case 3233:
                switch (current) {
                    case 'd' :
                        state = 3234;
                        return true;
                }
                break;
            case 3234:
                switch (current) {
                    case ';' : // angmsd;
                        match = "\u2221";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 3235;
                        return true;
                }
                break;
            case 3235:
                switch (current) {
                    case 'a' :
                        state = 3236;
                        return true;
                    case 'b' :
                        state = 3237;
                        return true;
                    case 'c' :
                        state = 3238;
                        return true;
                    case 'd' :
                        state = 3239;
                        return true;
                    case 'e' :
                        state = 3240;
                        return true;
                    case 'f' :
                        state = 3241;
                        return true;
                    case 'g' :
                        state = 3242;
                        return true;
                    case 'h' :
                        state = 3243;
                        return true;
                }
                break;
            case 3236:
                switch (current) {
                    case ';' : // angmsdaa;
                        match = "\u29A8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3237:
                switch (current) {
                    case ';' : // angmsdab;
                        match = "\u29A9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3238:
                switch (current) {
                    case ';' : // angmsdac;
                        match = "\u29AA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3239:
                switch (current) {
                    case ';' : // angmsdad;
                        match = "\u29AB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3240:
                switch (current) {
                    case ';' : // angmsdae;
                        match = "\u29AC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3241:
                switch (current) {
                    case ';' : // angmsdaf;
                        match = "\u29AD";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3242:
                switch (current) {
                    case ';' : // angmsdag;
                        match = "\u29AE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3243:
                switch (current) {
                    case ';' : // angmsdah;
                        match = "\u29AF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3244:
                switch (current) {
                    case 't' :
                        state = 3245;
                        return true;
                }
                break;
            case 3245:
                switch (current) {
                    case ';' : // angrt;
                        match = "\u221F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'v' :
                        state = 3246;
                        return true;
                }
                break;
            case 3246:
                switch (current) {
                    case 'b' :
                        state = 3247;
                        return true;
                }
                break;
            case 3247:
                switch (current) {
                    case ';' : // angrtvb;
                        match = "\u22BE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'd' :
                        state = 3248;
                        return true;
                }
                break;
            case 3248:
                switch (current) {
                    case ';' : // angrtvbd;
                        match = "\u299D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3249:
                switch (current) {
                    case 'p' :
                        state = 3250;
                        return true;
                    case 't' :
                        state = 3252;
                        return true;
                }
                break;
            case 3250:
                switch (current) {
                    case 'h' :
                        state = 3251;
                        return true;
                }
                break;
            case 3251:
                switch (current) {
                    case ';' : // angsph;
                        match = "\u2222";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3252:
                switch (current) {
                    case ';' : // angst;
                        match = "\u00C5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3253:
                switch (current) {
                    case 'a' :
                        state = 3254;
                        return true;
                }
                break;
            case 3254:
                switch (current) {
                    case 'r' :
                        state = 3255;
                        return true;
                }
                break;
            case 3255:
                switch (current) {
                    case 'r' :
                        state = 3256;
                        return true;
                }
                break;
            case 3256:
                switch (current) {
                    case ';' : // angzarr;
                        match = "\u237C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3257:
                switch (current) {
                    case 'g' :
                        state = 3258;
                        return true;
                    case 'p' :
                        state = 3261;
                        return true;
                }
                break;
            case 3258:
                switch (current) {
                    case 'o' :
                        state = 3259;
                        return true;
                }
                break;
            case 3259:
                switch (current) {
                    case 'n' :
                        state = 3260;
                        return true;
                }
                break;
            case 3260:
                switch (current) {
                    case ';' : // aogon;
                        match = "\u0105";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3261:
                switch (current) {
                    case 'f' :
                        state = 3262;
                        return true;
                }
                break;
            case 3262:
                switch (current) {
                    case ';' : // aopf;
                        match = "\uD835\uDD52";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3263:
                switch (current) {
                    case ';' : // ap;
                        match = "\u2248";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 3264;
                        return true;
                    case 'a' :
                        state = 3265;
                        return true;
                    case 'e' :
                        state = 3269;
                        return true;
                    case 'i' :
                        state = 3270;
                        return true;
                    case 'o' :
                        state = 3272;
                        return true;
                    case 'p' :
                        state = 3274;
                        return true;
                }
                break;
            case 3264:
                switch (current) {
                    case ';' : // apE;
                        match = "\u2A70";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3265:
                switch (current) {
                    case 'c' :
                        state = 3266;
                        return true;
                }
                break;
            case 3266:
                switch (current) {
                    case 'i' :
                        state = 3267;
                        return true;
                }
                break;
            case 3267:
                switch (current) {
                    case 'r' :
                        state = 3268;
                        return true;
                }
                break;
            case 3268:
                switch (current) {
                    case ';' : // apacir;
                        match = "\u2A6F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3269:
                switch (current) {
                    case ';' : // ape;
                        match = "\u224A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3270:
                switch (current) {
                    case 'd' :
                        state = 3271;
                        return true;
                }
                break;
            case 3271:
                switch (current) {
                    case ';' : // apid;
                        match = "\u224B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3272:
                switch (current) {
                    case 's' :
                        state = 3273;
                        return true;
                }
                break;
            case 3273:
                switch (current) {
                    case ';' : // apos;
                        match = "'";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3274:
                switch (current) {
                    case 'r' :
                        state = 3275;
                        return true;
                }
                break;
            case 3275:
                switch (current) {
                    case 'o' :
                        state = 3276;
                        return true;
                }
                break;
            case 3276:
                switch (current) {
                    case 'x' :
                        state = 3277;
                        return true;
                }
                break;
            case 3277:
                switch (current) {
                    case ';' : // approx;
                        match = "\u2248";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3278;
                        return true;
                }
                break;
            case 3278:
                switch (current) {
                    case 'q' :
                        state = 3279;
                        return true;
                }
                break;
            case 3279:
                switch (current) {
                    case ';' : // approxeq;
                        match = "\u224A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3280:
                switch (current) {
                    case 'i' :
                        state = 3281;
                        return true;
                }
                break;
            case 3281:
                switch (current) {
                    case 'n' :
                        state = 3282;
                        return true;
                }
                break;
            case 3282:
                switch (current) {
                    case 'g' : // aring
                        match = "\u00E5";
                        matchLength = consumedCount;
                        state = 3283;
                        return true;
                }
                break;
            case 3283:
                switch (current) {
                    case ';' : // aring;
                        match = "\u00E5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3284:
                switch (current) {
                    case 'c' :
                        state = 3285;
                        return true;
                    case 't' :
                        state = 3287;
                        return true;
                    case 'y' :
                        state = 3288;
                        return true;
                }
                break;
            case 3285:
                switch (current) {
                    case 'r' :
                        state = 3286;
                        return true;
                }
                break;
            case 3286:
                switch (current) {
                    case ';' : // ascr;
                        match = "\uD835\uDCB6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3287:
                switch (current) {
                    case ';' : // ast;
                        match = "*";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3288:
                switch (current) {
                    case 'm' :
                        state = 3289;
                        return true;
                }
                break;
            case 3289:
                switch (current) {
                    case 'p' :
                        state = 3290;
                        return true;
                }
                break;
            case 3290:
                switch (current) {
                    case ';' : // asymp;
                        match = "\u2248";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3291;
                        return true;
                }
                break;
            case 3291:
                switch (current) {
                    case 'q' :
                        state = 3292;
                        return true;
                }
                break;
            case 3292:
                switch (current) {
                    case ';' : // asympeq;
                        match = "\u224D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3293:
                switch (current) {
                    case 'i' :
                        state = 3294;
                        return true;
                }
                break;
            case 3294:
                switch (current) {
                    case 'l' :
                        state = 3295;
                        return true;
                }
                break;
            case 3295:
                switch (current) {
                    case 'd' :
                        state = 3296;
                        return true;
                }
                break;
            case 3296:
                switch (current) {
                    case 'e' : // atilde
                        match = "\u00E3";
                        matchLength = consumedCount;
                        state = 3297;
                        return true;
                }
                break;
            case 3297:
                switch (current) {
                    case ';' : // atilde;
                        match = "\u00E3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3298:
                switch (current) {
                    case 'm' :
                        state = 3299;
                        return true;
                }
                break;
            case 3299:
                switch (current) {
                    case 'l' : // auml
                        match = "\u00E4";
                        matchLength = consumedCount;
                        state = 3300;
                        return true;
                }
                break;
            case 3300:
                switch (current) {
                    case ';' : // auml;
                        match = "\u00E4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3301:
                switch (current) {
                    case 'c' :
                        state = 3302;
                        return true;
                    case 'i' :
                        state = 3308;
                        return true;
                }
                break;
            case 3302:
                switch (current) {
                    case 'o' :
                        state = 3303;
                        return true;
                }
                break;
            case 3303:
                switch (current) {
                    case 'n' :
                        state = 3304;
                        return true;
                }
                break;
            case 3304:
                switch (current) {
                    case 'i' :
                        state = 3305;
                        return true;
                }
                break;
            case 3305:
                switch (current) {
                    case 'n' :
                        state = 3306;
                        return true;
                }
                break;
            case 3306:
                switch (current) {
                    case 't' :
                        state = 3307;
                        return true;
                }
                break;
            case 3307:
                switch (current) {
                    case ';' : // awconint;
                        match = "\u2233";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3308:
                switch (current) {
                    case 'n' :
                        state = 3309;
                        return true;
                }
                break;
            case 3309:
                switch (current) {
                    case 't' :
                        state = 3310;
                        return true;
                }
                break;
            case 3310:
                switch (current) {
                    case ';' : // awint;
                        match = "\u2A11";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3311:
                switch (current) {
                    case 'N' :
                        state = 3312;
                        return true;
                    case 'a' :
                        state = 3315;
                        return true;
                    case 'b' :
                        state = 3348;
                        return true;
                    case 'c' :
                        state = 3355;
                        return true;
                    case 'd' :
                        state = 3360;
                        return true;
                    case 'e' :
                        state = 3364;
                        return true;
                    case 'f' :
                        state = 3389;
                        return true;
                    case 'i' :
                        state = 3391;
                        return true;
                    case 'k' :
                        state = 3449;
                        return true;
                    case 'l' :
                        state = 3454;
                        return true;
                    case 'n' :
                        state = 3503;
                        return true;
                    case 'o' :
                        state = 3511;
                        return true;
                    case 'p' :
                        state = 3584;
                        return true;
                    case 'r' :
                        state = 3589;
                        return true;
                    case 's' :
                        state = 3597;
                        return true;
                    case 'u' :
                        state = 3613;
                        return true;
                }
                break;
            case 3312:
                switch (current) {
                    case 'o' :
                        state = 3313;
                        return true;
                }
                break;
            case 3313:
                switch (current) {
                    case 't' :
                        state = 3314;
                        return true;
                }
                break;
            case 3314:
                switch (current) {
                    case ';' : // bNot;
                        match = "\u2AED";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3315:
                switch (current) {
                    case 'c' :
                        state = 3316;
                        return true;
                    case 'r' :
                        state = 3339;
                        return true;
                }
                break;
            case 3316:
                switch (current) {
                    case 'k' :
                        state = 3317;
                        return true;
                }
                break;
            case 3317:
                switch (current) {
                    case 'c' :
                        state = 3318;
                        return true;
                    case 'e' :
                        state = 3322;
                        return true;
                    case 'p' :
                        state = 3329;
                        return true;
                    case 's' :
                        state = 3334;
                        return true;
                }
                break;
            case 3318:
                switch (current) {
                    case 'o' :
                        state = 3319;
                        return true;
                }
                break;
            case 3319:
                switch (current) {
                    case 'n' :
                        state = 3320;
                        return true;
                }
                break;
            case 3320:
                switch (current) {
                    case 'g' :
                        state = 3321;
                        return true;
                }
                break;
            case 3321:
                switch (current) {
                    case ';' : // backcong;
                        match = "\u224C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3322:
                switch (current) {
                    case 'p' :
                        state = 3323;
                        return true;
                }
                break;
            case 3323:
                switch (current) {
                    case 's' :
                        state = 3324;
                        return true;
                }
                break;
            case 3324:
                switch (current) {
                    case 'i' :
                        state = 3325;
                        return true;
                }
                break;
            case 3325:
                switch (current) {
                    case 'l' :
                        state = 3326;
                        return true;
                }
                break;
            case 3326:
                switch (current) {
                    case 'o' :
                        state = 3327;
                        return true;
                }
                break;
            case 3327:
                switch (current) {
                    case 'n' :
                        state = 3328;
                        return true;
                }
                break;
            case 3328:
                switch (current) {
                    case ';' : // backepsilon;
                        match = "\u03F6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3329:
                switch (current) {
                    case 'r' :
                        state = 3330;
                        return true;
                }
                break;
            case 3330:
                switch (current) {
                    case 'i' :
                        state = 3331;
                        return true;
                }
                break;
            case 3331:
                switch (current) {
                    case 'm' :
                        state = 3332;
                        return true;
                }
                break;
            case 3332:
                switch (current) {
                    case 'e' :
                        state = 3333;
                        return true;
                }
                break;
            case 3333:
                switch (current) {
                    case ';' : // backprime;
                        match = "\u2035";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3334:
                switch (current) {
                    case 'i' :
                        state = 3335;
                        return true;
                }
                break;
            case 3335:
                switch (current) {
                    case 'm' :
                        state = 3336;
                        return true;
                }
                break;
            case 3336:
                switch (current) {
                    case ';' : // backsim;
                        match = "\u223D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3337;
                        return true;
                }
                break;
            case 3337:
                switch (current) {
                    case 'q' :
                        state = 3338;
                        return true;
                }
                break;
            case 3338:
                switch (current) {
                    case ';' : // backsimeq;
                        match = "\u22CD";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3339:
                switch (current) {
                    case 'v' :
                        state = 3340;
                        return true;
                    case 'w' :
                        state = 3343;
                        return true;
                }
                break;
            case 3340:
                switch (current) {
                    case 'e' :
                        state = 3341;
                        return true;
                }
                break;
            case 3341:
                switch (current) {
                    case 'e' :
                        state = 3342;
                        return true;
                }
                break;
            case 3342:
                switch (current) {
                    case ';' : // barvee;
                        match = "\u22BD";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3343:
                switch (current) {
                    case 'e' :
                        state = 3344;
                        return true;
                }
                break;
            case 3344:
                switch (current) {
                    case 'd' :
                        state = 3345;
                        return true;
                }
                break;
            case 3345:
                switch (current) {
                    case ';' : // barwed;
                        match = "\u2305";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'g' :
                        state = 3346;
                        return true;
                }
                break;
            case 3346:
                switch (current) {
                    case 'e' :
                        state = 3347;
                        return true;
                }
                break;
            case 3347:
                switch (current) {
                    case ';' : // barwedge;
                        match = "\u2305";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3348:
                switch (current) {
                    case 'r' :
                        state = 3349;
                        return true;
                }
                break;
            case 3349:
                switch (current) {
                    case 'k' :
                        state = 3350;
                        return true;
                }
                break;
            case 3350:
                switch (current) {
                    case ';' : // bbrk;
                        match = "\u23B5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 't' :
                        state = 3351;
                        return true;
                }
                break;
            case 3351:
                switch (current) {
                    case 'b' :
                        state = 3352;
                        return true;
                }
                break;
            case 3352:
                switch (current) {
                    case 'r' :
                        state = 3353;
                        return true;
                }
                break;
            case 3353:
                switch (current) {
                    case 'k' :
                        state = 3354;
                        return true;
                }
                break;
            case 3354:
                switch (current) {
                    case ';' : // bbrktbrk;
                        match = "\u23B6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3355:
                switch (current) {
                    case 'o' :
                        state = 3356;
                        return true;
                    case 'y' :
                        state = 3359;
                        return true;
                }
                break;
            case 3356:
                switch (current) {
                    case 'n' :
                        state = 3357;
                        return true;
                }
                break;
            case 3357:
                switch (current) {
                    case 'g' :
                        state = 3358;
                        return true;
                }
                break;
            case 3358:
                switch (current) {
                    case ';' : // bcong;
                        match = "\u224C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3359:
                switch (current) {
                    case ';' : // bcy;
                        match = "\u0431";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3360:
                switch (current) {
                    case 'q' :
                        state = 3361;
                        return true;
                }
                break;
            case 3361:
                switch (current) {
                    case 'u' :
                        state = 3362;
                        return true;
                }
                break;
            case 3362:
                switch (current) {
                    case 'o' :
                        state = 3363;
                        return true;
                }
                break;
            case 3363:
                switch (current) {
                    case ';' : // bdquo;
                        match = "\u201E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3364:
                switch (current) {
                    case 'c' :
                        state = 3365;
                        return true;
                    case 'm' :
                        state = 3370;
                        return true;
                    case 'p' :
                        state = 3375;
                        return true;
                    case 'r' :
                        state = 3378;
                        return true;
                    case 't' :
                        state = 3382;
                        return true;
                }
                break;
            case 3365:
                switch (current) {
                    case 'a' :
                        state = 3366;
                        return true;
                }
                break;
            case 3366:
                switch (current) {
                    case 'u' :
                        state = 3367;
                        return true;
                }
                break;
            case 3367:
                switch (current) {
                    case 's' :
                        state = 3368;
                        return true;
                }
                break;
            case 3368:
                switch (current) {
                    case ';' : // becaus;
                        match = "\u2235";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3369;
                        return true;
                }
                break;
            case 3369:
                switch (current) {
                    case ';' : // because;
                        match = "\u2235";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3370:
                switch (current) {
                    case 'p' :
                        state = 3371;
                        return true;
                }
                break;
            case 3371:
                switch (current) {
                    case 't' :
                        state = 3372;
                        return true;
                }
                break;
            case 3372:
                switch (current) {
                    case 'y' :
                        state = 3373;
                        return true;
                }
                break;
            case 3373:
                switch (current) {
                    case 'v' :
                        state = 3374;
                        return true;
                }
                break;
            case 3374:
                switch (current) {
                    case ';' : // bemptyv;
                        match = "\u29B0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3375:
                switch (current) {
                    case 's' :
                        state = 3376;
                        return true;
                }
                break;
            case 3376:
                switch (current) {
                    case 'i' :
                        state = 3377;
                        return true;
                }
                break;
            case 3377:
                switch (current) {
                    case ';' : // bepsi;
                        match = "\u03F6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3378:
                switch (current) {
                    case 'n' :
                        state = 3379;
                        return true;
                }
                break;
            case 3379:
                switch (current) {
                    case 'o' :
                        state = 3380;
                        return true;
                }
                break;
            case 3380:
                switch (current) {
                    case 'u' :
                        state = 3381;
                        return true;
                }
                break;
            case 3381:
                switch (current) {
                    case ';' : // bernou;
                        match = "\u212C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3382:
                switch (current) {
                    case 'a' :
                        state = 3383;
                        return true;
                    case 'h' :
                        state = 3384;
                        return true;
                    case 'w' :
                        state = 3385;
                        return true;
                }
                break;
            case 3383:
                switch (current) {
                    case ';' : // beta;
                        match = "\u03B2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3384:
                switch (current) {
                    case ';' : // beth;
                        match = "\u2136";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3385:
                switch (current) {
                    case 'e' :
                        state = 3386;
                        return true;
                }
                break;
            case 3386:
                switch (current) {
                    case 'e' :
                        state = 3387;
                        return true;
                }
                break;
            case 3387:
                switch (current) {
                    case 'n' :
                        state = 3388;
                        return true;
                }
                break;
            case 3388:
                switch (current) {
                    case ';' : // between;
                        match = "\u226C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3389:
                switch (current) {
                    case 'r' :
                        state = 3390;
                        return true;
                }
                break;
            case 3390:
                switch (current) {
                    case ';' : // bfr;
                        match = "\uD835\uDD1F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3391:
                switch (current) {
                    case 'g' :
                        state = 3392;
                        return true;
                }
                break;
            case 3392:
                switch (current) {
                    case 'c' :
                        state = 3393;
                        return true;
                    case 'o' :
                        state = 3401;
                        return true;
                    case 's' :
                        state = 3414;
                        return true;
                    case 't' :
                        state = 3422;
                        return true;
                    case 'u' :
                        state = 3436;
                        return true;
                    case 'v' :
                        state = 3441;
                        return true;
                    case 'w' :
                        state = 3444;
                        return true;
                }
                break;
            case 3393:
                switch (current) {
                    case 'a' :
                        state = 3394;
                        return true;
                    case 'i' :
                        state = 3396;
                        return true;
                    case 'u' :
                        state = 3399;
                        return true;
                }
                break;
            case 3394:
                switch (current) {
                    case 'p' :
                        state = 3395;
                        return true;
                }
                break;
            case 3395:
                switch (current) {
                    case ';' : // bigcap;
                        match = "\u22C2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3396:
                switch (current) {
                    case 'r' :
                        state = 3397;
                        return true;
                }
                break;
            case 3397:
                switch (current) {
                    case 'c' :
                        state = 3398;
                        return true;
                }
                break;
            case 3398:
                switch (current) {
                    case ';' : // bigcirc;
                        match = "\u25EF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3399:
                switch (current) {
                    case 'p' :
                        state = 3400;
                        return true;
                }
                break;
            case 3400:
                switch (current) {
                    case ';' : // bigcup;
                        match = "\u22C3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3401:
                switch (current) {
                    case 'd' :
                        state = 3402;
                        return true;
                    case 'p' :
                        state = 3405;
                        return true;
                    case 't' :
                        state = 3409;
                        return true;
                }
                break;
            case 3402:
                switch (current) {
                    case 'o' :
                        state = 3403;
                        return true;
                }
                break;
            case 3403:
                switch (current) {
                    case 't' :
                        state = 3404;
                        return true;
                }
                break;
            case 3404:
                switch (current) {
                    case ';' : // bigodot;
                        match = "\u2A00";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3405:
                switch (current) {
                    case 'l' :
                        state = 3406;
                        return true;
                }
                break;
            case 3406:
                switch (current) {
                    case 'u' :
                        state = 3407;
                        return true;
                }
                break;
            case 3407:
                switch (current) {
                    case 's' :
                        state = 3408;
                        return true;
                }
                break;
            case 3408:
                switch (current) {
                    case ';' : // bigoplus;
                        match = "\u2A01";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3409:
                switch (current) {
                    case 'i' :
                        state = 3410;
                        return true;
                }
                break;
            case 3410:
                switch (current) {
                    case 'm' :
                        state = 3411;
                        return true;
                }
                break;
            case 3411:
                switch (current) {
                    case 'e' :
                        state = 3412;
                        return true;
                }
                break;
            case 3412:
                switch (current) {
                    case 's' :
                        state = 3413;
                        return true;
                }
                break;
            case 3413:
                switch (current) {
                    case ';' : // bigotimes;
                        match = "\u2A02";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3414:
                switch (current) {
                    case 'q' :
                        state = 3415;
                        return true;
                    case 't' :
                        state = 3419;
                        return true;
                }
                break;
            case 3415:
                switch (current) {
                    case 'c' :
                        state = 3416;
                        return true;
                }
                break;
            case 3416:
                switch (current) {
                    case 'u' :
                        state = 3417;
                        return true;
                }
                break;
            case 3417:
                switch (current) {
                    case 'p' :
                        state = 3418;
                        return true;
                }
                break;
            case 3418:
                switch (current) {
                    case ';' : // bigsqcup;
                        match = "\u2A06";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3419:
                switch (current) {
                    case 'a' :
                        state = 3420;
                        return true;
                }
                break;
            case 3420:
                switch (current) {
                    case 'r' :
                        state = 3421;
                        return true;
                }
                break;
            case 3421:
                switch (current) {
                    case ';' : // bigstar;
                        match = "\u2605";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3422:
                switch (current) {
                    case 'r' :
                        state = 3423;
                        return true;
                }
                break;
            case 3423:
                switch (current) {
                    case 'i' :
                        state = 3424;
                        return true;
                }
                break;
            case 3424:
                switch (current) {
                    case 'a' :
                        state = 3425;
                        return true;
                }
                break;
            case 3425:
                switch (current) {
                    case 'n' :
                        state = 3426;
                        return true;
                }
                break;
            case 3426:
                switch (current) {
                    case 'g' :
                        state = 3427;
                        return true;
                }
                break;
            case 3427:
                switch (current) {
                    case 'l' :
                        state = 3428;
                        return true;
                }
                break;
            case 3428:
                switch (current) {
                    case 'e' :
                        state = 3429;
                        return true;
                }
                break;
            case 3429:
                switch (current) {
                    case 'd' :
                        state = 3430;
                        return true;
                    case 'u' :
                        state = 3434;
                        return true;
                }
                break;
            case 3430:
                switch (current) {
                    case 'o' :
                        state = 3431;
                        return true;
                }
                break;
            case 3431:
                switch (current) {
                    case 'w' :
                        state = 3432;
                        return true;
                }
                break;
            case 3432:
                switch (current) {
                    case 'n' :
                        state = 3433;
                        return true;
                }
                break;
            case 3433:
                switch (current) {
                    case ';' : // bigtriangledown;
                        match = "\u25BD";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3434:
                switch (current) {
                    case 'p' :
                        state = 3435;
                        return true;
                }
                break;
            case 3435:
                switch (current) {
                    case ';' : // bigtriangleup;
                        match = "\u25B3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3436:
                switch (current) {
                    case 'p' :
                        state = 3437;
                        return true;
                }
                break;
            case 3437:
                switch (current) {
                    case 'l' :
                        state = 3438;
                        return true;
                }
                break;
            case 3438:
                switch (current) {
                    case 'u' :
                        state = 3439;
                        return true;
                }
                break;
            case 3439:
                switch (current) {
                    case 's' :
                        state = 3440;
                        return true;
                }
                break;
            case 3440:
                switch (current) {
                    case ';' : // biguplus;
                        match = "\u2A04";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3441:
                switch (current) {
                    case 'e' :
                        state = 3442;
                        return true;
                }
                break;
            case 3442:
                switch (current) {
                    case 'e' :
                        state = 3443;
                        return true;
                }
                break;
            case 3443:
                switch (current) {
                    case ';' : // bigvee;
                        match = "\u22C1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3444:
                switch (current) {
                    case 'e' :
                        state = 3445;
                        return true;
                }
                break;
            case 3445:
                switch (current) {
                    case 'd' :
                        state = 3446;
                        return true;
                }
                break;
            case 3446:
                switch (current) {
                    case 'g' :
                        state = 3447;
                        return true;
                }
                break;
            case 3447:
                switch (current) {
                    case 'e' :
                        state = 3448;
                        return true;
                }
                break;
            case 3448:
                switch (current) {
                    case ';' : // bigwedge;
                        match = "\u22C0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3449:
                switch (current) {
                    case 'a' :
                        state = 3450;
                        return true;
                }
                break;
            case 3450:
                switch (current) {
                    case 'r' :
                        state = 3451;
                        return true;
                }
                break;
            case 3451:
                switch (current) {
                    case 'o' :
                        state = 3452;
                        return true;
                }
                break;
            case 3452:
                switch (current) {
                    case 'w' :
                        state = 3453;
                        return true;
                }
                break;
            case 3453:
                switch (current) {
                    case ';' : // bkarow;
                        match = "\u290D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3454:
                switch (current) {
                    case 'a' :
                        state = 3455;
                        return true;
                    case 'k' :
                        state = 3494;
                        return true;
                    case 'o' :
                        state = 3500;
                        return true;
                }
                break;
            case 3455:
                switch (current) {
                    case 'c' :
                        state = 3456;
                        return true;
                    case 'n' :
                        state = 3492;
                        return true;
                }
                break;
            case 3456:
                switch (current) {
                    case 'k' :
                        state = 3457;
                        return true;
                }
                break;
            case 3457:
                switch (current) {
                    case 'l' :
                        state = 3458;
                        return true;
                    case 's' :
                        state = 3465;
                        return true;
                    case 't' :
                        state = 3471;
                        return true;
                }
                break;
            case 3458:
                switch (current) {
                    case 'o' :
                        state = 3459;
                        return true;
                }
                break;
            case 3459:
                switch (current) {
                    case 'z' :
                        state = 3460;
                        return true;
                }
                break;
            case 3460:
                switch (current) {
                    case 'e' :
                        state = 3461;
                        return true;
                }
                break;
            case 3461:
                switch (current) {
                    case 'n' :
                        state = 3462;
                        return true;
                }
                break;
            case 3462:
                switch (current) {
                    case 'g' :
                        state = 3463;
                        return true;
                }
                break;
            case 3463:
                switch (current) {
                    case 'e' :
                        state = 3464;
                        return true;
                }
                break;
            case 3464:
                switch (current) {
                    case ';' : // blacklozenge;
                        match = "\u29EB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3465:
                switch (current) {
                    case 'q' :
                        state = 3466;
                        return true;
                }
                break;
            case 3466:
                switch (current) {
                    case 'u' :
                        state = 3467;
                        return true;
                }
                break;
            case 3467:
                switch (current) {
                    case 'a' :
                        state = 3468;
                        return true;
                }
                break;
            case 3468:
                switch (current) {
                    case 'r' :
                        state = 3469;
                        return true;
                }
                break;
            case 3469:
                switch (current) {
                    case 'e' :
                        state = 3470;
                        return true;
                }
                break;
            case 3470:
                switch (current) {
                    case ';' : // blacksquare;
                        match = "\u25AA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3471:
                switch (current) {
                    case 'r' :
                        state = 3472;
                        return true;
                }
                break;
            case 3472:
                switch (current) {
                    case 'i' :
                        state = 3473;
                        return true;
                }
                break;
            case 3473:
                switch (current) {
                    case 'a' :
                        state = 3474;
                        return true;
                }
                break;
            case 3474:
                switch (current) {
                    case 'n' :
                        state = 3475;
                        return true;
                }
                break;
            case 3475:
                switch (current) {
                    case 'g' :
                        state = 3476;
                        return true;
                }
                break;
            case 3476:
                switch (current) {
                    case 'l' :
                        state = 3477;
                        return true;
                }
                break;
            case 3477:
                switch (current) {
                    case 'e' :
                        state = 3478;
                        return true;
                }
                break;
            case 3478:
                switch (current) {
                    case ';' : // blacktriangle;
                        match = "\u25B4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'd' :
                        state = 3479;
                        return true;
                    case 'l' :
                        state = 3483;
                        return true;
                    case 'r' :
                        state = 3487;
                        return true;
                }
                break;
            case 3479:
                switch (current) {
                    case 'o' :
                        state = 3480;
                        return true;
                }
                break;
            case 3480:
                switch (current) {
                    case 'w' :
                        state = 3481;
                        return true;
                }
                break;
            case 3481:
                switch (current) {
                    case 'n' :
                        state = 3482;
                        return true;
                }
                break;
            case 3482:
                switch (current) {
                    case ';' : // blacktriangledown;
                        match = "\u25BE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3483:
                switch (current) {
                    case 'e' :
                        state = 3484;
                        return true;
                }
                break;
            case 3484:
                switch (current) {
                    case 'f' :
                        state = 3485;
                        return true;
                }
                break;
            case 3485:
                switch (current) {
                    case 't' :
                        state = 3486;
                        return true;
                }
                break;
            case 3486:
                switch (current) {
                    case ';' : // blacktriangleleft;
                        match = "\u25C2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3487:
                switch (current) {
                    case 'i' :
                        state = 3488;
                        return true;
                }
                break;
            case 3488:
                switch (current) {
                    case 'g' :
                        state = 3489;
                        return true;
                }
                break;
            case 3489:
                switch (current) {
                    case 'h' :
                        state = 3490;
                        return true;
                }
                break;
            case 3490:
                switch (current) {
                    case 't' :
                        state = 3491;
                        return true;
                }
                break;
            case 3491:
                switch (current) {
                    case ';' : // blacktriangleright;
                        match = "\u25B8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3492:
                switch (current) {
                    case 'k' :
                        state = 3493;
                        return true;
                }
                break;
            case 3493:
                switch (current) {
                    case ';' : // blank;
                        match = "\u2423";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3494:
                switch (current) {
                    case '1' :
                        state = 3495;
                        return true;
                    case '3' :
                        state = 3498;
                        return true;
                }
                break;
            case 3495:
                switch (current) {
                    case '2' :
                        state = 3496;
                        return true;
                    case '4' :
                        state = 3497;
                        return true;
                }
                break;
            case 3496:
                switch (current) {
                    case ';' : // blk12;
                        match = "\u2592";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3497:
                switch (current) {
                    case ';' : // blk14;
                        match = "\u2591";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3498:
                switch (current) {
                    case '4' :
                        state = 3499;
                        return true;
                }
                break;
            case 3499:
                switch (current) {
                    case ';' : // blk34;
                        match = "\u2593";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3500:
                switch (current) {
                    case 'c' :
                        state = 3501;
                        return true;
                }
                break;
            case 3501:
                switch (current) {
                    case 'k' :
                        state = 3502;
                        return true;
                }
                break;
            case 3502:
                switch (current) {
                    case ';' : // block;
                        match = "\u2588";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3503:
                switch (current) {
                    case 'e' :
                        state = 3504;
                        return true;
                    case 'o' :
                        state = 3509;
                        return true;
                }
                break;
            case 3504:
                switch (current) {
                    case ';' : // bne;
                        match = "=\u20E5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'q' :
                        state = 3505;
                        return true;
                }
                break;
            case 3505:
                switch (current) {
                    case 'u' :
                        state = 3506;
                        return true;
                }
                break;
            case 3506:
                switch (current) {
                    case 'i' :
                        state = 3507;
                        return true;
                }
                break;
            case 3507:
                switch (current) {
                    case 'v' :
                        state = 3508;
                        return true;
                }
                break;
            case 3508:
                switch (current) {
                    case ';' : // bnequiv;
                        match = "\u2261\u20E5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3509:
                switch (current) {
                    case 't' :
                        state = 3510;
                        return true;
                }
                break;
            case 3510:
                switch (current) {
                    case ';' : // bnot;
                        match = "\u2310";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3511:
                switch (current) {
                    case 'p' :
                        state = 3512;
                        return true;
                    case 't' :
                        state = 3514;
                        return true;
                    case 'w' :
                        state = 3518;
                        return true;
                    case 'x' :
                        state = 3522;
                        return true;
                }
                break;
            case 3512:
                switch (current) {
                    case 'f' :
                        state = 3513;
                        return true;
                }
                break;
            case 3513:
                switch (current) {
                    case ';' : // bopf;
                        match = "\uD835\uDD53";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3514:
                switch (current) {
                    case ';' : // bot;
                        match = "\u22A5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 't' :
                        state = 3515;
                        return true;
                }
                break;
            case 3515:
                switch (current) {
                    case 'o' :
                        state = 3516;
                        return true;
                }
                break;
            case 3516:
                switch (current) {
                    case 'm' :
                        state = 3517;
                        return true;
                }
                break;
            case 3517:
                switch (current) {
                    case ';' : // bottom;
                        match = "\u22A5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3518:
                switch (current) {
                    case 't' :
                        state = 3519;
                        return true;
                }
                break;
            case 3519:
                switch (current) {
                    case 'i' :
                        state = 3520;
                        return true;
                }
                break;
            case 3520:
                switch (current) {
                    case 'e' :
                        state = 3521;
                        return true;
                }
                break;
            case 3521:
                switch (current) {
                    case ';' : // bowtie;
                        match = "\u22C8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3522:
                switch (current) {
                    case 'D' :
                        state = 3523;
                        return true;
                    case 'H' :
                        state = 3528;
                        return true;
                    case 'U' :
                        state = 3533;
                        return true;
                    case 'V' :
                        state = 3538;
                        return true;
                    case 'b' :
                        state = 3545;
                        return true;
                    case 'd' :
                        state = 3548;
                        return true;
                    case 'h' :
                        state = 3553;
                        return true;
                    case 'm' :
                        state = 3558;
                        return true;
                    case 'p' :
                        state = 3563;
                        return true;
                    case 't' :
                        state = 3567;
                        return true;
                    case 'u' :
                        state = 3572;
                        return true;
                    case 'v' :
                        state = 3577;
                        return true;
                }
                break;
            case 3523:
                switch (current) {
                    case 'L' :
                        state = 3524;
                        return true;
                    case 'R' :
                        state = 3525;
                        return true;
                    case 'l' :
                        state = 3526;
                        return true;
                    case 'r' :
                        state = 3527;
                        return true;
                }
                break;
            case 3524:
                switch (current) {
                    case ';' : // boxDL;
                        match = "\u2557";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3525:
                switch (current) {
                    case ';' : // boxDR;
                        match = "\u2554";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3526:
                switch (current) {
                    case ';' : // boxDl;
                        match = "\u2556";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3527:
                switch (current) {
                    case ';' : // boxDr;
                        match = "\u2553";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3528:
                switch (current) {
                    case ';' : // boxH;
                        match = "\u2550";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'D' :
                        state = 3529;
                        return true;
                    case 'U' :
                        state = 3530;
                        return true;
                    case 'd' :
                        state = 3531;
                        return true;
                    case 'u' :
                        state = 3532;
                        return true;
                }
                break;
            case 3529:
                switch (current) {
                    case ';' : // boxHD;
                        match = "\u2566";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3530:
                switch (current) {
                    case ';' : // boxHU;
                        match = "\u2569";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3531:
                switch (current) {
                    case ';' : // boxHd;
                        match = "\u2564";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3532:
                switch (current) {
                    case ';' : // boxHu;
                        match = "\u2567";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3533:
                switch (current) {
                    case 'L' :
                        state = 3534;
                        return true;
                    case 'R' :
                        state = 3535;
                        return true;
                    case 'l' :
                        state = 3536;
                        return true;
                    case 'r' :
                        state = 3537;
                        return true;
                }
                break;
            case 3534:
                switch (current) {
                    case ';' : // boxUL;
                        match = "\u255D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3535:
                switch (current) {
                    case ';' : // boxUR;
                        match = "\u255A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3536:
                switch (current) {
                    case ';' : // boxUl;
                        match = "\u255C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3537:
                switch (current) {
                    case ';' : // boxUr;
                        match = "\u2559";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3538:
                switch (current) {
                    case ';' : // boxV;
                        match = "\u2551";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'H' :
                        state = 3539;
                        return true;
                    case 'L' :
                        state = 3540;
                        return true;
                    case 'R' :
                        state = 3541;
                        return true;
                    case 'h' :
                        state = 3542;
                        return true;
                    case 'l' :
                        state = 3543;
                        return true;
                    case 'r' :
                        state = 3544;
                        return true;
                }
                break;
            case 3539:
                switch (current) {
                    case ';' : // boxVH;
                        match = "\u256C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3540:
                switch (current) {
                    case ';' : // boxVL;
                        match = "\u2563";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3541:
                switch (current) {
                    case ';' : // boxVR;
                        match = "\u2560";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3542:
                switch (current) {
                    case ';' : // boxVh;
                        match = "\u256B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3543:
                switch (current) {
                    case ';' : // boxVl;
                        match = "\u2562";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3544:
                switch (current) {
                    case ';' : // boxVr;
                        match = "\u255F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3545:
                switch (current) {
                    case 'o' :
                        state = 3546;
                        return true;
                }
                break;
            case 3546:
                switch (current) {
                    case 'x' :
                        state = 3547;
                        return true;
                }
                break;
            case 3547:
                switch (current) {
                    case ';' : // boxbox;
                        match = "\u29C9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3548:
                switch (current) {
                    case 'L' :
                        state = 3549;
                        return true;
                    case 'R' :
                        state = 3550;
                        return true;
                    case 'l' :
                        state = 3551;
                        return true;
                    case 'r' :
                        state = 3552;
                        return true;
                }
                break;
            case 3549:
                switch (current) {
                    case ';' : // boxdL;
                        match = "\u2555";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3550:
                switch (current) {
                    case ';' : // boxdR;
                        match = "\u2552";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3551:
                switch (current) {
                    case ';' : // boxdl;
                        match = "\u2510";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3552:
                switch (current) {
                    case ';' : // boxdr;
                        match = "\u250C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3553:
                switch (current) {
                    case ';' : // boxh;
                        match = "\u2500";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'D' :
                        state = 3554;
                        return true;
                    case 'U' :
                        state = 3555;
                        return true;
                    case 'd' :
                        state = 3556;
                        return true;
                    case 'u' :
                        state = 3557;
                        return true;
                }
                break;
            case 3554:
                switch (current) {
                    case ';' : // boxhD;
                        match = "\u2565";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3555:
                switch (current) {
                    case ';' : // boxhU;
                        match = "\u2568";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3556:
                switch (current) {
                    case ';' : // boxhd;
                        match = "\u252C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3557:
                switch (current) {
                    case ';' : // boxhu;
                        match = "\u2534";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3558:
                switch (current) {
                    case 'i' :
                        state = 3559;
                        return true;
                }
                break;
            case 3559:
                switch (current) {
                    case 'n' :
                        state = 3560;
                        return true;
                }
                break;
            case 3560:
                switch (current) {
                    case 'u' :
                        state = 3561;
                        return true;
                }
                break;
            case 3561:
                switch (current) {
                    case 's' :
                        state = 3562;
                        return true;
                }
                break;
            case 3562:
                switch (current) {
                    case ';' : // boxminus;
                        match = "\u229F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3563:
                switch (current) {
                    case 'l' :
                        state = 3564;
                        return true;
                }
                break;
            case 3564:
                switch (current) {
                    case 'u' :
                        state = 3565;
                        return true;
                }
                break;
            case 3565:
                switch (current) {
                    case 's' :
                        state = 3566;
                        return true;
                }
                break;
            case 3566:
                switch (current) {
                    case ';' : // boxplus;
                        match = "\u229E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3567:
                switch (current) {
                    case 'i' :
                        state = 3568;
                        return true;
                }
                break;
            case 3568:
                switch (current) {
                    case 'm' :
                        state = 3569;
                        return true;
                }
                break;
            case 3569:
                switch (current) {
                    case 'e' :
                        state = 3570;
                        return true;
                }
                break;
            case 3570:
                switch (current) {
                    case 's' :
                        state = 3571;
                        return true;
                }
                break;
            case 3571:
                switch (current) {
                    case ';' : // boxtimes;
                        match = "\u22A0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3572:
                switch (current) {
                    case 'L' :
                        state = 3573;
                        return true;
                    case 'R' :
                        state = 3574;
                        return true;
                    case 'l' :
                        state = 3575;
                        return true;
                    case 'r' :
                        state = 3576;
                        return true;
                }
                break;
            case 3573:
                switch (current) {
                    case ';' : // boxuL;
                        match = "\u255B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3574:
                switch (current) {
                    case ';' : // boxuR;
                        match = "\u2558";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3575:
                switch (current) {
                    case ';' : // boxul;
                        match = "\u2518";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3576:
                switch (current) {
                    case ';' : // boxur;
                        match = "\u2514";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3577:
                switch (current) {
                    case ';' : // boxv;
                        match = "\u2502";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'H' :
                        state = 3578;
                        return true;
                    case 'L' :
                        state = 3579;
                        return true;
                    case 'R' :
                        state = 3580;
                        return true;
                    case 'h' :
                        state = 3581;
                        return true;
                    case 'l' :
                        state = 3582;
                        return true;
                    case 'r' :
                        state = 3583;
                        return true;
                }
                break;
            case 3578:
                switch (current) {
                    case ';' : // boxvH;
                        match = "\u256A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3579:
                switch (current) {
                    case ';' : // boxvL;
                        match = "\u2561";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3580:
                switch (current) {
                    case ';' : // boxvR;
                        match = "\u255E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3581:
                switch (current) {
                    case ';' : // boxvh;
                        match = "\u253C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3582:
                switch (current) {
                    case ';' : // boxvl;
                        match = "\u2524";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3583:
                switch (current) {
                    case ';' : // boxvr;
                        match = "\u251C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3584:
                switch (current) {
                    case 'r' :
                        state = 3585;
                        return true;
                }
                break;
            case 3585:
                switch (current) {
                    case 'i' :
                        state = 3586;
                        return true;
                }
                break;
            case 3586:
                switch (current) {
                    case 'm' :
                        state = 3587;
                        return true;
                }
                break;
            case 3587:
                switch (current) {
                    case 'e' :
                        state = 3588;
                        return true;
                }
                break;
            case 3588:
                switch (current) {
                    case ';' : // bprime;
                        match = "\u2035";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3589:
                switch (current) {
                    case 'e' :
                        state = 3590;
                        return true;
                    case 'v' :
                        state = 3593;
                        return true;
                }
                break;
            case 3590:
                switch (current) {
                    case 'v' :
                        state = 3591;
                        return true;
                }
                break;
            case 3591:
                switch (current) {
                    case 'e' :
                        state = 3592;
                        return true;
                }
                break;
            case 3592:
                switch (current) {
                    case ';' : // breve;
                        match = "\u02D8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3593:
                switch (current) {
                    case 'b' :
                        state = 3594;
                        return true;
                }
                break;
            case 3594:
                switch (current) {
                    case 'a' :
                        state = 3595;
                        return true;
                }
                break;
            case 3595:
                switch (current) {
                    case 'r' : // brvbar
                        match = "\u00A6";
                        matchLength = consumedCount;
                        state = 3596;
                        return true;
                }
                break;
            case 3596:
                switch (current) {
                    case ';' : // brvbar;
                        match = "\u00A6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3597:
                switch (current) {
                    case 'c' :
                        state = 3598;
                        return true;
                    case 'e' :
                        state = 3600;
                        return true;
                    case 'i' :
                        state = 3603;
                        return true;
                    case 'o' :
                        state = 3606;
                        return true;
                }
                break;
            case 3598:
                switch (current) {
                    case 'r' :
                        state = 3599;
                        return true;
                }
                break;
            case 3599:
                switch (current) {
                    case ';' : // bscr;
                        match = "\uD835\uDCB7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3600:
                switch (current) {
                    case 'm' :
                        state = 3601;
                        return true;
                }
                break;
            case 3601:
                switch (current) {
                    case 'i' :
                        state = 3602;
                        return true;
                }
                break;
            case 3602:
                switch (current) {
                    case ';' : // bsemi;
                        match = "\u204F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3603:
                switch (current) {
                    case 'm' :
                        state = 3604;
                        return true;
                }
                break;
            case 3604:
                switch (current) {
                    case ';' : // bsim;
                        match = "\u223D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3605;
                        return true;
                }
                break;
            case 3605:
                switch (current) {
                    case ';' : // bsime;
                        match = "\u22CD";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3606:
                switch (current) {
                    case 'l' :
                        state = 3607;
                        return true;
                }
                break;
            case 3607:
                switch (current) {
                    case ';' : // bsol;
                        match = "\\";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'b' :
                        state = 3608;
                        return true;
                    case 'h' :
                        state = 3609;
                        return true;
                }
                break;
            case 3608:
                switch (current) {
                    case ';' : // bsolb;
                        match = "\u29C5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3609:
                switch (current) {
                    case 's' :
                        state = 3610;
                        return true;
                }
                break;
            case 3610:
                switch (current) {
                    case 'u' :
                        state = 3611;
                        return true;
                }
                break;
            case 3611:
                switch (current) {
                    case 'b' :
                        state = 3612;
                        return true;
                }
                break;
            case 3612:
                switch (current) {
                    case ';' : // bsolhsub;
                        match = "\u27C8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3613:
                switch (current) {
                    case 'l' :
                        state = 3614;
                        return true;
                    case 'm' :
                        state = 3618;
                        return true;
                }
                break;
            case 3614:
                switch (current) {
                    case 'l' :
                        state = 3615;
                        return true;
                }
                break;
            case 3615:
                switch (current) {
                    case ';' : // bull;
                        match = "\u2022";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3616;
                        return true;
                }
                break;
            case 3616:
                switch (current) {
                    case 't' :
                        state = 3617;
                        return true;
                }
                break;
            case 3617:
                switch (current) {
                    case ';' : // bullet;
                        match = "\u2022";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3618:
                switch (current) {
                    case 'p' :
                        state = 3619;
                        return true;
                }
                break;
            case 3619:
                switch (current) {
                    case ';' : // bump;
                        match = "\u224E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 3620;
                        return true;
                    case 'e' :
                        state = 3621;
                        return true;
                }
                break;
            case 3620:
                switch (current) {
                    case ';' : // bumpE;
                        match = "\u2AAE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3621:
                switch (current) {
                    case ';' : // bumpe;
                        match = "\u224F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'q' :
                        state = 3622;
                        return true;
                }
                break;
            case 3622:
                switch (current) {
                    case ';' : // bumpeq;
                        match = "\u224F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3623:
                switch (current) {
                    case 'a' :
                        state = 3624;
                        return true;
                    case 'c' :
                        state = 3652;
                        return true;
                    case 'd' :
                        state = 3671;
                        return true;
                    case 'e' :
                        state = 3674;
                        return true;
                    case 'f' :
                        state = 3690;
                        return true;
                    case 'h' :
                        state = 3692;
                        return true;
                    case 'i' :
                        state = 3703;
                        return true;
                    case 'l' :
                        state = 3752;
                        return true;
                    case 'o' :
                        state = 3759;
                        return true;
                    case 'r' :
                        state = 3797;
                        return true;
                    case 's' :
                        state = 3804;
                        return true;
                    case 't' :
                        state = 3812;
                        return true;
                    case 'u' :
                        state = 3816;
                        return true;
                    case 'w' :
                        state = 3900;
                        return true;
                    case 'y' :
                        state = 3910;
                        return true;
                }
                break;
            case 3624:
                switch (current) {
                    case 'c' :
                        state = 3625;
                        return true;
                    case 'p' :
                        state = 3629;
                        return true;
                    case 'r' :
                        state = 3647;
                        return true;
                }
                break;
            case 3625:
                switch (current) {
                    case 'u' :
                        state = 3626;
                        return true;
                }
                break;
            case 3626:
                switch (current) {
                    case 't' :
                        state = 3627;
                        return true;
                }
                break;
            case 3627:
                switch (current) {
                    case 'e' :
                        state = 3628;
                        return true;
                }
                break;
            case 3628:
                switch (current) {
                    case ';' : // cacute;
                        match = "\u0107";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3629:
                switch (current) {
                    case ';' : // cap;
                        match = "\u2229";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 3630;
                        return true;
                    case 'b' :
                        state = 3633;
                        return true;
                    case 'c' :
                        state = 3638;
                        return true;
                    case 'd' :
                        state = 3643;
                        return true;
                    case 's' :
                        state = 3646;
                        return true;
                }
                break;
            case 3630:
                switch (current) {
                    case 'n' :
                        state = 3631;
                        return true;
                }
                break;
            case 3631:
                switch (current) {
                    case 'd' :
                        state = 3632;
                        return true;
                }
                break;
            case 3632:
                switch (current) {
                    case ';' : // capand;
                        match = "\u2A44";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3633:
                switch (current) {
                    case 'r' :
                        state = 3634;
                        return true;
                }
                break;
            case 3634:
                switch (current) {
                    case 'c' :
                        state = 3635;
                        return true;
                }
                break;
            case 3635:
                switch (current) {
                    case 'u' :
                        state = 3636;
                        return true;
                }
                break;
            case 3636:
                switch (current) {
                    case 'p' :
                        state = 3637;
                        return true;
                }
                break;
            case 3637:
                switch (current) {
                    case ';' : // capbrcup;
                        match = "\u2A49";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3638:
                switch (current) {
                    case 'a' :
                        state = 3639;
                        return true;
                    case 'u' :
                        state = 3641;
                        return true;
                }
                break;
            case 3639:
                switch (current) {
                    case 'p' :
                        state = 3640;
                        return true;
                }
                break;
            case 3640:
                switch (current) {
                    case ';' : // capcap;
                        match = "\u2A4B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3641:
                switch (current) {
                    case 'p' :
                        state = 3642;
                        return true;
                }
                break;
            case 3642:
                switch (current) {
                    case ';' : // capcup;
                        match = "\u2A47";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3643:
                switch (current) {
                    case 'o' :
                        state = 3644;
                        return true;
                }
                break;
            case 3644:
                switch (current) {
                    case 't' :
                        state = 3645;
                        return true;
                }
                break;
            case 3645:
                switch (current) {
                    case ';' : // capdot;
                        match = "\u2A40";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3646:
                switch (current) {
                    case ';' : // caps;
                        match = "\u2229\uFE00";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3647:
                switch (current) {
                    case 'e' :
                        state = 3648;
                        return true;
                    case 'o' :
                        state = 3650;
                        return true;
                }
                break;
            case 3648:
                switch (current) {
                    case 't' :
                        state = 3649;
                        return true;
                }
                break;
            case 3649:
                switch (current) {
                    case ';' : // caret;
                        match = "\u2041";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3650:
                switch (current) {
                    case 'n' :
                        state = 3651;
                        return true;
                }
                break;
            case 3651:
                switch (current) {
                    case ';' : // caron;
                        match = "\u02C7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3652:
                switch (current) {
                    case 'a' :
                        state = 3653;
                        return true;
                    case 'e' :
                        state = 3659;
                        return true;
                    case 'i' :
                        state = 3663;
                        return true;
                    case 'u' :
                        state = 3666;
                        return true;
                }
                break;
            case 3653:
                switch (current) {
                    case 'p' :
                        state = 3654;
                        return true;
                    case 'r' :
                        state = 3656;
                        return true;
                }
                break;
            case 3654:
                switch (current) {
                    case 's' :
                        state = 3655;
                        return true;
                }
                break;
            case 3655:
                switch (current) {
                    case ';' : // ccaps;
                        match = "\u2A4D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3656:
                switch (current) {
                    case 'o' :
                        state = 3657;
                        return true;
                }
                break;
            case 3657:
                switch (current) {
                    case 'n' :
                        state = 3658;
                        return true;
                }
                break;
            case 3658:
                switch (current) {
                    case ';' : // ccaron;
                        match = "\u010D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3659:
                switch (current) {
                    case 'd' :
                        state = 3660;
                        return true;
                }
                break;
            case 3660:
                switch (current) {
                    case 'i' :
                        state = 3661;
                        return true;
                }
                break;
            case 3661:
                switch (current) {
                    case 'l' : // ccedil
                        match = "\u00E7";
                        matchLength = consumedCount;
                        state = 3662;
                        return true;
                }
                break;
            case 3662:
                switch (current) {
                    case ';' : // ccedil;
                        match = "\u00E7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3663:
                switch (current) {
                    case 'r' :
                        state = 3664;
                        return true;
                }
                break;
            case 3664:
                switch (current) {
                    case 'c' :
                        state = 3665;
                        return true;
                }
                break;
            case 3665:
                switch (current) {
                    case ';' : // ccirc;
                        match = "\u0109";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3666:
                switch (current) {
                    case 'p' :
                        state = 3667;
                        return true;
                }
                break;
            case 3667:
                switch (current) {
                    case 's' :
                        state = 3668;
                        return true;
                }
                break;
            case 3668:
                switch (current) {
                    case ';' : // ccups;
                        match = "\u2A4C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 's' :
                        state = 3669;
                        return true;
                }
                break;
            case 3669:
                switch (current) {
                    case 'm' :
                        state = 3670;
                        return true;
                }
                break;
            case 3670:
                switch (current) {
                    case ';' : // ccupssm;
                        match = "\u2A50";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3671:
                switch (current) {
                    case 'o' :
                        state = 3672;
                        return true;
                }
                break;
            case 3672:
                switch (current) {
                    case 't' :
                        state = 3673;
                        return true;
                }
                break;
            case 3673:
                switch (current) {
                    case ';' : // cdot;
                        match = "\u010B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3674:
                switch (current) {
                    case 'd' :
                        state = 3675;
                        return true;
                    case 'm' :
                        state = 3678;
                        return true;
                    case 'n' :
                        state = 3683;
                        return true;
                }
                break;
            case 3675:
                switch (current) {
                    case 'i' :
                        state = 3676;
                        return true;
                }
                break;
            case 3676:
                switch (current) {
                    case 'l' : // cedil
                        match = "\u00B8";
                        matchLength = consumedCount;
                        state = 3677;
                        return true;
                }
                break;
            case 3677:
                switch (current) {
                    case ';' : // cedil;
                        match = "\u00B8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3678:
                switch (current) {
                    case 'p' :
                        state = 3679;
                        return true;
                }
                break;
            case 3679:
                switch (current) {
                    case 't' :
                        state = 3680;
                        return true;
                }
                break;
            case 3680:
                switch (current) {
                    case 'y' :
                        state = 3681;
                        return true;
                }
                break;
            case 3681:
                switch (current) {
                    case 'v' :
                        state = 3682;
                        return true;
                }
                break;
            case 3682:
                switch (current) {
                    case ';' : // cemptyv;
                        match = "\u29B2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3683:
                switch (current) {
                    case 't' : // cent
                        match = "\u00A2";
                        matchLength = consumedCount;
                        state = 3684;
                        return true;
                }
                break;
            case 3684:
                switch (current) {
                    case ';' : // cent;
                        match = "\u00A2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3685;
                        return true;
                }
                break;
            case 3685:
                switch (current) {
                    case 'r' :
                        state = 3686;
                        return true;
                }
                break;
            case 3686:
                switch (current) {
                    case 'd' :
                        state = 3687;
                        return true;
                }
                break;
            case 3687:
                switch (current) {
                    case 'o' :
                        state = 3688;
                        return true;
                }
                break;
            case 3688:
                switch (current) {
                    case 't' :
                        state = 3689;
                        return true;
                }
                break;
            case 3689:
                switch (current) {
                    case ';' : // centerdot;
                        match = "\u00B7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3690:
                switch (current) {
                    case 'r' :
                        state = 3691;
                        return true;
                }
                break;
            case 3691:
                switch (current) {
                    case ';' : // cfr;
                        match = "\uD835\uDD20";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3692:
                switch (current) {
                    case 'c' :
                        state = 3693;
                        return true;
                    case 'e' :
                        state = 3695;
                        return true;
                    case 'i' :
                        state = 3702;
                        return true;
                }
                break;
            case 3693:
                switch (current) {
                    case 'y' :
                        state = 3694;
                        return true;
                }
                break;
            case 3694:
                switch (current) {
                    case ';' : // chcy;
                        match = "\u0447";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3695:
                switch (current) {
                    case 'c' :
                        state = 3696;
                        return true;
                }
                break;
            case 3696:
                switch (current) {
                    case 'k' :
                        state = 3697;
                        return true;
                }
                break;
            case 3697:
                switch (current) {
                    case ';' : // check;
                        match = "\u2713";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'm' :
                        state = 3698;
                        return true;
                }
                break;
            case 3698:
                switch (current) {
                    case 'a' :
                        state = 3699;
                        return true;
                }
                break;
            case 3699:
                switch (current) {
                    case 'r' :
                        state = 3700;
                        return true;
                }
                break;
            case 3700:
                switch (current) {
                    case 'k' :
                        state = 3701;
                        return true;
                }
                break;
            case 3701:
                switch (current) {
                    case ';' : // checkmark;
                        match = "\u2713";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3702:
                switch (current) {
                    case ';' : // chi;
                        match = "\u03C7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3703:
                switch (current) {
                    case 'r' :
                        state = 3704;
                        return true;
                }
                break;
            case 3704:
                switch (current) {
                    case ';' : // cir;
                        match = "\u25CB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 3705;
                        return true;
                    case 'c' :
                        state = 3706;
                        return true;
                    case 'e' :
                        state = 3739;
                        return true;
                    case 'f' :
                        state = 3740;
                        return true;
                    case 'm' :
                        state = 3745;
                        return true;
                    case 's' :
                        state = 3748;
                        return true;
                }
                break;
            case 3705:
                switch (current) {
                    case ';' : // cirE;
                        match = "\u29C3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3706:
                switch (current) {
                    case ';' : // circ;
                        match = "\u02C6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3707;
                        return true;
                    case 'l' :
                        state = 3709;
                        return true;
                }
                break;
            case 3707:
                switch (current) {
                    case 'q' :
                        state = 3708;
                        return true;
                }
                break;
            case 3708:
                switch (current) {
                    case ';' : // circeq;
                        match = "\u2257";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3709:
                switch (current) {
                    case 'e' :
                        state = 3710;
                        return true;
                }
                break;
            case 3710:
                switch (current) {
                    case 'a' :
                        state = 3711;
                        return true;
                    case 'd' :
                        state = 3725;
                        return true;
                }
                break;
            case 3711:
                switch (current) {
                    case 'r' :
                        state = 3712;
                        return true;
                }
                break;
            case 3712:
                switch (current) {
                    case 'r' :
                        state = 3713;
                        return true;
                }
                break;
            case 3713:
                switch (current) {
                    case 'o' :
                        state = 3714;
                        return true;
                }
                break;
            case 3714:
                switch (current) {
                    case 'w' :
                        state = 3715;
                        return true;
                }
                break;
            case 3715:
                switch (current) {
                    case 'l' :
                        state = 3716;
                        return true;
                    case 'r' :
                        state = 3720;
                        return true;
                }
                break;
            case 3716:
                switch (current) {
                    case 'e' :
                        state = 3717;
                        return true;
                }
                break;
            case 3717:
                switch (current) {
                    case 'f' :
                        state = 3718;
                        return true;
                }
                break;
            case 3718:
                switch (current) {
                    case 't' :
                        state = 3719;
                        return true;
                }
                break;
            case 3719:
                switch (current) {
                    case ';' : // circlearrowleft;
                        match = "\u21BA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3720:
                switch (current) {
                    case 'i' :
                        state = 3721;
                        return true;
                }
                break;
            case 3721:
                switch (current) {
                    case 'g' :
                        state = 3722;
                        return true;
                }
                break;
            case 3722:
                switch (current) {
                    case 'h' :
                        state = 3723;
                        return true;
                }
                break;
            case 3723:
                switch (current) {
                    case 't' :
                        state = 3724;
                        return true;
                }
                break;
            case 3724:
                switch (current) {
                    case ';' : // circlearrowright;
                        match = "\u21BB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3725:
                switch (current) {
                    case 'R' :
                        state = 3726;
                        return true;
                    case 'S' :
                        state = 3727;
                        return true;
                    case 'a' :
                        state = 3728;
                        return true;
                    case 'c' :
                        state = 3731;
                        return true;
                    case 'd' :
                        state = 3735;
                        return true;
                }
                break;
            case 3726:
                switch (current) {
                    case ';' : // circledR;
                        match = "\u00AE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3727:
                switch (current) {
                    case ';' : // circledS;
                        match = "\u24C8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3728:
                switch (current) {
                    case 's' :
                        state = 3729;
                        return true;
                }
                break;
            case 3729:
                switch (current) {
                    case 't' :
                        state = 3730;
                        return true;
                }
                break;
            case 3730:
                switch (current) {
                    case ';' : // circledast;
                        match = "\u229B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3731:
                switch (current) {
                    case 'i' :
                        state = 3732;
                        return true;
                }
                break;
            case 3732:
                switch (current) {
                    case 'r' :
                        state = 3733;
                        return true;
                }
                break;
            case 3733:
                switch (current) {
                    case 'c' :
                        state = 3734;
                        return true;
                }
                break;
            case 3734:
                switch (current) {
                    case ';' : // circledcirc;
                        match = "\u229A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3735:
                switch (current) {
                    case 'a' :
                        state = 3736;
                        return true;
                }
                break;
            case 3736:
                switch (current) {
                    case 's' :
                        state = 3737;
                        return true;
                }
                break;
            case 3737:
                switch (current) {
                    case 'h' :
                        state = 3738;
                        return true;
                }
                break;
            case 3738:
                switch (current) {
                    case ';' : // circleddash;
                        match = "\u229D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3739:
                switch (current) {
                    case ';' : // cire;
                        match = "\u2257";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3740:
                switch (current) {
                    case 'n' :
                        state = 3741;
                        return true;
                }
                break;
            case 3741:
                switch (current) {
                    case 'i' :
                        state = 3742;
                        return true;
                }
                break;
            case 3742:
                switch (current) {
                    case 'n' :
                        state = 3743;
                        return true;
                }
                break;
            case 3743:
                switch (current) {
                    case 't' :
                        state = 3744;
                        return true;
                }
                break;
            case 3744:
                switch (current) {
                    case ';' : // cirfnint;
                        match = "\u2A10";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3745:
                switch (current) {
                    case 'i' :
                        state = 3746;
                        return true;
                }
                break;
            case 3746:
                switch (current) {
                    case 'd' :
                        state = 3747;
                        return true;
                }
                break;
            case 3747:
                switch (current) {
                    case ';' : // cirmid;
                        match = "\u2AEF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3748:
                switch (current) {
                    case 'c' :
                        state = 3749;
                        return true;
                }
                break;
            case 3749:
                switch (current) {
                    case 'i' :
                        state = 3750;
                        return true;
                }
                break;
            case 3750:
                switch (current) {
                    case 'r' :
                        state = 3751;
                        return true;
                }
                break;
            case 3751:
                switch (current) {
                    case ';' : // cirscir;
                        match = "\u29C2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3752:
                switch (current) {
                    case 'u' :
                        state = 3753;
                        return true;
                }
                break;
            case 3753:
                switch (current) {
                    case 'b' :
                        state = 3754;
                        return true;
                }
                break;
            case 3754:
                switch (current) {
                    case 's' :
                        state = 3755;
                        return true;
                }
                break;
            case 3755:
                switch (current) {
                    case ';' : // clubs;
                        match = "\u2663";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'u' :
                        state = 3756;
                        return true;
                }
                break;
            case 3756:
                switch (current) {
                    case 'i' :
                        state = 3757;
                        return true;
                }
                break;
            case 3757:
                switch (current) {
                    case 't' :
                        state = 3758;
                        return true;
                }
                break;
            case 3758:
                switch (current) {
                    case ';' : // clubsuit;
                        match = "\u2663";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3759:
                switch (current) {
                    case 'l' :
                        state = 3760;
                        return true;
                    case 'm' :
                        state = 3765;
                        return true;
                    case 'n' :
                        state = 3781;
                        return true;
                    case 'p' :
                        state = 3789;
                        return true;
                }
                break;
            case 3760:
                switch (current) {
                    case 'o' :
                        state = 3761;
                        return true;
                }
                break;
            case 3761:
                switch (current) {
                    case 'n' :
                        state = 3762;
                        return true;
                }
                break;
            case 3762:
                switch (current) {
                    case ';' : // colon;
                        match = ":";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3763;
                        return true;
                }
                break;
            case 3763:
                switch (current) {
                    case ';' : // colone;
                        match = "\u2254";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'q' :
                        state = 3764;
                        return true;
                }
                break;
            case 3764:
                switch (current) {
                    case ';' : // coloneq;
                        match = "\u2254";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3765:
                switch (current) {
                    case 'm' :
                        state = 3766;
                        return true;
                    case 'p' :
                        state = 3769;
                        return true;
                }
                break;
            case 3766:
                switch (current) {
                    case 'a' :
                        state = 3767;
                        return true;
                }
                break;
            case 3767:
                switch (current) {
                    case ';' : // comma;
                        match = ",";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 't' :
                        state = 3768;
                        return true;
                }
                break;
            case 3768:
                switch (current) {
                    case ';' : // commat;
                        match = "@";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3769:
                switch (current) {
                    case ';' : // comp;
                        match = "\u2201";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'f' :
                        state = 3770;
                        return true;
                    case 'l' :
                        state = 3772;
                        return true;
                }
                break;
            case 3770:
                switch (current) {
                    case 'n' :
                        state = 3771;
                        return true;
                }
                break;
            case 3771:
                switch (current) {
                    case ';' : // compfn;
                        match = "\u2218";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3772:
                switch (current) {
                    case 'e' :
                        state = 3773;
                        return true;
                }
                break;
            case 3773:
                switch (current) {
                    case 'm' :
                        state = 3774;
                        return true;
                    case 'x' :
                        state = 3778;
                        return true;
                }
                break;
            case 3774:
                switch (current) {
                    case 'e' :
                        state = 3775;
                        return true;
                }
                break;
            case 3775:
                switch (current) {
                    case 'n' :
                        state = 3776;
                        return true;
                }
                break;
            case 3776:
                switch (current) {
                    case 't' :
                        state = 3777;
                        return true;
                }
                break;
            case 3777:
                switch (current) {
                    case ';' : // complement;
                        match = "\u2201";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3778:
                switch (current) {
                    case 'e' :
                        state = 3779;
                        return true;
                }
                break;
            case 3779:
                switch (current) {
                    case 's' :
                        state = 3780;
                        return true;
                }
                break;
            case 3780:
                switch (current) {
                    case ';' : // complexes;
                        match = "\u2102";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3781:
                switch (current) {
                    case 'g' :
                        state = 3782;
                        return true;
                    case 'i' :
                        state = 3786;
                        return true;
                }
                break;
            case 3782:
                switch (current) {
                    case ';' : // cong;
                        match = "\u2245";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'd' :
                        state = 3783;
                        return true;
                }
                break;
            case 3783:
                switch (current) {
                    case 'o' :
                        state = 3784;
                        return true;
                }
                break;
            case 3784:
                switch (current) {
                    case 't' :
                        state = 3785;
                        return true;
                }
                break;
            case 3785:
                switch (current) {
                    case ';' : // congdot;
                        match = "\u2A6D";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3786:
                switch (current) {
                    case 'n' :
                        state = 3787;
                        return true;
                }
                break;
            case 3787:
                switch (current) {
                    case 't' :
                        state = 3788;
                        return true;
                }
                break;
            case 3788:
                switch (current) {
                    case ';' : // conint;
                        match = "\u222E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3789:
                switch (current) {
                    case 'f' :
                        state = 3790;
                        return true;
                    case 'r' :
                        state = 3791;
                        return true;
                    case 'y' : // copy
                        match = "\u00A9";
                        matchLength = consumedCount;
                        state = 3794;
                        return true;
                }
                break;
            case 3790:
                switch (current) {
                    case ';' : // copf;
                        match = "\uD835\uDD54";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3791:
                switch (current) {
                    case 'o' :
                        state = 3792;
                        return true;
                }
                break;
            case 3792:
                switch (current) {
                    case 'd' :
                        state = 3793;
                        return true;
                }
                break;
            case 3793:
                switch (current) {
                    case ';' : // coprod;
                        match = "\u2210";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3794:
                switch (current) {
                    case ';' : // copy;
                        match = "\u00A9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 's' :
                        state = 3795;
                        return true;
                }
                break;
            case 3795:
                switch (current) {
                    case 'r' :
                        state = 3796;
                        return true;
                }
                break;
            case 3796:
                switch (current) {
                    case ';' : // copysr;
                        match = "\u2117";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3797:
                switch (current) {
                    case 'a' :
                        state = 3798;
                        return true;
                    case 'o' :
                        state = 3801;
                        return true;
                }
                break;
            case 3798:
                switch (current) {
                    case 'r' :
                        state = 3799;
                        return true;
                }
                break;
            case 3799:
                switch (current) {
                    case 'r' :
                        state = 3800;
                        return true;
                }
                break;
            case 3800:
                switch (current) {
                    case ';' : // crarr;
                        match = "\u21B5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3801:
                switch (current) {
                    case 's' :
                        state = 3802;
                        return true;
                }
                break;
            case 3802:
                switch (current) {
                    case 's' :
                        state = 3803;
                        return true;
                }
                break;
            case 3803:
                switch (current) {
                    case ';' : // cross;
                        match = "\u2717";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3804:
                switch (current) {
                    case 'c' :
                        state = 3805;
                        return true;
                    case 'u' :
                        state = 3807;
                        return true;
                }
                break;
            case 3805:
                switch (current) {
                    case 'r' :
                        state = 3806;
                        return true;
                }
                break;
            case 3806:
                switch (current) {
                    case ';' : // cscr;
                        match = "\uD835\uDCB8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3807:
                switch (current) {
                    case 'b' :
                        state = 3808;
                        return true;
                    case 'p' :
                        state = 3810;
                        return true;
                }
                break;
            case 3808:
                switch (current) {
                    case ';' : // csub;
                        match = "\u2ACF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3809;
                        return true;
                }
                break;
            case 3809:
                switch (current) {
                    case ';' : // csube;
                        match = "\u2AD1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3810:
                switch (current) {
                    case ';' : // csup;
                        match = "\u2AD0";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3811;
                        return true;
                }
                break;
            case 3811:
                switch (current) {
                    case ';' : // csupe;
                        match = "\u2AD2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3812:
                switch (current) {
                    case 'd' :
                        state = 3813;
                        return true;
                }
                break;
            case 3813:
                switch (current) {
                    case 'o' :
                        state = 3814;
                        return true;
                }
                break;
            case 3814:
                switch (current) {
                    case 't' :
                        state = 3815;
                        return true;
                }
                break;
            case 3815:
                switch (current) {
                    case ';' : // ctdot;
                        match = "\u22EF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3816:
                switch (current) {
                    case 'd' :
                        state = 3817;
                        return true;
                    case 'e' :
                        state = 3823;
                        return true;
                    case 'l' :
                        state = 3828;
                        return true;
                    case 'p' :
                        state = 3833;
                        return true;
                    case 'r' :
                        state = 3850;
                        return true;
                    case 'v' :
                        state = 3894;
                        return true;
                    case 'w' :
                        state = 3897;
                        return true;
                }
                break;
            case 3817:
                switch (current) {
                    case 'a' :
                        state = 3818;
                        return true;
                }
                break;
            case 3818:
                switch (current) {
                    case 'r' :
                        state = 3819;
                        return true;
                }
                break;
            case 3819:
                switch (current) {
                    case 'r' :
                        state = 3820;
                        return true;
                }
                break;
            case 3820:
                switch (current) {
                    case 'l' :
                        state = 3821;
                        return true;
                    case 'r' :
                        state = 3822;
                        return true;
                }
                break;
            case 3821:
                switch (current) {
                    case ';' : // cudarrl;
                        match = "\u2938";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3822:
                switch (current) {
                    case ';' : // cudarrr;
                        match = "\u2935";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3823:
                switch (current) {
                    case 'p' :
                        state = 3824;
                        return true;
                    case 's' :
                        state = 3826;
                        return true;
                }
                break;
            case 3824:
                switch (current) {
                    case 'r' :
                        state = 3825;
                        return true;
                }
                break;
            case 3825:
                switch (current) {
                    case ';' : // cuepr;
                        match = "\u22DE";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3826:
                switch (current) {
                    case 'c' :
                        state = 3827;
                        return true;
                }
                break;
            case 3827:
                switch (current) {
                    case ';' : // cuesc;
                        match = "\u22DF";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                }
                break;
            case 3828:
                switch (current) {
                    case 'a' :
                        state = 3829;
                        return true;
                }
                break;
            case 3829:
                switch (current) {
                    case 'r' :
                        state = 3830;
                        return true;
                }
                break;
            case 3830:
                switch (current) {
                    case 'r' :
                        state = 3831;
                        return true;
                }
                break;
            case 3831:
                switch (current) {
                    case ';' : // cularr;
                        match = "\u21B6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'p' :
                        state = 3832;