/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security;

import java.security.acl.Group;
import java.util.Iterator;
import java.util.Set;

import javax.naming.InitialContext;
import javax.security.auth.Subject;

import org.jboss.logging.Logger;
import org.jboss.security.config.ApplicationPolicy;
import org.jboss.security.config.SecurityConfiguration;

//$Id$

/**
 *  Security Utility Class
 *  @author Anil.Saldhana@redhat.com
 *  @since  May 9, 2007 
 *  @version $Revision$
 */
public class SecurityUtil
{
   private static Logger log = Logger.getLogger(SecurityUtil.class);
   
   /**
    * Strip the security domain of prefix (java:jaas or java:jbsx)
    * @param securityDomain
    * @return
    */
   public static String unprefixSecurityDomain(String securityDomain)
   {
      String result = null;
      if(securityDomain != null)
      {
         if(securityDomain.startsWith(SecurityConstants.JAAS_CONTEXT_ROOT))
            result = securityDomain.substring(SecurityConstants.JAAS_CONTEXT_ROOT.length() + 1);
         else
            if(securityDomain.startsWith(SecurityConstants.JASPI_CONTEXT_ROOT))
               result = securityDomain.substring(SecurityConstants.JASPI_CONTEXT_ROOT.length() + 1); 
            else
               result = securityDomain;
      } 
      return result;

   }

   /**
    * Get the Subject roles by looking for a Group called 'Roles'
    * @param theSubject - the Subject to search for roles
    * @return the Group contain the subject roles if found, null otherwise
    */ 
   public static Group getSubjectRoles(Subject theSubject)
   {
      if(theSubject == null)
         throw new IllegalArgumentException("Subject is null");
      Set subjectGroups = theSubject.getPrincipals(Group.class);
      Iterator iter = subjectGroups.iterator();
      Group roles = null;
      while( iter.hasNext() )
      {
         Group grp = (Group) iter.next();
         String name = grp.getName();
         if( name.equals("Roles") )
            roles = grp;
      }
      return roles;
   } 

   /**
    * Obtain the Application Policy
    * 
    * @param domainName Security Domain  
    * @return
    */ 
   public static ApplicationPolicy getApplicationPolicy(String domainName)
   {
      return SecurityConfiguration.getApplicationPolicy(domainName); 
   }

   public static AuthenticationManager getAuthenticationManager(String securityDomain)
   {
      String securityMgrURL = "/securityMgr";
      String lookupURL = null;
      if(securityDomain.startsWith(SecurityConstants.JAAS_CONTEXT_ROOT))
         lookupURL = securityDomain + securityMgrURL;
      else
         lookupURL = SecurityConstants.JAAS_CONTEXT_ROOT + "/" 
         + securityDomain + securityMgrURL; 
      AuthenticationManager am = null;
      try
      {
         InitialContext ic = new InitialContext();
         am = (AuthenticationManager)ic.lookup(lookupURL);
      }
      catch(Exception e)
      {
         if(log.isTraceEnabled())
            log.trace("Error in obtaining AuthenticationManager",e);
      }
      return am;
   }

   /**
    * Do a JNDI lookup to obtain the authorization manager
    * @param securityDomain
    * @return
    */
   public static AuthorizationManager getAuthorizationManager(String securityDomain)
   {
      String authorizationMgrURL = "/authorizationMgr";
      String lookupURL = null;
      if(securityDomain.startsWith(SecurityConstants.JAAS_CONTEXT_ROOT))
         lookupURL = securityDomain + authorizationMgrURL;
      else
         lookupURL = SecurityConstants.JAAS_CONTEXT_ROOT + "/" + securityDomain + authorizationMgrURL; 
      AuthorizationManager am = null;
      try
      {
         InitialContext ic = new InitialContext();
         am = (AuthorizationManager)ic.lookup(lookupURL);
      }
      catch(Exception e)
      {
         if(log.isTraceEnabled())
            log.trace("Error in obtaining AuthorizationMgr",e);
      }
      return am;
   }
}
