package io.smallrye.reactive.messaging.pulsar;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;

/**
 * Extract the incoming configuration for the {@code smallrye-pulsar} connector.
*/
public class PulsarConnectorIncomingConfiguration extends PulsarConnectorCommonConfiguration {

  /**
   * Creates a new PulsarConnectorIncomingConfiguration.
   */
  public PulsarConnectorIncomingConfiguration(Config config) {
    super(config);
    validate();
  }

  /**
  * Gets the consumer-configuration value from the configuration.
  * Attribute Name: consumer-configuration
  * Description: Identifier of a CDI bean that provides the default Pulsar consumer configuration for this channel. The channel configuration can still override any attribute. The bean must have a type of Map<String, Object> and must use the @io.smallrye.common.annotation.Identifier qualifier to set the identifier.
  * @return the consumer-configuration
  */
  public Optional<String> getConsumerConfiguration() {
    return config.getOptionalValue("consumer-configuration", String.class);
  }

  /**
  * Gets the ack-strategy value from the configuration.
  * Attribute Name: ack-strategy
  * Description: Specify the commit strategy to apply when a message produced from a record is acknowledged. Values can be `ack`, `cumulative`.
  * Default Value: ack
  * @return the ack-strategy
  */
  public String getAckStrategy() {
    return config.getOptionalValue("ack-strategy", String.class)
     .orElse("ack");
  }

  /**
  * Gets the failure-strategy value from the configuration.
  * Attribute Name: failure-strategy
  * Description: Specify the failure strategy to apply when a message produced from a record is acknowledged negatively (nack). Values can be `nack` (default), `fail`, `ignore` or `reconsume-later
  * Default Value: nack
  * @return the failure-strategy
  */
  public String getFailureStrategy() {
    return config.getOptionalValue("failure-strategy", String.class)
     .orElse("nack");
  }

  /**
  * Gets the reconsumeLater.delay value from the configuration.
  * Attribute Name: reconsumeLater.delay
  * Description: Default delay for reconsume failure-strategy, in seconds
  * Default Value: 3
  * @return the reconsumeLater.delay
  */
  public Long getReconsumeLaterDelay() {
    return config.getOptionalValue("reconsumeLater.delay", Long.class)
     .orElse(Long.valueOf("3"));
  }

  /**
  * Gets the negativeAck.redeliveryBackoff value from the configuration.
  * Attribute Name: negativeAck.redeliveryBackoff
  * Description: Comma separated values for configuring negative ack MultiplierRedeliveryBackoff, min delay, max delay, multiplier.
  * @return the negativeAck.redeliveryBackoff
  */
  public Optional<String> getNegativeAckRedeliveryBackoff() {
    return config.getOptionalValue("negativeAck.redeliveryBackoff", String.class);
  }

  /**
  * Gets the ackTimeout.redeliveryBackoff value from the configuration.
  * Attribute Name: ackTimeout.redeliveryBackoff
  * Description: Comma separated values for configuring ack timeout MultiplierRedeliveryBackoff, min delay, max delay, multiplier.
  * @return the ackTimeout.redeliveryBackoff
  */
  public Optional<String> getAckTimeoutRedeliveryBackoff() {
    return config.getOptionalValue("ackTimeout.redeliveryBackoff", String.class);
  }

  /**
  * Gets the deadLetterPolicy.maxRedeliverCount value from the configuration.
  * Attribute Name: deadLetterPolicy.maxRedeliverCount
  * Description: Maximum number of times that a message will be redelivered before being sent to the dead letter topic
  * @return the deadLetterPolicy.maxRedeliverCount
  */
  public Optional<Integer> getDeadLetterPolicyMaxRedeliverCount() {
    return config.getOptionalValue("deadLetterPolicy.maxRedeliverCount", Integer.class);
  }

  /**
  * Gets the deadLetterPolicy.deadLetterTopic value from the configuration.
  * Attribute Name: deadLetterPolicy.deadLetterTopic
  * Description: Name of the dead letter topic where the failing messages will be sent
  * @return the deadLetterPolicy.deadLetterTopic
  */
  public Optional<String> getDeadLetterPolicyDeadLetterTopic() {
    return config.getOptionalValue("deadLetterPolicy.deadLetterTopic", String.class);
  }

  /**
  * Gets the deadLetterPolicy.retryLetterTopic value from the configuration.
  * Attribute Name: deadLetterPolicy.retryLetterTopic
  * Description: Name of the retry topic where the failing messages will be sent
  * @return the deadLetterPolicy.retryLetterTopic
  */
  public Optional<String> getDeadLetterPolicyRetryLetterTopic() {
    return config.getOptionalValue("deadLetterPolicy.retryLetterTopic", String.class);
  }

  /**
  * Gets the deadLetterPolicy.initialSubscriptionName value from the configuration.
  * Attribute Name: deadLetterPolicy.initialSubscriptionName
  * Description: Name of the initial subscription name of the dead letter topic
  * @return the deadLetterPolicy.initialSubscriptionName
  */
  public Optional<String> getDeadLetterPolicyInitialSubscriptionName() {
    return config.getOptionalValue("deadLetterPolicy.initialSubscriptionName", String.class);
  }

  /**
  * Gets the batchReceive value from the configuration.
  * Attribute Name: batchReceive
  * Description: Whether batch receive is used to consume messages
  * Default Value: false
  * @return the batchReceive
  */
  public Boolean getBatchReceive() {
    return config.getOptionalValue("batchReceive", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  public void validate() {
    super.validate();
  }
}
