/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package de.knightsoftnet.validators.shared;

import de.knightsoftnet.validators.shared.impl.EmptyIfOtherHasValueValidator;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import jakarta.validation.Constraint;
import jakarta.validation.Payload;

/**
 * The annotated bean must contain at least two properties:
 * <ul>
 * <li>a field that has to be checked (option <code>field</code>)</li>
 * <li>a field which entry is compared against a value (option <code>fieldCompare</code> and
 * <code>valueCompare</code>)</li>
 * </ul>
 * if the entry of <code>fieldCompare</code> matches <code>valueCompare</code>, <code>field</code>
 * must be empty (null or "").<br>
 * Supported types are beans, <code>null</code> elements are considered valid.<br>
 *
 * @author Manfred Tremmel
 *
 */
@Documented
@Constraint(validatedBy = EmptyIfOtherHasValueValidator.class)
@Target({ElementType.TYPE, ElementType.ANNOTATION_TYPE})
@Retention(RetentionPolicy.RUNTIME)
public @interface EmptyIfOtherHasValue {
  /**
   * localized message.
   *
   * @return localized validation message
   */
  String message() default "{de.knightsoftnet.validators.shared.EmptyIfOtherHasValue.message}";

  /**
   * groups to use.
   *
   * @return array of validation groups
   */
  Class<?>[] groups() default {};

  /**
   * field name to check.
   *
   * @return field/path of the value
   */
  String field();

  /**
   * field name to compare.
   *
   * @return field/path of the value to compare
   */
  String fieldCompare();

  /**
   * value to compare with field name to compare.
   *
   * @return value to compare
   */
  String[] valueCompare();

  /**
   * payload whatever.
   *
   * @return payload class
   */
  Class<? extends Payload>[] payload() default {};

  /**
   * Defines several {@code @MustBeEmptyIfOtherHasValue} annotations on the same element.
   */
  @Target({ElementType.TYPE, ElementType.ANNOTATION_TYPE})
  @Retention(RetentionPolicy.RUNTIME)
  @Documented
  @interface List {
    /**
     * must be empty if other has the given value.
     *
     * @return value
     */
    EmptyIfOtherHasValue[] value();
  }
}
