package graphql.validation.schemawiring;

import graphql.PublicApi;
import graphql.schema.DataFetcher;
import graphql.schema.GraphQLFieldDefinition;
import graphql.schema.GraphQLFieldsContainer;
import graphql.schema.idl.SchemaDirectiveWiring;
import graphql.schema.idl.SchemaDirectiveWiringEnvironment;
import graphql.validation.interpolation.MessageInterpolator;
import graphql.validation.rules.OnValidationErrorStrategy;
import graphql.validation.rules.TargetedValidationRules;
import graphql.validation.rules.ValidationRules;

import java.util.Locale;

/**
 * A {@link SchemaDirectiveWiring} that can be used to inject validation rules into the data fetchers
 * when the graphql schema is being built.  It will use the validation rules and ask each one of they apply to the field and or its
 * arguments.
 * <p>
 * If there are rules that apply then it will it will change the {@link DataFetcher} of that field so that rules get run
 * BEFORE the original field fetch is run.
 */
@PublicApi
public class ValidationSchemaWiring implements SchemaDirectiveWiring {

    private final ValidationRules ruleCandidates;

    public ValidationSchemaWiring(ValidationRules ruleCandidates) {
        this.ruleCandidates = ruleCandidates;
    }

    @Override
    public GraphQLFieldDefinition onField(SchemaDirectiveWiringEnvironment<GraphQLFieldDefinition> env) {
        GraphQLFieldsContainer fieldsContainer = env.getFieldsContainer();
        GraphQLFieldDefinition fieldDefinition = env.getFieldDefinition();

        OnValidationErrorStrategy errorStrategy = ruleCandidates.getOnValidationErrorStrategy();
        MessageInterpolator messageInterpolator = ruleCandidates.getMessageInterpolator();
        Locale locale = ruleCandidates.getLocale();

        final DataFetcher<?> currentDF = env.getCodeRegistry().getDataFetcher(fieldsContainer, fieldDefinition);
        final DataFetcher<?> newDF = buildValidatingDataFetcher(errorStrategy, messageInterpolator, currentDF, locale);

        env.getCodeRegistry().dataFetcher(fieldsContainer, fieldDefinition, newDF);

        return fieldDefinition;
    }

    private DataFetcher<Object> buildValidatingDataFetcher(OnValidationErrorStrategy errorStrategy,
                                                           MessageInterpolator messageInterpolator,
                                                           DataFetcher<?> currentDF,
                                                           final Locale defaultLocale) {
        return new FieldValidatorDataFetcher(
                errorStrategy,
                messageInterpolator,
                currentDF,
                defaultLocale,
                ruleCandidates
        );
    }

}
