package com.datadog.android.telemetry.model

import com.datadog.android.core.`internal`.utils.JsonSerializer
import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import com.google.gson.JsonPrimitive
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Any
import kotlin.Array
import kotlin.Boolean
import kotlin.Long
import kotlin.Number
import kotlin.String
import kotlin.collections.ArrayList
import kotlin.collections.List
import kotlin.collections.MutableMap
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Schema of common properties of Telemetry events
 * @param dd Internal properties
 * @param date Start of the event in ms from epoch
 * @param service The SDK generating the telemetry event
 * @param source The source of this event
 * @param version The version of the SDK generating the telemetry event
 * @param application Application properties
 * @param session Session properties
 * @param view View properties
 * @param action Action properties
 * @param effectiveSampleRate The actual percentage of telemetry usage per event
 * @param experimentalFeatures Enabled experimental features
 * @param telemetry The telemetry usage information
 */
public data class TelemetryUsageEvent(
    public val dd: Dd,
    public val date: Long,
    public val service: String,
    public val source: Source,
    public val version: String,
    public val application: Application? = null,
    public val session: Session? = null,
    public val view: View? = null,
    public val action: Action? = null,
    public val effectiveSampleRate: Number? = null,
    public val experimentalFeatures: List<String>? = null,
    public val telemetry: Telemetry,
) {
    public val type: String = "telemetry"

    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.add("_dd", dd.toJson())
        json.addProperty("type", type)
        json.addProperty("date", date)
        json.addProperty("service", service)
        json.add("source", source.toJson())
        json.addProperty("version", version)
        application?.let { applicationNonNull ->
            json.add("application", applicationNonNull.toJson())
        }
        session?.let { sessionNonNull ->
            json.add("session", sessionNonNull.toJson())
        }
        view?.let { viewNonNull ->
            json.add("view", viewNonNull.toJson())
        }
        action?.let { actionNonNull ->
            json.add("action", actionNonNull.toJson())
        }
        effectiveSampleRate?.let { effectiveSampleRateNonNull ->
            json.addProperty("effective_sample_rate", effectiveSampleRateNonNull)
        }
        experimentalFeatures?.let { experimentalFeaturesNonNull ->
            val experimentalFeaturesArray = JsonArray(experimentalFeaturesNonNull.size)
            experimentalFeaturesNonNull.forEach { experimentalFeaturesArray.add(it) }
            json.add("experimental_features", experimentalFeaturesArray)
        }
        json.add("telemetry", telemetry.toJson())
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): TelemetryUsageEvent {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryUsageEvent",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): TelemetryUsageEvent {
            try {
                val dd = jsonObject.get("_dd").asJsonObject.let {
                    Dd.fromJsonObject(it)
                }
                val type = jsonObject.get("type").asString
                val date = jsonObject.get("date").asLong
                val service = jsonObject.get("service").asString
                val source = Source.fromJson(jsonObject.get("source").asString)
                val version = jsonObject.get("version").asString
                val application = jsonObject.get("application")?.asJsonObject?.let {
                    Application.fromJsonObject(it)
                }
                val session = jsonObject.get("session")?.asJsonObject?.let {
                    Session.fromJsonObject(it)
                }
                val view = jsonObject.get("view")?.asJsonObject?.let {
                    View.fromJsonObject(it)
                }
                val action = jsonObject.get("action")?.asJsonObject?.let {
                    Action.fromJsonObject(it)
                }
                val effectiveSampleRate = jsonObject.get("effective_sample_rate")?.asNumber
                val experimentalFeatures =
                        jsonObject.get("experimental_features")?.asJsonArray?.let { jsonArray ->
                    val collection = ArrayList<String>(jsonArray.size())
                    jsonArray.forEach {
                        collection.add(it.asString)
                    }
                    collection
                }
                val telemetry = jsonObject.get("telemetry").asJsonObject.let {
                    Telemetry.fromJsonObject(it)
                }
                check(type == "telemetry")
                return TelemetryUsageEvent(dd, date, service, source, version, application, session,
                        view, action, effectiveSampleRate, experimentalFeatures, telemetry)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryUsageEvent",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryUsageEvent",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryUsageEvent",
                    e
                )
            }
        }
    }

    /**
     * Internal properties
     */
    public class Dd {
        public val formatVersion: Long = 2L

        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("format_version", formatVersion)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Dd {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Dd {
                try {
                    val formatVersion = jsonObject.get("format_version").asLong
                    check(formatVersion == 2.0.toLong())
                    return Dd()
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }
        }
    }

    /**
     * Application properties
     * @param id UUID of the application
     */
    public data class Application(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Application {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Application {
                try {
                    val id = jsonObject.get("id").asString
                    return Application(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session properties
     * @param id UUID of the session
     */
    public data class Session(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Session {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Session {
                try {
                    val id = jsonObject.get("id").asString
                    return Session(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }
        }
    }

    /**
     * View properties
     * @param id UUID of the view
     */
    public data class View(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): View {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): View {
                try {
                    val id = jsonObject.get("id").asString
                    return View(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }
        }
    }

    /**
     * Action properties
     * @param id UUID of the action
     */
    public data class Action(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Action {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Action {
                try {
                    val id = jsonObject.get("id").asString
                    return Action(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }
        }
    }

    /**
     * The telemetry usage information
     * @param device Device properties
     * @param os OS properties
     */
    public data class Telemetry(
        public val device: Device? = null,
        public val os: Os? = null,
        public val usage: Usage,
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public val type: String = "usage"

        public fun toJson(): JsonElement {
            val json = JsonObject()
            device?.let { deviceNonNull ->
                json.add("device", deviceNonNull.toJson())
            }
            os?.let { osNonNull ->
                json.add("os", osNonNull.toJson())
            }
            json.addProperty("type", type)
            json.add("usage", usage.toJson())
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.add(k, JsonSerializer.toJsonElement(v))
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("device", "os", "type",
                    "usage")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Telemetry {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Telemetry {
                try {
                    val device = jsonObject.get("device")?.asJsonObject?.let {
                        Device.fromJsonObject(it)
                    }
                    val os = jsonObject.get("os")?.asJsonObject?.let {
                        Os.fromJsonObject(it)
                    }
                    val type = jsonObject.get("type").asString
                    val usage = jsonObject.get("usage").asJsonObject.let {
                        Usage.fromJsonObject(it)
                    }
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value
                        }
                    }
                    check(type == "usage")
                    return Telemetry(device, os, usage, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device properties
     * @param architecture Architecture of the device
     * @param brand Brand of the device
     * @param model Model of the device
     */
    public data class Device(
        public val architecture: String? = null,
        public val brand: String? = null,
        public val model: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            architecture?.let { architectureNonNull ->
                json.addProperty("architecture", architectureNonNull)
            }
            brand?.let { brandNonNull ->
                json.addProperty("brand", brandNonNull)
            }
            model?.let { modelNonNull ->
                json.addProperty("model", modelNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Device {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Device {
                try {
                    val architecture = jsonObject.get("architecture")?.asString
                    val brand = jsonObject.get("brand")?.asString
                    val model = jsonObject.get("model")?.asString
                    return Device(architecture, brand, model)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }
        }
    }

    /**
     * OS properties
     * @param build Build of the OS
     * @param name Name of the OS
     * @param version Version of the OS
     */
    public data class Os(
        public val build: String? = null,
        public val name: String? = null,
        public val version: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            build?.let { buildNonNull ->
                json.addProperty("build", buildNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            version?.let { versionNonNull ->
                json.addProperty("version", versionNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Os {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Os {
                try {
                    val build = jsonObject.get("build")?.asString
                    val name = jsonObject.get("name")?.asString
                    val version = jsonObject.get("version")?.asString
                    return Os(build, name, version)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }
        }
    }

    public sealed class Usage {
        public abstract fun toJson(): JsonElement

        /**
         * @param trackingConsent The tracking consent value set by the user
         */
        public data class SetTrackingConsent(
            public val trackingConsent: TrackingConsent,
        ) : Usage() {
            public val feature: String = "set-tracking-consent"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                json.add("tracking_consent", trackingConsent.toJson())
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): SetTrackingConsent {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetTrackingConsent",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): SetTrackingConsent {
                    try {
                        val feature = jsonObject.get("feature").asString
                        val trackingConsent =
                                TrackingConsent.fromJson(jsonObject.get("tracking_consent").asString)
                        check(feature == "set-tracking-consent")
                        return SetTrackingConsent(trackingConsent)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetTrackingConsent",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetTrackingConsent",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetTrackingConsent",
                            e
                        )
                    }
                }
            }
        }

        public class StopSession : Usage() {
            public val feature: String = "stop-session"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): StopSession {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type StopSession",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): StopSession {
                    try {
                        val feature = jsonObject.get("feature").asString
                        check(feature == "stop-session")
                        return StopSession()
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type StopSession",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type StopSession",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type StopSession",
                            e
                        )
                    }
                }
            }
        }

        public class StartView : Usage() {
            public val feature: String = "start-view"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): StartView {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartView",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): StartView {
                    try {
                        val feature = jsonObject.get("feature").asString
                        check(feature == "start-view")
                        return StartView()
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartView",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartView",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartView",
                            e
                        )
                    }
                }
            }
        }

        public class AddAction : Usage() {
            public val feature: String = "add-action"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): AddAction {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddAction",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): AddAction {
                    try {
                        val feature = jsonObject.get("feature").asString
                        check(feature == "add-action")
                        return AddAction()
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddAction",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddAction",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddAction",
                            e
                        )
                    }
                }
            }
        }

        public class AddError : Usage() {
            public val feature: String = "add-error"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): AddError {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddError",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): AddError {
                    try {
                        val feature = jsonObject.get("feature").asString
                        check(feature == "add-error")
                        return AddError()
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddError",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddError",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddError",
                            e
                        )
                    }
                }
            }
        }

        public class SetGlobalContext : Usage() {
            public val feature: String = "set-global-context"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): SetGlobalContext {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetGlobalContext",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): SetGlobalContext {
                    try {
                        val feature = jsonObject.get("feature").asString
                        check(feature == "set-global-context")
                        return SetGlobalContext()
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetGlobalContext",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetGlobalContext",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetGlobalContext",
                            e
                        )
                    }
                }
            }
        }

        public class SetUser : Usage() {
            public val feature: String = "set-user"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): SetUser {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetUser",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): SetUser {
                    try {
                        val feature = jsonObject.get("feature").asString
                        check(feature == "set-user")
                        return SetUser()
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetUser",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetUser",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type SetUser",
                            e
                        )
                    }
                }
            }
        }

        public class AddFeatureFlagEvaluation : Usage() {
            public val feature: String = "add-feature-flag-evaluation"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): AddFeatureFlagEvaluation {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddFeatureFlagEvaluation",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): AddFeatureFlagEvaluation {
                    try {
                        val feature = jsonObject.get("feature").asString
                        check(feature == "add-feature-flag-evaluation")
                        return AddFeatureFlagEvaluation()
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddFeatureFlagEvaluation",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddFeatureFlagEvaluation",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddFeatureFlagEvaluation",
                            e
                        )
                    }
                }
            }
        }

        /**
         * @param isForced Whether the recording is allowed to start even on sessions sampled out of
         * replay
         */
        public data class StartSessionReplayRecording(
            public val isForced: Boolean? = null,
        ) : Usage() {
            public val feature: String = "start-session-replay-recording"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                isForced?.let { isForcedNonNull ->
                    json.addProperty("is_forced", isForcedNonNull)
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): StartSessionReplayRecording {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartSessionReplayRecording",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): StartSessionReplayRecording {
                    try {
                        val feature = jsonObject.get("feature").asString
                        val isForced = jsonObject.get("is_forced")?.asBoolean
                        check(feature == "start-session-replay-recording")
                        return StartSessionReplayRecording(isForced)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartSessionReplayRecording",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartSessionReplayRecording",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartSessionReplayRecording",
                            e
                        )
                    }
                }
            }
        }

        public class StartDurationVital : Usage() {
            public val feature: String = "start-duration-vital"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): StartDurationVital {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartDurationVital",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): StartDurationVital {
                    try {
                        val feature = jsonObject.get("feature").asString
                        check(feature == "start-duration-vital")
                        return StartDurationVital()
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartDurationVital",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartDurationVital",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type StartDurationVital",
                            e
                        )
                    }
                }
            }
        }

        public class StopDurationVital : Usage() {
            public val feature: String = "stop-duration-vital"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): StopDurationVital {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type StopDurationVital",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): StopDurationVital {
                    try {
                        val feature = jsonObject.get("feature").asString
                        check(feature == "stop-duration-vital")
                        return StopDurationVital()
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type StopDurationVital",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type StopDurationVital",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type StopDurationVital",
                            e
                        )
                    }
                }
            }
        }

        public class AddDurationVital : Usage() {
            public val feature: String = "add-duration-vital"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): AddDurationVital {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddDurationVital",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): AddDurationVital {
                    try {
                        val feature = jsonObject.get("feature").asString
                        check(feature == "add-duration-vital")
                        return AddDurationVital()
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddDurationVital",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddDurationVital",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddDurationVital",
                            e
                        )
                    }
                }
            }
        }

        /**
         * @param noView Whether the view is not available
         * @param noActiveView Whether the available view is not active
         * @param overwritten Whether the loading time was overwritten
         */
        public data class AddViewLoadingTime(
            public val noView: Boolean,
            public val noActiveView: Boolean,
            public val overwritten: Boolean,
        ) : Usage() {
            public val feature: String = "addViewLoadingTime"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("feature", feature)
                json.addProperty("no_view", noView)
                json.addProperty("no_active_view", noActiveView)
                json.addProperty("overwritten", overwritten)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): AddViewLoadingTime {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddViewLoadingTime",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): AddViewLoadingTime {
                    try {
                        val feature = jsonObject.get("feature").asString
                        val noView = jsonObject.get("no_view").asBoolean
                        val noActiveView = jsonObject.get("no_active_view").asBoolean
                        val overwritten = jsonObject.get("overwritten").asBoolean
                        check(feature == "addViewLoadingTime")
                        return AddViewLoadingTime(noView, noActiveView, overwritten)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddViewLoadingTime",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddViewLoadingTime",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type AddViewLoadingTime",
                            e
                        )
                    }
                }
            }
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Usage {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into one of type Usage",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Usage {
                val errors = mutableListOf<Throwable>()
                val asSetTrackingConsent = try {
                    SetTrackingConsent.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asStopSession = try {
                    StopSession.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asStartView = try {
                    StartView.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asAddAction = try {
                    AddAction.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asAddError = try {
                    AddError.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asSetGlobalContext = try {
                    SetGlobalContext.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asSetUser = try {
                    SetUser.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asAddFeatureFlagEvaluation = try {
                    AddFeatureFlagEvaluation.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asStartSessionReplayRecording = try {
                    StartSessionReplayRecording.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asStartDurationVital = try {
                    StartDurationVital.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asStopDurationVital = try {
                    StopDurationVital.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asAddDurationVital = try {
                    AddDurationVital.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asAddViewLoadingTime = try {
                    AddViewLoadingTime.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val result = arrayOf(
                    asSetTrackingConsent,
                    asStopSession,
                    asStartView,
                    asAddAction,
                    asAddError,
                    asSetGlobalContext,
                    asSetUser,
                    asAddFeatureFlagEvaluation,
                    asStartSessionReplayRecording,
                    asStartDurationVital,
                    asStopDurationVital,
                    asAddDurationVital,
                    asAddViewLoadingTime,
                ).firstOrNull { it != null }
                if (result == null) {
                    val message = "Unable to parse json into one of type \n" + "Usage\n" +
                        errors.joinToString("\n") { it.message.toString() }
                    throw JsonParseException(message)
                }
                return result
            }
        }
    }

    /**
     * The source of this event
     */
    public enum class Source(
        private val jsonValue: String,
    ) {
        ANDROID("android"),
        IOS("ios"),
        BROWSER("browser"),
        FLUTTER("flutter"),
        REACT_NATIVE("react-native"),
        UNITY("unity"),
        KOTLIN_MULTIPLATFORM("kotlin-multiplatform"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Source = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * The tracking consent value set by the user
     */
    public enum class TrackingConsent(
        private val jsonValue: String,
    ) {
        GRANTED("granted"),
        NOT_GRANTED("not-granted"),
        PENDING("pending"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): TrackingConsent = values().first {
                it.jsonValue == jsonString
            }
        }
    }
}
