package com.datadog.android.telemetry.model

import com.datadog.android.core.`internal`.utils.JsonSerializer
import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import com.google.gson.JsonPrimitive
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Any
import kotlin.Array
import kotlin.Boolean
import kotlin.Long
import kotlin.Number
import kotlin.String
import kotlin.collections.ArrayList
import kotlin.collections.List
import kotlin.collections.MutableMap
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Schema of common properties of Telemetry events
 * @param dd Internal properties
 * @param date Start of the event in ms from epoch
 * @param service The SDK generating the telemetry event
 * @param source The source of this event
 * @param version The version of the SDK generating the telemetry event
 * @param application Application properties
 * @param session Session properties
 * @param view View properties
 * @param action Action properties
 * @param effectiveSampleRate The actual percentage of telemetry usage per event
 * @param experimentalFeatures Enabled experimental features
 * @param telemetry The telemetry configuration information
 */
public data class TelemetryConfigurationEvent(
    public val dd: Dd,
    public val date: Long,
    public val service: String,
    public val source: Source,
    public val version: String,
    public val application: Application? = null,
    public val session: Session? = null,
    public val view: View? = null,
    public val action: Action? = null,
    public val effectiveSampleRate: Number? = null,
    public val experimentalFeatures: List<String>? = null,
    public val telemetry: Telemetry,
) {
    public val type: String = "telemetry"

    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.add("_dd", dd.toJson())
        json.addProperty("type", type)
        json.addProperty("date", date)
        json.addProperty("service", service)
        json.add("source", source.toJson())
        json.addProperty("version", version)
        application?.let { applicationNonNull ->
            json.add("application", applicationNonNull.toJson())
        }
        session?.let { sessionNonNull ->
            json.add("session", sessionNonNull.toJson())
        }
        view?.let { viewNonNull ->
            json.add("view", viewNonNull.toJson())
        }
        action?.let { actionNonNull ->
            json.add("action", actionNonNull.toJson())
        }
        effectiveSampleRate?.let { effectiveSampleRateNonNull ->
            json.addProperty("effective_sample_rate", effectiveSampleRateNonNull)
        }
        experimentalFeatures?.let { experimentalFeaturesNonNull ->
            val experimentalFeaturesArray = JsonArray(experimentalFeaturesNonNull.size)
            experimentalFeaturesNonNull.forEach { experimentalFeaturesArray.add(it) }
            json.add("experimental_features", experimentalFeaturesArray)
        }
        json.add("telemetry", telemetry.toJson())
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): TelemetryConfigurationEvent {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryConfigurationEvent",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): TelemetryConfigurationEvent {
            try {
                val dd = jsonObject.get("_dd").asJsonObject.let {
                    Dd.fromJsonObject(it)
                }
                val type = jsonObject.get("type").asString
                val date = jsonObject.get("date").asLong
                val service = jsonObject.get("service").asString
                val source = Source.fromJson(jsonObject.get("source").asString)
                val version = jsonObject.get("version").asString
                val application = jsonObject.get("application")?.asJsonObject?.let {
                    Application.fromJsonObject(it)
                }
                val session = jsonObject.get("session")?.asJsonObject?.let {
                    Session.fromJsonObject(it)
                }
                val view = jsonObject.get("view")?.asJsonObject?.let {
                    View.fromJsonObject(it)
                }
                val action = jsonObject.get("action")?.asJsonObject?.let {
                    Action.fromJsonObject(it)
                }
                val effectiveSampleRate = jsonObject.get("effective_sample_rate")?.asNumber
                val experimentalFeatures =
                        jsonObject.get("experimental_features")?.asJsonArray?.let { jsonArray ->
                    val collection = ArrayList<String>(jsonArray.size())
                    jsonArray.forEach {
                        collection.add(it.asString)
                    }
                    collection
                }
                val telemetry = jsonObject.get("telemetry").asJsonObject.let {
                    Telemetry.fromJsonObject(it)
                }
                check(type == "telemetry")
                return TelemetryConfigurationEvent(dd, date, service, source, version, application,
                        session, view, action, effectiveSampleRate, experimentalFeatures, telemetry)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryConfigurationEvent",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryConfigurationEvent",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryConfigurationEvent",
                    e
                )
            }
        }
    }

    /**
     * Internal properties
     */
    public class Dd {
        public val formatVersion: Long = 2L

        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("format_version", formatVersion)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Dd {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Dd {
                try {
                    val formatVersion = jsonObject.get("format_version").asLong
                    check(formatVersion == 2.0.toLong())
                    return Dd()
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }
        }
    }

    /**
     * Application properties
     * @param id UUID of the application
     */
    public data class Application(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Application {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Application {
                try {
                    val id = jsonObject.get("id").asString
                    return Application(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session properties
     * @param id UUID of the session
     */
    public data class Session(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Session {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Session {
                try {
                    val id = jsonObject.get("id").asString
                    return Session(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }
        }
    }

    /**
     * View properties
     * @param id UUID of the view
     */
    public data class View(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): View {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): View {
                try {
                    val id = jsonObject.get("id").asString
                    return View(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }
        }
    }

    /**
     * Action properties
     * @param id UUID of the action
     */
    public data class Action(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Action {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Action {
                try {
                    val id = jsonObject.get("id").asString
                    return Action(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }
        }
    }

    /**
     * The telemetry configuration information
     * @param device Device properties
     * @param os OS properties
     * @param configuration Configuration properties
     */
    public data class Telemetry(
        public val device: Device? = null,
        public val os: Os? = null,
        public val configuration: Configuration,
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public val type: String = "configuration"

        public fun toJson(): JsonElement {
            val json = JsonObject()
            device?.let { deviceNonNull ->
                json.add("device", deviceNonNull.toJson())
            }
            os?.let { osNonNull ->
                json.add("os", osNonNull.toJson())
            }
            json.addProperty("type", type)
            json.add("configuration", configuration.toJson())
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.add(k, JsonSerializer.toJsonElement(v))
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("device", "os", "type",
                    "configuration")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Telemetry {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Telemetry {
                try {
                    val device = jsonObject.get("device")?.asJsonObject?.let {
                        Device.fromJsonObject(it)
                    }
                    val os = jsonObject.get("os")?.asJsonObject?.let {
                        Os.fromJsonObject(it)
                    }
                    val type = jsonObject.get("type").asString
                    val configuration = jsonObject.get("configuration").asJsonObject.let {
                        Configuration.fromJsonObject(it)
                    }
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value
                        }
                    }
                    check(type == "configuration")
                    return Telemetry(device, os, configuration, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device properties
     * @param architecture Architecture of the device
     * @param brand Brand of the device
     * @param model Model of the device
     */
    public data class Device(
        public val architecture: String? = null,
        public val brand: String? = null,
        public val model: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            architecture?.let { architectureNonNull ->
                json.addProperty("architecture", architectureNonNull)
            }
            brand?.let { brandNonNull ->
                json.addProperty("brand", brandNonNull)
            }
            model?.let { modelNonNull ->
                json.addProperty("model", modelNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Device {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Device {
                try {
                    val architecture = jsonObject.get("architecture")?.asString
                    val brand = jsonObject.get("brand")?.asString
                    val model = jsonObject.get("model")?.asString
                    return Device(architecture, brand, model)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }
        }
    }

    /**
     * OS properties
     * @param build Build of the OS
     * @param name Name of the OS
     * @param version Version of the OS
     */
    public data class Os(
        public val build: String? = null,
        public val name: String? = null,
        public val version: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            build?.let { buildNonNull ->
                json.addProperty("build", buildNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            version?.let { versionNonNull ->
                json.addProperty("version", versionNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Os {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Os {
                try {
                    val build = jsonObject.get("build")?.asString
                    val name = jsonObject.get("name")?.asString
                    val version = jsonObject.get("version")?.asString
                    return Os(build, name, version)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }
        }
    }

    /**
     * Configuration properties
     * @param sessionSampleRate The percentage of sessions tracked
     * @param telemetrySampleRate The percentage of telemetry events sent
     * @param telemetryConfigurationSampleRate The percentage of telemetry configuration events sent
     * after being sampled by telemetry_sample_rate
     * @param telemetryUsageSampleRate The percentage of telemetry usage events sent after being
     * sampled by telemetry_sample_rate
     * @param traceSampleRate The percentage of requests traced
     * @param traceContextInjection The opt-in configuration to add trace context
     * @param premiumSampleRate The percentage of sessions with Browser RUM & Session Replay pricing
     * tracked (deprecated in favor of session_replay_sample_rate)
     * @param replaySampleRate The percentage of sessions with Browser RUM & Session Replay pricing
     * tracked (deprecated in favor of session_replay_sample_rate)
     * @param sessionReplaySampleRate The percentage of sessions with RUM & Session Replay pricing
     * tracked
     * @param trackingConsent The initial tracking consent value
     * @param startSessionReplayRecordingManually Whether the session replay start is handled
     * manually
     * @param startRecordingImmediately Whether Session Replay should automatically start a
     * recording when enabled
     * @param useProxy Whether a proxy is used
     * @param useBeforeSend Whether beforeSend callback function is used
     * @param silentMultipleInit Whether initialization fails silently if the SDK is already
     * initialized
     * @param trackSessionAcrossSubdomains Whether sessions across subdomains for the same site are
     * tracked
     * @param trackResources Whether resources are tracked
     * @param trackLongTask Whether long tasks are tracked
     * @param useCrossSiteSessionCookie Whether a secure cross-site session cookie is used
     * (deprecated)
     * @param usePartitionedCrossSiteSessionCookie Whether a partitioned secure cross-site session
     * cookie is used
     * @param useSecureSessionCookie Whether a secure session cookie is used
     * @param allowFallbackToLocalStorage Whether it is allowed to use LocalStorage when cookies are
     * not available
     * @param storeContextsAcrossPages Whether contexts are stored in local storage
     * @param allowUntrustedEvents Whether untrusted events are allowed
     * @param actionNameAttribute Attribute to be used to name actions
     * @param useAllowedTracingOrigins Whether the allowed tracing origins list is used (deprecated
     * in favor of use_allowed_tracing_urls)
     * @param useAllowedTracingUrls Whether the allowed tracing urls list is used
     * @param selectedTracingPropagators A list of selected tracing propagators
     * @param defaultPrivacyLevel Session replay default privacy level
     * @param textAndInputPrivacyLevel Session replay text and input privacy level
     * @param imagePrivacyLevel Session replay image privacy level
     * @param touchPrivacyLevel Session replay touch privacy level
     * @param enablePrivacyForActionName Privacy control for action name
     * @param useExcludedActivityUrls Whether the request origins list to ignore when computing the
     * page activity is used
     * @param useWorkerUrl Whether the Worker is loaded from an external URL
     * @param compressIntakeRequests Whether intake requests are compressed
     * @param trackFrustrations Whether user frustrations are tracked
     * @param trackViewsManually Whether the RUM views creation is handled manually
     * @param trackInteractions Whether user actions are tracked (deprecated in favor of
     * track_user_interactions)
     * @param trackUserInteractions Whether user actions are tracked
     * @param forwardErrorsToLogs Whether console.error logs, uncaught exceptions and network errors
     * are tracked
     * @param useLocalEncryption Whether local encryption is used
     * @param viewTrackingStrategy View tracking strategy
     * @param trackBackgroundEvents Whether RUM events are tracked when the application is in
     * Background
     * @param mobileVitalsUpdatePeriod The period between each Mobile Vital sample (in milliseconds)
     * @param trackErrors Whether error monitoring & crash reporting is enabled for the source
     * platform
     * @param trackNetworkRequests Whether automatic collection of network requests is enabled
     * @param useTracing Whether tracing features are enabled
     * @param trackNativeViews Whether native views are tracked (for cross platform SDKs)
     * @param trackNativeErrors Whether native error monitoring & crash reporting is enabled (for
     * cross platform SDKs)
     * @param trackNativeLongTasks Whether long task tracking is performed automatically
     * @param trackCrossPlatformLongTasks Whether long task tracking is performed automatically for
     * cross platform SDKs
     * @param useFirstPartyHosts Whether the client has provided a list of first party hosts
     * @param initializationType The type of initialization the SDK used, in case multiple are
     * supported
     * @param trackFlutterPerformance Whether Flutter build and raster time tracking is enabled
     * @param batchSize The window duration for batches sent by the SDK (in milliseconds)
     * @param batchUploadFrequency The upload frequency of batches (in milliseconds)
     * @param batchProcessingLevel Maximum number of batches processed sequentially without a delay
     * @param backgroundTasksEnabled Whether UIApplication background tasks are enabled
     * @param reactVersion The version of React used in a ReactNative application
     * @param reactNativeVersion The version of ReactNative used in a ReactNative application
     * @param dartVersion The version of Dart used in a Flutter application
     * @param unityVersion The version of Unity used in a Unity application
     * @param appHangThreshold The threshold used for iOS App Hangs monitoring (in milliseconds)
     * @param usePciIntake Whether logs are sent to the PCI-compliant intake
     * @param tracerApi The tracer API used by the SDK. Possible values: 'Datadog', 'OpenTelemetry',
     * 'OpenTracing'
     * @param tracerApiVersion The version of the tracer API used by the SDK. Eg. '0.1.0'
     * @param sendLogsAfterSessionExpiration Whether logs are sent after the session expiration
     * @param plugins The list of plugins enabled
     * @param isMainProcess Whether the SDK is initialised on the application's main or a secondary
     * process
     * @param collectFeatureFlagsOn The list of events that include feature flags collection
     */
    public data class Configuration(
        public val sessionSampleRate: Long? = null,
        public val telemetrySampleRate: Long? = null,
        public val telemetryConfigurationSampleRate: Long? = null,
        public val telemetryUsageSampleRate: Long? = null,
        public val traceSampleRate: Long? = null,
        public var traceContextInjection: TraceContextInjection? = null,
        public val premiumSampleRate: Long? = null,
        public val replaySampleRate: Long? = null,
        public var sessionReplaySampleRate: Long? = null,
        public val trackingConsent: TrackingConsent? = null,
        public var startSessionReplayRecordingManually: Boolean? = null,
        public var startRecordingImmediately: Boolean? = null,
        public var useProxy: Boolean? = null,
        public val useBeforeSend: Boolean? = null,
        public val silentMultipleInit: Boolean? = null,
        public val trackSessionAcrossSubdomains: Boolean? = null,
        public var trackResources: Boolean? = null,
        public var trackLongTask: Boolean? = null,
        public val useCrossSiteSessionCookie: Boolean? = null,
        public val usePartitionedCrossSiteSessionCookie: Boolean? = null,
        public val useSecureSessionCookie: Boolean? = null,
        public val allowFallbackToLocalStorage: Boolean? = null,
        public val storeContextsAcrossPages: Boolean? = null,
        public val allowUntrustedEvents: Boolean? = null,
        public val actionNameAttribute: String? = null,
        public val useAllowedTracingOrigins: Boolean? = null,
        public val useAllowedTracingUrls: Boolean? = null,
        public val selectedTracingPropagators: List<SelectedTracingPropagator>? = null,
        public var defaultPrivacyLevel: String? = null,
        public var textAndInputPrivacyLevel: String? = null,
        public var imagePrivacyLevel: String? = null,
        public var touchPrivacyLevel: String? = null,
        public var enablePrivacyForActionName: Boolean? = null,
        public val useExcludedActivityUrls: Boolean? = null,
        public val useWorkerUrl: Boolean? = null,
        public val compressIntakeRequests: Boolean? = null,
        public var trackFrustrations: Boolean? = null,
        public var trackViewsManually: Boolean? = null,
        public var trackInteractions: Boolean? = null,
        public var trackUserInteractions: Boolean? = null,
        public val forwardErrorsToLogs: Boolean? = null,
        public val forwardConsoleLogs: List<String>? = null,
        public val forwardReports: List<String>? = null,
        public val useLocalEncryption: Boolean? = null,
        public val viewTrackingStrategy: ViewTrackingStrategy? = null,
        public var trackBackgroundEvents: Boolean? = null,
        public var mobileVitalsUpdatePeriod: Long? = null,
        public var trackErrors: Boolean? = null,
        public var trackNetworkRequests: Boolean? = null,
        public val useTracing: Boolean? = null,
        public var trackNativeViews: Boolean? = null,
        public var trackNativeErrors: Boolean? = null,
        public var trackNativeLongTasks: Boolean? = null,
        public var trackCrossPlatformLongTasks: Boolean? = null,
        public var useFirstPartyHosts: Boolean? = null,
        public var initializationType: String? = null,
        public var trackFlutterPerformance: Boolean? = null,
        public val batchSize: Long? = null,
        public val batchUploadFrequency: Long? = null,
        public val batchProcessingLevel: Long? = null,
        public val backgroundTasksEnabled: Boolean? = null,
        public var reactVersion: String? = null,
        public var reactNativeVersion: String? = null,
        public var dartVersion: String? = null,
        public var unityVersion: String? = null,
        public val appHangThreshold: Long? = null,
        public var usePciIntake: Boolean? = null,
        public var tracerApi: String? = null,
        public var tracerApiVersion: String? = null,
        public var sendLogsAfterSessionExpiration: Boolean? = null,
        public val plugins: List<Plugin>? = null,
        public val isMainProcess: Boolean? = null,
        public val collectFeatureFlagsOn: List<CollectFeatureFlagsOn>? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            sessionSampleRate?.let { sessionSampleRateNonNull ->
                json.addProperty("session_sample_rate", sessionSampleRateNonNull)
            }
            telemetrySampleRate?.let { telemetrySampleRateNonNull ->
                json.addProperty("telemetry_sample_rate", telemetrySampleRateNonNull)
            }
            telemetryConfigurationSampleRate?.let { telemetryConfigurationSampleRateNonNull ->
                json.addProperty("telemetry_configuration_sample_rate",
                        telemetryConfigurationSampleRateNonNull)
            }
            telemetryUsageSampleRate?.let { telemetryUsageSampleRateNonNull ->
                json.addProperty("telemetry_usage_sample_rate", telemetryUsageSampleRateNonNull)
            }
            traceSampleRate?.let { traceSampleRateNonNull ->
                json.addProperty("trace_sample_rate", traceSampleRateNonNull)
            }
            traceContextInjection?.let { traceContextInjectionNonNull ->
                json.add("trace_context_injection", traceContextInjectionNonNull.toJson())
            }
            premiumSampleRate?.let { premiumSampleRateNonNull ->
                json.addProperty("premium_sample_rate", premiumSampleRateNonNull)
            }
            replaySampleRate?.let { replaySampleRateNonNull ->
                json.addProperty("replay_sample_rate", replaySampleRateNonNull)
            }
            sessionReplaySampleRate?.let { sessionReplaySampleRateNonNull ->
                json.addProperty("session_replay_sample_rate", sessionReplaySampleRateNonNull)
            }
            trackingConsent?.let { trackingConsentNonNull ->
                json.add("tracking_consent", trackingConsentNonNull.toJson())
            }
            startSessionReplayRecordingManually?.let { startSessionReplayRecordingManuallyNonNull ->
                json.addProperty("start_session_replay_recording_manually",
                        startSessionReplayRecordingManuallyNonNull)
            }
            startRecordingImmediately?.let { startRecordingImmediatelyNonNull ->
                json.addProperty("start_recording_immediately", startRecordingImmediatelyNonNull)
            }
            useProxy?.let { useProxyNonNull ->
                json.addProperty("use_proxy", useProxyNonNull)
            }
            useBeforeSend?.let { useBeforeSendNonNull ->
                json.addProperty("use_before_send", useBeforeSendNonNull)
            }
            silentMultipleInit?.let { silentMultipleInitNonNull ->
                json.addProperty("silent_multiple_init", silentMultipleInitNonNull)
            }
            trackSessionAcrossSubdomains?.let { trackSessionAcrossSubdomainsNonNull ->
                json.addProperty("track_session_across_subdomains",
                        trackSessionAcrossSubdomainsNonNull)
            }
            trackResources?.let { trackResourcesNonNull ->
                json.addProperty("track_resources", trackResourcesNonNull)
            }
            trackLongTask?.let { trackLongTaskNonNull ->
                json.addProperty("track_long_task", trackLongTaskNonNull)
            }
            useCrossSiteSessionCookie?.let { useCrossSiteSessionCookieNonNull ->
                json.addProperty("use_cross_site_session_cookie", useCrossSiteSessionCookieNonNull)
            }
            usePartitionedCrossSiteSessionCookie?.let { usePartitionedCrossSiteSessionCookieNonNull ->
                json.addProperty("use_partitioned_cross_site_session_cookie",
                        usePartitionedCrossSiteSessionCookieNonNull)
            }
            useSecureSessionCookie?.let { useSecureSessionCookieNonNull ->
                json.addProperty("use_secure_session_cookie", useSecureSessionCookieNonNull)
            }
            allowFallbackToLocalStorage?.let { allowFallbackToLocalStorageNonNull ->
                json.addProperty("allow_fallback_to_local_storage",
                        allowFallbackToLocalStorageNonNull)
            }
            storeContextsAcrossPages?.let { storeContextsAcrossPagesNonNull ->
                json.addProperty("store_contexts_across_pages", storeContextsAcrossPagesNonNull)
            }
            allowUntrustedEvents?.let { allowUntrustedEventsNonNull ->
                json.addProperty("allow_untrusted_events", allowUntrustedEventsNonNull)
            }
            actionNameAttribute?.let { actionNameAttributeNonNull ->
                json.addProperty("action_name_attribute", actionNameAttributeNonNull)
            }
            useAllowedTracingOrigins?.let { useAllowedTracingOriginsNonNull ->
                json.addProperty("use_allowed_tracing_origins", useAllowedTracingOriginsNonNull)
            }
            useAllowedTracingUrls?.let { useAllowedTracingUrlsNonNull ->
                json.addProperty("use_allowed_tracing_urls", useAllowedTracingUrlsNonNull)
            }
            selectedTracingPropagators?.let { selectedTracingPropagatorsNonNull ->
                val selectedTracingPropagatorsArray =
                        JsonArray(selectedTracingPropagatorsNonNull.size)
                selectedTracingPropagatorsNonNull.forEach {
                        selectedTracingPropagatorsArray.add(it.toJson()) }
                json.add("selected_tracing_propagators", selectedTracingPropagatorsArray)
            }
            defaultPrivacyLevel?.let { defaultPrivacyLevelNonNull ->
                json.addProperty("default_privacy_level", defaultPrivacyLevelNonNull)
            }
            textAndInputPrivacyLevel?.let { textAndInputPrivacyLevelNonNull ->
                json.addProperty("text_and_input_privacy_level", textAndInputPrivacyLevelNonNull)
            }
            imagePrivacyLevel?.let { imagePrivacyLevelNonNull ->
                json.addProperty("image_privacy_level", imagePrivacyLevelNonNull)
            }
            touchPrivacyLevel?.let { touchPrivacyLevelNonNull ->
                json.addProperty("touch_privacy_level", touchPrivacyLevelNonNull)
            }
            enablePrivacyForActionName?.let { enablePrivacyForActionNameNonNull ->
                json.addProperty("enable_privacy_for_action_name",
                        enablePrivacyForActionNameNonNull)
            }
            useExcludedActivityUrls?.let { useExcludedActivityUrlsNonNull ->
                json.addProperty("use_excluded_activity_urls", useExcludedActivityUrlsNonNull)
            }
            useWorkerUrl?.let { useWorkerUrlNonNull ->
                json.addProperty("use_worker_url", useWorkerUrlNonNull)
            }
            compressIntakeRequests?.let { compressIntakeRequestsNonNull ->
                json.addProperty("compress_intake_requests", compressIntakeRequestsNonNull)
            }
            trackFrustrations?.let { trackFrustrationsNonNull ->
                json.addProperty("track_frustrations", trackFrustrationsNonNull)
            }
            trackViewsManually?.let { trackViewsManuallyNonNull ->
                json.addProperty("track_views_manually", trackViewsManuallyNonNull)
            }
            trackInteractions?.let { trackInteractionsNonNull ->
                json.addProperty("track_interactions", trackInteractionsNonNull)
            }
            trackUserInteractions?.let { trackUserInteractionsNonNull ->
                json.addProperty("track_user_interactions", trackUserInteractionsNonNull)
            }
            forwardErrorsToLogs?.let { forwardErrorsToLogsNonNull ->
                json.addProperty("forward_errors_to_logs", forwardErrorsToLogsNonNull)
            }
            forwardConsoleLogs?.let { forwardConsoleLogsNonNull ->
                val forwardConsoleLogsArray = JsonArray(forwardConsoleLogsNonNull.size)
                forwardConsoleLogsNonNull.forEach { forwardConsoleLogsArray.add(it) }
                json.add("forward_console_logs", forwardConsoleLogsArray)
            }
            forwardReports?.let { forwardReportsNonNull ->
                val forwardReportsArray = JsonArray(forwardReportsNonNull.size)
                forwardReportsNonNull.forEach { forwardReportsArray.add(it) }
                json.add("forward_reports", forwardReportsArray)
            }
            useLocalEncryption?.let { useLocalEncryptionNonNull ->
                json.addProperty("use_local_encryption", useLocalEncryptionNonNull)
            }
            viewTrackingStrategy?.let { viewTrackingStrategyNonNull ->
                json.add("view_tracking_strategy", viewTrackingStrategyNonNull.toJson())
            }
            trackBackgroundEvents?.let { trackBackgroundEventsNonNull ->
                json.addProperty("track_background_events", trackBackgroundEventsNonNull)
            }
            mobileVitalsUpdatePeriod?.let { mobileVitalsUpdatePeriodNonNull ->
                json.addProperty("mobile_vitals_update_period", mobileVitalsUpdatePeriodNonNull)
            }
            trackErrors?.let { trackErrorsNonNull ->
                json.addProperty("track_errors", trackErrorsNonNull)
            }
            trackNetworkRequests?.let { trackNetworkRequestsNonNull ->
                json.addProperty("track_network_requests", trackNetworkRequestsNonNull)
            }
            useTracing?.let { useTracingNonNull ->
                json.addProperty("use_tracing", useTracingNonNull)
            }
            trackNativeViews?.let { trackNativeViewsNonNull ->
                json.addProperty("track_native_views", trackNativeViewsNonNull)
            }
            trackNativeErrors?.let { trackNativeErrorsNonNull ->
                json.addProperty("track_native_errors", trackNativeErrorsNonNull)
            }
            trackNativeLongTasks?.let { trackNativeLongTasksNonNull ->
                json.addProperty("track_native_long_tasks", trackNativeLongTasksNonNull)
            }
            trackCrossPlatformLongTasks?.let { trackCrossPlatformLongTasksNonNull ->
                json.addProperty("track_cross_platform_long_tasks",
                        trackCrossPlatformLongTasksNonNull)
            }
            useFirstPartyHosts?.let { useFirstPartyHostsNonNull ->
                json.addProperty("use_first_party_hosts", useFirstPartyHostsNonNull)
            }
            initializationType?.let { initializationTypeNonNull ->
                json.addProperty("initialization_type", initializationTypeNonNull)
            }
            trackFlutterPerformance?.let { trackFlutterPerformanceNonNull ->
                json.addProperty("track_flutter_performance", trackFlutterPerformanceNonNull)
            }
            batchSize?.let { batchSizeNonNull ->
                json.addProperty("batch_size", batchSizeNonNull)
            }
            batchUploadFrequency?.let { batchUploadFrequencyNonNull ->
                json.addProperty("batch_upload_frequency", batchUploadFrequencyNonNull)
            }
            batchProcessingLevel?.let { batchProcessingLevelNonNull ->
                json.addProperty("batch_processing_level", batchProcessingLevelNonNull)
            }
            backgroundTasksEnabled?.let { backgroundTasksEnabledNonNull ->
                json.addProperty("background_tasks_enabled", backgroundTasksEnabledNonNull)
            }
            reactVersion?.let { reactVersionNonNull ->
                json.addProperty("react_version", reactVersionNonNull)
            }
            reactNativeVersion?.let { reactNativeVersionNonNull ->
                json.addProperty("react_native_version", reactNativeVersionNonNull)
            }
            dartVersion?.let { dartVersionNonNull ->
                json.addProperty("dart_version", dartVersionNonNull)
            }
            unityVersion?.let { unityVersionNonNull ->
                json.addProperty("unity_version", unityVersionNonNull)
            }
            appHangThreshold?.let { appHangThresholdNonNull ->
                json.addProperty("app_hang_threshold", appHangThresholdNonNull)
            }
            usePciIntake?.let { usePciIntakeNonNull ->
                json.addProperty("use_pci_intake", usePciIntakeNonNull)
            }
            tracerApi?.let { tracerApiNonNull ->
                json.addProperty("tracer_api", tracerApiNonNull)
            }
            tracerApiVersion?.let { tracerApiVersionNonNull ->
                json.addProperty("tracer_api_version", tracerApiVersionNonNull)
            }
            sendLogsAfterSessionExpiration?.let { sendLogsAfterSessionExpirationNonNull ->
                json.addProperty("send_logs_after_session_expiration",
                        sendLogsAfterSessionExpirationNonNull)
            }
            plugins?.let { pluginsNonNull ->
                val pluginsArray = JsonArray(pluginsNonNull.size)
                pluginsNonNull.forEach { pluginsArray.add(it.toJson()) }
                json.add("plugins", pluginsArray)
            }
            isMainProcess?.let { isMainProcessNonNull ->
                json.addProperty("is_main_process", isMainProcessNonNull)
            }
            collectFeatureFlagsOn?.let { collectFeatureFlagsOnNonNull ->
                val collectFeatureFlagsOnArray = JsonArray(collectFeatureFlagsOnNonNull.size)
                collectFeatureFlagsOnNonNull.forEach { collectFeatureFlagsOnArray.add(it.toJson()) }
                json.add("collect_feature_flags_on", collectFeatureFlagsOnArray)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Configuration {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Configuration {
                try {
                    val sessionSampleRate = jsonObject.get("session_sample_rate")?.asLong
                    val telemetrySampleRate = jsonObject.get("telemetry_sample_rate")?.asLong
                    val telemetryConfigurationSampleRate =
                            jsonObject.get("telemetry_configuration_sample_rate")?.asLong
                    val telemetryUsageSampleRate =
                            jsonObject.get("telemetry_usage_sample_rate")?.asLong
                    val traceSampleRate = jsonObject.get("trace_sample_rate")?.asLong
                    val traceContextInjection =
                            jsonObject.get("trace_context_injection")?.asString?.let {
                        TraceContextInjection.fromJson(it)
                    }
                    val premiumSampleRate = jsonObject.get("premium_sample_rate")?.asLong
                    val replaySampleRate = jsonObject.get("replay_sample_rate")?.asLong
                    val sessionReplaySampleRate =
                            jsonObject.get("session_replay_sample_rate")?.asLong
                    val trackingConsent = jsonObject.get("tracking_consent")?.asString?.let {
                        TrackingConsent.fromJson(it)
                    }
                    val startSessionReplayRecordingManually =
                            jsonObject.get("start_session_replay_recording_manually")?.asBoolean
                    val startRecordingImmediately =
                            jsonObject.get("start_recording_immediately")?.asBoolean
                    val useProxy = jsonObject.get("use_proxy")?.asBoolean
                    val useBeforeSend = jsonObject.get("use_before_send")?.asBoolean
                    val silentMultipleInit = jsonObject.get("silent_multiple_init")?.asBoolean
                    val trackSessionAcrossSubdomains =
                            jsonObject.get("track_session_across_subdomains")?.asBoolean
                    val trackResources = jsonObject.get("track_resources")?.asBoolean
                    val trackLongTask = jsonObject.get("track_long_task")?.asBoolean
                    val useCrossSiteSessionCookie =
                            jsonObject.get("use_cross_site_session_cookie")?.asBoolean
                    val usePartitionedCrossSiteSessionCookie =
                            jsonObject.get("use_partitioned_cross_site_session_cookie")?.asBoolean
                    val useSecureSessionCookie =
                            jsonObject.get("use_secure_session_cookie")?.asBoolean
                    val allowFallbackToLocalStorage =
                            jsonObject.get("allow_fallback_to_local_storage")?.asBoolean
                    val storeContextsAcrossPages =
                            jsonObject.get("store_contexts_across_pages")?.asBoolean
                    val allowUntrustedEvents = jsonObject.get("allow_untrusted_events")?.asBoolean
                    val actionNameAttribute = jsonObject.get("action_name_attribute")?.asString
                    val useAllowedTracingOrigins =
                            jsonObject.get("use_allowed_tracing_origins")?.asBoolean
                    val useAllowedTracingUrls =
                            jsonObject.get("use_allowed_tracing_urls")?.asBoolean
                    val selectedTracingPropagators =
                            jsonObject.get("selected_tracing_propagators")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<SelectedTracingPropagator>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(SelectedTracingPropagator.fromJson(it.asString))
                        }
                        collection
                    }
                    val defaultPrivacyLevel = jsonObject.get("default_privacy_level")?.asString
                    val textAndInputPrivacyLevel =
                            jsonObject.get("text_and_input_privacy_level")?.asString
                    val imagePrivacyLevel = jsonObject.get("image_privacy_level")?.asString
                    val touchPrivacyLevel = jsonObject.get("touch_privacy_level")?.asString
                    val enablePrivacyForActionName =
                            jsonObject.get("enable_privacy_for_action_name")?.asBoolean
                    val useExcludedActivityUrls =
                            jsonObject.get("use_excluded_activity_urls")?.asBoolean
                    val useWorkerUrl = jsonObject.get("use_worker_url")?.asBoolean
                    val compressIntakeRequests =
                            jsonObject.get("compress_intake_requests")?.asBoolean
                    val trackFrustrations = jsonObject.get("track_frustrations")?.asBoolean
                    val trackViewsManually = jsonObject.get("track_views_manually")?.asBoolean
                    val trackInteractions = jsonObject.get("track_interactions")?.asBoolean
                    val trackUserInteractions = jsonObject.get("track_user_interactions")?.asBoolean
                    val forwardErrorsToLogs = jsonObject.get("forward_errors_to_logs")?.asBoolean
                    val forwardConsoleLogs =
                            jsonObject.get("forward_console_logs")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<String>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(it.asString)
                        }
                        collection
                    }
                    val forwardReports =
                            jsonObject.get("forward_reports")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<String>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(it.asString)
                        }
                        collection
                    }
                    val useLocalEncryption = jsonObject.get("use_local_encryption")?.asBoolean
                    val viewTrackingStrategy =
                            jsonObject.get("view_tracking_strategy")?.asString?.let {
                        ViewTrackingStrategy.fromJson(it)
                    }
                    val trackBackgroundEvents = jsonObject.get("track_background_events")?.asBoolean
                    val mobileVitalsUpdatePeriod =
                            jsonObject.get("mobile_vitals_update_period")?.asLong
                    val trackErrors = jsonObject.get("track_errors")?.asBoolean
                    val trackNetworkRequests = jsonObject.get("track_network_requests")?.asBoolean
                    val useTracing = jsonObject.get("use_tracing")?.asBoolean
                    val trackNativeViews = jsonObject.get("track_native_views")?.asBoolean
                    val trackNativeErrors = jsonObject.get("track_native_errors")?.asBoolean
                    val trackNativeLongTasks = jsonObject.get("track_native_long_tasks")?.asBoolean
                    val trackCrossPlatformLongTasks =
                            jsonObject.get("track_cross_platform_long_tasks")?.asBoolean
                    val useFirstPartyHosts = jsonObject.get("use_first_party_hosts")?.asBoolean
                    val initializationType = jsonObject.get("initialization_type")?.asString
                    val trackFlutterPerformance =
                            jsonObject.get("track_flutter_performance")?.asBoolean
                    val batchSize = jsonObject.get("batch_size")?.asLong
                    val batchUploadFrequency = jsonObject.get("batch_upload_frequency")?.asLong
                    val batchProcessingLevel = jsonObject.get("batch_processing_level")?.asLong
                    val backgroundTasksEnabled =
                            jsonObject.get("background_tasks_enabled")?.asBoolean
                    val reactVersion = jsonObject.get("react_version")?.asString
                    val reactNativeVersion = jsonObject.get("react_native_version")?.asString
                    val dartVersion = jsonObject.get("dart_version")?.asString
                    val unityVersion = jsonObject.get("unity_version")?.asString
                    val appHangThreshold = jsonObject.get("app_hang_threshold")?.asLong
                    val usePciIntake = jsonObject.get("use_pci_intake")?.asBoolean
                    val tracerApi = jsonObject.get("tracer_api")?.asString
                    val tracerApiVersion = jsonObject.get("tracer_api_version")?.asString
                    val sendLogsAfterSessionExpiration =
                            jsonObject.get("send_logs_after_session_expiration")?.asBoolean
                    val plugins = jsonObject.get("plugins")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<Plugin>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(Plugin.fromJsonObject(it.asJsonObject))
                        }
                        collection
                    }
                    val isMainProcess = jsonObject.get("is_main_process")?.asBoolean
                    val collectFeatureFlagsOn =
                            jsonObject.get("collect_feature_flags_on")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<CollectFeatureFlagsOn>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(CollectFeatureFlagsOn.fromJson(it.asString))
                        }
                        collection
                    }
                    return Configuration(sessionSampleRate, telemetrySampleRate,
                            telemetryConfigurationSampleRate, telemetryUsageSampleRate,
                            traceSampleRate, traceContextInjection, premiumSampleRate,
                            replaySampleRate, sessionReplaySampleRate, trackingConsent,
                            startSessionReplayRecordingManually, startRecordingImmediately,
                            useProxy, useBeforeSend, silentMultipleInit,
                            trackSessionAcrossSubdomains, trackResources, trackLongTask,
                            useCrossSiteSessionCookie, usePartitionedCrossSiteSessionCookie,
                            useSecureSessionCookie, allowFallbackToLocalStorage,
                            storeContextsAcrossPages, allowUntrustedEvents, actionNameAttribute,
                            useAllowedTracingOrigins, useAllowedTracingUrls,
                            selectedTracingPropagators, defaultPrivacyLevel,
                            textAndInputPrivacyLevel, imagePrivacyLevel, touchPrivacyLevel,
                            enablePrivacyForActionName, useExcludedActivityUrls, useWorkerUrl,
                            compressIntakeRequests, trackFrustrations, trackViewsManually,
                            trackInteractions, trackUserInteractions, forwardErrorsToLogs,
                            forwardConsoleLogs, forwardReports, useLocalEncryption,
                            viewTrackingStrategy, trackBackgroundEvents, mobileVitalsUpdatePeriod,
                            trackErrors, trackNetworkRequests, useTracing, trackNativeViews,
                            trackNativeErrors, trackNativeLongTasks, trackCrossPlatformLongTasks,
                            useFirstPartyHosts, initializationType, trackFlutterPerformance,
                            batchSize, batchUploadFrequency, batchProcessingLevel,
                            backgroundTasksEnabled, reactVersion, reactNativeVersion, dartVersion,
                            unityVersion, appHangThreshold, usePciIntake, tracerApi,
                            tracerApiVersion, sendLogsAfterSessionExpiration, plugins,
                            isMainProcess, collectFeatureFlagsOn)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param name The name of the plugin
     */
    public data class Plugin(
        public val name: String,
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("name", name)
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.add(k, JsonSerializer.toJsonElement(v))
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("name")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Plugin {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Plugin",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Plugin {
                try {
                    val name = jsonObject.get("name").asString
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value
                        }
                    }
                    return Plugin(name, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Plugin",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Plugin",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Plugin",
                        e
                    )
                }
            }
        }
    }

    /**
     * The source of this event
     */
    public enum class Source(
        private val jsonValue: String,
    ) {
        ANDROID("android"),
        IOS("ios"),
        BROWSER("browser"),
        FLUTTER("flutter"),
        REACT_NATIVE("react-native"),
        UNITY("unity"),
        KOTLIN_MULTIPLATFORM("kotlin-multiplatform"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Source = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * The opt-in configuration to add trace context
     */
    public enum class TraceContextInjection(
        private val jsonValue: String,
    ) {
        ALL("all"),
        SAMPLED("sampled"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): TraceContextInjection = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * The initial tracking consent value
     */
    public enum class TrackingConsent(
        private val jsonValue: String,
    ) {
        GRANTED("granted"),
        NOT_GRANTED("not-granted"),
        PENDING("pending"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): TrackingConsent = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    public enum class SelectedTracingPropagator(
        private val jsonValue: String,
    ) {
        DATADOG("datadog"),
        B3("b3"),
        B3MULTI("b3multi"),
        TRACECONTEXT("tracecontext"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): SelectedTracingPropagator = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * View tracking strategy
     */
    public enum class ViewTrackingStrategy(
        private val jsonValue: String,
    ) {
        ACTIVITYVIEWTRACKINGSTRATEGY("ActivityViewTrackingStrategy"),
        FRAGMENTVIEWTRACKINGSTRATEGY("FragmentViewTrackingStrategy"),
        MIXEDVIEWTRACKINGSTRATEGY("MixedViewTrackingStrategy"),
        NAVIGATIONVIEWTRACKINGSTRATEGY("NavigationViewTrackingStrategy"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): ViewTrackingStrategy = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    public enum class CollectFeatureFlagsOn(
        private val jsonValue: String,
    ) {
        VIEW("view"),
        ERROR("error"),
        VITAL("vital"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): CollectFeatureFlagsOn = values().first {
                it.jsonValue == jsonString
            }
        }
    }
}
