package com.datadog.android.rum.model

import com.datadog.android.core.`internal`.utils.JsonSerializer
import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import com.google.gson.JsonPrimitive
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Any
import kotlin.Array
import kotlin.Boolean
import kotlin.Long
import kotlin.Number
import kotlin.String
import kotlin.collections.ArrayList
import kotlin.collections.List
import kotlin.collections.MutableMap
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Schema of common properties of RUM events
 * View Container schema for views that are nested (webviews in mobile)
 * @param date Start of the event in ms from epoch
 * @param application Application properties
 * @param service The service name for this application
 * @param version The version for this application
 * @param buildVersion The build version for this application
 * @param buildId Generated unique ID of the application build. Unlike version or build_version this
 * field is not meant to be coming from the user, but rather generated by the tooling for each build.
 * @param session Session properties
 * @param source The source of this event
 * @param view View properties
 * View properties
 * @param usr User properties
 * @param connectivity Device connectivity properties
 * @param display Display properties
 * @param synthetics Synthetics properties
 * @param ciTest CI Visibility properties
 * @param os Operating system properties
 * @param device Device properties
 * @param dd Internal properties
 * Internal properties
 * @param context User provided context
 * @param container View Container properties (view wrapping the current view)
 * @param action Action properties
 */
public data class ActionEvent(
    public val date: Long,
    public val application: Application,
    public val service: String? = null,
    public val version: String? = null,
    public val buildVersion: String? = null,
    public val buildId: String? = null,
    public val session: ActionEventSession,
    public val source: ActionEventSource? = null,
    public val view: ActionEventView,
    public val usr: Usr? = null,
    public val connectivity: Connectivity? = null,
    public val display: Display? = null,
    public val synthetics: Synthetics? = null,
    public val ciTest: CiTest? = null,
    public val os: Os? = null,
    public val device: Device? = null,
    public val dd: Dd,
    public val context: Context? = null,
    public val container: Container? = null,
    public val action: ActionEventAction,
) {
    public val type: String = "action"

    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.addProperty("date", date)
        json.add("application", application.toJson())
        service?.let { serviceNonNull ->
            json.addProperty("service", serviceNonNull)
        }
        version?.let { versionNonNull ->
            json.addProperty("version", versionNonNull)
        }
        buildVersion?.let { buildVersionNonNull ->
            json.addProperty("build_version", buildVersionNonNull)
        }
        buildId?.let { buildIdNonNull ->
            json.addProperty("build_id", buildIdNonNull)
        }
        json.add("session", session.toJson())
        source?.let { sourceNonNull ->
            json.add("source", sourceNonNull.toJson())
        }
        json.add("view", view.toJson())
        usr?.let { usrNonNull ->
            json.add("usr", usrNonNull.toJson())
        }
        connectivity?.let { connectivityNonNull ->
            json.add("connectivity", connectivityNonNull.toJson())
        }
        display?.let { displayNonNull ->
            json.add("display", displayNonNull.toJson())
        }
        synthetics?.let { syntheticsNonNull ->
            json.add("synthetics", syntheticsNonNull.toJson())
        }
        ciTest?.let { ciTestNonNull ->
            json.add("ci_test", ciTestNonNull.toJson())
        }
        os?.let { osNonNull ->
            json.add("os", osNonNull.toJson())
        }
        device?.let { deviceNonNull ->
            json.add("device", deviceNonNull.toJson())
        }
        json.add("_dd", dd.toJson())
        context?.let { contextNonNull ->
            json.add("context", contextNonNull.toJson())
        }
        container?.let { containerNonNull ->
            json.add("container", containerNonNull.toJson())
        }
        json.addProperty("type", type)
        json.add("action", action.toJson())
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): ActionEvent {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type ActionEvent",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): ActionEvent {
            try {
                val date = jsonObject.get("date").asLong
                val application = jsonObject.get("application").asJsonObject.let {
                    Application.fromJsonObject(it)
                }
                val service = jsonObject.get("service")?.asString
                val version = jsonObject.get("version")?.asString
                val buildVersion = jsonObject.get("build_version")?.asString
                val buildId = jsonObject.get("build_id")?.asString
                val session = jsonObject.get("session").asJsonObject.let {
                    ActionEventSession.fromJsonObject(it)
                }
                val source = jsonObject.get("source")?.asString?.let {
                    ActionEventSource.fromJson(it)
                }
                val view = jsonObject.get("view").asJsonObject.let {
                    ActionEventView.fromJsonObject(it)
                }
                val usr = jsonObject.get("usr")?.asJsonObject?.let {
                    Usr.fromJsonObject(it)
                }
                val connectivity = jsonObject.get("connectivity")?.asJsonObject?.let {
                    Connectivity.fromJsonObject(it)
                }
                val display = jsonObject.get("display")?.asJsonObject?.let {
                    Display.fromJsonObject(it)
                }
                val synthetics = jsonObject.get("synthetics")?.asJsonObject?.let {
                    Synthetics.fromJsonObject(it)
                }
                val ciTest = jsonObject.get("ci_test")?.asJsonObject?.let {
                    CiTest.fromJsonObject(it)
                }
                val os = jsonObject.get("os")?.asJsonObject?.let {
                    Os.fromJsonObject(it)
                }
                val device = jsonObject.get("device")?.asJsonObject?.let {
                    Device.fromJsonObject(it)
                }
                val dd = jsonObject.get("_dd").asJsonObject.let {
                    Dd.fromJsonObject(it)
                }
                val context = jsonObject.get("context")?.asJsonObject?.let {
                    Context.fromJsonObject(it)
                }
                val container = jsonObject.get("container")?.asJsonObject?.let {
                    Container.fromJsonObject(it)
                }
                val type = jsonObject.get("type").asString
                val action = jsonObject.get("action").asJsonObject.let {
                    ActionEventAction.fromJsonObject(it)
                }
                check(type == "action")
                return ActionEvent(date, application, service, version, buildVersion, buildId,
                        session, source, view, usr, connectivity, display, synthetics, ciTest, os,
                        device, dd, context, container, action)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type ActionEvent",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type ActionEvent",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type ActionEvent",
                    e
                )
            }
        }
    }

    /**
     * Application properties
     * @param id UUID of the application
     */
    public data class Application(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Application {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Application {
                try {
                    val id = jsonObject.get("id").asString
                    return Application(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session properties
     * @param id UUID of the session
     * @param type Type of the session
     * @param hasReplay Whether this session has a replay
     */
    public data class ActionEventSession(
        public val id: String,
        public val type: ActionEventSessionType,
        public val hasReplay: Boolean? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            json.add("type", type.toJson())
            hasReplay?.let { hasReplayNonNull ->
                json.addProperty("has_replay", hasReplayNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): ActionEventSession {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventSession",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): ActionEventSession {
                try {
                    val id = jsonObject.get("id").asString
                    val type = ActionEventSessionType.fromJson(jsonObject.get("type").asString)
                    val hasReplay = jsonObject.get("has_replay")?.asBoolean
                    return ActionEventSession(id, type, hasReplay)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventSession",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventSession",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventSession",
                        e
                    )
                }
            }
        }
    }

    /**
     * View properties
     * View properties
     * @param id UUID of the view
     * @param referrer URL that linked to the initial view of the page
     * @param url URL of the view
     * @param name User defined name of the view
     * @param inForeground Is the action starting in the foreground (focus in browser)
     */
    public data class ActionEventView(
        public val id: String,
        public var referrer: String? = null,
        public var url: String,
        public var name: String? = null,
        public val inForeground: Boolean? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            referrer?.let { referrerNonNull ->
                json.addProperty("referrer", referrerNonNull)
            }
            json.addProperty("url", url)
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            inForeground?.let { inForegroundNonNull ->
                json.addProperty("in_foreground", inForegroundNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): ActionEventView {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventView",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): ActionEventView {
                try {
                    val id = jsonObject.get("id").asString
                    val referrer = jsonObject.get("referrer")?.asString
                    val url = jsonObject.get("url").asString
                    val name = jsonObject.get("name")?.asString
                    val inForeground = jsonObject.get("in_foreground")?.asBoolean
                    return ActionEventView(id, referrer, url, name, inForeground)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventView",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventView",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventView",
                        e
                    )
                }
            }
        }
    }

    /**
     * User properties
     * @param id Identifier of the user
     * @param name Name of the user
     * @param email Email of the user
     */
    public data class Usr(
        public val id: String? = null,
        public val name: String? = null,
        public val email: String? = null,
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            email?.let { emailNonNull ->
                json.addProperty("email", emailNonNull)
            }
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.add(k, JsonSerializer.toJsonElement(v))
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("id", "name", "email")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Usr {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Usr {
                try {
                    val id = jsonObject.get("id")?.asString
                    val name = jsonObject.get("name")?.asString
                    val email = jsonObject.get("email")?.asString
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value
                        }
                    }
                    return Usr(id, name, email, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device connectivity properties
     * @param status Status of the device connectivity
     * @param interfaces The list of available network interfaces
     * @param effectiveType Cellular connection type reflecting the measured network performance
     * @param cellular Cellular connectivity properties
     */
    public data class Connectivity(
        public val status: Status,
        public val interfaces: List<Interface>? = null,
        public val effectiveType: EffectiveType? = null,
        public val cellular: Cellular? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("status", status.toJson())
            interfaces?.let { interfacesNonNull ->
                val interfacesArray = JsonArray(interfacesNonNull.size)
                interfacesNonNull.forEach { interfacesArray.add(it.toJson()) }
                json.add("interfaces", interfacesArray)
            }
            effectiveType?.let { effectiveTypeNonNull ->
                json.add("effective_type", effectiveTypeNonNull.toJson())
            }
            cellular?.let { cellularNonNull ->
                json.add("cellular", cellularNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Connectivity {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Connectivity {
                try {
                    val status = Status.fromJson(jsonObject.get("status").asString)
                    val interfaces = jsonObject.get("interfaces")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<Interface>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(Interface.fromJson(it.asString))
                        }
                        collection
                    }
                    val effectiveType = jsonObject.get("effective_type")?.asString?.let {
                        EffectiveType.fromJson(it)
                    }
                    val cellular = jsonObject.get("cellular")?.asJsonObject?.let {
                        Cellular.fromJsonObject(it)
                    }
                    return Connectivity(status, interfaces, effectiveType, cellular)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                }
            }
        }
    }

    /**
     * Display properties
     * @param viewport The viewport represents the rectangular area that is currently being viewed.
     * Content outside the viewport is not visible onscreen until scrolled into view.
     */
    public data class Display(
        public val viewport: Viewport? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            viewport?.let { viewportNonNull ->
                json.add("viewport", viewportNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Display {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Display {
                try {
                    val viewport = jsonObject.get("viewport")?.asJsonObject?.let {
                        Viewport.fromJsonObject(it)
                    }
                    return Display(viewport)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                }
            }
        }
    }

    /**
     * Synthetics properties
     * @param testId The identifier of the current Synthetics test
     * @param resultId The identifier of the current Synthetics test results
     * @param injected Whether the event comes from a SDK instance injected by Synthetics
     */
    public data class Synthetics(
        public val testId: String,
        public val resultId: String,
        public val injected: Boolean? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("test_id", testId)
            json.addProperty("result_id", resultId)
            injected?.let { injectedNonNull ->
                json.addProperty("injected", injectedNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Synthetics {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Synthetics {
                try {
                    val testId = jsonObject.get("test_id").asString
                    val resultId = jsonObject.get("result_id").asString
                    val injected = jsonObject.get("injected")?.asBoolean
                    return Synthetics(testId, resultId, injected)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                }
            }
        }
    }

    /**
     * CI Visibility properties
     * @param testExecutionId The identifier of the current CI Visibility test execution
     */
    public data class CiTest(
        public val testExecutionId: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("test_execution_id", testExecutionId)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): CiTest {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): CiTest {
                try {
                    val testExecutionId = jsonObject.get("test_execution_id").asString
                    return CiTest(testExecutionId)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                }
            }
        }
    }

    /**
     * Operating system properties
     * @param name Operating system name, e.g. Android, iOS
     * @param version Full operating system version, e.g. 8.1.1
     * @param build Operating system build number, e.g. 15D21
     * @param versionMajor Major operating system version, e.g. 8
     */
    public data class Os(
        public val name: String,
        public val version: String,
        public val build: String? = null,
        public val versionMajor: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("name", name)
            json.addProperty("version", version)
            build?.let { buildNonNull ->
                json.addProperty("build", buildNonNull)
            }
            json.addProperty("version_major", versionMajor)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Os {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Os {
                try {
                    val name = jsonObject.get("name").asString
                    val version = jsonObject.get("version").asString
                    val build = jsonObject.get("build")?.asString
                    val versionMajor = jsonObject.get("version_major").asString
                    return Os(name, version, build, versionMajor)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device properties
     * @param type Device type info
     * @param name Device marketing name, e.g. Xiaomi Redmi Note 8 Pro, Pixel 5, etc.
     * @param model Device SKU model, e.g. Samsung SM-988GN, etc. Quite often name and model can be
     * the same.
     * @param brand Device marketing brand, e.g. Apple, OPPO, Xiaomi, etc.
     * @param architecture The CPU architecture of the device that is reporting the error
     */
    public data class Device(
        public val type: DeviceType,
        public val name: String? = null,
        public val model: String? = null,
        public val brand: String? = null,
        public val architecture: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("type", type.toJson())
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            model?.let { modelNonNull ->
                json.addProperty("model", modelNonNull)
            }
            brand?.let { brandNonNull ->
                json.addProperty("brand", brandNonNull)
            }
            architecture?.let { architectureNonNull ->
                json.addProperty("architecture", architectureNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Device {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Device {
                try {
                    val type = DeviceType.fromJson(jsonObject.get("type").asString)
                    val name = jsonObject.get("name")?.asString
                    val model = jsonObject.get("model")?.asString
                    val brand = jsonObject.get("brand")?.asString
                    val architecture = jsonObject.get("architecture")?.asString
                    return Device(type, name, model, brand, architecture)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }
        }
    }

    /**
     * Internal properties
     * Internal properties
     * @param session Session-related internal properties
     * @param configuration Subset of the SDK configuration options in use during its execution
     * @param browserSdkVersion Browser SDK version
     * @param action Action properties
     */
    public data class Dd(
        public val session: DdSession? = null,
        public val configuration: Configuration? = null,
        public val browserSdkVersion: String? = null,
        public val action: DdAction? = null,
    ) {
        public val formatVersion: Long = 2L

        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("format_version", formatVersion)
            session?.let { sessionNonNull ->
                json.add("session", sessionNonNull.toJson())
            }
            configuration?.let { configurationNonNull ->
                json.add("configuration", configurationNonNull.toJson())
            }
            browserSdkVersion?.let { browserSdkVersionNonNull ->
                json.addProperty("browser_sdk_version", browserSdkVersionNonNull)
            }
            action?.let { actionNonNull ->
                json.add("action", actionNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Dd {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Dd {
                try {
                    val formatVersion = jsonObject.get("format_version").asLong
                    val session = jsonObject.get("session")?.asJsonObject?.let {
                        DdSession.fromJsonObject(it)
                    }
                    val configuration = jsonObject.get("configuration")?.asJsonObject?.let {
                        Configuration.fromJsonObject(it)
                    }
                    val browserSdkVersion = jsonObject.get("browser_sdk_version")?.asString
                    val action = jsonObject.get("action")?.asJsonObject?.let {
                        DdAction.fromJsonObject(it)
                    }
                    check(formatVersion == 2.0.toLong())
                    return Dd(session, configuration, browserSdkVersion, action)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }
        }
    }

    /**
     * User provided context
     */
    public data class Context(
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            additionalProperties.forEach { (k, v) ->
                json.add(k, JsonSerializer.toJsonElement(v))
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Context {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Context {
                try {
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        additionalProperties[entry.key] = entry.value
                    }
                    return Context(additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                }
            }
        }
    }

    /**
     * View Container properties (view wrapping the current view)
     * @param view Attributes of the view's container
     * @param source Source of the parent view
     */
    public data class Container(
        public val view: ContainerView,
        public val source: ActionEventSource,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("view", view.toJson())
            json.add("source", source.toJson())
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Container {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Container",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Container {
                try {
                    val view = jsonObject.get("view").asJsonObject.let {
                        ContainerView.fromJsonObject(it)
                    }
                    val source = ActionEventSource.fromJson(jsonObject.get("source").asString)
                    return Container(view, source)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Container",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Container",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Container",
                        e
                    )
                }
            }
        }
    }

    /**
     * Action properties
     * @param type Type of the action
     * @param id UUID of the action
     * @param loadingTime Duration in ns to the action is considered loaded
     * @param target Action target properties
     * @param frustration Action frustration properties
     * @param error Properties of the errors of the action
     * @param crash Properties of the crashes of the action
     * @param longTask Properties of the long tasks of the action
     * @param resource Properties of the resources of the action
     */
    public data class ActionEventAction(
        public val type: ActionEventActionType,
        public val id: String? = null,
        public val loadingTime: Long? = null,
        public val target: ActionEventActionTarget? = null,
        public val frustration: Frustration? = null,
        public val error: Error? = null,
        public val crash: Crash? = null,
        public val longTask: LongTask? = null,
        public val resource: Resource? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("type", type.toJson())
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            loadingTime?.let { loadingTimeNonNull ->
                json.addProperty("loading_time", loadingTimeNonNull)
            }
            target?.let { targetNonNull ->
                json.add("target", targetNonNull.toJson())
            }
            frustration?.let { frustrationNonNull ->
                json.add("frustration", frustrationNonNull.toJson())
            }
            error?.let { errorNonNull ->
                json.add("error", errorNonNull.toJson())
            }
            crash?.let { crashNonNull ->
                json.add("crash", crashNonNull.toJson())
            }
            longTask?.let { longTaskNonNull ->
                json.add("long_task", longTaskNonNull.toJson())
            }
            resource?.let { resourceNonNull ->
                json.add("resource", resourceNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): ActionEventAction {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventAction",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): ActionEventAction {
                try {
                    val type = ActionEventActionType.fromJson(jsonObject.get("type").asString)
                    val id = jsonObject.get("id")?.asString
                    val loadingTime = jsonObject.get("loading_time")?.asLong
                    val target = jsonObject.get("target")?.asJsonObject?.let {
                        ActionEventActionTarget.fromJsonObject(it)
                    }
                    val frustration = jsonObject.get("frustration")?.asJsonObject?.let {
                        Frustration.fromJsonObject(it)
                    }
                    val error = jsonObject.get("error")?.asJsonObject?.let {
                        Error.fromJsonObject(it)
                    }
                    val crash = jsonObject.get("crash")?.asJsonObject?.let {
                        Crash.fromJsonObject(it)
                    }
                    val longTask = jsonObject.get("long_task")?.asJsonObject?.let {
                        LongTask.fromJsonObject(it)
                    }
                    val resource = jsonObject.get("resource")?.asJsonObject?.let {
                        Resource.fromJsonObject(it)
                    }
                    return ActionEventAction(type, id, loadingTime, target, frustration, error,
                            crash, longTask, resource)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventAction",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventAction",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventAction",
                        e
                    )
                }
            }
        }
    }

    /**
     * Cellular connectivity properties
     * @param technology The type of a radio technology used for cellular connection
     * @param carrierName The name of the SIM carrier
     */
    public data class Cellular(
        public val technology: String? = null,
        public val carrierName: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            technology?.let { technologyNonNull ->
                json.addProperty("technology", technologyNonNull)
            }
            carrierName?.let { carrierNameNonNull ->
                json.addProperty("carrier_name", carrierNameNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Cellular {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Cellular {
                try {
                    val technology = jsonObject.get("technology")?.asString
                    val carrierName = jsonObject.get("carrier_name")?.asString
                    return Cellular(technology, carrierName)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                }
            }
        }
    }

    /**
     * The viewport represents the rectangular area that is currently being viewed. Content outside
     * the viewport is not visible onscreen until scrolled into view.
     * @param width Width of the viewport (in pixels)
     * @param height Height of the viewport (in pixels)
     */
    public data class Viewport(
        public val width: Number,
        public val height: Number,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("width", width)
            json.addProperty("height", height)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Viewport {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Viewport {
                try {
                    val width = jsonObject.get("width").asNumber
                    val height = jsonObject.get("height").asNumber
                    return Viewport(width, height)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session-related internal properties
     * @param plan Session plan: 1 is the plan without replay, 2 is the plan with replay
     * (deprecated)
     * @param sessionPrecondition The precondition that led to the creation of the session
     */
    public data class DdSession(
        public val plan: Plan? = null,
        public val sessionPrecondition: SessionPrecondition? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            plan?.let { planNonNull ->
                json.add("plan", planNonNull.toJson())
            }
            sessionPrecondition?.let { sessionPreconditionNonNull ->
                json.add("session_precondition", sessionPreconditionNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): DdSession {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): DdSession {
                try {
                    val plan = jsonObject.get("plan")?.asString?.let {
                        Plan.fromJson(it)
                    }
                    val sessionPrecondition =
                            jsonObject.get("session_precondition")?.asString?.let {
                        SessionPrecondition.fromJson(it)
                    }
                    return DdSession(plan, sessionPrecondition)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                }
            }
        }
    }

    /**
     * Subset of the SDK configuration options in use during its execution
     * @param sessionSampleRate The percentage of sessions tracked
     * @param sessionReplaySampleRate The percentage of sessions with RUM & Session Replay pricing
     * tracked
     */
    public data class Configuration(
        public val sessionSampleRate: Number,
        public val sessionReplaySampleRate: Number? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("session_sample_rate", sessionSampleRate)
            sessionReplaySampleRate?.let { sessionReplaySampleRateNonNull ->
                json.addProperty("session_replay_sample_rate", sessionReplaySampleRateNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Configuration {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Configuration {
                try {
                    val sessionSampleRate = jsonObject.get("session_sample_rate").asNumber
                    val sessionReplaySampleRate =
                            jsonObject.get("session_replay_sample_rate")?.asNumber
                    return Configuration(sessionSampleRate, sessionReplaySampleRate)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                }
            }
        }
    }

    /**
     * Action properties
     * @param position Action position properties
     * @param target Target properties
     */
    public data class DdAction(
        public val position: Position? = null,
        public val target: DdActionTarget? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            position?.let { positionNonNull ->
                json.add("position", positionNonNull.toJson())
            }
            target?.let { targetNonNull ->
                json.add("target", targetNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): DdAction {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdAction",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): DdAction {
                try {
                    val position = jsonObject.get("position")?.asJsonObject?.let {
                        Position.fromJsonObject(it)
                    }
                    val target = jsonObject.get("target")?.asJsonObject?.let {
                        DdActionTarget.fromJsonObject(it)
                    }
                    return DdAction(position, target)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdAction",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdAction",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdAction",
                        e
                    )
                }
            }
        }
    }

    /**
     * Attributes of the view's container
     * @param id ID of the parent view
     */
    public data class ContainerView(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): ContainerView {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ContainerView",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): ContainerView {
                try {
                    val id = jsonObject.get("id").asString
                    return ContainerView(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ContainerView",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type ContainerView",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type ContainerView",
                        e
                    )
                }
            }
        }
    }

    /**
     * Action target properties
     * @param name Target name
     */
    public data class ActionEventActionTarget(
        public var name: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("name", name)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): ActionEventActionTarget {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventActionTarget",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): ActionEventActionTarget {
                try {
                    val name = jsonObject.get("name").asString
                    return ActionEventActionTarget(name)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventActionTarget",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventActionTarget",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type ActionEventActionTarget",
                        e
                    )
                }
            }
        }
    }

    /**
     * Action frustration properties
     * @param type Action frustration types
     */
    public data class Frustration(
        public val type: List<Type>,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            val typeArray = JsonArray(type.size)
            type.forEach { typeArray.add(it.toJson()) }
            json.add("type", typeArray)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Frustration {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Frustration",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Frustration {
                try {
                    val type = jsonObject.get("type").asJsonArray.let { jsonArray ->
                        val collection = ArrayList<Type>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(Type.fromJson(it.asString))
                        }
                        collection
                    }
                    return Frustration(type)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Frustration",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Frustration",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Frustration",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the errors of the action
     * @param count Number of errors that occurred on the action
     */
    public data class Error(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Error {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Error {
                try {
                    val count = jsonObject.get("count").asLong
                    return Error(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the crashes of the action
     * @param count Number of crashes that occurred on the action
     */
    public data class Crash(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Crash {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Crash",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Crash {
                try {
                    val count = jsonObject.get("count").asLong
                    return Crash(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Crash",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Crash",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Crash",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the long tasks of the action
     * @param count Number of long tasks that occurred on the action
     */
    public data class LongTask(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): LongTask {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type LongTask",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): LongTask {
                try {
                    val count = jsonObject.get("count").asLong
                    return LongTask(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type LongTask",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type LongTask",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type LongTask",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the resources of the action
     * @param count Number of resources that occurred on the action
     */
    public data class Resource(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Resource {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Resource {
                try {
                    val count = jsonObject.get("count").asLong
                    return Resource(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                }
            }
        }
    }

    /**
     * Action position properties
     * @param x X coordinate relative to the target element of the action (in pixels)
     * @param y Y coordinate relative to the target element of the action (in pixels)
     */
    public data class Position(
        public val x: Long,
        public val y: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("x", x)
            json.addProperty("y", y)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Position {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Position",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Position {
                try {
                    val x = jsonObject.get("x").asLong
                    val y = jsonObject.get("y").asLong
                    return Position(x, y)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Position",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Position",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Position",
                        e
                    )
                }
            }
        }
    }

    /**
     * Target properties
     * @param selector CSS selector path of the target element
     * @param width Width of the target element (in pixels)
     * @param height Height of the target element (in pixels)
     */
    public data class DdActionTarget(
        public val selector: String? = null,
        public val width: Long? = null,
        public val height: Long? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            selector?.let { selectorNonNull ->
                json.addProperty("selector", selectorNonNull)
            }
            width?.let { widthNonNull ->
                json.addProperty("width", widthNonNull)
            }
            height?.let { heightNonNull ->
                json.addProperty("height", heightNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): DdActionTarget {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdActionTarget",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): DdActionTarget {
                try {
                    val selector = jsonObject.get("selector")?.asString
                    val width = jsonObject.get("width")?.asLong
                    val height = jsonObject.get("height")?.asLong
                    return DdActionTarget(selector, width, height)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdActionTarget",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdActionTarget",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdActionTarget",
                        e
                    )
                }
            }
        }
    }

    /**
     * The source of this event
     */
    public enum class ActionEventSource(
        private val jsonValue: String,
    ) {
        ANDROID("android"),
        IOS("ios"),
        BROWSER("browser"),
        FLUTTER("flutter"),
        REACT_NATIVE("react-native"),
        ROKU("roku"),
        UNITY("unity"),
        KOTLIN_MULTIPLATFORM("kotlin-multiplatform"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): ActionEventSource = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Type of the session
     */
    public enum class ActionEventSessionType(
        private val jsonValue: String,
    ) {
        USER("user"),
        SYNTHETICS("synthetics"),
        CI_TEST("ci_test"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): ActionEventSessionType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Status of the device connectivity
     */
    public enum class Status(
        private val jsonValue: String,
    ) {
        CONNECTED("connected"),
        NOT_CONNECTED("not_connected"),
        MAYBE("maybe"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Status = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    public enum class Interface(
        private val jsonValue: String,
    ) {
        BLUETOOTH("bluetooth"),
        CELLULAR("cellular"),
        ETHERNET("ethernet"),
        WIFI("wifi"),
        WIMAX("wimax"),
        MIXED("mixed"),
        OTHER("other"),
        UNKNOWN("unknown"),
        NONE("none"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Interface = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Cellular connection type reflecting the measured network performance
     */
    public enum class EffectiveType(
        private val jsonValue: String,
    ) {
        SLOW_2G("slow_2g"),
        `2G`("2g"),
        `3G`("3g"),
        `4G`("4g"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): EffectiveType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Device type info
     */
    public enum class DeviceType(
        private val jsonValue: String,
    ) {
        MOBILE("mobile"),
        DESKTOP("desktop"),
        TABLET("tablet"),
        TV("tv"),
        GAMING_CONSOLE("gaming_console"),
        BOT("bot"),
        OTHER("other"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): DeviceType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Type of the action
     */
    public enum class ActionEventActionType(
        private val jsonValue: String,
    ) {
        CUSTOM("custom"),
        CLICK("click"),
        TAP("tap"),
        SCROLL("scroll"),
        SWIPE("swipe"),
        APPLICATION_START("application_start"),
        BACK("back"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): ActionEventActionType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Session plan: 1 is the plan without replay, 2 is the plan with replay (deprecated)
     */
    public enum class Plan(
        private val jsonValue: Number,
    ) {
        PLAN_1(1),
        PLAN_2(2),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Plan = values().first {
                it.jsonValue.toString() == jsonString
            }
        }
    }

    /**
     * The precondition that led to the creation of the session
     */
    public enum class SessionPrecondition(
        private val jsonValue: String,
    ) {
        USER_APP_LAUNCH("user_app_launch"),
        INACTIVITY_TIMEOUT("inactivity_timeout"),
        MAX_DURATION("max_duration"),
        BACKGROUND_LAUNCH("background_launch"),
        PREWARM("prewarm"),
        FROM_NON_INTERACTIVE_SESSION("from_non_interactive_session"),
        EXPLICIT_STOP("explicit_stop"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): SessionPrecondition = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    public enum class Type(
        private val jsonValue: String,
    ) {
        RAGE_CLICK("rage_click"),
        DEAD_CLICK("dead_click"),
        ERROR_CLICK("error_click"),
        RAGE_TAP("rage_tap"),
        ERROR_TAP("error_tap"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Type = values().first {
                it.jsonValue == jsonString
            }
        }
    }
}
