/*
 * Decompiled with CFR 0.152.
 */
package com.apptasticsoftware.rssreader;

import com.apptasticsoftware.rssreader.Channel;
import com.apptasticsoftware.rssreader.Enclosure;
import com.apptasticsoftware.rssreader.Image;
import com.apptasticsoftware.rssreader.Item;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.Spliterators;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import java.util.zip.GZIPInputStream;
import javax.net.ssl.SSLContext;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

public abstract class AbstractRssReader<C extends Channel, I extends Item> {
    private static final String LOG_GROUP = "com.apptasticsoftware.rssreader";
    private final HttpClient httpClient;
    private String userAgent = "";
    private final Map<String, String> headers = new HashMap<String, String>();
    private final HashMap<String, BiConsumer<C, String>> channelTags = new HashMap();
    private final HashMap<String, Map<String, BiConsumer<C, String>>> channelAttributes = new HashMap();
    private final HashMap<String, BiConsumer<I, String>> itemTags = new HashMap();
    private final HashMap<String, Map<String, BiConsumer<I, String>>> itemAttributes = new HashMap();
    private final HashMap<String, BiConsumer<Image, String>> imageTags = new HashMap();
    private boolean isInitialized;

    protected AbstractRssReader() {
        this.httpClient = this.createHttpClient();
    }

    protected AbstractRssReader(HttpClient httpClient) {
        Objects.requireNonNull(httpClient, "Http client must not be null");
        this.httpClient = httpClient;
    }

    protected abstract C createChannel();

    protected abstract I createItem();

    protected void initialize() {
        this.registerChannelTags();
        this.registerChannelAttributes();
        this.registerItemTags();
        this.registerItemAttributes();
        this.registerImageTags();
    }

    protected void registerChannelTags() {
        this.channelTags.put("title", Channel::setTitle);
        this.channelTags.put("description", Channel::setDescription);
        this.channelTags.put("subtitle", Channel::setDescription);
        this.channelTags.put("link", Channel::setLink);
        this.channelTags.put("category", Channel::addCategory);
        this.channelTags.put("language", Channel::setLanguage);
        this.channelTags.put("copyright", Channel::setCopyright);
        this.channelTags.put("rights", Channel::setCopyright);
        this.channelTags.put("generator", Channel::setGenerator);
        this.channelTags.put("ttl", Channel::setTtl);
        this.channelTags.put("pubDate", Channel::setPubDate);
        this.channelTags.put("lastBuildDate", Channel::setLastBuildDate);
        this.channelTags.put("updated", Channel::setLastBuildDate);
        this.channelTags.put("managingEditor", Channel::setManagingEditor);
        this.channelTags.put("webMaster", Channel::setWebMaster);
        this.channelTags.put("docs", Channel::setDocs);
        this.channelTags.put("rating", Channel::setRating);
    }

    protected void registerChannelAttributes() {
        this.channelAttributes.computeIfAbsent("link", k -> new HashMap()).put("href", Channel::setLink);
    }

    protected void registerItemTags() {
        this.itemTags.put("guid", Item::setGuid);
        this.itemTags.put("id", Item::setGuid);
        this.itemTags.put("title", Item::setTitle);
        this.itemTags.put("description", Item::setDescription);
        this.itemTags.put("summary", Item::setDescription);
        this.itemTags.put("content", Item::setDescription);
        this.itemTags.put("link", Item::setLink);
        this.itemTags.put("author", Item::setAuthor);
        this.itemTags.put("category", Item::addCategory);
        this.itemTags.put("pubDate", Item::setPubDate);
        this.itemTags.put("published", Item::setPubDate);
        this.itemTags.put("updated", (i, v) -> {
            if (i.getPubDate().isEmpty()) {
                i.setPubDate((String)v);
            }
        });
        this.itemTags.put("comments", Item::setComments);
    }

    protected void registerItemAttributes() {
        this.itemAttributes.computeIfAbsent("link", k -> new HashMap()).put("href", Item::setLink);
        this.itemAttributes.computeIfAbsent("guid", k -> new HashMap()).put("isPermaLink", (i, v) -> i.setIsPermaLink(Boolean.parseBoolean(v)));
        Map enclosureAttributes = this.itemAttributes.computeIfAbsent("enclosure", k -> new HashMap());
        enclosureAttributes.put("url", (i, v) -> i.getEnclosure().ifPresent(e -> e.setUrl((String)v)));
        enclosureAttributes.put("type", (i, v) -> i.getEnclosure().ifPresent(e -> e.setType((String)v)));
        enclosureAttributes.put("length", (i, v) -> i.getEnclosure().ifPresent(e -> this.mapLong((String)v, e::setLength)));
    }

    protected void registerImageTags() {
        this.imageTags.put("title", Image::setTitle);
        this.imageTags.put("link", Image::setLink);
        this.imageTags.put("url", Image::setUrl);
        this.imageTags.put("description", Image::setDescription);
        this.imageTags.put("height", (i, v) -> this.mapInteger((String)v, i::setHeight));
        this.imageTags.put("width", (i, v) -> this.mapInteger((String)v, i::setWidth));
    }

    protected void mapBoolean(String text, Consumer<Boolean> func) {
        if ("true".equals(text = text.toLowerCase()) || "yes".equals(text)) {
            func.accept(Boolean.TRUE);
        } else if ("false".equals(text) || "no".equals(text)) {
            func.accept(Boolean.FALSE);
        }
    }

    protected void mapInteger(String text, Consumer<Integer> func) {
        AbstractRssReader.mapNumber(text, func, Integer::valueOf);
    }

    protected void mapLong(String text, Consumer<Long> func) {
        AbstractRssReader.mapNumber(text, func, Long::valueOf);
    }

    private static <T> void mapNumber(String text, Consumer<T> func, Function<String, T> convert) {
        block3: {
            if (text != null && !text.isBlank()) {
                try {
                    func.accept(convert.apply(text));
                }
                catch (NumberFormatException e) {
                    Logger logger = Logger.getLogger(LOG_GROUP);
                    if (!logger.isLoggable(Level.WARNING)) break block3;
                    logger.log(Level.WARNING, () -> String.format("Failed to convert %s. Message: %s", text, e.getMessage()));
                }
            }
        }
    }

    public AbstractRssReader<C, I> setUserAgent(String userAgent) {
        Objects.requireNonNull(userAgent, "User-agent must not be null");
        this.userAgent = userAgent;
        return this;
    }

    public AbstractRssReader<C, I> addHeader(String key, String value) {
        Objects.requireNonNull(key, "Key must not be null");
        Objects.requireNonNull(value, "Value must not be null");
        this.headers.put(key, value);
        return this;
    }

    public AbstractRssReader<C, I> addItemExtension(String tag, BiConsumer<I, String> consumer) {
        Objects.requireNonNull(tag, "Item tag must not be null");
        Objects.requireNonNull(consumer, "Item consumer must not be null");
        this.itemTags.put(tag, consumer);
        return this;
    }

    public AbstractRssReader<C, I> addItemExtension(String tag, String attribute, BiConsumer<I, String> consumer) {
        Objects.requireNonNull(tag, "Item tag must not be null");
        Objects.requireNonNull(attribute, "Item attribute must not be null");
        Objects.requireNonNull(consumer, "Item consumer must not be null");
        this.itemAttributes.computeIfAbsent(tag, k -> new HashMap()).put(attribute, consumer);
        return this;
    }

    public AbstractRssReader<C, I> addChannelExtension(String tag, BiConsumer<C, String> consumer) {
        Objects.requireNonNull(tag, "Channel tag must not be null");
        Objects.requireNonNull(consumer, "Channel consumer must not be null");
        this.channelTags.put(tag, consumer);
        return this;
    }

    public AbstractRssReader<C, I> addChannelExtension(String tag, String attribute, BiConsumer<C, String> consumer) {
        Objects.requireNonNull(tag, "Channel tag must not be null");
        Objects.requireNonNull(attribute, "Channel attribute must not be null");
        Objects.requireNonNull(consumer, "Channel consumer must not be null");
        this.channelAttributes.computeIfAbsent(tag, k -> new HashMap()).put(attribute, consumer);
        return this;
    }

    public Stream<I> read(String url) throws IOException {
        Objects.requireNonNull(url, "URL must not be null");
        try {
            return this.readAsync(url).get(1L, TimeUnit.MINUTES);
        }
        catch (CompletionException e) {
            try {
                throw e.getCause();
            }
            catch (IOException e2) {
                throw e2;
            }
            catch (Throwable e2) {
                throw new AssertionError((Object)e2);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException(e);
        }
        catch (ExecutionException | TimeoutException e) {
            throw new IOException(e);
        }
    }

    public Stream<Item> read(Collection<String> urls) {
        Objects.requireNonNull(urls, "URLs collection must not be null");
        if (!this.isInitialized) {
            this.initialize();
            this.isInitialized = true;
        }
        return ((Stream)urls.stream().parallel()).map(url -> {
            try {
                return Map.entry(url, this.readAsync((String)url));
            }
            catch (Exception e) {
                Logger logger = Logger.getLogger(LOG_GROUP);
                if (logger.isLoggable(Level.WARNING)) {
                    logger.log(Level.WARNING, () -> String.format("Failed read URL %s. Message: %s", url, e.getMessage()));
                }
                return null;
            }
        }).filter(Objects::nonNull).flatMap(f -> {
            try {
                return (Stream)((CompletableFuture)f.getValue()).join();
            }
            catch (Exception e) {
                Logger logger = Logger.getLogger(LOG_GROUP);
                if (logger.isLoggable(Level.WARNING)) {
                    logger.log(Level.WARNING, () -> String.format("Failed to read URL %s. Message: %s", f.getKey(), e.getMessage()));
                }
                return null;
            }
        });
    }

    public Stream<I> read(InputStream inputStream) {
        Objects.requireNonNull(inputStream, "Input stream must not be null");
        if (!this.isInitialized) {
            this.initialize();
            this.isInitialized = true;
        }
        RssItemIterator itemIterator = new RssItemIterator(inputStream);
        return (Stream)StreamSupport.stream(Spliterators.spliteratorUnknownSize(itemIterator, 16), false).onClose(itemIterator::close);
    }

    public CompletableFuture<Stream<I>> readAsync(String url) {
        Objects.requireNonNull(url, "URL must not be null");
        if (!this.isInitialized) {
            this.initialize();
            this.isInitialized = true;
        }
        return this.sendAsyncRequest(url).thenApply(this.processResponse());
    }

    protected CompletableFuture<HttpResponse<InputStream>> sendAsyncRequest(String url) {
        HttpRequest.Builder builder = HttpRequest.newBuilder(URI.create(url)).timeout(Duration.ofSeconds(25L)).header("Accept-Encoding", "gzip");
        if (!this.userAgent.isBlank()) {
            builder.header("User-Agent", this.userAgent);
        }
        this.headers.forEach(builder::header);
        return this.httpClient.sendAsync(builder.GET().build(), HttpResponse.BodyHandlers.ofInputStream());
    }

    private Function<HttpResponse<InputStream>, Stream<I>> processResponse() {
        return response -> {
            try {
                if (response.statusCode() >= 400 && response.statusCode() < 600) {
                    throw new IOException("Response http status code: " + response.statusCode());
                }
                InputStream inputStream = (InputStream)response.body();
                if (Optional.of("gzip").equals(response.headers().firstValue("Content-Encoding"))) {
                    inputStream = new GZIPInputStream(inputStream);
                }
                inputStream = new BufferedInputStream(inputStream);
                this.removeBadData(inputStream);
                RssItemIterator itemIterator = new RssItemIterator(inputStream);
                return (Stream)StreamSupport.stream(Spliterators.spliteratorUnknownSize(itemIterator, 16), false).onClose(itemIterator::close);
            }
            catch (IOException e) {
                throw new CompletionException(e);
            }
        };
    }

    private void removeBadData(InputStream inputStream) throws IOException {
        int secondChar;
        inputStream.mark(2);
        int firstChar = inputStream.read();
        if (firstChar != 65279 && firstChar != 13 && firstChar != 10 && !Character.isWhitespace(firstChar)) {
            inputStream.reset();
        } else if ((firstChar == 13 || Character.isWhitespace(firstChar)) && (secondChar = inputStream.read()) != 10 && !Character.isWhitespace(secondChar)) {
            inputStream.reset();
            inputStream.read();
        }
    }

    private HttpClient createHttpClient() {
        HttpClient client;
        try {
            SSLContext context = SSLContext.getInstance("TLSv1.3");
            context.init(null, null, null);
            client = HttpClient.newBuilder().sslContext(context).connectTimeout(Duration.ofSeconds(25L)).followRedirects(HttpClient.Redirect.ALWAYS).build();
        }
        catch (KeyManagementException | NoSuchAlgorithmException e) {
            client = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(25L)).followRedirects(HttpClient.Redirect.ALWAYS).build();
        }
        return client;
    }

    class RssItemIterator
    implements Iterator<I> {
        private final StringBuilder textBuilder;
        private final InputStream is;
        private XMLStreamReader reader;
        private C channel;
        private Image image;
        private I item;
        private I nextItem;
        private boolean isChannelPart;
        private boolean isImagePart;
        private boolean isItemPart;
        private String elementName;

        public RssItemIterator(InputStream is) {
            block2: {
                this.image = null;
                this.item = null;
                this.isChannelPart = false;
                this.isImagePart = false;
                this.isItemPart = false;
                this.elementName = null;
                this.is = is;
                this.nextItem = null;
                this.textBuilder = new StringBuilder();
                try {
                    XMLInputFactory xmlInFact = XMLInputFactory.newInstance();
                    xmlInFact.setProperty("javax.xml.stream.isSupportingExternalEntities", Boolean.FALSE);
                    xmlInFact.setProperty("javax.xml.stream.supportDTD", Boolean.FALSE);
                    xmlInFact.setProperty("javax.xml.stream.isCoalescing", true);
                    this.reader = xmlInFact.createXMLStreamReader(is);
                }
                catch (XMLStreamException e) {
                    Logger logger = Logger.getLogger(AbstractRssReader.LOG_GROUP);
                    if (!logger.isLoggable(Level.WARNING)) break block2;
                    logger.log(Level.WARNING, "Failed to process XML. ", e);
                }
            }
        }

        public void close() {
            block2: {
                try {
                    this.reader.close();
                    this.is.close();
                }
                catch (IOException | XMLStreamException e) {
                    Logger logger = Logger.getLogger(AbstractRssReader.LOG_GROUP);
                    if (!logger.isLoggable(Level.WARNING)) break block2;
                    logger.log(Level.WARNING, "Failed to close XML stream. ", e);
                }
            }
        }

        void peekNext() {
            if (this.nextItem == null) {
                try {
                    this.nextItem = this.next();
                }
                catch (NoSuchElementException e) {
                    this.nextItem = null;
                }
            }
        }

        @Override
        public boolean hasNext() {
            this.peekNext();
            return this.nextItem != null;
        }

        @Override
        public I next() {
            block6: {
                if (this.nextItem != null) {
                    Object next = this.nextItem;
                    this.nextItem = null;
                    return next;
                }
                try {
                    while (this.reader.hasNext()) {
                        boolean itemParsed;
                        int type = this.reader.next();
                        if (type == 4 || type == 12) {
                            this.parseCharacters();
                            continue;
                        }
                        if (type == 1) {
                            this.parseStartElement();
                            this.parseAttributes();
                            continue;
                        }
                        if (type != 2 || !(itemParsed = this.parseEndElement())) continue;
                        return this.item;
                    }
                }
                catch (XMLStreamException e) {
                    Logger logger = Logger.getLogger(AbstractRssReader.LOG_GROUP);
                    if (!logger.isLoggable(Level.WARNING)) break block6;
                    logger.log(Level.WARNING, "Failed to parse XML. ", e);
                }
            }
            this.close();
            throw new NoSuchElementException();
        }

        void parseStartElement() {
            this.textBuilder.setLength(0);
            this.elementName = this.reader.getLocalName();
            String prefix = this.reader.getPrefix();
            String nsLocalName = this.toNsName(prefix, this.elementName);
            if ("channel".equals(nsLocalName) || "feed".equals(nsLocalName)) {
                this.channel = AbstractRssReader.this.createChannel();
                ((Channel)this.channel).setTitle("");
                ((Channel)this.channel).setDescription("");
                ((Channel)this.channel).setLink("");
                this.isChannelPart = true;
            } else if ("item".equals(nsLocalName) || "entry".equals(nsLocalName)) {
                this.item = AbstractRssReader.this.createItem();
                ((Item)this.item).setChannel((Channel)this.channel);
                this.isChannelPart = false;
                this.isImagePart = false;
                this.isItemPart = true;
            } else if ("enclosure".equals(nsLocalName)) {
                ((Item)this.item).setEnclosure(new Enclosure());
            } else if ("image".equals(nsLocalName)) {
                this.image = new Image();
                ((Channel)this.channel).setImage(this.image);
                this.isImagePart = true;
            }
        }

        void parseAttributes() {
            Map consumers;
            String localName = this.reader.getLocalName();
            String prefix = this.reader.getPrefix();
            String nsLocalName = this.toNsName(prefix, localName);
            if (this.isChannelPart) {
                Map consumers2 = AbstractRssReader.this.channelAttributes.get(nsLocalName);
                if (consumers2 != null && this.channel != null) {
                    consumers2.forEach((attributeName, consumer) -> {
                        Optional<String> attributeValue = Optional.ofNullable(this.reader.getAttributeValue(null, (String)attributeName));
                        attributeValue.ifPresent(v -> consumer.accept(this.channel, v));
                    });
                }
            } else if (this.isItemPart && (consumers = AbstractRssReader.this.itemAttributes.get(nsLocalName)) != null && this.item != null) {
                consumers.forEach((attributeName, consumer) -> {
                    Optional<String> attributeValue = Optional.ofNullable(this.reader.getAttributeValue(null, (String)attributeName));
                    attributeValue.ifPresent(v -> consumer.accept(this.item, v));
                });
            }
        }

        boolean parseEndElement() {
            String localName = this.reader.getLocalName();
            String prefix = this.reader.getPrefix();
            String nsLocalName = this.toNsName(prefix, localName);
            String text = this.textBuilder.toString().trim();
            if ("image".equals(nsLocalName)) {
                this.isImagePart = false;
            } else if (this.isImagePart) {
                this.parseImageCharacters(this.image, this.elementName, text);
            } else if (this.isChannelPart) {
                this.parseChannelCharacters(this.channel, prefix, this.elementName, text);
            } else {
                this.parseItemCharacters(this.item, prefix, this.elementName, text);
            }
            this.textBuilder.setLength(0);
            this.elementName = "";
            return "item".equals(nsLocalName) || "entry".equals(nsLocalName);
        }

        void parseCharacters() {
            String text = this.reader.getText();
            if (text.isBlank()) {
                return;
            }
            this.textBuilder.append(text);
        }

        void parseChannelCharacters(C channel, String prefix, String elementName, String text) {
            if (channel == null || text.isEmpty()) {
                return;
            }
            String nsElementName = this.toNsName(prefix, elementName);
            BiConsumer consumer = AbstractRssReader.this.channelTags.get(nsElementName);
            if (consumer != null) {
                consumer.accept(channel, text);
            }
        }

        void parseImageCharacters(Image image, String elementName, String text) {
            if (image == null || text.isEmpty()) {
                return;
            }
            BiConsumer<Image, String> consumer = AbstractRssReader.this.imageTags.get(elementName);
            if (consumer != null) {
                consumer.accept(image, text);
            }
        }

        void parseItemCharacters(I item, String prefix, String elementName, String text) {
            if (item == null || text.isEmpty()) {
                return;
            }
            String nsElementName = this.toNsName(prefix, elementName);
            BiConsumer consumer = AbstractRssReader.this.itemTags.get(nsElementName);
            if (consumer != null) {
                consumer.accept(item, text);
            }
        }

        String toNsName(String prefix, String name) {
            return prefix.isEmpty() ? name : prefix + ":" + name;
        }
    }
}

